/* $Id: mat_badpixel-test.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_badpixel-test.c $
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <string.h>

#include <cpl.h>
#include <cpl_test.h>
#include "../mat_gendata.h"
#include "../mat_frame.h"
#include "../mat_badpixel.h"
#include "../mat_imagingdetector.h"

int main(int argc, char ** argv) {
    /* const char *fctid = "mat_badpixel-test"; */
    /* cpl_errorstate prestate; */

    /* create a dummy mat_detector for a HAWAII-2RG */
    mat_detector  *det = NULL;
    /* create a dummy mat_imagingdetector for a full-frame detector bad pixel map */
    mat_imagingdetector *imgdet = NULL;

    /* create a dummy mat_frame containing an artificial median map */
    cpl_image *medianimg = NULL;
    cpl_image *milist[1];
    mat_imgreg medianreg = {
	1, 1, 1, milist
    };
    mat_imgreg *mrlist[1];
    mat_frame median = {
	0.0, 0.6, {0.0, 0.0, 0.0, 0.0}, {0.0, 0.0, 0.0, 0.0}, 0, {'C', 'S'}, 1, mrlist
    };

    /* create a dummy mat_frame containing an artificial variance map */
    cpl_image *varianceimg = NULL;
    cpl_image *vilist[1];
    mat_imgreg variancereg = {
	1, 1, 1, vilist
    };
    mat_imgreg *vrlist[1];
    mat_frame variance = {
	0.0, 0.6, {0.0, 0.0, 0.0, 0.0}, {0.0, 0.0, 0.0, 0.0}, 0, {'C', 'S'}, 1, vrlist
    };

    mat_badpixel *bpm = NULL;
    cpl_frameset *fs = NULL;
    (void) argc;
    (void) argv;

    cpl_test_init("mhein@mpifr-bonn.mpg.de",CPL_MSG_WARNING);
    /* prestate = cpl_errorstate_get(); */

    /* create and fill the mat_detector data structure for a HAWAII-2RG */
    det = mat_detector_new();
    cpl_test_nonnull(det);
    if (det == NULL)
    {
	cpl_test(cpl_memory_is_empty());
	return cpl_test_end(0);
    }
    det->type = MAT_HAWAII2RG_DET;             /* detector type as an enumeration (MAT_*_DET) */
    det->nr = 1;               /* the detector number HAWAII2RG: 1, Aquarius: 2, other:1 */
    strncpy(det->id, "4711", 80);           /* DET CHIP ID */
    strncpy(det->name, "MATISSE-L/M-Band", 80);         /* DET CHIP NAME */
    det->nx = 2048;               /* total detector width in pixel */
    det->ny = 2048;               /* total detector height in pixel */
    det->read_id = 1;          /* DET READ CURID */
    strncpy(det->read_name, "CDS slow", 80);    /* DET READ CURNAME */
    det->channel_nrows = 1;    /* number of channel rows (HAWAII-2RG: 1, Aquarius: 2, other: 1) */
    det->channel_ncolumns = 32; /* number of channel columns (HAWAII-2RG: 32, Aquarius: 32, other: 1) */
    det->channel_nx = 64;       /* width of a detector channel (calculated) */
    det->channel_ny = 2048;       /* height of a detector channel (calculated) */
    det->frame_left = 128;       /* left frame width (HAWAII-2RG: 128, Aquarius: 32, other: 0) */
    det->frame_right = 32;      /* right frame width (HAWAII-2RG: 128, Aquarius: 32, other: 0) */
    det->frame_bottom = 32;     /* bottom frame height (HAWAII-2RG: 32, Aquarius: 32, other: 0) */
    det->frame_top = 128;        /* top frame height (HAWAII-2RG: 32, Aquarius: 32, other: 0) */
    det->data_nregions = 1;    /* number of data sub-windows */
    det->data_nx = 2048;          /* width of the one data sub-window */
    det->data_ny = 2048;          /* height of the one data sub-window */
    /* default pixel timing for a HAWAII-2RG detector (100 kHz pixel clock) */
    det->horizontal_timing[0] = 640.0; /* 640 us for all 64 pixels of a detector channel */
    det->horizontal_timing[1] = 0.0;  /* no dependency on the region width */
    det->horizontal_timing[2] = 0.0;  /* no dependency on the region height */
    det->horizontal_timing[3] = 0.0;  /* no dependency on the region size */
    det->vertical_timing[0] = 0.0;    /* no constant part */
    det->vertical_timing[1] = 0.0;    /* no dependency on the region width */
    det->vertical_timing[2] = 640.0;  /* 640 us for all 64 pixels of a detector channel */
    det->vertical_timing[2] = 0.0;    /* no dependency on the region size */

    /* create and fill the mat_imagingdetector data structure for a full frame sub-window setup */
    imgdet = mat_imagingdetector_new(NULL, 1);
    cpl_test_nonnull(imgdet);
    if (imgdet == NULL)
    {
	mat_detector_delete(det);
	cpl_test(cpl_memory_is_empty());
	return cpl_test_end(0);
    }
    //strncpy(imgdet->origin, "calculated", SIZE_MAX_KEYWORDS);                        /* origin of the data          */
    strncpy(imgdet->instrument, "MATISSE", SIZE_MAX_KEYWORDS);                    /* instrument name             */
    imgdet->dateobsmjd = 56304.6773611;                    /* observation date (MJD)      */
    strncpy(imgdet->dateobs, "20130114T080741", SIZE_MAX_KEYWORDS);                       /* observation date            */
    //strncpy(imgdet->date, "20130114T080741", SIZE_MAX_KEYWORDS);                          /* HDU creation date           */
    strncpy(imgdet->dcsdictionaryid, "MATISSE", SIZE_MAX_KEYWORDS);               /* data dic name               */
    strncpy(imgdet->dcsid, "0.0", SIZE_MAX_KEYWORDS);                         /* DCS software version string */
    imgdet->list_region[0]->numregion = 1;                      /* region number               */
    strncpy(imgdet->list_region[0]->regionname, "FF", 3);                     /* origin of the data          */
    imgdet->list_region[0]->numdetector = 1;                    /* detector number             */
    imgdet->list_region[0]->numbeam[0] = 1;                        /* beam numbers                */
    imgdet->list_region[0]->numbeam[1] = 0;                        /* beam numbers                */
    imgdet->list_region[0]->numbeam[2] = 0;                        /* beam numbers                */
    imgdet->list_region[0]->numbeam[3] = 0;                        /* beam numbers                */
    imgdet->list_region[0]->correlation = 0;                    /* correlation type            */
    imgdet->list_region[0]->corner[0] = 1;                      /* corner for this region      */
    imgdet->list_region[0]->corner[1] = 1;                      /* corner for this region      */
    imgdet->list_region[0]->gain = 1.0;                           /* conversion factor           */
    strncpy(imgdet->list_region[0]->ctype[0], "PIXEL", 8);                       /* types of coordinates        */
    strncpy(imgdet->list_region[0]->ctype[1], "PIXEL", 8);                       /* types of coordinates        */
    imgdet->list_region[0]->naxis[0] = 2048;                       /* dimension of each axis      */
    imgdet->list_region[0]->naxis[1] = 2048;                       /* dimension of each axis      */


    medianimg = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
    milist[0] = medianimg;
    mrlist[0] = &medianreg;
    varianceimg = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
    vilist[0] = varianceimg;
    vrlist[0] = &variancereg;

    bpm = mat_badpixel_new(det, imgdet);
    cpl_test_nonnull(bpm);
    cpl_test_nonnull(medianimg);
    cpl_test_nonnull(varianceimg);
    if ((bpm != NULL) && (medianimg != NULL) && (varianceimg != NULL))
    {
	int x, y, c, r;
	srand(42L);
	/* fill the median image with values evenly distributed between 1000 and 1511 */
	for (x = 1; x <= 2048; x++)
	{
	    for (y = 1; y <= 2048; y++)
	    {
		cpl_image_set(medianimg, x, y, 1000.0 + (double)(rand()%512));
	    }
	}
	/* set the intensity for two pixels to a value outside the valid range */
	cpl_image_set(medianimg, 47, 11, 8765.0);
	cpl_image_set(medianimg, 11, 47, 31.0);
	/* fill the variance image with values evenly distributed between 8000 and 9023 */
	for (x = 1; x <= 2048; x++)
	{
	    for (y = 1; y <= 2048; y++)
	    {
		cpl_image_set(varianceimg, x, y, 8000.0 + (double)(rand()%1024));
	    }
	}
	/* set the intensity for two pixels to a value outside the valid range */
	cpl_image_set(varianceimg, 247, 321, 28765.0);
	cpl_image_set(varianceimg, 321, 247, 765.0);
	/* iterate over all detector channels and detect the four bad pixels */
	for (c = 0; c < 32; c++)
	{
	    cpl_test(mat_badpixel_detect_global_window(&median, 5.0, bpm,
						       0,
						       1 + c*det->channel_nx, 1,
						       det->channel_nx, det->channel_ny) == CPL_ERROR_NONE);
	}
	for (c = 0; c < 32; c++)
	{
	    cpl_test(mat_badpixel_detect_local_window(&variance, 5.0, bpm,
						      0,
						      1 + c*det->channel_nx, 1,
						      det->channel_nx, det->channel_ny) == CPL_ERROR_NONE);
	}
	/* and now check the results */
	bpm->nbbad = 0;
	for (r = 0; r < bpm->imgdet->nbregion; r++)
	{
	    bpm->nbbad += cpl_mask_count(bpm->list_badpixel[r]);
	}
	bpm->nbgood = bpm->nbtotal - bpm->nbbad;

	cpl_msg_info(cpl_func, "bpm->nbbad = %d", bpm->nbbad);
	cpl_test(bpm->nbtotal == (2048*2048));
	cpl_test(bpm->nbgood == (bpm->nbtotal - 4)); /* four bad pixels! */
	cpl_test(bpm->nbbad == 4);
	cpl_test(mat_badpixel_get(bpm, 0, 47, 11) == CPL_BINARY_1);
	cpl_test(mat_badpixel_get(bpm, 0, 11, 47) == CPL_BINARY_1);
	cpl_test(mat_badpixel_get(bpm, 0, 247, 321) == CPL_BINARY_1);
	cpl_test(mat_badpixel_get(bpm, 0, 321, 247) == CPL_BINARY_1);
    }
    /* try to save the bad pixel map in a file */
    fs = cpl_frameset_new();
    if (fs != NULL)
    {
	cpl_test(mat_badpixel_save(bpm, "bpm.fits", NULL, NULL, fs) == CPL_ERROR_NONE);
	cpl_frameset_delete(fs);
    }

    cpl_test(mat_badpixel_delete(bpm) == CPL_ERROR_NONE);
    bpm = NULL;
    cpl_image_delete(medianimg);
    milist[0] = NULL;
    cpl_image_delete(varianceimg);
    vilist[0] = NULL;
    mat_detector_delete(det);
    det = NULL;
    mat_imagingdetector_delete(imgdet);
    imgdet = NULL;
    // Check for memory leak
    cpl_test(cpl_memory_is_empty());
    return cpl_test_end(0);
}
