/* $Id: mat_obsflat-test.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_obsflat-test.c $
 */

#include <cpl.h>
#include <cpl_test.h>
#include <sys/stat.h>

#include "../mat_drl.h"
#include "../mat_obsflat.h"
#include "../mat_imagingdetector.h"
#include "../mat_detector.h"
#include <stdio.h>
#define OBSFLAT_TEST_FILENAME "obsflattest.fits"


static cpl_error_code mat_obsflat_new_test(void);
static cpl_error_code mat_obsflat_delete_test(void);
static cpl_error_code mat_obsflat_save_test(void);
static cpl_error_code mat_obsflat_load_test(void);

//######################-Implements-############################################

/**
   @brief Creates a mat_imagingdetector structure with a test pattern
   @param nbregions the number of regions
   @return pointer to new allocated mat_imagingdetector structure

   The method creates a mat_imagingdetector structure with nbregions regions. 
   The test pattern is, that the x-dimesion goes from 1 to nbregions and the 
   y-dimension from nbregions to 1
*/
static mat_imagingdetector* create_test_mat_imagingdetector(int nbregions){
    int i = 0;
    mat_imagingdetector* imgdet = NULL;

    imgdet = mat_imagingdetector_new(NULL, nbregions);
    if(imgdet == NULL){
	return NULL;
    }
    //set the dimensions for the regions
    for(i = 0; i < nbregions; ++i){
	imgdet->list_region[i]->naxis[0] = i + 1;
	imgdet->list_region[i]->naxis[1] = nbregions - i;
    }

    return imgdet;
}

/**
   @brief Fills the whole given cpl_image with a value
   @param image pointer to a cpl_image
   @param value the value for each pixel in the image

   The method sets every pixel in the image to the given value
*/
static void fill_image(cpl_image* image, double value){
    int i = 0, j = 0;

    for(i = 0; i < cpl_image_get_size_y(image); ++i){
	for(j = 0; j < cpl_image_get_size_x(image); ++j){
	    cpl_image_set(image, j + 1,i + 1, value);
	}
    }
}

/**
   @brief Creates the Aquarius detector layout structure for the tests
   @return pointer to new allocated mat_detector structure

   A mat_detector structure is allocated and the values are set to 
   the Aquarius layout.
*/
static mat_detector* create_test_mat_detector(void){
    mat_detector* det = NULL;

    det = mat_detector_new();
    det->nr = 2;
    snprintf(det->name, 80, "AQ");
    det->nx = 1024;
    det->ny = 1024;
    det->channel_nrows = 2;
    det->channel_ncolumns = 32;
    det->frame_left = 32;
    det->frame_right = 32;
    det->frame_bottom = 32;
    det->frame_top = 32;
    return det;
}


//###############################-Test-Methods-#################################
/**
   @brief Test the creation of a new mat_obsflat structure.
   @return CPL_ERROR_NONE on success

   Creates an empty obsflat map. Checks the correct creation of the images
   with the
   dimensions provided by the mat_detector and mat_imagingdetector structures.
*/
cpl_error_code mat_obsflat_new_test(void){
    mat_obsflat* obsflat = NULL;
    mat_detector* det = NULL;
    mat_imagingdetector* imgdet = NULL;
    int i = 0;
    int image_width = 0, image_height = 0;
    static const int nbregions = 42;
    //Aquarius Detector testvalues
    static const int channel_nrows = 2;
    static const int channel_ncolumns = 32;

    //create a normal test with all input data allocated and a testset with
    //normal values
    det = mat_detector_new();
    imgdet =create_test_mat_imagingdetector(nbregions);

    //set the detector values
    det->channel_ncolumns = channel_ncolumns;
    det->channel_nrows = channel_nrows;

    obsflat =  mat_obsflat_new(det,imgdet);
    cpl_test_nonnull(obsflat);

    //check the values
    cpl_test_nonnull(obsflat->det);
    cpl_test_nonnull(obsflat->imgdet);
    cpl_test_eq(obsflat->imgdet->nbregion, nbregions);

    cpl_test_nonnull(obsflat->channelgain);
    cpl_test_eq(cpl_vector_get_size(obsflat->channelgain), 
		channel_nrows*channel_ncolumns);

    cpl_test_nonnull(obsflat->channeloffset);
    cpl_test_eq(cpl_vector_get_size(obsflat->channeloffset), 
		channel_nrows*channel_ncolumns);

    cpl_test_nonnull(obsflat->channelnoise);
    cpl_test_eq(cpl_vector_get_size(obsflat->channelnoise), 
		channel_nrows*channel_ncolumns);

    //check the images
    cpl_test_nonnull(obsflat->list_obsflat);
    cpl_test_nonnull(obsflat->list_bias);

    for(i = 0; i < obsflat->imgdet->nbregion; ++i){
	cpl_test_nonnull(obsflat->list_obsflat[i]);
	image_width = cpl_image_get_size_x(obsflat->list_obsflat[i]);
	image_height = cpl_image_get_size_y(obsflat->list_obsflat[i]);

	cpl_test_eq(image_width, imgdet->list_region[i]->naxis[0]);
	cpl_test_eq(image_height, imgdet->list_region[i]->naxis[1]);

	cpl_test_nonnull(obsflat->list_bias[i]);
	image_width = cpl_image_get_size_x(obsflat->list_bias[i]);
	image_height = cpl_image_get_size_y(obsflat->list_bias[i]);

	cpl_test_eq(image_width, imgdet->list_region[i]->naxis[0]);
	cpl_test_eq(image_height, imgdet->list_region[i]->naxis[1]);
    }

    //free all memory used in this test
    mat_detector_delete(det);
    mat_imagingdetector_delete(imgdet);

    mat_obsflat_delete(obsflat);

    // Check for memory leak
    cpl_test(cpl_memory_is_empty());

    return CPL_ERROR_NONE;
}

/**
   @brief Test the deletion of a mat_obsflat structure.
   @return CPL_ERROR_NONE on success

   Creates an empty obsflat map and then deletes it. Check if all allocated 
   memory is cleared.
*/
cpl_error_code mat_obsflat_delete_test(void){
    mat_obsflat* obsflat = NULL;
    mat_imagingdetector* imgdet = NULL;
    cpl_error_code returncode = CPL_ERROR_NONE;
    static const int vectorsize = 32;
    static const int nbregions = 42;
    cpl_image* temp_image = NULL;
    int i = 0;

    //fill the obflat structure with default values
    obsflat = cpl_calloc(1, sizeof(mat_obsflat));
    obsflat->channelgain = cpl_vector_new(vectorsize);
    obsflat->channelnoise = cpl_vector_new(vectorsize);
    obsflat->channeloffset = cpl_vector_new(vectorsize);

    //create default image lists
    obsflat->list_bias = cpl_calloc(nbregions, sizeof(cpl_image*));
    obsflat->list_obsflat = cpl_calloc(nbregions, sizeof(cpl_image*));

    for(i = 0; i < nbregions; ++i){
	temp_image = cpl_image_new(i+1, nbregions - i, CPL_TYPE_DOUBLE);
	if(temp_image == NULL){
	    //ERROR
	    mat_obsflat_delete(obsflat);
	    return CPL_ERROR_UNSPECIFIED;
	}
	obsflat->list_bias[i] = temp_image;
	temp_image = cpl_image_new(i+1, nbregions - i, CPL_TYPE_DOUBLE);
	if(temp_image == NULL){
	    //ERROR
	    mat_obsflat_delete(obsflat);
	    return CPL_ERROR_UNSPECIFIED;
	}
	obsflat->list_obsflat[i] = temp_image;
    }

    imgdet =create_test_mat_imagingdetector(nbregions);
    obsflat->imgdet = imgdet;

    returncode = mat_obsflat_delete(obsflat);
    cpl_test(returncode == CPL_ERROR_NONE);

    // Check for memory leak
    cpl_test(cpl_memory_is_empty());
    return CPL_ERROR_NONE;

}

/**
   @brief Test the save method of a mat_obsflat structure.
   @return CPL_ERROR_NONE on success

   Creates an empty obsflat map and then fills it with a specific pattern.
   This pattern consists of the numbers 0 to nbregions-1 for the bias
   and nbregions to 2*nbregions -1 for the flatfied values.
*/
cpl_error_code mat_obsflat_save_test(void){
    mat_obsflat* obsflat = NULL;
    mat_imagingdetector* imgdet = NULL;
    cpl_error_code returncode = CPL_ERROR_NONE;
    static const int nbregions = 42;
    cpl_image* temp_image = NULL;
    cpl_frameset* newframeset = 0;
    cpl_parameterlist* parlist = 0;
    mat_detector* det = NULL;
    int i = 0;
    int vectorsize = 0;
    const char* filename = OBSFLAT_TEST_FILENAME;

    obsflat = cpl_calloc(1, sizeof(mat_obsflat));

    //create testdetector
    det = create_test_mat_detector();
    obsflat->det = det;
    //fill the obflat structure with default values
    vectorsize = det->channel_ncolumns*det->channel_nrows;
    obsflat->channelgain = cpl_vector_new(vectorsize);
    obsflat->channelnoise = cpl_vector_new(vectorsize);
    obsflat->channeloffset = cpl_vector_new(vectorsize);
    for(i = 0; i < vectorsize; ++i){
	cpl_vector_set(obsflat->channelgain, i, (double)i);
	cpl_vector_set(obsflat->channelnoise, i, (double)(vectorsize + i));
	cpl_vector_set(obsflat->channeloffset, i, (double)(2*vectorsize + i));
    }

    //create default image lists
    obsflat->list_bias = cpl_calloc(nbregions, sizeof(cpl_image*));
    obsflat->list_obsflat = cpl_calloc(nbregions, sizeof(cpl_image*));



    imgdet =create_test_mat_imagingdetector(nbregions);
    obsflat->imgdet = imgdet;

    for(i = 0; i < nbregions; ++i){
	temp_image = cpl_image_new(imgdet->list_region[i]->naxis[0], 
				   imgdet->list_region[i]->naxis[1], 
				   CPL_TYPE_DOUBLE);
	if(temp_image == NULL){
	    //ERROR
	    mat_obsflat_delete(obsflat);
	    return CPL_ERROR_UNSPECIFIED;
	}
	fill_image(temp_image, (double) i);
	obsflat->list_bias[i] = temp_image;
	temp_image = cpl_image_new(imgdet->list_region[i]->naxis[0], 
				   imgdet->list_region[i]->naxis[1], 
				   CPL_TYPE_DOUBLE);
	if(temp_image == NULL){
	    mat_obsflat_delete(obsflat);
	    return CPL_ERROR_UNSPECIFIED;
	}
	fill_image(temp_image, (double) (nbregions + i));
	obsflat->list_obsflat[i] = temp_image;
    }

    newframeset = cpl_frameset_new();
    parlist = cpl_parameterlist_new();
    returncode = mat_obsflat_save(obsflat, filename, NULL, 
				       parlist, newframeset);

    cpl_test(returncode == CPL_ERROR_NONE);

    //clean
    cpl_frameset_delete(newframeset);
    cpl_parameterlist_delete(parlist);
    mat_obsflat_delete(obsflat);

    // Check for memory leak
    cpl_test(cpl_memory_is_empty());

    return CPL_ERROR_NONE;

}
/**
   @brief Test the load method of a mat_obsflat structure.
   @return CPL_ERROR_NONE on success

   The FITS File with the pattern following pattern is loaded into a 
   mat_obsflat structure.
   This pattern consists of the numbers 0 to nbregions-1 for the bias
   and nbregions to 2*nbregions -1 for the flatfied values.
   The dimensions of the images must match to the Aquarius layout.
*/
cpl_error_code mat_obsflat_load_test(void){
    const char* filename = OBSFLAT_TEST_FILENAME;
    mat_obsflat* obsflat = NULL;
    cpl_frame* obsframe = NULL;
    mat_imagingdetector* p_imgdet = NULL;
    int i = 0;
    int image_width = 0;
    int image_height = 0;
    double absflux_value = 0;


    obsframe = cpl_frame_new();
    cpl_frame_set_filename(obsframe, filename);

    obsflat = mat_obsflat_load(obsframe);

    //test allocation
    cpl_test_nonnull(obsflat);
    if(obsflat == NULL){
	cpl_frame_delete(obsframe);
	return CPL_ERROR_UNSPECIFIED;
    }

    cpl_test_nonnull(obsflat->imgdet);
    if(obsflat->imgdet == NULL){
	mat_obsflat_delete(obsflat);
	cpl_frame_delete(obsframe);
	return CPL_ERROR_UNSPECIFIED;
    }

    cpl_test_nonnull(obsflat->keywords);
    if(obsflat->keywords == NULL){
	mat_obsflat_delete(obsflat);
	cpl_frame_delete(obsframe);
	return CPL_ERROR_UNSPECIFIED;
    }

    cpl_test_nonnull(obsflat->det);
    if(obsflat->det == NULL){
	mat_obsflat_delete(obsflat);
	cpl_frame_delete(obsframe);
	return CPL_ERROR_UNSPECIFIED;
    }

    cpl_test_nonnull(obsflat->list_obsflat);
    if(obsflat->list_obsflat == NULL){
	mat_obsflat_delete(obsflat);
	cpl_frame_delete(obsframe);
	return CPL_ERROR_UNSPECIFIED;
    }

    cpl_test_nonnull(obsflat->list_bias);
    if(obsflat->list_bias == NULL){
	mat_obsflat_delete(obsflat);
	cpl_frame_delete(obsframe);
	return CPL_ERROR_UNSPECIFIED;
    }

    cpl_error_reset();
    //test the values
    p_imgdet = obsflat->imgdet;
    for(i = 0; i < p_imgdet->nbregion; ++i){
	cpl_test_nonnull(obsflat->list_bias[i]);
	if(obsflat->list_bias[i] == NULL){
	    mat_obsflat_delete(obsflat);
	    cpl_frame_delete(obsframe);
	    return CPL_ERROR_UNSPECIFIED;
	}

	cpl_test_nonnull(obsflat->list_obsflat[i]);
	if(obsflat->list_obsflat[i] == NULL){
	    mat_obsflat_delete(obsflat);
	    cpl_frame_delete(obsframe);
	    return CPL_ERROR_UNSPECIFIED;
	}

	cpl_test_nonnull(p_imgdet->list_region[i]);
	if(p_imgdet->list_region[i] == NULL){
	    mat_obsflat_delete(obsflat);
	    cpl_frame_delete(obsframe);
	    return CPL_ERROR_UNSPECIFIED;
	}
	image_width = cpl_image_get_size_x(obsflat->list_bias[i]);
	image_height = cpl_image_get_size_y(obsflat->list_bias[i]);

	cpl_test_eq(image_width, p_imgdet->list_region[i]->naxis[0]);
	cpl_test_eq(image_height, p_imgdet->list_region[i]->naxis[1]);

	image_width = cpl_image_get_size_x(obsflat->list_obsflat[i]);
	image_height = cpl_image_get_size_y(obsflat->list_obsflat[i]);

	cpl_test_eq(image_width, p_imgdet->list_region[i]->naxis[0]);
	cpl_test_eq(image_height, p_imgdet->list_region[i]->naxis[1]);

	absflux_value = cpl_image_get_absflux(obsflat->list_obsflat[i]);
	cpl_test_eq(absflux_value, 
		    image_width*image_height*(p_imgdet->nbregion + i));
	absflux_value = cpl_image_get_absflux(obsflat->list_bias[i]);
	cpl_test_eq(absflux_value, image_width*image_height*(i));
    }

    //clean
    mat_obsflat_delete(obsflat);
    cpl_frame_delete(obsframe);

    // Check for memory leak
    cpl_test(cpl_memory_is_empty());

    return CPL_ERROR_NONE;

}


int main(int argc, char ** argv){
    (void) argc;
    (void) argv;
    cpl_test_init("tkroener@mpifr.de",CPL_MSG_WARNING);
    mat_obsflat_new_test();
    mat_obsflat_delete_test();
    mat_obsflat_save_test();
    mat_obsflat_load_test();
    return cpl_test_end(0);

}


























