/* $Id: mat_oifits-test.c,v0.7 2015-11-16 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
/*
 * $Author: mheininger $
 * $Date: 2016/11/16 16:52:00 $
 * $Revision: 0.7 $
 * $Name: mat_oifits-test.c $
 */

#include <string.h>
#include <stdio.h>

#include <cpl.h>
#include <cpl_test.h>
#include <sys/stat.h>

#include "../mat_oifits.h"

static mat_oitarget *mat_oitarget_create_test(void)
{
  mat_oitarget  *tar = NULL;

  tar = mat_oitarget_new();
  cpl_test_nonnull(tar);
  if (tar == NULL) return NULL;
  tar->targetid = 1;
  tar->targetname = cpl_strdup("HD87643");
  tar->equinox = 2000.0;
  tar->raep0 = 151.12615;
  tar->decep0 = -58.66447;
  tar->raerr = 0.0;
  tar->decerr = 0.0;
  tar->sysvel = 0.0;
  tar->veltyp = cpl_strdup("UNKNOWN");
  tar->veldef = cpl_strdup("OPTICAL");
  tar->pmra = 0.0;
  tar->pmdec = 0.0;
  tar->pmraerr = 0.0;
  tar->pmdecerr = 0.0;
  tar->parallax = 0.0;
  tar->paraerr = 0.0;
  tar->spectyp = cpl_strdup("UNKNOWN");
  tar->category = cpl_strdup("CAL");
  return tar;
}

/**
   @brief Creates a new ARRAY binary table with a well defined content.
   @returns The newly created mat_array data structure or NULL.

   This static function creates a simple ARRAY data structure which
   is later used to check the ARRAY related functions of the mat_oifits
   data structure.
 */
static mat_array *mat_array_create_test_amber(void)
{
  mat_array   *arr = NULL;
  mat_station *sta = NULL;

  arr = mat_array_new(3); // create an array with 3 stations (ATs)
  cpl_test_nonnull(arr);
  if (arr == NULL) return NULL;
  cpl_test_nonnull(arr->list_station);
  if (arr->list_station == NULL)
    {
      mat_array_delete(arr);
      return NULL;
    }
  arr->telescope = cpl_strdup("VLTI");
  arr->dcsdictionaryid = cpl_strdup("AMBER");
  arr->dcsid = cpl_strdup("DCS-1.1");
  arr->arrayname = cpl_strdup("VLTI-AMBER");
  arr->cooframe = cpl_strdup("GEOCENTRIC");
  arr->arrayx =  1951952.0;
  arr->arrayy = -5483173.0;
  arr->arrayz = -2668147.0;
  sta = arr->list_station[0];
  strncpy(sta->telname, "AT3", sizeof(sta->telname));
  strncpy(sta->staname, "K0", sizeof(sta->staname));
  sta->staindex = 28;
  sta->diameter = 1.8;
  sta->staxyz[0] = -106.409452128499;
  sta->staxyz[1] = 14.1889084662963;
  sta->staxyz[2] = 0.006309541;
  sta = arr->list_station[1];
  strncpy(sta->telname, "AT4", sizeof(sta->telname));
  strncpy(sta->staname, "G1", sizeof(sta->staname));
  sta->staindex = 18;
  sta->diameter = 1.8;
  sta->staxyz[0] = -66.715727329499;
  sta->staxyz[1] = 95.5014776162963;
  sta->staxyz[2] = 0;
  sta = arr->list_station[2];
  strncpy(sta->telname, "AT2", sizeof(sta->telname));
  strncpy(sta->staname, "A0", sizeof(sta->staname));
  sta->staindex = 1;
  sta->diameter = 1.8;
  sta->staxyz[0] = 14.619949864501;
  sta->staxyz[1] = 55.7985168502963;
  sta->staxyz[2] = -0.013313335;
  return arr;
}

static mat_oiwavelength *mat_oiwavelength_create_test_amber(void)
{
  mat_oiwavelength *wave = NULL;

  wave = mat_oiwavelength_new(3);
  cpl_test_nonnull(wave);
  if (wave == NULL) return NULL;
  cpl_test_nonnull(wave->effwave);
  cpl_test_nonnull(wave->effband);
  if ((wave->effwave == NULL) || (wave->effband == NULL))
    {
      mat_oiwavelength_delete(wave);
      return NULL;
    }
  wave->insname = cpl_strdup("AMBER");
  wave->effwave[0] = 2.214271E-06;
  wave->effband[0] = 3.337536E-08;
  wave->effwave[1] = 2.180809E-06;
  wave->effband[1] = 3.341748E-08;
  wave->effwave[2] = 2.147435E-06;
  wave->effband[2] = 3.343054E-08;
  return wave;
}

static mat_oivis *mat_oivis_create_test_amber(void)
{
  mat_oivis    *vis = NULL;
  mat_viselem  *el  = NULL;

  vis = mat_oivis_new(0, 3);
  cpl_test_nonnull(vis);
  cpl_test_null(vis->list_vis);
  /* if (vis == NULL) return NULL; */
  vis->dateobs = cpl_strdup("2008-03-01T02:01:57.1002");
  vis->arrayname = cpl_strdup("VLTI-AMBER");
  vis->insname = cpl_strdup("AMBER");

  // the 1. visibility element (for all three wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 1);
  cpl_test_eq_ptr(vis->list_vis[0], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->ucoord = -17.0703104905503;
  el->vcoord = -80.5530052239785;
  el->stationindex[0] = 28;
  el->stationindex[1] = 18;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -0.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  el->visamp[1] = 0.0;
  el->visamperr[1] = 0.0;
  el->visphi[1] = -8.6666627459412;
  el->visphierr[1] = 18.6040440191475;
  el->bandflag[1] = 0;

  el->visamp[2] = 0.0;
  el->visamperr[2] = 0.0;
  el->visphi[2] = -2.30538844993919;
  el->visphierr[2] = 15.5272866144247;
  el->bandflag[2] = 0;

  // the 2. visibility element (for all three wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 2);
  cpl_test_eq_ptr(vis->list_vis[1], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->ucoord = -78.4898005390535;
  el->vcoord = -4.21202660701083;
  el->stationindex[0] = 18;
  el->stationindex[1] = 1;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -26.6975220964386;
  el->visphierr[0] = 6.30290642100673;
  el->bandflag[0] = 0;

  el->visamp[1] = 0.0;
  el->visamperr[1] = 0.0;
  el->visphi[1] = -23.3315156400845;
  el->visphierr[1] = 2.12095173490497;
  el->bandflag[1] = 0;

  el->visamp[2] = 0.0;
  el->visamperr[2] = 0.0;
  el->visphi[2] = -33.9451655295921;
  el->visphierr[2] = 5.93250794413644;
  el->bandflag[2] = 0;

  // the 3. visibility element (for all three wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 3);
  cpl_test_eq_ptr(vis->list_vis[2], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->ucoord = -95.5601110364095;
  el->vcoord = -84.7650317906203;
  el->stationindex[0] = 1;
  el->stationindex[1] = 28;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -156.139049651167;
  el->visphierr[0] = 30.8682778921438;
  el->bandflag[0] = 0;

  el->visamp[1] = 0.0;
  el->visamperr[1] = 0.0;
  el->visphi[1] = -104.829831837738;
  el->visphierr[1] = 39.7838730797034;
  el->bandflag[1] = 0;

  el->visamp[2] = 0.0;
  el->visamperr[2] = 0.0;
  el->visphi[2] = -157.650217318357;
  el->visphierr[2] = 39.9370744063605;
  el->bandflag[2] = 0;

  // returns the just ceated and filled OI_VIS binary table
  return vis;
}

static mat_oivis2 *mat_oivis2_create_test_amber(void)
{
  mat_oivis2    *vis2 = NULL;
  mat_vis2elem  *el   = NULL;

  vis2 = mat_oivis2_new(0, 3);
  cpl_test_nonnull(vis2);
  cpl_test_null(vis2->list_vis2);
  /* if (vis2 == NULL) return NULL; */
  vis2->dateobs = cpl_strdup("2008-03-01T02:01:57.1002");
  vis2->arrayname = cpl_strdup("VLTI-AMBER");
  vis2->insname = cpl_strdup("AMBER");

  // the 1. visibility element (for all three wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 1);
  cpl_test_eq_ptr(vis2->list_vis2[0], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->ucoord = -17.0703104905503;
  el->vcoord = -80.5530052239785;
  el->stationindex[0] = 28;
  el->stationindex[1] = 18;

  el->vis2[0] = 0.0659429108568084;
  el->vis2err[0] = 0.00856092089959347;
  el->bandflag[0] = 0;

  el->vis2[1] = 0.0866980170434447;
  el->vis2err[1] = 0.010932154024135;
  el->bandflag[1] = 0;

  el->vis2[2] = 0.0961729887197798;
  el->vis2err[2] = 0.0118491129695247;
  el->bandflag[2] = 0;

  // the 2. visibility element (for all three wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 2);
  cpl_test_eq_ptr(vis2->list_vis2[1], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->ucoord = -78.4898005390535;
  el->vcoord = -4.21202660701083;
  el->stationindex[0] = 18;
  el->stationindex[1] = 1;

  el->vis2[0] = 0.12068857130843;
  el->vis2err[0] = 0.014743340023794;
  el->bandflag[0] = 0;

  el->vis2[1] = 0.121565991985528;
  el->vis2err[1] = 0.0151982459773075;
  el->bandflag[1] = 0;

  el->vis2[2] = 0.120984298715595;
  el->vis2err[2] = 0.0154301900444324;
  el->bandflag[2] = 0;

  // the 3. visibility element (for all three wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 3);
  cpl_test_eq_ptr(vis2->list_vis2[2], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->ucoord = -95.5601110364095;
  el->vcoord = -84.7650317906203;
  el->stationindex[0] = 1;
  el->stationindex[1] = 28;

  el->vis2[0] = 0.0159990594078288;
  el->vis2err[0] = 0.00220486473948673;
  el->bandflag[0] = 0;

  el->vis2[1] = 0.0250420418103495;
  el->vis2err[1] = 0.00328488930376672;
  el->bandflag[1] = 0;

  el->vis2[2] = 0.0311027719528187;
  el->vis2err[2] = 0.00408597836692518;
  el->bandflag[2] = 0;

  // returns the just ceated and filled OI_VIS binary table
  return vis2;
}

static mat_oit3 *mat_oit3_create_test_amber(void)
{
  mat_oit3    *t3 = NULL;
  mat_t3elem  *el = NULL;

  t3 = mat_oit3_new(0, 3);
  cpl_test_nonnull(t3);
  cpl_test_null(t3->list_t3);
  /* if (t3 == NULL) return NULL; */
  t3->dateobs = cpl_strdup("2008-03-01T02:01:57.1002");
  t3->arrayname = cpl_strdup("VLTI-AMBER");
  t3->insname = cpl_strdup("AMBER");
  // the 1. T3 element (for all three wavelength)
  el = mat_oit3_elem_new(t3);
  cpl_test_nonnull(el);
  cpl_test_nonnull(t3->list_t3);
  if (el == NULL)
    {
      mat_oit3_delete(t3);
      return NULL;
    }
  cpl_test_eq(t3->nbt3, 1);
  cpl_test_eq_ptr(t3->list_t3[0], el);
  el->targetid = 1;
  el->time = 7650.96163898706;
  el->dateobsmjd = 54526.0885527967;
  el->exptime = 50.0495223999023;
  el->u1coord = -17.0703104905503;
  el->v1coord = -80.5530052239785;
  el->u2coord = -78.4898005390535;
  el->v2coord = -4.21202660701083;
  el->stationindex[0] = 28;
  el->stationindex[1] = 18;
  el->stationindex[2] = 1;

  el->t3phi[0] = 24.2031279605295;
  el->t3phierr[0] = 13.8304653775751;
  el->t3amp[0] = 0.0;
  el->t3amperr[0] = 0.0;
  el->bandflag[0] = 0;

  el->t3phi[1] = 27.9795095472788;
  el->t3phierr[1] = 18.6951682458425;
  el->t3amp[1] = 0.0;
  el->t3amperr[1] = 0.0;
  el->bandflag[1] = 0;

  el->t3phi[2] = 23.6747854810202;
  el->t3phierr[2] = 16.0054639204603;
  el->t3amp[2] = 0.0;
  el->t3amperr[2] = 0.0;
  el->bandflag[2] = 0;

  // returns the just created and filled OT_T3 binary table
  return t3;
}

static mat_oifits *mat_oifits_test_create_amber(void)
{
  mat_oifits        *oifits = NULL;
  mat_oitarget      *target = NULL;
  mat_array         *array  = NULL;
  mat_oiwavelength  *oiwave = NULL;
  mat_oivis         *oivis  = NULL;
  mat_oivis2        *oivis2 = NULL;
  mat_oit3          *oit3   = NULL;
  mat_oifits_tupel  *tupel  = NULL;

  oifits = mat_oifits_new();
  cpl_test_nonnull(oifits);
  if (oifits == NULL) return NULL;
  oifits->keywords = cpl_propertylist_new();
  // create and add the OI_TARGET binary table
  target = mat_oitarget_create_test();
  cpl_test_nonnull(target);
  if (target == NULL) return oifits;
  mat_oifits_add_target(oifits, target);
  cpl_test_eq_ptr(oifits->oitarget, target);
  // create and add the ARRAY binary table
  array = mat_array_create_test_amber();
  cpl_test_nonnull(array);
  if (array == NULL) return oifits;
  mat_oifits_add_array(oifits, array);
  cpl_test_eq_ptr(oifits->array, array);
  cpl_test_eq(oifits->nbarray, 1);
  cpl_test_nonnull(oifits->array_list);
  if (oifits->array_list == NULL) return oifits;
  cpl_test_eq_ptr(oifits->array_list[0], array);
  // create and add the OI_WAVELENGTH binary table
  oiwave = mat_oiwavelength_create_test_amber();
  cpl_test_nonnull(oiwave);
  if (oiwave == NULL) return oifits;
  mat_oifits_add_wavelength(oifits, oiwave);
  cpl_test_eq_ptr(oifits->oiwave, oiwave);
  cpl_test_eq(oifits->nbwave, 1);
  cpl_test_nonnull(oifits->wave_list);
  if (oifits->wave_list == NULL) return oifits;
  cpl_test_eq_ptr(oifits->wave_list[0], oiwave);
  // create and add the OI_VIS binary table
  oivis = mat_oivis_create_test_amber();
  cpl_test_nonnull(oivis);
  if (oivis == NULL) return oifits;
  mat_oifits_add_vis(oifits, oivis);
  cpl_test_eq_ptr(oifits->oivis, oivis);
  cpl_test_eq(oifits->nbtupel, 1);
  cpl_test_eq(oifits->sel_tupel, 0);
  cpl_test_nonnull(oifits->tupel_list);
  if (oifits->tupel_list == NULL) return oifits;
  tupel = oifits->tupel_list[0];
  cpl_test_nonnull(tupel);
  if (tupel == NULL) return oifits;
  cpl_test_eq_string(tupel->arrname, oivis->arrayname);
  cpl_test_eq_string(tupel->insname, oivis->insname);
  cpl_test_eq_ptr(tupel->oivis, oivis);
  // create and add the OI_VIS2 binary table
  oivis2 = mat_oivis2_create_test_amber();
  cpl_test_nonnull(oivis2);
  if (oivis2 == NULL) return oifits;
  mat_oifits_add_vis2(oifits, oivis2);
  cpl_test_eq_ptr(oifits->oivis2, oivis2);
  cpl_test_eq(oifits->nbtupel, 1);
  cpl_test_eq(oifits->sel_tupel, 0);
  cpl_test_nonnull(oifits->tupel_list);
  if (oifits->tupel_list == NULL) return oifits;
  tupel = oifits->tupel_list[0];
  cpl_test_nonnull(tupel);
  if (tupel == NULL) return oifits;
  cpl_test_eq_string(tupel->arrname, oivis2->arrayname);
  cpl_test_eq_string(tupel->insname, oivis2->insname);
  cpl_test_eq_ptr(tupel->oivis2, oivis2);
  // create and add the OI_T3 binary table
  oit3 = mat_oit3_create_test_amber();
  cpl_test_nonnull(oit3);
  if (oit3 == NULL) return oifits;
  mat_oifits_add_t3(oifits, oit3);
  cpl_test_eq_ptr(oifits->oit3, oit3);
  cpl_test_eq(oifits->nbtupel, 1);
  cpl_test_eq(oifits->sel_tupel, 0);
  cpl_test_nonnull(oifits->tupel_list);
  if (oifits->tupel_list == NULL) return oifits;
  tupel = oifits->tupel_list[0];
  cpl_test_nonnull(tupel);
  if (tupel == NULL) return oifits;
  cpl_test_eq_string(tupel->arrname, oit3->arrayname);
  cpl_test_eq_string(tupel->insname, oit3->insname);
  cpl_test_eq_ptr(tupel->oit3, oit3);

  cpl_test_null(tupel->oitf2);
  // returns the just created and filled OI-FITS data structure
  return oifits;
}


/**
   @brief Creates a new ARRAY binary table with a well defined content.
   @returns The newly created mat_array data structure or NULL.

   This static function creates a simple ARRAY data structure which
   is later used to check the ARRAY related functions of the mat_oifits
   data structure.
 */
static mat_array *mat_array_create_test_matisse(void)
{
  mat_array   *arr = NULL;
  mat_station *sta = NULL;

  arr = mat_array_new(4); // create an array with 4 stations (ATs)
  cpl_test_nonnull(arr);
  if (arr == NULL) return NULL;
  cpl_test_nonnull(arr->list_station);
  if (arr->list_station == NULL)
    {
      mat_array_delete(arr);
      return NULL;
    }
  arr->telescope = cpl_strdup("VLTI");
  arr->dcsdictionaryid = cpl_strdup("MATISSE");
  arr->dcsid = cpl_strdup("DCS-1.1");
  arr->arrayname = cpl_strdup("VLTI-MATISSE");
  arr->cooframe = cpl_strdup("GEOCENTRIC");
  arr->arrayx =  1951952.0;
  arr->arrayy = -5483173.0;
  arr->arrayz = -2668147.0;
  sta = arr->list_station[0];
  strncpy(sta->telname, "AT3", sizeof(sta->telname));
  strncpy(sta->staname, "D0", sizeof(sta->staname));
  sta->staindex = 28;
  sta->diameter = 1.8;
  sta->staxyz[0] = -106.409452128499;
  sta->staxyz[1] =   14.1889084662963;
  sta->staxyz[2] =    0.006309541;
  sta = arr->list_station[1];
  strncpy(sta->telname, "AT4", sizeof(sta->telname));
  strncpy(sta->staname, "G1", sizeof(sta->staname));
  sta->staindex = 18;
  sta->diameter = 1.8;
  sta->staxyz[0] = -66.715727329499;
  sta->staxyz[1] =  95.5014776162963;
  sta->staxyz[2] =   0.0;
  sta = arr->list_station[2];
  strncpy(sta->telname, "AT1", sizeof(sta->telname));
  strncpy(sta->staname, "H0", sizeof(sta->staname));
  sta->staindex = 1;
  sta->diameter = 1.8;
  sta->staxyz[0] = 14.619949864501;
  sta->staxyz[1] = 55.7985168502963;
  sta->staxyz[2] = -0.013313335;
  sta = arr->list_station[3];
  strncpy(sta->telname, "AT2", sizeof(sta->telname));
  strncpy(sta->staname, "I1", sizeof(sta->staname));
  sta->staindex = 1;
  sta->diameter = 1.8;
  sta->staxyz[0] = -15.6302358974051;
  sta->staxyz[1] =  45.3787849093006;
  sta->staxyz[2] =  -0.012968482;
  return arr;
}

static mat_oiwavelength *mat_oiwavelength_create_test_matisse(void)
{
  mat_oiwavelength *wave = NULL;

  wave = mat_oiwavelength_new(1);
  cpl_test_nonnull(wave);
  if (wave == NULL) return NULL;
  cpl_test_nonnull(wave->effwave);
  cpl_test_nonnull(wave->effband);
  if ((wave->effwave == NULL) || (wave->effband == NULL))
    {
      mat_oiwavelength_delete(wave);
      return NULL;
    }
  wave->insname = cpl_strdup("MATISSE");
  wave->effwave[0] = 4.214271E-06;
  wave->effband[0] = 3.337536E-08;
  return wave;
}

static mat_oivis2 *mat_oivis2_create_test_matisse(void)
{
  mat_oivis2    *vis2 = NULL;
  mat_vis2elem  *el   = NULL;

  vis2 = mat_oivis2_new(0, 1);
  cpl_test_nonnull(vis2);
  cpl_test_null(vis2->list_vis2);
  /* if (vis2 == NULL) return NULL; */
  vis2->dateobs = cpl_strdup("2015-03-06T08:40:35");
  vis2->arrayname = cpl_strdup("VLTI-MATISSE");
  vis2->insname = cpl_strdup("MATISSE");

  // the 1. visibility element (for one wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 1);
  cpl_test_eq_ptr(vis2->list_vis2[0], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -0.418886237936506;
  el->vcoord = -11.1438926844056;
  el->stationindex[0] = 1;
  el->stationindex[1] = 2;

  el->vis2[0] = 0.305248038035335;
  el->vis2err[0] = 0.0087867278962683;
  el->bandflag[0] = 0;

  // the 2. visibility element (for one wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 2);
  cpl_test_eq_ptr(vis2->list_vis2[1], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -9.8443856954422;
  el->vcoord = -31.941900352947;
  el->stationindex[0] = 1;
  el->stationindex[1] = 3;

  el->vis2[0] = 0.00784530205064533;
  el->vis2err[0] = 0.000327879224013711;
  el->bandflag[0] = 0;

  // the 3. visibility element (for one wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 3);
  cpl_test_eq_ptr(vis2->list_vis2[2], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -1.27699752426426;
  el->vcoord = -33.4513855921862;
  el->stationindex[0] = 1;
  el->stationindex[1] = 4;

  el->vis2[0] = 0.00713537392946795;
  el->vis2err[0] = 0.000353946119517788;
  el->bandflag[0] = 0;

  // the 4. visibility element (for one wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 4);
  cpl_test_eq_ptr(vis2->list_vis2[3], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -0.858111286327752;
  el->vcoord = -22.3074929077806;
  el->stationindex[0] = 2;
  el->stationindex[1] = 3;

  el->vis2[0] = 0.0980322973931264;
  el->vis2err[0] = 0.00299272223043889;
  el->bandflag[0] = 0;

  // the 5. visibility element (for one wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 5);
  cpl_test_eq_ptr(vis2->list_vis2[4], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = 8.56738817117794;
  el->vcoord = -1.5094852392392;
  el->stationindex[0] = 2;
  el->stationindex[1] = 4;

  el->vis2[0] = 0.295119943366124;
  el->vis2err[0] = 0.00891755081704421;
  el->bandflag[0] = 0;

  // the 6. visibility element (for one wavelength)
  el = mat_oivis2_elem_new(vis2);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis2->list_vis2);
  if (el == NULL)
    {
      mat_oivis2_delete(vis2);
      return NULL;
    }
  cpl_test_eq(vis2->nbvis2, 6);
  cpl_test_eq_ptr(vis2->list_vis2[5], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = 8.98627440911445;
  el->vcoord = 9.63440744516642;
  el->stationindex[0] = 3;
  el->stationindex[1] = 4;

  el->vis2[0] = 0.19326352668098;
  el->vis2err[0] = 0.00582081272454077;
  el->bandflag[0] = 0;

  // returns the just ceated and filled OI_VIS binary table
  return vis2;
}

static mat_oit3 *mat_oit3_create_test_matisse(void)
{
  mat_oit3    *t3 = NULL;
  mat_t3elem  *el = NULL;

  t3 = mat_oit3_new(0, 1);
  cpl_test_nonnull(t3);
  cpl_test_null(t3->list_t3);
  /* if (t3 == NULL) return NULL; */
  t3->dateobs = cpl_strdup("2015-03-06T08:40:35");
  t3->arrayname = cpl_strdup("VLTI-MATISSE");
  t3->insname = cpl_strdup("MATISSE");
  // the 1. T3 element (for one wavelength)
  el = mat_oit3_elem_new(t3);
  cpl_test_nonnull(el);
  cpl_test_nonnull(t3->list_t3);
  if (el == NULL)
    {
      mat_oit3_delete(t3);
      return NULL;
    }
  cpl_test_eq(t3->nbt3, 1);
  cpl_test_eq_ptr(t3->list_t3[0], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->u1coord = -0.858111286327752;
  el->v1coord = -22.3074929077806;
  el->u2coord = -0.418886237936506;
  el->v2coord = -11.1438926844056;
  el->stationindex[0] = 2;
  el->stationindex[1] = 3;
  el->stationindex[2] = 4;

  el->t3phi[0] = -4.37129469257616;
  el->t3phierr[0] = 1.87281665316759;
  el->t3amp[0] = 0.0155522089137633;
  el->t3amperr[0] = 0.000541289018517712;
  el->bandflag[0] = 0;

  // the 2. T3 element (for one wavelength)
  el = mat_oit3_elem_new(t3);
  cpl_test_nonnull(el);
  cpl_test_nonnull(t3->list_t3);
  if (el == NULL)
    {
      mat_oit3_delete(t3);
      return NULL;
    }
  cpl_test_eq(t3->nbt3, 2);
  cpl_test_eq_ptr(t3->list_t3[1], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->u1coord = -9.8443856954422;
  el->v1coord = -31.941900352947;
  el->u2coord = 8.56738817117794;
  el->v2coord = -1.5094852392392;
  el->stationindex[0] = 1;
  el->stationindex[1] = 3;
  el->stationindex[2] = 4;

  el->t3phi[0] = -1.58918236890349;
  el->t3phierr[0] = 2.48992169435559;
  el->t3amp[0] = 0.00452771154721223;
  el->t3amperr[0] = 0.000216356217119923;
  el->bandflag[0] = 0;

  // the 3. T3 element (for one wavelength)
  el = mat_oit3_elem_new(t3);
  cpl_test_nonnull(el);
  cpl_test_nonnull(t3->list_t3);
  if (el == NULL)
    {
      mat_oit3_delete(t3);
      return NULL;
    }
  cpl_test_eq(t3->nbt3, 3);
  cpl_test_eq_ptr(t3->list_t3[2], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->u1coord = -9.8443856954422;
  el->v1coord = -31.941900352947;
  el->u2coord = 8.98627440911445;
  el->v2coord = 9.63440744516642;
  el->stationindex[0] = 1;
  el->stationindex[1] = 2;
  el->stationindex[2] = 4;

  el->t3phi[0] = 4.19874380422652;
  el->t3phierr[0] = 0.814377182037615;
  el->t3amp[0] = 0.0129373053855216;
  el->t3amperr[0] = 0.000416272478169222;
  el->bandflag[0] = 0;

  // the 4. T3 element (for one wavelength)
  el = mat_oit3_elem_new(t3);
  cpl_test_nonnull(el);
  cpl_test_nonnull(t3->list_t3);
  if (el == NULL)
    {
      mat_oit3_delete(t3);
      return NULL;
    }
  cpl_test_eq(t3->nbt3, 4);
  cpl_test_eq_ptr(t3->list_t3[3], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->u1coord = 8.98627440911445;
  el->v1coord = 9.63440744516642;
  el->u2coord = -0.418886237936506;
  el->v2coord = -11.1438926844056;
  el->stationindex[0] = 1;
  el->stationindex[1] = 2;
  el->stationindex[2] = 3;

  el->t3phi[0] = 0.638240134203443;
  el->t3phierr[0] = 0.404267619577992;
  el->t3amp[0] = 0.13308818546362;
  el->t3amperr[0] = 0.00397834961277706;
  el->bandflag[0] = 0;

  // returns the just created and filled OT_T3 binary table
  return t3;
}

static mat_oivis *mat_oivis_create_test_matisse(void)
{
  mat_oivis    *vis = NULL;
  mat_viselem  *el   = NULL;

  vis = mat_oivis_new(0, 1);
  cpl_test_nonnull(vis);
  cpl_test_null(vis->list_vis);
  /* if (vis == NULL) return NULL; */
  vis->dateobs = cpl_strdup("2015-03-06T08:40:35");
  vis->arrayname = cpl_strdup("VLTI-MATISSE");
  vis->insname = cpl_strdup("MATISSE");

  // the 1. visibility element (for one wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 1);
  cpl_test_eq_ptr(vis->list_vis[0], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -0.418886237936506;
  el->vcoord = -11.1438926844056;
  el->stationindex[0] = 1;
  el->stationindex[1] = 2;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -1.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  // the 2. visibility element (for one wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 2);
  cpl_test_eq_ptr(vis->list_vis[1], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -9.8443856954422;
  el->vcoord = -31.941900352947;
  el->stationindex[0] = 1;
  el->stationindex[1] = 3;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -2.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  // the 3. visibility element (for one wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 3);
  cpl_test_eq_ptr(vis->list_vis[2], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -1.27699752426426;
  el->vcoord = -33.4513855921862;
  el->stationindex[0] = 1;
  el->stationindex[1] = 4;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -3.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  // the 4. visibility element (for one wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 4);
  cpl_test_eq_ptr(vis->list_vis[3], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = -0.858111286327752;
  el->vcoord = -22.3074929077806;
  el->stationindex[0] = 2;
  el->stationindex[1] = 3;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -4.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  // the 5. visibility element (for one wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 5);
  cpl_test_eq_ptr(vis->list_vis[4], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = 8.56738817117794;
  el->vcoord = -1.5094852392392;
  el->stationindex[0] = 2;
  el->stationindex[1] = 4;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -5.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  // the 6. visibility element (for one wavelength)
  el = mat_oivis_elem_new(vis);
  cpl_test_nonnull(el);
  cpl_test_nonnull(vis->list_vis);
  if (el == NULL)
    {
      mat_oivis_delete(vis);
      return NULL;
    }
  cpl_test_eq(vis->nbvis, 6);
  cpl_test_eq_ptr(vis->list_vis[5], el);
  el->targetid = 1;
  el->time = 0.0;
  el->dateobsmjd = 56679.0390650552;
  el->exptime = 156.159992218018;
  el->ucoord = 8.98627440911445;
  el->vcoord = 9.63440744516642;
  el->stationindex[0] = 3;
  el->stationindex[1] = 4;

  el->visamp[0] = 0.0;
  el->visamperr[0] = 0.0;
  el->visphi[0] = -6.992892415061318;
  el->visphierr[0] = 18.6949277823478;
  el->bandflag[0] = 0;

  // returns the just ceated and filled OI_VIS binary table
  return vis;
}

static mat_oifits *mat_oifits_test_create_matisse(void)
{
  mat_oifits        *oifits = NULL;
  mat_oitarget      *target = NULL;
  mat_array         *array  = NULL;
  mat_oiwavelength  *oiwave = NULL;
  mat_oivis         *oivis  = NULL;
  mat_oivis2        *oivis2 = NULL;
  mat_oit3          *oit3   = NULL;
  mat_oifits_tupel  *tupel  = NULL;

  oifits = mat_oifits_new();
  cpl_test_nonnull(oifits);
  if (oifits == NULL) return NULL;
  oifits->keywords = cpl_propertylist_new();
  // create and add the OI_TARGET binary table
  target = mat_oitarget_create_test();
  cpl_test_nonnull(target);
  if (target == NULL) return oifits;
  mat_oifits_add_target(oifits, target);
  cpl_test_eq_ptr(oifits->oitarget, target);
  // create and add the ARRAY binary table
  array = mat_array_create_test_matisse();
  cpl_test_nonnull(array);
  if (array == NULL) return oifits;
  mat_oifits_add_array(oifits, array);
  cpl_test_eq_ptr(oifits->array, array);
  cpl_test_eq(oifits->nbarray, 1);
  cpl_test_nonnull(oifits->array_list);
  if (oifits->array_list == NULL) return oifits;
  cpl_test_eq_ptr(oifits->array_list[0], array);
  // create and add the OI_WAVELENGTH binary table
  oiwave = mat_oiwavelength_create_test_matisse();
  cpl_test_nonnull(oiwave);
  if (oiwave == NULL) return oifits;
  mat_oifits_add_wavelength(oifits, oiwave);
  cpl_test_eq_ptr(oifits->oiwave, oiwave);
  cpl_test_eq(oifits->nbwave, 1);
  cpl_test_nonnull(oifits->wave_list);
  if (oifits->wave_list == NULL) return oifits;
  cpl_test_eq_ptr(oifits->wave_list[0], oiwave);
  // create and add the OI_VIS binary table
  oivis = mat_oivis_create_test_matisse();
  cpl_test_nonnull(oivis);
  if (oivis == NULL) return oifits;
  mat_oifits_add_vis(oifits, oivis);
  cpl_test_eq_ptr(oifits->oivis, oivis);
  cpl_test_eq(oifits->nbtupel, 1);
  cpl_test_eq(oifits->sel_tupel, 0);
  cpl_test_nonnull(oifits->tupel_list);
  if (oifits->tupel_list == NULL) return oifits;
  tupel = oifits->tupel_list[0];
  cpl_test_nonnull(tupel);
  if (tupel == NULL) return oifits;
  cpl_test_eq_string(tupel->arrname, oivis->arrayname);
  cpl_test_eq_string(tupel->insname, oivis->insname);
  cpl_test_eq_ptr(tupel->oivis, oivis);
  // create and add the OI_VIS2 binary table
  oivis2 = mat_oivis2_create_test_matisse();
  cpl_test_nonnull(oivis2);
  if (oivis2 == NULL) return oifits;
  mat_oifits_add_vis2(oifits, oivis2);
  cpl_test_eq_ptr(oifits->oivis2, oivis2);
  cpl_test_eq(oifits->nbtupel, 1);
  cpl_test_eq(oifits->sel_tupel, 0);
  cpl_test_nonnull(oifits->tupel_list);
  if (oifits->tupel_list == NULL) return oifits;
  tupel = oifits->tupel_list[0];
  cpl_test_nonnull(tupel);
  if (tupel == NULL) return oifits;
  cpl_test_eq_string(tupel->arrname, oivis2->arrayname);
  cpl_test_eq_string(tupel->insname, oivis2->insname);
  cpl_test_eq_ptr(tupel->oivis2, oivis2);
  // create and add the OI_T3 binary table
  oit3 = mat_oit3_create_test_matisse();
  cpl_test_nonnull(oit3);
  if (oit3 == NULL) return oifits;
  mat_oifits_add_t3(oifits, oit3);
  cpl_test_eq_ptr(oifits->oit3, oit3);
  cpl_test_eq(oifits->nbtupel, 1);
  cpl_test_eq(oifits->sel_tupel, 0);
  cpl_test_nonnull(oifits->tupel_list);
  if (oifits->tupel_list == NULL) return oifits;
  tupel = oifits->tupel_list[0];
  cpl_test_nonnull(tupel);
  if (tupel == NULL) return oifits;
  cpl_test_eq_string(tupel->arrname, oit3->arrayname);
  cpl_test_eq_string(tupel->insname, oit3->insname);
  cpl_test_eq_ptr(tupel->oit3, oit3);

  //cpl_test_null(tupel->oivis);
  //cpl_test_null(tupel->oitf2);
  // returns the just ceated and filled OI-FITS dta structure
  return oifits;
}

static int mat_oifits_test_merge(mat_oifits *dst, mat_oifits *src)
{
  // Move all ARRAY, OI_WAVELENGTH and OI_SPECTRUM binary tables from the source OI_FITS file
  // to the destination OI_FITS file
  while (src->nbarray != 0)
    {
      int onbs = src->nbarray;
      int onbd = dst->nbarray;
      mat_array *array = src->array_list[0];
      mat_oifits_remove_array(src, array);
      mat_oifits_add_array(dst, array);
      cpl_test_eq(src->nbarray, onbs - 1);
      cpl_test_eq(dst->nbarray, onbd + 1);
    }
  while (src->nbwave != 0)
    {
      int onbs = src->nbwave;
      int onbd = dst->nbwave;
      mat_oiwavelength *wave = src->wave_list[0];
      mat_oifits_remove_wavelength(src, wave);
      mat_oifits_add_wavelength(dst, wave);
      cpl_test_eq(src->nbwave, onbs - 1);
      cpl_test_eq(dst->nbwave, onbd + 1);
    }
  while (src->nbspect != 0)
    {
      int onbs = src->nbspect;
      int onbd = dst->nbspect;
      mat_oispectrum *spect = src->spect_list[0];
      mat_oifits_remove_spectrum(src, spect);
      mat_oifits_add_spectrum(dst, spect);
      cpl_test_eq(src->nbspect, onbs - 1);
      cpl_test_eq(dst->nbspect, onbd + 1);
    }

  // Move all tupels from the source to the destination OI_FITS file
  while (src->nbtupel != 0)
    {
      // Select the first tupel
      mat_oifits_select_tupel(src, 0);
      // Move the OI_VIS2 binary table
      if (src->oivis2 != NULL)
	{
	  mat_oivis2  *vis2 = src->oivis2;
	  mat_oifits_remove_vis2(src, vis2);
	  cpl_test_null(src->oivis2);
	  mat_oifits_add_vis2(dst, vis2);
	}
      // Move the OI_T3 binary table
      if (src->oit3 != NULL)
	{
	  mat_oit3  *t3 = src->oit3;
	  mat_oifits_remove_t3(src, t3);
	  cpl_test_null(src->oit3);
	  mat_oifits_add_t3(dst, t3);
	}
      // Move the OI_VIS binary table
      if (src->oivis != NULL)
	{
	  mat_oivis  *vis = src->oivis;
	  mat_oifits_remove_vis(src, vis);
	  cpl_test_null(src->oivis);
	  mat_oifits_add_vis(dst, vis);
	}
      // Move the OI_TF2 binary table
      if (src->oitf2 != NULL)
	{
	  mat_oitf2  *tf2 = src->oitf2;
	  mat_oifits_remove_tf2(src, tf2);
	  cpl_test_null(src->oitf2);
	  mat_oifits_add_tf2(dst, tf2);
	}
    }
  // Check the result
  cpl_test_eq(src->nbarray, 0);
  cpl_test_eq(src->nbwave, 0);
  cpl_test_eq(src->nbspect, 0);
  cpl_test_eq(src->nbtupel, 0);
  cpl_test_eq(dst->nbarray, 2);
  cpl_test_eq(dst->nbwave, 2);
  cpl_test_eq(dst->nbspect, 0);
  cpl_test_eq(dst->nbtupel, 2);
  // select the first tupel (must be the AMBER measurement)
  mat_oifits_select_tupel(dst, 0);
  cpl_test_nonnull(dst->array);
  cpl_test_nonnull(dst->oiwave);
  cpl_test_null(dst->oispect);
  cpl_test_nonnull(dst->oivis2);
  cpl_test_nonnull(dst->oit3);
  cpl_test_nonnull(dst->oivis);
  cpl_test_null(dst->oitf2);
  if (dst->oiwave != NULL) // paranoia!
    {
      cpl_test_eq(dst->oiwave->nbchannel, 3); // three wavelength
    }
  if (dst->oivis2 != NULL) // paranoia!
    {
      cpl_test_eq(dst->oivis2->nbchannel, 3); // three wavelength
      cpl_test_eq(dst->oivis2->nbvis2, 3); // three baselines
    }
  if (dst->oit3 != NULL) // paranoia!
    {
      cpl_test_eq(dst->oit3->nbchannel, 3); // three wavelength
      cpl_test_eq(dst->oit3->nbt3, 1); // one triple
    }
  if (dst->oivis != NULL) // paranoia!
    {
      cpl_test_eq(dst->oivis->nbchannel, 3); // three wavelength
      cpl_test_eq(dst->oivis->nbvis, 3); // three baselines
    }
  // select the second tupel (must be the MATISSE measurement)
  mat_oifits_select_tupel(dst, 1);
  cpl_test_nonnull(dst->array);
  cpl_test_nonnull(dst->oiwave);
  cpl_test_null(dst->oispect);
  cpl_test_nonnull(dst->oivis2);
  cpl_test_nonnull(dst->oit3);
  cpl_test_nonnull(dst->oivis);
  cpl_test_null(dst->oitf2);
  if (dst->oiwave != NULL) // paranoia!
    {
      cpl_test_eq(dst->oiwave->nbchannel, 1); // one wavelength
    }
  if (dst->oivis2 != NULL) // paranoia!
    {
      cpl_test_eq(dst->oivis2->nbchannel, 1); // one wavelength
      cpl_test_eq(dst->oivis2->nbvis2, 6); // six baselines
    }
  if (dst->oit3 != NULL) // paranoia!
    {
      cpl_test_eq(dst->oit3->nbchannel, 1); // one wavelength
      cpl_test_eq(dst->oit3->nbt3, 4); // four triple
    }
  if (dst->oivis != NULL) // paranoia!
    {
      cpl_test_eq(dst->oivis->nbchannel, 1); // one wavelength
      cpl_test_eq(dst->oivis->nbvis, 6); // six baselines
    }
  return 1;
}

static void mat_compare_array(mat_array *array1, mat_array *array2)
{
  int i;

  cpl_test_eq_string(array1->telescope, array2->telescope);
  cpl_test_eq_string(array1->dcsdictionaryid, array2->dcsdictionaryid);
  cpl_test_eq_string(array1->dcsid, array2->dcsid);
  cpl_test_eq_string(array1->arrayname, array2->arrayname);
  cpl_test_eq_string(array1->cooframe, array2->cooframe);
  cpl_test_rel(array1->arrayx, array2->arrayx, 1e-4);
  cpl_test_rel(array1->arrayy, array2->arrayy, 1e-4);
  cpl_test_rel(array1->arrayz, array2->arrayz, 1e-4);
  cpl_test_eq(array1->nbstation, array2->nbstation);
  for (i = 0; (i < array1->nbstation) && (i < array2->nbstation); i++)
    {
      mat_station *station1 = array1->list_station[i];
      mat_station *station2 = array2->list_station[i];
      cpl_test_eq(station1->staindex, station2->staindex);
      cpl_test_rel(station1->diameter, station2->diameter, 1e-4);
      cpl_test_rel(station1->staxyz[0], station2->staxyz[0], 1e-4);
      cpl_test_rel(station1->staxyz[1], station2->staxyz[1], 1e-4);
      cpl_test_rel(station1->staxyz[2], station2->staxyz[2], 1e-4);
    }
}

static void mat_compare_oiwavelength(mat_oiwavelength *wave1, mat_oiwavelength *wave2)
{
  int i;

  cpl_test_eq_string(wave1->insname, wave2->insname);
  cpl_test_eq(wave1->nbchannel, wave2->nbchannel);
  for (i = 0; (i < wave1->nbchannel) && (i < wave2->nbchannel); i++)
    {
      cpl_test_rel(wave1->effwave[i], wave2->effwave[i], 1e-6);
      cpl_test_rel(wave1->effband[i], wave2->effband[i], 1e-6);
    }
}

static void mat_compare_oivis2(mat_oivis2 *vis21, mat_oivis2 *vis22)
{
  int i, j;

  if ((vis21 == NULL) && (vis22 == NULL)) return;
  cpl_test_nonnull(vis21);
  cpl_test_nonnull(vis22);
  if ((vis21 == NULL) || (vis22 == NULL)) return;
  cpl_test_eq_string(vis21->dateobs, vis22->dateobs);
  cpl_test_eq_string(vis21->arrayname, vis22->arrayname);
  cpl_test_eq_string(vis21->insname, vis22->insname);
  cpl_test_eq(vis21->nbchannel, vis22->nbchannel);
  cpl_test_eq(vis21->nbvis2, vis22->nbvis2);
  for (i = 0; (i < vis21->nbvis2) && (i < vis22->nbvis2); i++)
    {
      mat_vis2elem  *el1 = vis21->list_vis2[i];
      mat_vis2elem  *el2 = vis22->list_vis2[i];
      cpl_test_eq(el1->targetid, el2->targetid);
      cpl_test_rel(el1->time, el2->time, 1e-6);
      cpl_test_rel(el1->dateobsmjd, el2->dateobsmjd, 1e-6);
      cpl_test_rel(el1->exptime, el2->exptime, 1e-6);
      cpl_test_rel(el1->ucoord, el2->ucoord, 1e-6);
      cpl_test_rel(el1->vcoord, el2->vcoord, 1e-6);
      cpl_test_eq(el1->stationindex[0], el2->stationindex[0]);
      cpl_test_eq(el1->stationindex[1], el2->stationindex[1]);
      for (j = 0; (j < vis21->nbchannel) && (j < vis22->nbchannel); j++)
	{
	  cpl_test_rel(el1->vis2[j], el2->vis2[j], 1e-8);
	  cpl_test_rel(el1->vis2err[j], el2->vis2err[j], 1e-8);
	  cpl_test_eq(el1->bandflag[j], el2->bandflag[j]);
	}
    }
}

static void mat_compare_oit3(mat_oit3 *t31, mat_oit3 *t32)
{
  int i, j;

  if ((t31 == NULL) && (t32 == NULL)) return;
  cpl_test_nonnull(t31);
  cpl_test_nonnull(t32);
  if ((t31 == NULL) || (t32 == NULL)) return;
  cpl_test_eq_string(t31->dateobs, t32->dateobs);
  cpl_test_eq_string(t31->arrayname, t32->arrayname);
  cpl_test_eq_string(t31->insname, t32->insname);
  cpl_test_eq(t31->nbchannel, t32->nbchannel);
  cpl_test_eq(t31->nbt3, t32->nbt3);
  for (i = 0; (i < t31->nbt3) && (i < t32->nbt3); i++)
    {
      mat_t3elem  *el1 = t31->list_t3[i];
      mat_t3elem  *el2 = t32->list_t3[i];
      cpl_test_eq(el1->targetid, el2->targetid);
      cpl_test_rel(el1->time, el2->time, 1e-6);
      cpl_test_rel(el1->dateobsmjd, el2->dateobsmjd, 1e-6);
      cpl_test_rel(el1->exptime, el2->exptime, 1e-6);
      cpl_test_rel(el1->u1coord, el2->u1coord, 1e-6);
      cpl_test_rel(el1->v1coord, el2->v1coord, 1e-6);
      cpl_test_rel(el1->u2coord, el2->u2coord, 1e-6);
      cpl_test_rel(el1->v2coord, el2->v2coord, 1e-6);
      cpl_test_eq(el1->stationindex[0], el2->stationindex[0]);
      cpl_test_eq(el1->stationindex[1], el2->stationindex[1]);
      cpl_test_eq(el1->stationindex[2], el2->stationindex[2]);
      for (j = 0; (j < t31->nbchannel) && (j < t32->nbchannel); j++)
	{
	  cpl_test_rel(el1->t3amp[j], el2->t3amp[j], 1e-8);
	  cpl_test_rel(el1->t3amperr[j], el2->t3amperr[j], 1e-8);
	  cpl_test_rel(el1->t3phi[j], el2->t3phi[j], 1e-8);
	  cpl_test_rel(el1->t3phierr[j], el2->t3phierr[j], 1e-8);
	  cpl_test_eq(el1->bandflag[j], el2->bandflag[j]);
	}
    }
}

static void mat_compare_oivis(mat_oivis *vis1, mat_oivis *vis2)
{
  int i, j;

  if ((vis1 == NULL) && (vis2 == NULL)) return;
  cpl_test_nonnull(vis1);
  cpl_test_nonnull(vis2);
  if ((vis1 == NULL) || (vis2 == NULL)) return;
  cpl_test_eq_string(vis1->dateobs, vis2->dateobs);
  cpl_test_eq_string(vis1->arrayname, vis2->arrayname);
  cpl_test_eq_string(vis1->insname, vis2->insname);
  cpl_test_eq(vis1->nbchannel, vis2->nbchannel);
  cpl_test_eq(vis1->nbvis, vis2->nbvis);
  for (i = 0; (i < vis1->nbvis) && (i < vis2->nbvis); i++)
    {
      mat_viselem  *el1 = vis1->list_vis[i];
      mat_viselem  *el2 = vis2->list_vis[i];
      cpl_test_eq(el1->targetid, el2->targetid);
      cpl_test_rel(el1->time, el2->time, 1e-6);
      cpl_test_rel(el1->dateobsmjd, el2->dateobsmjd, 1e-6);
      cpl_test_rel(el1->exptime, el2->exptime, 1e-6);
      cpl_test_rel(el1->ucoord, el2->ucoord, 1e-6);
      cpl_test_rel(el1->vcoord, el2->vcoord, 1e-6);
      cpl_test_eq(el1->stationindex[0], el2->stationindex[0]);
      cpl_test_eq(el1->stationindex[1], el2->stationindex[1]);
      for (j = 0; (j < vis1->nbchannel) && (j < vis2->nbchannel); j++)
	{
	  cpl_test_rel(el1->visamp[j], el2->visamp[j], 1e-8);
	  cpl_test_rel(el1->visamperr[j], el2->visamperr[j], 1e-8);
	  cpl_test_rel(el1->visphi[j], el2->visphi[j], 1e-8);
	  cpl_test_rel(el1->visphierr[j], el2->visphierr[j], 1e-8);
	  cpl_test_eq(el1->bandflag[j], el2->bandflag[j]);
	}
    }
}

static void mat_oifits_compare(mat_oifits *oifits1, mat_oifits *oifits2)
{
  int i;

  cpl_test(mat_oitarget_compare(oifits1->oitarget, oifits2->oitarget));
  cpl_test_eq(oifits1->nbarray, oifits2->nbarray);
  for (i = 0; (i < oifits1->nbarray) && (i < oifits2->nbarray); i++)
    {
      mat_compare_array(oifits1->array_list[i], oifits2->array_list[i]);
    }
  cpl_test_eq(oifits1->nbwave, oifits2->nbwave);
  for (i = 0; (i < oifits1->nbwave) && (i < oifits2->nbwave); i++)
    {
      mat_compare_oiwavelength(oifits1->wave_list[i], oifits2->wave_list[i]);
    }
  cpl_test_eq(oifits1->nbspect, oifits2->nbspect);
  cpl_test_eq(oifits1->nbtupel, oifits2->nbtupel);
  for (i = 0; (i < oifits1->nbtupel) && (i < oifits2->nbtupel); i++)
    {
      mat_oifits_select_tupel(oifits1, i);
      mat_oifits_select_tupel(oifits2, i);
      mat_compare_oivis2(oifits1->oivis2, oifits2->oivis2);
      mat_compare_oit3(oifits1->oit3, oifits2->oit3);
      mat_compare_oivis(oifits1->oivis, oifits2->oivis);
    }
}

static int mat_oifits_test_fileio(mat_oifits *oifits)
{
  cpl_frame         *frame    = NULL;
  cpl_frameset      *frameset = NULL;
  cpl_parameterlist *parlist  = NULL;
  mat_oifits        *hoifits  = NULL;

  // We need at least one existing file in the frameset!
  // There is no way to create a product all by its own!

  char *filename = NULL;
  filename = cpl_sprintf("%s/unit_05.fits", MAT_TESTDATADIR);
  frameset = cpl_frameset_new();
  if (frameset == NULL)
    {
	  cpl_free(filename);
      return 0;
    }
  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, filename);
  cpl_free(filename);
  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
  cpl_frame_set_tag(frame, "TAG");
  cpl_frameset_insert(frameset, frame);
  parlist = cpl_parameterlist_new();
  if (parlist == NULL)
    {
      cpl_frameset_delete(frameset);
      return 0;
    }
  cpl_test(mat_oifits_save(oifits, frameset, frameset, parlist, "oifits-test", (char *)"oifits_mat_oifits.fits", 9) == CPL_ERROR_NONE);
  cpl_frameset_delete(frameset);
  cpl_parameterlist_delete(parlist);

  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, "oifits_mat_oifits.fits");
  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
  cpl_frame_set_tag(frame, "TAG");
  hoifits = mat_oifits_load(frame);
  cpl_test_nonnull(hoifits);
  mat_oifits_compare(oifits, hoifits);
  mat_oifits_delete(hoifits);
  cpl_frame_delete(frame);
  return 1;
}

int main(int argc, char ** argv)
{
  mat_oifits *oif_amber = NULL;
  mat_oifits *oif_matisse = NULL;

  (void) argc;
  (void) argv;

  cpl_test_init("mhein@mpifr-bonn.mpg.de",CPL_MSG_WARNING);
  // Try to create OI-FITS data structures of an AMBER and an MATISSE observation
  oif_amber = mat_oifits_test_create_amber();
  oif_matisse = mat_oifits_test_create_matisse();
  /* // Check the merge of the MATISSE measurements to the AMBER measurements */
  mat_oifits_test_merge(oif_amber, oif_matisse);
  /* // Check the OI_FITS file IO */
  mat_oifits_test_fileio(oif_amber);
  /* // deletes the OF-FITS data structures */
  mat_oifits_delete(oif_amber);
  mat_oifits_delete(oif_matisse);
  // Check for memory leak
  cpl_test(cpl_memory_is_empty());
  return cpl_test_end(0);
}
