/* $Id: mat_statistics-test.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*
 * uniform_double, poisson, normal partially based on numpy/random/mtrand
 * Copyright 2005 Robert Kern (robert.kern@gmail.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *
 * PCG generator implementation is licensed under Apache License 2.0
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_statistics-test.c $
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <math.h>
#include <sys/time.h>
#include <time.h>
#include <stdint.h>

#include <cpl.h>
#include <cpl_test.h>
#include <sys/stat.h>
#include "../mat_gendata.h"
#include "../mat_frame.h"
#include "../mat_statistics.h"

/* random number generators from ESO/HDRL (hdrl_random.c) */

struct mat_random_state_{
    uint64_t state;
    uint64_t inc;
    uint64_t has_normal;
    double normal;
};

typedef struct mat_random_state_ mat_random_state;

/* ---------------------------------------------------------------------------*/
/**
 * @brief generate uniform 64 bit random numbers
 *
 * @param rng random number generator state structure
 *
 * @return uniformly distributed 32 bit unsigned integer
 */
/* ---------------------------------------------------------------------------*/
static uint32_t mat_random_uniform_uint32(mat_random_state * rng)
{
    /* from:
     * PCG: A Family of Simple Fast Space-Efficient Statistically Good
     * Algorithms for Random Number Generation */
    uint64_t oldstate = rng->state;
    // Calculate output function (XSH RR), uses old state for max ILP
    uint32_t xorshifted = ((oldstate >> 18u) ^ oldstate) >> 27u;
    uint32_t rot = oldstate >> 59u;
    // Advance internal state
    rng->state = oldstate * 6364136223846793005ULL + (rng->inc|1);
    return (xorshifted >> rot) | (xorshifted << ((-rot) & 31));
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief create random number generator state
 *
 * @param type type, currently needs to be 1 for pcg generator
 * @param seed seed array, length depends on used generator, if NULL uses
 *             rand() to seed
 *             for pcg: two integers state and streamid
 *
 * @return random number generator state structure, must be deleted with
 *         mat_random_state_delete
 */
/* ---------------------------------------------------------------------------*/
static mat_random_state * mat_random_state_new(int type, uint64_t * seed)
{
  mat_random_state *state = NULL;
  uint64_t          s1, s2;

  cpl_error_ensure(type == 1, CPL_ERROR_UNSUPPORTED_MODE,
		   return NULL, "type needs to be 1");
  state = cpl_calloc(sizeof(*state), 1);
  s1 = seed ? seed[0] : (uint64_t)rand();
  s2 = seed ? seed[1] : (uint64_t)rand();
  state->state = 0;
  state->inc = s2;
  mat_random_uniform_uint32(state);
  state->state += s1;
  mat_random_uniform_uint32(state);
  return state;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief delete random number generator state structure
 *
 * @param state
 */
/* ---------------------------------------------------------------------------*/
static void mat_random_state_delete(mat_random_state * state)
{
    cpl_free(state);
}

static double mat_random_uniform_double_one(mat_random_state * state)
{
    /* shifts : 67108864 = 0x4000000, 9007199254740992 = 0x20000000000000 */
    uint64_t a = mat_random_uniform_uint32(state) >> 5;
    uint64_t b = mat_random_uniform_uint32(state) >> 6;
    return (a * 67108864.0 + b) / 9007199254740992.0;
}

static uint64_t mat_random_poisson_low(mat_random_state * state, double lam)
{
    const double explam = exp(-lam);
    double prod = mat_random_uniform_double_one(state);
    uint64_t r = 0;

    while (prod > explam) {
        r++;
        prod *= mat_random_uniform_double_one(state);
    }

    return r;
}

/*
 * The transformed rejection method for generating Poisson random variables
 * W. Hoermann
 * Insurance: Mathematics and Economics 12, 39-45 (1993)
 */
static uint64_t mat_random_poisson_ptrs(mat_random_state * state, double lam)
{
    const double slam = sqrt(lam);
    const double loglam = log(lam);
    const double b = 0.931 + 2.53*slam;
    const double a = -0.059 + 0.02483*b;
    const double invalpha = 1.1239 + 1.1328/(b-3.4);
    const double vr = 0.9277 - 3.6224/(b-2);

    while (1) {
#ifdef HAVE_LGAMMA_R
        int sgngam;
        double lgamk;
#else
        double lgamk;
#endif
        double U = mat_random_uniform_double_one(state) - 0.5;
        double V = mat_random_uniform_double_one(state);
        double us = 0.5 - fabs(U);
        cpl_boolean eval = (us >= 0.07);
        double aux = 2 * a / us;

        int64_t k = (long)floor((aux + b)*U + lam + 0.43);
        if (eval && V <= vr) {
            return k;
        }
        if ((k < 0) ||
            ((us < 0.013) && (V > us))) {
            continue;
        }
#ifdef HAVE_LGAMMA_R
        lgamk = lgamma_r(k + 1, &sgngam);
#else
        lgamk = lgamma(k + 1);
#endif
        if ((log(V) + log(invalpha) - log(a/(us*us)+b)) <=
            (-lam + k * loglam - lgamk)) {
            return k;
        }
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief generatore poisson distributed values
 *
 * @param state   rng state
 * @param lam     lambda/mean parameter of poisson distribution
 *
 * @return poisson distributed value
 */
/* ---------------------------------------------------------------------------*/
static uint64_t mat_random_poisson(mat_random_state * state, double lam)
{
    if (lam >= 10.) {
        return mat_random_poisson_ptrs(state, lam);
    }
    else if (lam == 0.) {
        return 0;
    }
    else if (lam < 0.) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "lam must not be negative");
        return 0;
    }
    else {
        return mat_random_poisson_low(state, lam);
    }
}



static mat_gendata *mat_create_test_data(void)
{
  cpl_propertylist  *plist_gendat;
  cpl_propertylist  *plist_imgdet;
  mat_gendata       *raw;
  mat_region        *reg;
  int                i, j, k;
  cpl_size           x, y;
  //mat_random         info_poisson;
  mat_random_state *state_poisson;

  /* Create the main data structure for a RAW fits file (including property list for the primary HDU). */
  plist_gendat = cpl_propertylist_new();
  cpl_propertylist_append_bool(plist_gendat, "SIMPLE", 1);
  cpl_propertylist_append_int(plist_gendat, "BITPIX", 8);
  cpl_propertylist_append_int(plist_gendat, "NAXIS", 0);
  cpl_propertylist_append_bool(plist_gendat, "EXTEND", 1);
  cpl_propertylist_append_string(plist_gendat, "DATE", "2012-10-22");
  cpl_propertylist_append_string(plist_gendat, "ESO TPL ID", "MATISSE_gen_cal_det");
  cpl_propertylist_append_int(plist_gendat, "ESO DET READ CURID", 1);
  cpl_propertylist_append_string(plist_gendat, "ESO DET READ CURNAME", "SLOW");
  cpl_propertylist_append_string(plist_gendat, "ESO INS MODE", "4Tn_LsL_NsL");
  cpl_propertylist_append_string(plist_gendat, "ESO DPR CATG", "CALIB");
  cpl_propertylist_append_string(plist_gendat, "ESO DPR TECH", "IMAGE");
  cpl_propertylist_append_string(plist_gendat, "ESO DPR TYPE", "FLAT");
  cpl_propertylist_append_string(plist_gendat, "ESO INS BSL1 ST", "T");
  cpl_propertylist_append_string(plist_gendat, "ESO INS BSL2 ST", "T");
  cpl_propertylist_append_string(plist_gendat, "ESO INS BSL3 ST", "T");
  cpl_propertylist_append_string(plist_gendat, "ESO INS BSL4 ST", "T");
  cpl_propertylist_append_string(plist_gendat, "ESO DET CHIP ID", "4712");
  cpl_propertylist_append_string(plist_gendat, "ESO DET CHIP NAME", "HAWAII2RG");
  cpl_propertylist_append_string(plist_gendat, "ESO INS DIL", "LOW");
  /* create the main structure */
  raw = (mat_gendata *)cpl_calloc(1, sizeof(mat_gendata));
  if (raw == NULL) {
    cpl_msg_error(cpl_func, "could not allocate memory for the test data");
    return NULL;
  }
  raw->keywords = plist_gendat;
  /* Create and fill the imaging detector binary table (including property list). */
  plist_imgdet = cpl_propertylist_new();
  cpl_propertylist_append_string(plist_imgdet, "XTENSION", "BINTABLE");
  cpl_propertylist_append_int(plist_imgdet, "BITPIX", 8);
  cpl_propertylist_append_int(plist_imgdet, "NAXIS", 2);
  cpl_propertylist_append_int(plist_imgdet, "NAXIS1", 86);
  cpl_propertylist_append_int(plist_imgdet, "NAXIS2", 1);
  cpl_propertylist_append_int(plist_imgdet, "PCOUNT", 0);
  cpl_propertylist_append_int(plist_imgdet, "GCOUNT", 1);
  cpl_propertylist_append_int(plist_imgdet, "TFIELDS", 11);
  cpl_propertylist_append_string(plist_imgdet, "EXTNAME", "IMAGING_DETECTOR");
  cpl_propertylist_append_string(plist_imgdet, "ORIGIN", "SIMULATOR");
  cpl_propertylist_append_string(plist_imgdet, "INSTRUME", "MATISSE ");
  cpl_propertylist_append_double(plist_imgdet, "MJD-OBS", 56222.806);
  cpl_propertylist_append_string(plist_imgdet, "DATE-OBS", "2012-10-22T07:20:37");
  cpl_propertylist_append_string(plist_imgdet, "DATE", "2012-10-22T07:20:37");
  cpl_propertylist_append_string(plist_imgdet, "ESO DET DID", "ESO-VLT-DIC-NGC-1.00");
  cpl_propertylist_append_string(plist_imgdet, "ESO DET ID", "v 1.00");
  cpl_propertylist_append_int(plist_imgdet, "NDETECT", 2);
  cpl_propertylist_append_int(plist_imgdet, "NREGION", 1);
  cpl_propertylist_append_int(plist_imgdet, "MAX_COEF", 0);
  cpl_propertylist_append_int(plist_imgdet, "NUM_DIM", 2);
  cpl_propertylist_append_int(plist_imgdet, "MAXTEL", 4);
  raw->imgdet = mat_imagingdetector_new(plist_imgdet, 1);
  cpl_propertylist_delete(plist_imgdet);
  if (raw->imgdet == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the imaging detector binary table");
      mat_gendata_delete(raw);
      return NULL;
    }
  reg = raw->imgdet->list_region[0];
  mat_region_set_name(reg, "C", 0);
  reg->numdetector = 1;
  reg->numbeam[0] = 1;
  reg->numbeam[1] = 1;
  reg->numbeam[2] = 1;
  reg->numbeam[3] = 1;
  reg->correlation = 0;
  mat_region_set_area(reg, 1, 1, 2048, 2048);
  reg->gain = 2.6;
  /* Create and fill the imaging data binary table. */
  raw->imgdata = mat_imagingdata_new(raw->imgdet, 50, CPL_TYPE_FLOAT);
  for (i = 0; i < raw->imgdata->nbtel; i++)
    {
      raw->imgdata->staindex[i] = i;
      raw->imgdata->armindex[i] = i;
    }
  raw->imgdata->maxstep = 0;
  raw->imgdata->maxins = 0;
  //rnd_set_seed(&info_poisson, -43);
  state_poisson = mat_random_state_new(1, NULL);
  for (i = 0; i < raw->imgdata->nbframe; i++)
    {
      mat_frame *frame = raw->imgdata->list_frame[i];
      frame->time = 56222.806 + 7.0/3600.0/24.0*(double)i;
      frame->exptime = 1.4;
      frame->opd[0] = 0.0;
      frame->opd[1] = 0.0;
      frame->opd[2] = 0.0;
      frame->opd[3] = 0.0;
      frame->localopd[0] = 0.0;
      frame->localopd[1] = 0.0;
      frame->localopd[2] = 0.0;
      frame->localopd[3] = 0.0;
      frame->stepphase = 0;
      frame->tartype[0] = 'B';
      frame->tartype[1] = '\0';
      for (j = 0; j < frame->nbsubwin; j++)
	{
	  mat_imgreg *ireg = frame->list_subwin[j];
	  for (k = 0; k < ireg->nbimgreg; k++)
	    {
	      cpl_image *img = ireg->imgreg[k];
	      cpl_size nx = cpl_image_get_size_x(img);
	      cpl_size ny = cpl_image_get_size_y(img);
	      for (y = 1; y <= ny; y++)
		{
		  for (x = 1; x <= nx; x++)
		    {
		      double intensity = 10000.0;
		      double value, dist;
		      if (x <= 64) intensity = 10;
		      if (x >= 2048 - 64) intensity = 10;
		      if (y <= 32) intensity = 10;
		      if (y >= 2048 - 32) intensity = 10;
		      //value = rnd_poisson(&info_poisson, intensity);
		      value = (double)mat_random_poisson(state_poisson, intensity);
		      dist = (intensity - value);
		      value = intensity - dist;
		      if ((x == 1313) && (y == 1111))
			cpl_msg_info(cpl_func, " int %d %.1f", i, value);
		      cpl_image_set(img, x, y, value/3.14);
		    }
		}
	    }
	}
    }
  mat_random_state_delete(state_poisson);
  return raw;
}

int main(int argc, char ** argv) {
    /* const char *fctid = "mat_statistics-test"; */
    /* cpl_errorstate prestate; */
    mat_gendata *raw = NULL;
    mat_detector det;
    mat_frame *median = NULL;
    mat_frame *variance = NULL;
    double detgain;
    double detnoise;
    cpl_vector *channelgain = NULL;
    cpl_vector *channeloffset = NULL;
    cpl_vector *channelnoise = NULL;
    (void) argc;
    (void) argv;

    cpl_test_init("mhein@mpifr-bonn.mpg.de",CPL_MSG_WARNING);
    /* prestate = cpl_errorstate_get(); */
    raw = mat_create_test_data();
    cpl_test_nonnull(raw);
    if (raw != NULL)
    {
	cpl_error_reset();
	mat_detector_decode_raw(&det, raw);
	median = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
	cpl_test_nonnull(median);
	if (median != NULL)
	{
	    variance = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
	    //cpl_test_nonnull(variance);
	    if (variance != NULL)
	    {
	      cpl_test((mat_calc_statistics_cosmics(raw, median, NULL, 0) == CPL_ERROR_NULL_INPUT));
		cpl_error_reset();
		cpl_test((mat_calc_statistics_cosmics(raw, median, variance, 0) == CPL_ERROR_NONE));
		channelgain = cpl_vector_new(det.channel_ncolumns*det.channel_nrows);
		channeloffset = cpl_vector_new(det.channel_ncolumns*det.channel_nrows);
		channelnoise = cpl_vector_new(det.channel_ncolumns*det.channel_nrows);
		cpl_test_nonnull(channelgain);
		cpl_test_nonnull(channeloffset);
		cpl_test_nonnull(channelnoise);
		if (channelgain != NULL)
		{
		    cpl_test((mat_calc_gain(median->list_subwin[0]->imgreg[0],
					    variance->list_subwin[0]->imgreg[0],
					    NULL,
					    NULL,
					    det.channel_ncolumns,
					    det.channel_nrows,
					    &detgain,
					    NULL) == CPL_ERROR_NONE));
		    cpl_test((mat_calc_gain(median->list_subwin[0]->imgreg[0],
					    variance->list_subwin[0]->imgreg[0],
					    NULL,
					    NULL,
					    det.channel_ncolumns,
					    det.channel_nrows,
					    &detgain,
					    channelgain) == CPL_ERROR_NONE));
		    cpl_msg_info(cpl_func, "detgain = %f", detgain);
		    cpl_test_rel(detgain, 3.14, 0.02);
		    cpl_msg_info(cpl_func, "channelgain[7] = %f", cpl_vector_get(channelgain, 7));
		    cpl_test_rel(cpl_vector_get(channelgain, 7), 3.14, 0.02);
		}
		if ((channelgain != NULL) && (channeloffset != NULL) && (channelnoise != NULL))
		{
		    cpl_test((mat_calc_noise(variance->list_subwin[0]->imgreg[0],
					     NULL,
					     det.channel_ncolumns,
					     det.channel_nrows,
					     detgain,
					     &detnoise,
					     NULL,
					     NULL, 1.0) == CPL_ERROR_NONE));
		    cpl_test((mat_calc_noise(variance->list_subwin[0]->imgreg[0],
					     NULL,
					     det.channel_ncolumns,
					     det.channel_nrows,
					     detgain,
					     &detnoise,
					     channelgain,
					     channelnoise,
					     1.0) == CPL_ERROR_NONE));
		    cpl_test((mat_calc_offset(median->list_subwin[0]->imgreg[0],
					      NULL,
					      det.channel_ncolumns,
					      det.channel_nrows,
					      channeloffset) == CPL_ERROR_NONE));
		    cpl_msg_info(cpl_func, "detnoise = %f", detnoise);
		    cpl_test_rel(detnoise, 100.0, 0.02);
		    cpl_msg_info(cpl_func, "channeloffset[7] = %f", cpl_vector_get(channeloffset, 7));
		    cpl_test_rel(cpl_vector_get(channeloffset, 7), 3185.0, 0.02);
		    cpl_msg_info(cpl_func, "channelnoise[7] = %f", cpl_vector_get(channelnoise, 7));
		    cpl_test_rel(cpl_vector_get(channelnoise, 7), 100.0, 0.02);
		}
		if (channelgain != NULL)
		{
		    cpl_vector_delete(channelgain);
		    channelgain = NULL;
		}
		if (channeloffset != NULL)
		{
		    cpl_vector_delete(channeloffset);
		    channeloffset = NULL;
		}
		if (channelnoise != NULL)
		{
		    cpl_vector_delete(channelnoise);
		    channelnoise = NULL;
		}
		mat_frame_delete(variance);
		variance = NULL;
	    }
	    mat_frame_delete(median);
	    median = NULL;
	}
	mat_gendata_delete(raw);
	raw = NULL;
    }
// Test attribute keywords
// Check for memory leak
    cpl_test(cpl_memory_is_empty());
    return cpl_test_end(0);
}
