/* $Id: mat_cal_oifits.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_cal_oifits.c $
 */
/*------------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_cal_vis_lib.h"
#include "mat_cal_cphase_lib.h"
#include "mat_cal_dphase_lib.h"
#include "mat_merge_results_lib.h"

/*------------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_cal_oifits"
#define NB_BCD 4



/* Plugin detailed description */
static const char *
mat_cal_oifits_help = "This plugin calls all data reduction steps which \
calibrate visibility, closure phase and differential phase.\n"
  "It produces  OIFITS files containing all the calibrated interferometric measurements.\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    TARGET_RAW_INT         Science OIFITS              Yes \n"
  "    CALIB_RAW_INT          Calibrator OIFITS           Yes \n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    TARGET_CAL_INT         Calibrated OIFITS\n";
static int mat_cal_oifits_create(cpl_plugin *);
static int mat_cal_oifits_exec(cpl_plugin *);
static int mat_cal_oifits_destroy(cpl_plugin *);
static int mat_cal_oifits(cpl_parameterlist *parlist, cpl_frameset *frames);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_cal_oifits",
		  "All calibrate steps",
		  mat_cal_oifits_help,
		  "Philippe Berio",
		  PACKAGE_BUGREPORT,
		  "GPL",
		  mat_cal_oifits_create,
		  mat_cal_oifits_exec,
		  mat_cal_oifits_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

static int mat_cal_oifits_create(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameter *p,*p1,*p2;
  recipe->parameters = cpl_parameterlist_new();

  p = cpl_parameter_new_value("matisse.mat_cal_vis.tfKeep",
			      CPL_TYPE_INT, 
			      "store interpolate function", 
			      "matisse.mat_cal_oifits",
			      0,0,1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "tfKeep") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  p1 = cpl_parameter_new_value("matisse.mat_cal_vis.tfInterp",
			       CPL_TYPE_INT,
			       "transfer function interpolation method (0: average, 2:linear function)",
			       "matisse.mat_cal_oifits",
			       0,0,2);
  cpl_parameter_set_alias(p1, CPL_PARAMETER_MODE_CLI, "tfInterp") ;
  cpl_parameterlist_append(recipe->parameters, p1) ;

  p2 = cpl_parameter_new_value("matisse.mat_merge_results.cumulBlock",
			       CPL_TYPE_BOOL, "cumul all blocks of an OB", 
			       "matisse.mat_cal_oifits",TRUE);
  cpl_parameter_set_alias(p2, CPL_PARAMETER_MODE_CLI, "cumulBlock") ;
  cpl_parameterlist_append(recipe->parameters, p2) ; 


  cpl_msg_info("mat_cal_oifits_create", "Parameters created.");

  return 0;
}
static int mat_cal_oifits_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}


static int mat_cal_oifits_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe=(cpl_recipe *)plugin;
  } else {
    return -1;
  }
  return mat_cal_oifits(recipe->parameters,recipe->frames);
  
}

void frameset_sort(cpl_frameset *frames)
{
  cpl_frameset *framesort=NULL;

  framesort=cpl_frameset_duplicate(frames);
  cpl_frameset_erase(framesort,"TARGET_RAW_INT");
  cpl_frameset_erase(frames,"CALIB_RAW_INT");
  cpl_frameset_join(frames,framesort);
}

int mat_cal_oifits(cpl_parameterlist *parlist, cpl_frameset *frames) {
 
  cpl_frameset *frame_result=NULL;
  cpl_frame *cur_frame=NULL;
  cpl_frame *frame_calvis2 = NULL;
  cpl_frame *frame_calcphase = NULL;
  cpl_frame *frame_caldphase = NULL;
  cpl_size size_before;
  cpl_size size_after;
  mat_oifits *oifits=NULL;
  char *pszFileTag=NULL;
  cpl_frameset_iterator *it=NULL;
  char *bcd1Status=NULL;
  char *chopMode=NULL;
  int proScience=0;
  cpl_parameter *p=NULL;
  cpl_frameset *framesInIn=NULL;
  cpl_frameset *framesOutOut=NULL;
  cpl_frameset *framesInOut=NULL;
  cpl_frameset *framesOutIn=NULL;
  cpl_frameset *framesInInChop=NULL;
  cpl_frameset *framesOutOutChop=NULL;
  cpl_frameset *framesInOutChop=NULL;
  cpl_frameset *framesOutInChop=NULL;
  cpl_propertylist *pheader = NULL;
  int flagChopData = 0;

  p = cpl_parameter_new_value("matisse.mat_merge_results.bcdMode",CPL_TYPE_INT, "Select frames with BCD OUT-OUT (0) \
or frames with BCD IN-IN (1) or frames with BCD IN-OUT (2) or frames with BCD OUT-IN (3)", "matisse.mat_merge_results",0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bcdMode") ;
  cpl_parameterlist_append(parlist, p) ; 
   
  // sort frameset frames TARGET_RAW_INT first and CALIB_RAW_INT after
  frameset_sort(frames);

  cpl_frame *frameDup=NULL;
  framesInIn=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1)
  	{
  	  bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	  if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	    {
  	      chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	    }
  	  else
  	    {
  	      chopMode=cpl_sprintf("F");
  	    }
  	  if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "IN-IN") == 0)
  	    {
  	      frameDup=cpl_frame_duplicate(cur_frame);
  	      cpl_frameset_insert(framesInIn,frameDup);
  	    }
  	}
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0)
  	{
  	  bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	  if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	    {
  	      chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	    }
  	  else
  	    {
  	      chopMode=cpl_sprintf("F");
  	    }
  	  if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "IN-IN") == 0)
  	    {
  	      frameDup=cpl_frame_duplicate(cur_frame);
  	      cpl_frameset_insert(framesInIn,frameDup);
  	    }
  	}
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  
  framesOutOut=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
     /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "OUT-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutOut,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "OUT-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutOut,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  
  framesInOut=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
      	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "IN-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesInOut,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0 ) {
      	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "IN-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesInOut,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  
  framesOutIn=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "OUT-IN") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutIn,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "OUT-IN") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutIn,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);


  framesInInChop=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "IN-IN") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesInInChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "IN-IN") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesInInChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  
  framesOutOutChop=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "OUT-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutOutChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "OUT-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutOutChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  
  framesInOutChop=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "IN-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesInOutChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "IN-OUT") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesInOutChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);

  
  framesOutInChop=cpl_frameset_new();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 1) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "OUT-IN") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutInChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      /*first file, references values*/
      pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
      proScience = cpl_propertylist_get_bool(pheader,"ESO PRO SCIENCE");
      if (cpl_propertylist_has(pheader,"ESO CFG BCD MODE") && proScience == 0) {
  	bcd1Status = (char *) cpl_propertylist_get_string(pheader,"ESO CFG BCD MODE");
  	if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
  	  {
  	    chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
  	  }
  	else
  	  {
  	    chopMode=cpl_sprintf("F");
  	  }
  	if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "OUT-IN") == 0) {
  	  frameDup=cpl_frame_duplicate(cur_frame);
  	  cpl_frameset_insert(framesOutInChop,frameDup);
  	}
      }
      cpl_propertylist_delete(pheader);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  
  for(int ibcd=0;ibcd<2*NB_BCD;ibcd++) {
    if ((ibcd==0 && cpl_frameset_count_tags(framesInIn,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesInIn,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==1 && cpl_frameset_count_tags(framesOutOut,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesOutOut,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==2 && cpl_frameset_count_tags(framesInOut,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesInOut,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==3 && cpl_frameset_count_tags(framesOutIn,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesOutIn,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==4 && cpl_frameset_count_tags(framesInInChop,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesInInChop,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==5 && cpl_frameset_count_tags(framesOutOutChop,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesOutOutChop,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==6 && cpl_frameset_count_tags(framesInOutChop,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesInOutChop,"TARGET_RAW_INT") >= 1) ||
    	(ibcd==7 && cpl_frameset_count_tags(framesOutInChop,"CALIB_RAW_INT") >= 1 && cpl_frameset_count_tags(framesOutInChop,"TARGET_RAW_INT") >= 1)
    	) {
      if (ibcd==0) {
    	cpl_msg_info(cpl_func,"Processing frames BCD IN-IN - No Chop");
      }
      if (ibcd==1) {
    	cpl_msg_info(cpl_func,"Processing frames BCD OUT-OUT - No Chop");
      }
      if (ibcd==2) {
    	cpl_msg_info(cpl_func,"Processing frames BCD IN-OUT - No Chop");
      }
      if (ibcd==3) {
    	cpl_msg_info(cpl_func,"Processing frames BCD OUT-IN - No Chop");
      }
      if (ibcd==4) {
    	cpl_msg_info(cpl_func,"Processing frames BCD IN-IN - Chop");
	flagChopData=1;
      }
      if (ibcd==5) {
    	cpl_msg_info(cpl_func,"Processing frames BCD OUT-OUT - Chop");
	flagChopData=1;
      }
      if (ibcd==6) {
    	cpl_msg_info(cpl_func,"Processing frames BCD IN-OUT - Chop");
	flagChopData=1;
      }
      if (ibcd==7) {
    	cpl_msg_info(cpl_func,"Processing frames BCD OUT-IN - Chop");
	flagChopData=1;
      }
      // load oifits of TARGET_RAW_INT
      if (ibcd==0) {
    	cur_frame=cpl_frameset_find(framesInIn,"TARGET_RAW_INT");
      }
      if (ibcd==1) {
    	cur_frame=cpl_frameset_find(framesOutOut,"TARGET_RAW_INT");
      }
      if (ibcd==2) {
    	cur_frame=cpl_frameset_find(framesInOut,"TARGET_RAW_INT");
      }
      if (ibcd==3) {
    	cur_frame=cpl_frameset_find(framesOutIn,"TARGET_RAW_INT");
      }
      if (ibcd==4) {
    	cur_frame=cpl_frameset_find(framesInInChop,"TARGET_RAW_INT");
	flagChopData=1;
      }
      if (ibcd==5) {
    	cur_frame=cpl_frameset_find(framesOutOutChop,"TARGET_RAW_INT");
	flagChopData=1;
      }
      if (ibcd==6) {
    	cur_frame=cpl_frameset_find(framesInOutChop,"TARGET_RAW_INT");
	flagChopData=1;
      }
      if (ibcd==7) {
    	cur_frame=cpl_frameset_find(framesOutInChop,"TARGET_RAW_INT");
	flagChopData=1;
      }
      if (cur_frame == NULL) {
    	cpl_msg_error(cpl_func,"No TARGET_RAW_INT in mat_cal_oifits");
    	cpl_parameter_delete(p);
    	return CPL_ERROR_ILLEGAL_INPUT;
      } else {
    	oifits=mat_oifits_load(cur_frame);
    	if (cpl_propertylist_has(oifits->keywords,"ESO CFG BCD MODE")) {
    	  bcd1Status = (char *) cpl_propertylist_get_string(oifits->keywords,"ESO CFG BCD MODE");
    	  if ( cpl_propertylist_has(oifits->keywords,"ESO ISS CHOP ST") )
    	    {
    	      chopMode = (char *) cpl_propertylist_get_string(oifits->keywords,"ESO ISS CHOP ST");
    	    }
    	  else
    	    {
    	      chopMode=cpl_sprintf("F");
    	    }
    	  if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "OUT-OUT") == 0) {
    	    cpl_parameter_set_int(p,0);
    	  }
    	  if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "IN-IN") == 0) {
    	    cpl_parameter_set_int(p,1);
    	  }
    	  if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "IN-OUT") == 0) {
    	    cpl_parameter_set_int(p,2);
    	  }
    	  if (strcmp(chopMode,"F") == 0 && strcmp(bcd1Status, "OUT-IN") == 0) {
    	    cpl_parameter_set_int(p,3);
    	  }
    	  if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "OUT-OUT") == 0) {
    	    cpl_parameter_set_int(p,4);
    	  }
    	  if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "IN-IN") == 0) {
    	    cpl_parameter_set_int(p,5);
    	  }
    	  if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "IN-OUT") == 0) {
    	    cpl_parameter_set_int(p,6);
    	  }
    	  if (strcmp(chopMode,"T") == 0 && strcmp(bcd1Status, "OUT-IN") == 0) {
    	    cpl_parameter_set_int(p,7);
    	  }
    	} else {
    	  cpl_msg_error(cpl_func,"No BCD keywords");
    	  mat_oifits_delete(oifits);
    	  cpl_parameter_delete(p);
    	  return CPL_ERROR_INCOMPATIBLE_INPUT;
    	}
      }

      
			     
      cpl_msg_info(cpl_func,"****************************************************");
      cpl_msg_info(cpl_func,"Starting mat_cal_vis");
      cpl_msg_info(cpl_func,"****************************************************");

      if (ibcd==0) {
      	size_before=cpl_frameset_get_size(framesInIn);
      	if (mat_cal_vis_lib(parlist, framesInIn,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesInIn);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInIn,cpl_frameset_get_size(framesInIn)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInIn,cpl_frameset_get_size(framesInIn)-1));
      	}
      	cpl_frameset_erase(framesInIn,"CAL_VIS2");
      }
      if (ibcd==1) {
      	size_before=cpl_frameset_get_size(framesOutOut);
      	if (mat_cal_vis_lib(parlist, framesOutOut,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesOutOut);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOut,cpl_frameset_get_size(framesOutOut)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOut,cpl_frameset_get_size(framesOutOut)-1));
      	}
      	cpl_frameset_erase(framesOutOut,"CAL_VIS2");
      }
      if (ibcd==2) {
      	size_before=cpl_frameset_get_size(framesInOut);
      	if (mat_cal_vis_lib(parlist, framesInOut,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesInOut);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInOut,cpl_frameset_get_size(framesInOut)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInOut,cpl_frameset_get_size(framesInOut)-1));
      	}
      	cpl_frameset_erase(framesInOut,"CAL_VIS2");
      }
      if (ibcd==3) {
      	size_before=cpl_frameset_get_size(framesOutIn);
      	if (mat_cal_vis_lib(parlist, framesOutIn,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesOutIn);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutIn,cpl_frameset_get_size(framesOutIn)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutIn,cpl_frameset_get_size(framesOutIn)-1));
      	}
      	cpl_frameset_erase(framesOutIn,"CAL_VIS2");
      }
      if (ibcd==4) {
      	size_before=cpl_frameset_get_size(framesInInChop);
      	if (mat_cal_vis_lib(parlist, framesInInChop,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesInInChop);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInInChop,cpl_frameset_get_size(framesInInChop)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInInChop,cpl_frameset_get_size(framesInInChop)-1));
      	}
      	cpl_frameset_erase(framesInInChop,"CAL_VIS2");
      }
      if (ibcd==5) {
      	size_before=cpl_frameset_get_size(framesOutOutChop);
      	if (mat_cal_vis_lib(parlist, framesOutOutChop,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesOutOutChop);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOutChop,cpl_frameset_get_size(framesOutOutChop)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOutChop,cpl_frameset_get_size(framesOutOutChop)-1));
      	}
      	cpl_frameset_erase(framesOutOutChop,"CAL_VIS2");
      }
      if (ibcd==6) {
      	size_before=cpl_frameset_get_size(framesInOutChop);
      	if (mat_cal_vis_lib(parlist, framesInOutChop,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesInOutChop);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInOutChop,cpl_frameset_get_size(framesInOutChop)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesInOutChop,cpl_frameset_get_size(framesInOutChop)-1));
      	}
      	cpl_frameset_erase(framesInOutChop,"CAL_VIS2");
      }
      if (ibcd==7) {
      	size_before=cpl_frameset_get_size(framesOutInChop);
      	if (mat_cal_vis_lib(parlist, framesOutInChop,RECIPE_NAME)) {
      	  cpl_msg_error(cpl_func,"Error in mat_cal_vis_lib");
      	  return CPL_ERROR_ILLEGAL_OUTPUT;
      	}
      	size_after=cpl_frameset_get_size(framesOutInChop);
      	if (size_after > size_before+1) {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutInChop,cpl_frameset_get_size(framesOutInChop)-2));
      	} else {
      	  frame_calvis2=cpl_frame_duplicate(cpl_frameset_get_position(framesOutInChop,cpl_frameset_get_size(framesOutInChop)-1));
      	}
      	cpl_frameset_erase(framesOutInChop,"CAL_VIS2");
      }

      
      cpl_msg_info(cpl_func,"****************************************************");
      cpl_msg_info(cpl_func,"Starting mat_cal_cphase");
      cpl_msg_info(cpl_func,"****************************************************");
      if (ibcd==0) {
      	size_before=cpl_frameset_get_size(framesInIn);
      	if (mat_cal_cphase_lib(parlist, framesInIn,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInIn);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInIn,size_after-1));
      	  cpl_frameset_erase(framesInIn,"CAL_CPHASE");
      	}
      }
      if (ibcd==1) {
      	size_before=cpl_frameset_get_size(framesOutOut);
      	if (mat_cal_cphase_lib(parlist, framesOutOut,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutOut);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOut,size_after-1));
      	  cpl_frameset_erase(framesOutOut,"CAL_CPHASE");
      	}
      }
      if (ibcd==2) {
      	size_before=cpl_frameset_get_size(framesInOut);
      	if (mat_cal_cphase_lib(parlist, framesInOut,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInOut);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInOut,size_after-1));
      	  cpl_frameset_erase(framesInOut,"CAL_CPHASE");
      	}
      }
      if (ibcd==3) {
      	size_before=cpl_frameset_get_size(framesOutIn);
      	if (mat_cal_cphase_lib(parlist, framesOutIn,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutIn);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutIn,size_after-1));
      	  cpl_frameset_erase(framesOutIn,"CAL_CPHASE");
      	}
      }
      if (ibcd==4) {
      	size_before=cpl_frameset_get_size(framesInInChop);
      	if (mat_cal_cphase_lib(parlist, framesInInChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInInChop);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInInChop,size_after-1));
      	  cpl_frameset_erase(framesInInChop,"CAL_CPHASE");
      	}
      }
      if (ibcd==5) {
      	size_before=cpl_frameset_get_size(framesOutOutChop);
      	if (mat_cal_cphase_lib(parlist, framesOutOutChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutOutChop);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOutChop,size_after-1));
      	  cpl_frameset_erase(framesOutOutChop,"CAL_CPHASE");
      	}
      }
      if (ibcd==6) {
      	size_before=cpl_frameset_get_size(framesInOutChop);
      	if (mat_cal_cphase_lib(parlist, framesInOutChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInOutChop);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInOutChop,size_after-1));
      	  cpl_frameset_erase(framesInOutChop,"CAL_CPHASE");
      	}
      }
      if (ibcd==7) {
      	size_before=cpl_frameset_get_size(framesOutInChop);
      	if (mat_cal_cphase_lib(parlist, framesOutInChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_cphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutInChop);
      	if (size_after > size_before) {
      	  frame_calcphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutInChop,size_after-1));
      	  cpl_frameset_erase(framesOutInChop,"CAL_CPHASE");
      	}
      }

   
      cpl_msg_info(cpl_func,"****************************************************");
      cpl_msg_info(cpl_func,"Starting mat_cal_dphase");
      cpl_msg_info(cpl_func,"****************************************************");
      if (ibcd==0) {
      	size_before=cpl_frameset_get_size(framesInIn);
      	if (mat_cal_dphase_lib(parlist, framesInIn,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInIn);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInIn,size_after-1));
      	  cpl_frameset_erase(framesInIn,"CAL_DPHASE");
      	}
      }
      if (ibcd==1) {
      	size_before=cpl_frameset_get_size(framesOutOut);
      	if (mat_cal_dphase_lib(parlist, framesOutOut,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutOut);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOut,size_after-1));
      	  cpl_frameset_erase(framesOutOut,"CAL_DPHASE");
      	}
      }
      if (ibcd==2) {
      	size_before=cpl_frameset_get_size(framesInOut);
      	if (mat_cal_dphase_lib(parlist, framesInOut,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInOut);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInOut,size_after-1));
      	  cpl_frameset_erase(framesInOut,"CAL_DPHASE");
      	}
      }
      if (ibcd==3) {
      	size_before=cpl_frameset_get_size(framesOutIn);
      	if (mat_cal_dphase_lib(parlist, framesOutIn,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutIn);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutIn,size_after-1));
      	  cpl_frameset_erase(framesOutIn,"CAL_DPHASE");
      	}
      }
      if (ibcd==4) {
      	size_before=cpl_frameset_get_size(framesInInChop);
      	if (mat_cal_dphase_lib(parlist, framesInInChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInInChop);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInInChop,size_after-1));
      	  cpl_frameset_erase(framesInInChop,"CAL_DPHASE");
      	}
      }
      if (ibcd==5) {
      	size_before=cpl_frameset_get_size(framesOutOutChop);
      	if (mat_cal_dphase_lib(parlist, framesOutOutChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutOutChop);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutOutChop,size_after-1));
      	  cpl_frameset_erase(framesOutOutChop,"CAL_DPHASE");
      	}
      }
      if (ibcd==6) {
      	size_before=cpl_frameset_get_size(framesInOutChop);
      	if (mat_cal_dphase_lib(parlist, framesInOutChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesInOutChop);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesInOutChop,size_after-1));
      	  cpl_frameset_erase(framesInOutChop,"CAL_DPHASE");
      	}
      }
      if (ibcd==7) {
      	size_before=cpl_frameset_get_size(framesOutInChop);
      	if (mat_cal_dphase_lib(parlist, framesOutInChop,RECIPE_NAME)) {
      	  cpl_msg_warning(cpl_func,"Warning in mat_cal_dphase_lib");
      	}
      	size_after=cpl_frameset_get_size(framesOutInChop);
      	if (size_after > size_before) {
      	  frame_caldphase=cpl_frame_duplicate(cpl_frameset_get_position(framesOutInChop,size_after-1));
      	  cpl_frameset_erase(framesOutInChop,"CAL_DPHASE");
      	}
      }


      frame_result=cpl_frameset_new();
      cpl_frameset_insert(frame_result,frame_calvis2);
      if (frame_calcphase!=NULL) {
      	cpl_frameset_insert(frame_result,frame_calcphase);
      }
      if (frame_caldphase!=NULL) {
      	cpl_frameset_insert(frame_result,frame_caldphase);
      }




      // Adding names of calibrator in primary header
      int icpt=0;
      char *jsdcname=NULL;
      char *tplstart=NULL;
      char *keyname=NULL;
      char *keyvalue=NULL;
      char *p_filetag=NULL;
      cpl_propertylist *keycal=NULL;
	
      if (ibcd==0) //framesInIn
	{
	  it= cpl_frameset_iterator_new(framesInIn);
	}
      else if (ibcd==1) //framesOutOut
	{
	  it= cpl_frameset_iterator_new(framesOutOut);
	}
      else if (ibcd==2) //framesInOut
	{
	  it= cpl_frameset_iterator_new(framesInOut);
	}
      else if (ibcd==3) //framesOutIn
	{
	  it= cpl_frameset_iterator_new(framesOutIn);
	}
      else if (ibcd==4) //framesInInChop
	{
	  it= cpl_frameset_iterator_new(framesInInChop);
	}
      else if (ibcd==5) //framesOutOutChop
	{
	  it= cpl_frameset_iterator_new(framesOutOutChop);
	}
      else if (ibcd==6) //framesInOutChop
	{
	  it= cpl_frameset_iterator_new(framesInOutChop);
	}
      else if (ibcd==7) //framesOutInChop
	{
	  it= cpl_frameset_iterator_new(framesOutInChop);
	}

      keycal=cpl_propertylist_new();
      do {
	cur_frame = cpl_frameset_iterator_get(it);
	if (cur_frame != NULL) {
	  /*first file, references values*/
	  pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
	  p_filetag  = (char *)cpl_frame_get_tag(cur_frame);
	  if (!strcmp(p_filetag,"CALIB_RAW_INT"))
	    {
	      jsdcname=(char *)cpl_propertylist_get_string(pheader,"ESO PRO JSDC NAME");
	      tplstart=(char *)cpl_propertylist_get_string(pheader,"ESO TPL START");
	      keyvalue=cpl_sprintf("%s --- %s",jsdcname,tplstart);
	      keyname=cpl_sprintf("HIERARCH ESO PRO JSDC CAL%d",icpt+1);

	      cpl_propertylist_append_string(keycal, keyname, keyvalue);
	      
	      cpl_free(keyname);
	      cpl_free(keyvalue);
	      icpt++;
	    }
	  cpl_propertylist_delete(pheader);
	}
      } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
      cpl_frameset_iterator_delete(it);

      cpl_msg_info(cpl_func,"****************************************************");
      cpl_msg_info(cpl_func,"Starting mat_merge_results");
      cpl_msg_info(cpl_func,"****************************************************");
      if (mat_merge_results_lib(frame_result, parlist,RECIPE_NAME,keycal)) {
      	cpl_msg_error(cpl_func,"Error in mat_merge_results_lib");
      	return CPL_ERROR_ILLEGAL_OUTPUT;
      }
      cpl_propertylist_delete(keycal);

      // Adding RAW_SPECTRUM in the merged OIFITS
      size_after=cpl_frameset_get_size(frame_result);
      cur_frame=cpl_frameset_get_position(frame_result,size_after-1);
      char *filename=NULL;
      filename=(char *)cpl_frame_get_filename(cur_frame);
      mat_oispectrum_append(filename,oifits->oispect);
      cpl_frameset_join(frames,frame_result);


      /*  back to the good group */
      /* required for mat_cal_oifits otherwise these files are not moved in */
      /* the output-dir at the end of the recipes */
      it= cpl_frameset_iterator_new(frame_result);
      do {
      	cur_frame = cpl_frameset_iterator_get(it);
      	if (cur_frame != NULL) {
      	  pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
      	  if (strcmp(pszFileTag,"CAL_VIS2") == 0  ||
      	      strcmp(pszFileTag,"CAL_CPHASE") == 0 ||
      	      strcmp(pszFileTag,"CAL_DPHASE") == 0 ) {
      	    cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_PRODUCT);
      	  }
      	}
      } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
      cpl_frameset_iterator_delete(it);
      mat_oifits_delete(oifits);
      cpl_frameset_delete(frame_result);
    } else {
      if (ibcd == 0) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD IN-IN - No Chop");
      }
      if (ibcd == 1) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD OUT-OUT - No Chop");
      }
      if (ibcd == 2) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD IN-OUT - No Chop");
      }
      if (ibcd == 3) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD OUT-IN - No Chop");
      }
      if (ibcd == 4) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD IN-IN - Chop");
      }
      if (ibcd == 5) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD OUT-OUT - Chop");
      }
      if (ibcd == 6) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD IN-OUT - Chop");
      }
      if (ibcd == 7) {
	cpl_msg_warning(cpl_func,"OIFITS missing for BCD OUT-IN - Chop");
      }
    }

  }
  cpl_frameset_delete(framesInIn);
  cpl_frameset_delete(framesOutOut);
  cpl_frameset_delete(framesInOut);
  cpl_frameset_delete(framesOutIn);
  cpl_frameset_delete(framesInInChop);
  cpl_frameset_delete(framesOutOutChop);
  cpl_frameset_delete(framesInOutChop);
  cpl_frameset_delete(framesOutInChop);


  // BCD Calibration
  int BCD[4][6]={{0,1,2,3,4,5},  //OUT-OUT (0)
	   {0,1,4,5,2,3},  //OUT-IN  (1)
	   {0,1,3,2,5,4},  //IN-OUT  (2)
	   {0,1,5,4,3,2}};  //IN-IN  (3)
  int BCDsign[4][6]={{1,1,1,1,1,1},   //OUT-OUT (0)
	       {1,-1,1,1,1,1},  //OUT-IN  (1)
	       {-1,1,1,1,1,1},  //IN-OUT  (2) 
	       {-1,-1,1,1,1,1}}; //IN-IN  (3)
  int BCDcp[4][4]={{0, 1, 2, 3},  //OUT-OUT (0)
	     {3, 1, 2, 0},  //OUT-IN  (1)
	     {0, 2, 1, 3},  //IN-OUT  (2)
	     {3, 2, 1, 0}}; //IN-IN  (3)
  /* int uv1[4]={2,1,1,4};  */
  /* int uv2[4]={0,2,3,0}; */
  int BCDcpsign[4][4]={{ 1,  1,  1,  1},  //OUT-OUT (0)
		 {1,  -1,  -1, 1},  //OUT-IN  (1)
		 { -1, 1, 1,  -1},  //IN-OUT  (2)
		 {-1, -1, -1, -1}}; //IN-IN  (3)
  int BCDfluxL[4][4]={{0,1,3,2},
		{0,1,2,3},
		{1,0,3,2},
		{1,0,2,3}};
  int BCDfluxN[4][4]={{3,2,0,1},
		{3,2,1,0},
		{2,3,0,1},
		{2,3,1,0}};
  int uv1[4][4]={{2,4,3,5},{1,1,1,1},{1,1,1,1},{4,2,5,3}};
  int uv2[4][4]={{0,0,0,0},{2,4,3,5},{3,5,2,4},{0,0,0,0}};
    
  cpl_parameter	*cumulBlock = NULL;
  cumulBlock = cpl_parameterlist_find(parlist,"matisse.mat_merge_results.cumulBlock");
  if ( cpl_parameter_get_bool(cumulBlock) )
    {
      cpl_msg_info(cpl_func,"------ Entering BCD Calibration -----------");
      cpl_msg_info(cpl_func,"Data with chopping %d",flagChopData);
      mat_oifits *oifitsCalib=NULL;
      mat_oifits *oifitstemp=NULL;
      int ibcd=0;
      int nbobvis=0;
      int nbobphi=0;
      char *detName=NULL;
      double *phasorRePhi=NULL;
      double *phasorImPhi=NULL;
      double *phasorReT3=NULL;
      double *phasorImT3=NULL;
      cpl_matrix *vis22=NULL;
      double foo1=0.;
      cpl_matrix *visamp2=NULL;
      double foo2=0.;
      
      it= cpl_frameset_iterator_new(frames);
      do
	{
	  cur_frame = cpl_frameset_iterator_get(it);
	  if (cur_frame != NULL)
	    {
	      pszFileTag  = (char *)cpl_frame_get_tag(cur_frame);
	      if (!strcmp(pszFileTag,"TARGET_CAL_INT"))
		{
		  cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
		  pheader=cpl_propertylist_load(cpl_frame_get_filename(cur_frame),0);
		  if ( cpl_propertylist_has(pheader,"ESO ISS CHOP ST") )
		    {
		      chopMode = (char *) cpl_propertylist_get_string(pheader,"ESO ISS CHOP ST");
		    }
		  else
		    {
		      chopMode=cpl_sprintf("F");
		    }
		  
		  if (oifitsCalib==NULL)
		    {
		      oifitsCalib=mat_oifits_load(cur_frame);
		      phasorRePhi=cpl_malloc(oifitsCalib->oiwave->nbchannel*oifitsCalib->oivis->nbvis*sizeof(double));
		      phasorImPhi=cpl_malloc(oifitsCalib->oiwave->nbchannel*oifitsCalib->oivis->nbvis*sizeof(double));
		      phasorReT3=cpl_malloc(oifitsCalib->oiwave->nbchannel*oifitsCalib->oit3->nbt3*sizeof(double));
		      phasorImT3=cpl_malloc(oifitsCalib->oiwave->nbchannel*oifitsCalib->oit3->nbt3*sizeof(double));
		      vis22=cpl_matrix_new(oifitsCalib->oivis->nbvis,oifitsCalib->oivis2->nbchannel);
		      visamp2=cpl_matrix_new(oifitsCalib->oivis->nbvis,oifitsCalib->oivis2->nbchannel);
		      for(int ix=0;ix<oifitsCalib->oivis->nbvis;ix++)
			{
			  for(int jx=0;jx<oifitsCalib->oiwave->nbchannel;jx++)
			    {
			      phasorRePhi[jx+oifitsCalib->oiwave->nbchannel*ix]=0.;
			      phasorImPhi[jx+oifitsCalib->oiwave->nbchannel*ix]=0.;
			    }
			}
		      for(int ix=0;ix<oifitsCalib->oit3->nbt3;ix++)
			{
			  for(int jx=0;jx<oifitsCalib->oiwave->nbchannel;jx++)
			    {
			      phasorReT3[jx+oifitsCalib->oiwave->nbchannel*ix]=0.;
			      phasorImT3[jx+oifitsCalib->oiwave->nbchannel*ix]=0.;
			    }
			}

		      detName=(char *)cpl_propertylist_get_string(oifitsCalib->keywords,"ESO DET CHIP NAME");
		      // OIVIS2 set to zero relevant quantity
		      for(int ix=0;ix<oifitsCalib->oivis2->nbvis2;ix++)
			{
			  oifitsCalib->oivis2->list_vis2[ix]->time=0.;
			  oifitsCalib->oivis2->list_vis2[ix]->dateobsmjd=0.;
			  oifitsCalib->oivis2->list_vis2[ix]->exptime=0.;
			  oifitsCalib->oivis2->list_vis2[ix]->ucoord=0.;
			  oifitsCalib->oivis2->list_vis2[ix]->vcoord=0.;
			  for(int jx=0;jx<oifitsCalib->oivis2->nbchannel;jx++)
			    {
			      oifitsCalib->oivis2->list_vis2[ix]->vis2[jx]=0.;
			      oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]=0.;
			    }
			}
	  
		      // OIVIS set to zero relevant quantity
		      for(int ix=0;ix<oifitsCalib->oivis->nbvis;ix++)
			{
			  oifitsCalib->oivis->list_vis[ix]->time=0.;
			  oifitsCalib->oivis->list_vis[ix]->dateobsmjd=0.;
			  oifitsCalib->oivis->list_vis[ix]->exptime=0.;
			  oifitsCalib->oivis->list_vis[ix]->ucoord=0.;
			  oifitsCalib->oivis->list_vis[ix]->vcoord=0.;
			  for(int jx=0;jx<oifitsCalib->oivis->nbchannel;jx++)
			    {
			      oifitsCalib->oivis->list_vis[ix]->visamp[jx]=0.;
			      oifitsCalib->oivis->list_vis[ix]->visamperr[jx]=0.;
			      oifitsCalib->oivis->list_vis[ix]->visphi[jx]=0.;
			      oifitsCalib->oivis->list_vis[ix]->visphierr[jx]=0.;
			    }
			}

		      // OIT3 set to zero relevant quantity
		      for(int ix=0;ix<oifitsCalib->oit3->nbt3;ix++)
			{
			  oifitsCalib->oit3->list_t3[ix]->time=0.;
			  oifitsCalib->oit3->list_t3[ix]->dateobsmjd=0.;
			  oifitsCalib->oit3->list_t3[ix]->exptime=0.;
			  oifitsCalib->oit3->list_t3[ix]->u1coord=0.;
			  oifitsCalib->oit3->list_t3[ix]->v1coord=0.;
			  oifitsCalib->oit3->list_t3[ix]->u2coord=0.;
			  oifitsCalib->oit3->list_t3[ix]->v2coord=0.;
			  for(int jx=0;jx<oifitsCalib->oit3->nbchannel;jx++)
			    {
			      oifitsCalib->oit3->list_t3[ix]->t3phi[jx]=0.;
			      oifitsCalib->oit3->list_t3[ix]->t3phierr[jx]=0.;
			    }
			}

		      if (oifitsCalib->oispect != NULL)
			{
			  // OISPECTRUM set to zero relevant quantity
			  for(int ix=0;ix<oifitsCalib->oispect->nbflux;ix++)
			    {
			      oifitsCalib->oispect->list_flux[ix]->time=0.;
			      oifitsCalib->oispect->list_flux[ix]->dateobsmjd=0.;
			      oifitsCalib->oispect->list_flux[ix]->exptime=0.;
			      for(int jx=0;jx<oifitsCalib->oispect->nbchannel;jx++)
				{
				  oifitsCalib->oispect->list_flux[ix]->fluxdata[jx]=0.;
				  oifitsCalib->oispect->list_flux[ix]->fluxerr[jx]=0.;
				}
			    }
			}

		    }

		  oifitstemp=mat_oifits_load(cur_frame);
		  bcd1Status = (char *) cpl_propertylist_get_string(oifitstemp->keywords,"ESO CFG BCD MODE");
		  if (!strcmp(bcd1Status,"OUT-OUT"))
		    {
		      ibcd=0;
		    }
		  else if (!strcmp(bcd1Status,"OUT-IN"))
		    {
		      ibcd=1;
		    }
		  else if (!strcmp(bcd1Status,"IN-OUT"))
		    {
		      ibcd=2;
		    }
		  else if (!strcmp(bcd1Status,"IN-IN"))
		    {
		      ibcd=3;
		    }
		  // OIVIS2
		  if (!strcmp(detName, "AQUARIUS") || flagChopData==0 || (flagChopData==1 && !strcmp(chopMode,"T")) )
		    {
		      nbobvis+=1;
		      for(int ix=0;ix<oifitsCalib->oivis2->nbvis2;ix++)
			{
			  oifitsCalib->oivis2->list_vis2[ix]->time+=oifitstemp->oivis2->list_vis2[ix]->time;
			  oifitsCalib->oivis2->list_vis2[ix]->dateobsmjd+=oifitstemp->oivis2->list_vis2[ix]->dateobsmjd;
			  oifitsCalib->oivis2->list_vis2[ix]->exptime+=oifitstemp->oivis2->list_vis2[ix]->exptime;
			  oifitsCalib->oivis2->list_vis2[ix]->ucoord+=BCDsign[ibcd][ix]*oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->ucoord;
			  oifitsCalib->oivis2->list_vis2[ix]->vcoord+=BCDsign[ibcd][ix]*oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->vcoord;
			  for(int jx=0;jx<oifitsCalib->oivis2->nbchannel;jx++)
			    {
			      oifitsCalib->oivis2->list_vis2[ix]->vis2[jx]+=oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->vis2[jx];
			      oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]+=pow(oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->vis2err[jx],2.0);
			      foo1=cpl_matrix_get(vis22,ix,jx)+pow(oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->vis2[jx],2.0);
			      cpl_matrix_set(vis22,ix,jx,foo1);
			    }
			  if (ibcd==0)
			    {
			      oifitsCalib->oivis2->list_vis2[ix]->stationindex[0]=oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->stationindex[0];
			      oifitsCalib->oivis2->list_vis2[ix]->stationindex[1]=oifitstemp->oivis2->list_vis2[BCD[ibcd][ix]]->stationindex[1];
			    }
			}
		    }
		  // OIVIS
		  if (!strcmp(detName, "AQUARIUS") || flagChopData==0 || (flagChopData==1 && !strcmp(chopMode,"T")) )
		    {
		      for(int ix=0;ix<oifitsCalib->oivis->nbvis;ix++)
			{
			  oifitsCalib->oivis->list_vis[ix]->time+=oifitstemp->oivis->list_vis[ix]->time;
			  oifitsCalib->oivis->list_vis[ix]->dateobsmjd+=oifitstemp->oivis->list_vis[ix]->dateobsmjd;
			  oifitsCalib->oivis->list_vis[ix]->exptime+=oifitstemp->oivis->list_vis[ix]->exptime;
			  oifitsCalib->oivis->list_vis[ix]->ucoord+=BCDsign[ibcd][ix]*oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->ucoord;
			  oifitsCalib->oivis->list_vis[ix]->vcoord+=BCDsign[ibcd][ix]*oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->vcoord;
			  for(int jx=0;jx<oifitsCalib->oivis->nbchannel;jx++)
			    {
			      oifitsCalib->oivis->list_vis[ix]->visamp[jx]+=oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visamp[jx];
			      oifitsCalib->oivis->list_vis[ix]->visamperr[jx]+=pow(oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visamperr[jx],2.0);
			      phasorRePhi[jx+oifitsCalib->oivis->nbchannel*ix] += cos(BCDsign[ibcd][ix]*oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visphi[jx]*M_PI/180.);
			      phasorImPhi[jx+oifitsCalib->oivis->nbchannel*ix] += sin(BCDsign[ibcd][ix]*oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visphi[jx]*M_PI/180.);
			      //oifitsCalib->oivis->list_vis[ix]->visphi[jx]+=BCDsign[ibcd][ix]*oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visphi[jx];
			      oifitsCalib->oivis->list_vis[ix]->visphierr[jx]+=pow(oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visphierr[jx],2.0);
			      foo2=cpl_matrix_get(visamp2,ix,jx)+pow(oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->visamp[jx],2.0);
			      cpl_matrix_set(visamp2,ix,jx,foo2);
			    }
			  if (ibcd==0)
			    {
			      oifitsCalib->oivis->list_vis[ix]->stationindex[0]=oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->stationindex[0];
			      oifitsCalib->oivis->list_vis[ix]->stationindex[1]=oifitstemp->oivis->list_vis[BCD[ibcd][ix]]->stationindex[1];
			    }
			}
		    }
		  // OIT3
		  if (!strcmp(detName, "AQUARIUS") || flagChopData==0 || (flagChopData==1 && !strcmp(chopMode,"F")) )
		    {
		      nbobphi+=1;
		      for(int ix=0;ix<oifitsCalib->oit3->nbt3;ix++)
			{
			  oifitsCalib->oit3->list_t3[ix]->time+=oifitstemp->oit3->list_t3[ix]->time;
			  oifitsCalib->oit3->list_t3[ix]->dateobsmjd+=oifitstemp->oit3->list_t3[ix]->dateobsmjd;
			  oifitsCalib->oit3->list_t3[ix]->exptime+=oifitstemp->oit3->list_t3[ix]->exptime;
			  oifitsCalib->oit3->list_t3[ix]->u1coord+=BCDsign[ibcd][uv1[ix][ibcd]]*oifitstemp->oivis2->list_vis2[uv1[ix][ibcd]]->ucoord;
			  oifitsCalib->oit3->list_t3[ix]->v1coord+=BCDsign[ibcd][uv1[ix][ibcd]]*oifitstemp->oivis2->list_vis2[uv1[ix][ibcd]]->vcoord;
			  oifitsCalib->oit3->list_t3[ix]->u2coord+=BCDsign[ibcd][uv2[ix][ibcd]]*oifitstemp->oivis2->list_vis2[uv2[ix][ibcd]]->ucoord;
			  oifitsCalib->oit3->list_t3[ix]->v2coord+=BCDsign[ibcd][uv2[ix][ibcd]]*oifitstemp->oivis2->list_vis2[uv2[ix][ibcd]]->vcoord;
			  /* oifitsCalib->oit3->list_t3[ix]->u1coord+=BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[uv1[ix]]->u1coord; */
			  /* oifitsCalib->oit3->list_t3[ix]->v1coord+=BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[uv1[ix]]->v1coord; */
			  /* oifitsCalib->oit3->list_t3[ix]->u2coord+=BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[uv2[ix]]->u2coord; */
			  /* oifitsCalib->oit3->list_t3[ix]->v2coord+=BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[uv2[ix]]->v2coord; */
			  for(int jx=0;jx<oifitsCalib->oit3->nbchannel;jx++)
			    {
			      phasorReT3[jx+oifitsCalib->oit3->nbchannel*ix] += cos(BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->t3phi[jx]*M_PI/180.);
			      phasorImT3[jx+oifitsCalib->oit3->nbchannel*ix] += sin(BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->t3phi[jx]*M_PI/180.);
			      //oifitsCalib->oit3->list_t3[ix]->t3phi[jx]+=BCDcpsign[ibcd][ix]*oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->t3phi[jx];
			      oifitsCalib->oit3->list_t3[ix]->t3phierr[jx]+=pow(oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->t3phierr[jx],2.0);
			    }
			  if (ibcd==0)
			    {
			      oifitsCalib->oit3->list_t3[ix]->stationindex[0]=oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->stationindex[0];
			      oifitsCalib->oit3->list_t3[ix]->stationindex[1]=oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->stationindex[1];
			      oifitsCalib->oit3->list_t3[ix]->stationindex[2]=oifitstemp->oit3->list_t3[BCDcp[ibcd][ix]]->stationindex[2];
			    }
			}
		    }
		  
		  if (oifitsCalib->oispect != NULL)
		    {
		      // OIFLUX
		      if (!strcmp(detName, "AQUARIUS") || flagChopData==0 || (flagChopData==1 && !strcmp(chopMode,"T")) )
			{
			  for(int ix=0;ix<oifitsCalib->oispect->nbflux;ix++)
			    {
			      oifitsCalib->oispect->list_flux[ix]->time+=oifitstemp->oispect->list_flux[ix]->time;
			      oifitsCalib->oispect->list_flux[ix]->dateobsmjd+=oifitstemp->oispect->list_flux[ix]->dateobsmjd;
			      oifitsCalib->oispect->list_flux[ix]->exptime+=oifitstemp->oispect->list_flux[ix]->exptime;
			      if (!strcmp(detName, "AQUARIUS"))
				{
				  for(int jx=0;jx<oifitsCalib->oispect->nbchannel;jx++)
				    {
				      oifitsCalib->oispect->list_flux[ix]->fluxdata[jx]+=oifitstemp->oispect->list_flux[BCDfluxN[ibcd][ix]]->fluxdata[jx];
				      oifitsCalib->oispect->list_flux[ix]->fluxerr[jx]+=pow(oifitstemp->oispect->list_flux[BCDfluxN[ibcd][ix]]->fluxerr[jx],2.0);
				    }
				}
			      else
				{
				  for(int jx=0;jx<oifitsCalib->oispect->nbchannel;jx++)
				    {
				      oifitsCalib->oispect->list_flux[ix]->fluxdata[jx]+=oifitstemp->oispect->list_flux[BCDfluxL[ibcd][ix]]->fluxdata[jx];
				      oifitsCalib->oispect->list_flux[ix]->fluxerr[jx]+=pow(oifitstemp->oispect->list_flux[BCDfluxL[ibcd][ix]]->fluxerr[jx],2.0);
				    }
				}
			      if (ibcd==0)
				{
				  oifitsCalib->oispect->list_flux[ix]->stationindex=oifitstemp->oispect->list_flux[ix]->stationindex;
				}		    
			    }
			}
		    }
		  cpl_propertylist_delete(pheader);
		  mat_oifits_delete(oifitstemp);
		}
	    }
	} while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
      cpl_frameset_iterator_delete(it);

      
      // Normalization

      // OIVIS2
      for(int ix=0;ix<oifitsCalib->oivis2->nbvis2;ix++)
	{
	  oifitsCalib->oivis2->list_vis2[ix]->time/=nbobvis;
	  oifitsCalib->oivis2->list_vis2[ix]->dateobsmjd/=nbobvis;
	  oifitsCalib->oivis2->list_vis2[ix]->ucoord/=nbobvis;
	  oifitsCalib->oivis2->list_vis2[ix]->vcoord/=nbobvis;
	  for(int jx=0;jx<oifitsCalib->oivis2->nbchannel;jx++)
	    {
	      oifitsCalib->oivis2->list_vis2[ix]->vis2[jx]/=nbobvis;
	      oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]=(oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]+cpl_matrix_get(vis22,ix,jx))/nbobvis;
	      oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]-=pow(oifitsCalib->oivis2->list_vis2[ix]->vis2[jx],2.0);
	      oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]=sqrt(oifitsCalib->oivis2->list_vis2[ix]->vis2err[jx]);
	    }
	}
      // OIVIS
      for(int ix=0;ix<oifitsCalib->oivis->nbvis;ix++)
	{
	  oifitsCalib->oivis->list_vis[ix]->time/=nbobvis;
	  oifitsCalib->oivis->list_vis[ix]->dateobsmjd/=nbobvis;
	  oifitsCalib->oivis->list_vis[ix]->ucoord/=nbobvis;
	  oifitsCalib->oivis->list_vis[ix]->vcoord/=nbobvis;
	  for(int jx=0;jx<oifitsCalib->oivis->nbchannel;jx++)
	    {
	      oifitsCalib->oivis->list_vis[ix]->visamp[jx]/=nbobvis;
	      oifitsCalib->oivis->list_vis[ix]->visamperr[jx]=(oifitsCalib->oivis->list_vis[ix]->visamperr[jx]+cpl_matrix_get(visamp2,ix,jx))/(nbobvis);
	      oifitsCalib->oivis->list_vis[ix]->visamperr[jx]-=pow(oifitsCalib->oivis->list_vis[ix]->visamp[jx],2.0);
	      oifitsCalib->oivis->list_vis[ix]->visamperr[jx]=sqrt(oifitsCalib->oivis->list_vis[ix]->visamperr[jx]);
	      oifitsCalib->oivis->list_vis[ix]->visphi[jx]=(180./M_PI)*atan2(phasorImPhi[jx+oifitsCalib->oivis->nbchannel*ix],phasorRePhi[jx+oifitsCalib->oivis->nbchannel*ix]);
	      //oifitsCalib->oivis->list_vis[ix]->visphi[jx]/=nbobvis;
	      oifitsCalib->oivis->list_vis[ix]->visphierr[jx]=sqrt(oifitsCalib->oivis->list_vis[ix]->visphierr[jx])/sqrt(nbobvis);
	    }
	}
      // OIT3
      for(int ix=0;ix<oifitsCalib->oit3->nbt3;ix++)
	{
	  oifitsCalib->oit3->list_t3[ix]->time/=nbobphi;
	  oifitsCalib->oit3->list_t3[ix]->dateobsmjd/=nbobphi;
	  oifitsCalib->oit3->list_t3[ix]->u1coord/=nbobphi;
	  oifitsCalib->oit3->list_t3[ix]->v1coord/=nbobphi;
	  oifitsCalib->oit3->list_t3[ix]->u2coord/=nbobphi;
	  oifitsCalib->oit3->list_t3[ix]->v2coord/=nbobphi;
	  for(int jx=0;jx<oifitsCalib->oit3->nbchannel;jx++)
	    {
	      oifitsCalib->oit3->list_t3[ix]->t3phi[jx]=(180./M_PI)*atan2(phasorImT3[jx+oifitsCalib->oit3->nbchannel*ix],phasorReT3[jx+oifitsCalib->oit3->nbchannel*ix]);
	      //oifitsCalib->oit3->list_t3[ix]->t3phi[jx]/=nbobphi;
	      oifitsCalib->oit3->list_t3[ix]->t3phierr[jx]=sqrt(oifitsCalib->oit3->list_t3[ix]->t3phierr[jx])/sqrt(nbobphi);
	    }
	}
      if (oifitsCalib->oispect != NULL)
	{
	  // OIFLUX   
	  for(int ix=0;ix<oifitsCalib->oispect->nbflux;ix++)
	    {
	      oifitsCalib->oispect->list_flux[ix]->time/=nbobvis;
	      oifitsCalib->oispect->list_flux[ix]->dateobsmjd/=nbobvis;
	      for(int jx=0;jx<oifitsCalib->oispect->nbchannel;jx++)
		{
		  oifitsCalib->oispect->list_flux[ix]->fluxdata[jx]/=nbobvis;
		  oifitsCalib->oispect->list_flux[ix]->fluxerr[jx]=sqrt(oifitsCalib->oispect->list_flux[ix]->fluxerr[jx])/sqrt(nbobvis);
		}
	    }
	}
      cpl_free(phasorRePhi);
      cpl_free(phasorImPhi);
      cpl_free(phasorReT3);
      cpl_free(phasorImT3);
      cpl_matrix_delete(vis22);
      cpl_matrix_delete(visamp2);

      // Save final oifits
      cpl_propertylist_update_bool(oifitsCalib->keywords, "ESO DRS BCD COMBINED", 1);
      mat_oifits_save(oifitsCalib,frames,frames,parlist,RECIPE_NAME,"TARGET_CAL_INT_noBCD.fits",9);      
      mat_oifits_delete(oifitsCalib);
      cpl_msg_info(cpl_func,"------ End of BCD Calibration -----------");
    }


  
  it= cpl_frameset_iterator_new(frames);
  do
    {
      cur_frame = cpl_frameset_iterator_get(it);
      if (cur_frame != NULL)
	{
	  pszFileTag  = (char *)cpl_frame_get_tag(cur_frame);
	  if (!strcmp(pszFileTag,"TARGET_CAL_INT"))
	    {
	      cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_PRODUCT);
	    }
	}
    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  it= cpl_frameset_iterator_new(frames);
  return 0;
}
