/* $Id: mat_est_aphase.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2016/04/11 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_aphase.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MATISSE_LICENCE         "GPL"

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <complex.h>
#include <stdio.h>

#include <cpl.h>

#include "mat_error.h"
#include "mat_oifits.h"
#include "mat_utils.h"
#include "mat_image.h"

/*-----------------------------------------------------------------------------
  Defines
  -----------------------------------------------------------------------------*/
#define MAT_DO_TARGET_CAL_INT   "TARGET_CAL_INT"
#define MAT_DO_REFERENCE_IMAGE  "REFERENCE_IMAGE"

#define MAT_MODE_IMAGE      0
#define MAT_MODE_STAR       1

#define MAT_NB_REFERENCE    256
#define MAT_NB_WINDOW       256

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/

typedef struct {
  double      lfrom; /*!< Shortest wavelength for the input data in [um]. [1.0] */
  double      lto;   /*!< Longest wavelength for the input data in [um]. [15.0] */
  cpl_image  *img;   /*!< This cpl image contains the loaded and rescaled reference image. */
  cpl_image  *fimg;  /*!< The FFT of the scaled reference image (double complex). */
} mat_reference_image;

typedef struct {
  double      lfrom;     /*!< Shortest wavelength of the window in [um]. [1.0] */
  double      lto;       /*!< Longest wavelength of the window in [um]. [15.0] */
  double      n2pi;      /*!< A 2PI phase offset count. */
  //double      n2pi_best; /*!< The best 2PI phase offset count. */
} mat_lambda_window;

/**
   @brief This data strucure is a replacement for global variables needed for the absolute phase estimation.

   This data structure contain all plugin parameters for easy access. Also temporary information needed for loading the calibrated
   interferometric data (e.g. factors for lambda scaling) is stored in such a data structure. All images and arrays needed for
   one absolute phase estimation run are stored.
*/
typedef struct {
  cpl_parameterlist *parlist;    /*!< Easy access to the plugin parameter list. */
  cpl_frameset      *frameset;   /*!< Easy access to the plugin frameset. */
  // ***** size of the reconstruction (number of pixels and field of view)
  double          FOV;           /*!< Field of view for the reconstructed image in [mas]. [40.0] */
  int             npix;          /*!< Size of the reconstructed image in pixels. Powers of 2 should be used (speeds up the FFT), but this is not enforced. [256] */
  double          dxrek;         /*!< pixel size of the reconstruction [mas], calculated as FOV/npix. */
  // ***** wavelength filter and uv-scaling
  char           *kwd_lambda_min;
  char           *kwd_lambda_max;
  char           *kwd_lambda_center;
  char           *kwd_lambda_width;
  //int             lambda_count;
  double          lambda_scale;
  cpl_vector     *lambda_list;   /*!< This vector contains a list of wavelength from the current OI_WAVELENGTH binary table. */
  cpl_vector     *pixel_scale;  /*!< Scale factors for baseline -> pixel. */
  cpl_vector     *lambda_image;  /*!< Relation wavelength -> reference image. */
  cpl_vector     *lambda_window; /*!< Relation wavelength -> wavelength window index or -1. */
  // ***** reference images for wavelength ranges
  int                  nref;                   /*!< Number of reference images. */
  mat_reference_image  refs[MAT_NB_REFERENCE]; /*!< reference images. */
  int                  nwin;                   /*!< Number of wavelength windows. */
  mat_lambda_window    wins[MAT_NB_WINDOW];    /*!< Wavelength windows (continous lambda ranges with reference windows). */
  // ***** The necessary reference image of the target created by using an image reconstruction with the bispectrum method
  int             ref_mode;     /*!< The mode for reading/creating the reference image. 0 = read from file, 1 = point source. [0] */
  double          ref_param;    /*!< Additional parameter for the reference image creation (mode=0 -> scale [mas/px]). */
  // ***** The original OI-Fits file with the differential phase and the new OI-Fits file with the absolute phase
  char           *fname;         /*!< The file name of the first OI-FITS file on the SOF. */
  mat_oifits     *oifits;        /*!< The calibrated interferometric data (input = with differential phase, output = absolute phase). */
  double          phi_error;     /*!< Differential phase error used instead of the measured error (0.0 -> measured error). */
  char            vis_name[256]; /*!< ASCII file for the differential and absolute phases. [] */
  // ***** Temporary variables which are used inside a loop ofer all T3 triplets
  cpl_vector     *phim;          /*!< Differential phase difference for baseline 1 (u). */
  cpl_vector     *phie;          /*!< Error of the differential phase for baseline 1 (u). */
  cpl_vector     *phir;          /*!< Reconstructed phase difference for baseline 1 (u). */
  cpl_vector     *phid;          /*!< Phase difference for baseline 1 (u). */
  cpl_vector     *wn;            /*!< Wave number 1/lambda. */
  int             best_valid;
  double          best_offset;
  double          best_slope;
  double          best_chi2;
} mat_est_aphase_info;

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_est_aphase_create(cpl_plugin *);
static int mat_est_aphase_exec(cpl_plugin *);
static int mat_est_aphase_destroy(cpl_plugin *);
static int mat_est_aphase(cpl_parameterlist *, cpl_frameset *);

/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static char mat_est_aphase_description[] =
  "This plugin uses one or several reference images to estimate the absolute phase from the differential "
  "phases given in an OI-FITS file.\n"
  "\n"
  "It supports the following image reconstruction workflow:\n"
  "\n"
  "  1. Calculate one or more reference images using the mat_cal_imarec plugin and the bispectrum image reconstruction method.\n"
  "  2. Estimate the absolute phase from the differential phase and these reference images using the mat_est_aphase plugin.\n"
  "  3. Reconstruct the final image or images using the mat_cal_imarec plugin and the complex visibility image reconstruction method.\n"
  "\n"
  "The following pseudocode illustrates the overall scheme of this plugin:\n"
  "\n"
  "BEGIN\n"
  "   Read all reference images and map them into the internal mas to pixel space;\n"
  "   Read the OI-FITS file with the differential phase;\n"
  "   FOR EACH differential visibility baseline DO\n"
  "      Calculate the difference between the differential phase and the reconstructed phase from the reference images;\n"
  "      Fit a straight line to these phase differences;\n"
  "      The absolute phase is equal to differential phase minus the phase difference from the fit;\n"
  "   ENDFOR\n"
  "   Store the result (differential phases replaced by the absolute phase) in an OI-FITS file;\n"
  "END\n"
  "\n"
  "In the SOF file, the calibrated interferometric data files are classified as TARGET_CAL_INT.\n"
  "\n"
  "In the SOF file the reference images are classified as REFERENCE_IMAGE. From each of these "
  "files, the images in the primary header unit (if a cube is present all image planes are used) "
  "are read in as reference images. The order of the files in the SOF file is important, because "
  "it is mapped to the sequence of wavelength windows given with the --lambda_list option.\n"
  "\n"
  "It is possible to write an ASCII file (option --vis_name) which will contain:\n"
  "  - for each wavelength the following line will be created:\n"
  "    lambda <i>  = <wavelength [m]> reference = <reference image or -1> window = <reference window or -1> scale m -> pixel = <m to pixel factor>\n"
  "  - for each baseline the following blocks will be created:\n"
  "     + for each reference image the following line will be created:\n"
  "       ref <datemjd> <ucoord [m]> <vcoord [m]> <lambda [m]> <reconstructed phase [rad]> <phase difference [rad]>\n"
  "     + a line for the fit of the straight line to the phase differences:\n"
  "       # fit ref <datemjd> <ucoord [m]> <vcoord [m]> offset <offset value [rad]> slope <slope value [rad] chi2 <chi2 value>\n"
  "     + for each differential phase for the baseline the following line will be created:\n"
  "       phase <datemjd> <ucoord [m]> <vcoord [m]> <lambda [m]> <measured differential phase [rad]> <phase error [rad]> <fitted phase difference [rad]>\n"
  "\n"
  "An option (--phi_error) enables the use of the measured differential phase errors (--phi_error=0.0, default) "
  "or an artificial (fixed) phase error (for example 1.0).\n"
  "\n"
  "Input files:\n"
  "\n"
  "  DO category:      Explanation:                       Required:\n"
  "  REFERENCE_IMAGE   Reference image(s)                 Yes\n"
  "  TARGET_CAL_INT    Calibrated interferometric data    Yes\n"
  "                    (without absolute phases)\n"
  "\n"
  "Output files:\n"
  "\n"
  "  DO category:      Explanation:\n"
  "  TARGET_CAL_INT    Calibrated interferometric data\n"
  "                    (with absolute phases)"
  ;

/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
int cpl_plugin_get_info(cpl_pluginlist * list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
  cpl_plugin *plugin = &recipe->interface;

  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_est_aphase",
		  "Estimates the absolute phase from reference images and differential phases",
		  mat_est_aphase_description,
		  "Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  MATISSE_LICENCE,
		  mat_est_aphase_create,
		  mat_est_aphase_exec,
		  mat_est_aphase_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
static int mat_est_aphase_create(cpl_plugin * plugin)
{
  cpl_recipe    *recipe;
  cpl_parameter *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */

  /* --fov */
  p = cpl_parameter_new_range("matisse.mat_est_aphase.fov",
			      CPL_TYPE_DOUBLE,
			      "Field of view for the internal image in [mas].",
			      "matisse.mat_est_aphase",
			      40.0, 1.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fov") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --npix */
  p = cpl_parameter_new_value("matisse.mat_est_aphase.npix",
			      CPL_TYPE_INT,
			      "Size of the internal image in pixels. "
			      "Powers of 2 should be used (speeds up the FFT), "
			      "but this is not enforced.",
			      "matisse.mat_est_aphase",
			      256);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "npix") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --kwd_lambda_min */
  p = cpl_parameter_new_value("matisse.mat_est_aphase.kwd_lambda_min",
			      CPL_TYPE_STRING,
			      "Keyword name for the lower wavelength stored in the reference image file",
			      "matisse.mat_est_aphase",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kwd_lambda_min");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --kwd_lambda_max*/
  p = cpl_parameter_new_value("matisse.mat_est_aphase.kwd_lambda_max",
			      CPL_TYPE_STRING,
			      "Keyword name for the upper wavelength stored in the reference image file",
			      "matisse.mat_est_aphase",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kwd_lambda_max");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --kwd_lambda_center*/
  p = cpl_parameter_new_value("matisse.mat_est_aphase.kwd_lambda_center",
			      CPL_TYPE_STRING,
			      "Keyword name for the central wavelength stored in the reference image file",
			      "matisse.mat_est_aphase",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kwd_lambda_center");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --kwd_lambda_width*/
  p = cpl_parameter_new_value("matisse.mat_est_aphase.kwd_lambda_width",
			      CPL_TYPE_STRING,
			      "Keyword name for the band width stored in the reference image file",
			      "matisse.mat_est_aphase",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kwd_lambda_width");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --lambda_unit*/
  p = cpl_parameter_new_value("matisse.mat_est_aphase.lambda_unit",
			      CPL_TYPE_STRING,
			      "Unit used for the wavelength parameter and keyword values [m, mm, um, nm]",
			      "matisse.mat_est_aphase",
			      "um");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambda_unit");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --lambda_list */
  p = cpl_parameter_new_value("matisse.mat_est_aphase.lambda_list",
			      CPL_TYPE_STRING,
			      "A list of lambda ranges (pairs of lower and upper wavelength). "
			      "It is a sequence of comma separated floating point numbers or \'none\'. ",
			      "matisse.mat_est_aphase",
			      "none");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambda_list");
  cpl_parameterlist_append(recipe->parameters, p);

  /* --ref_mode */
  p = cpl_parameter_new_range("matisse.mat_est_aphase.ref_mode",
			      CPL_TYPE_INT,
			      "The mode for reading/creating the reference image. "
			      "0 = read from file, 1 = point source.",
			      "matisse.mat_est_aphase",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ref_mode") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --ref_param */
  p = cpl_parameter_new_range("matisse.mat_est_aphase.ref_param",
			      CPL_TYPE_DOUBLE,
			      "Additional parameter for the reference image creation "
			      "(mode=0 -> scale [mas/px]).",
			      "matisse.mat_est_aphase",
			      0.0, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ref_param") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --phi_error */
  p = cpl_parameter_new_range("matisse.mat_est_aphase.phi_error",
			      CPL_TYPE_DOUBLE,
			      "Differential phase error used instead of the measured error "
			      "(phi_error=0.0 -> use measured error [rad]).",
			      "matisse.mat_est_aphase",
			      0.0, 0.0, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "phi_error") ;
  cpl_parameterlist_append(recipe->parameters, p) ;

  /* --vis_name */
  p = cpl_parameter_new_value("matisse.mat_est_aphase.vis_name",
			      CPL_TYPE_STRING,
			      "ASCII file for differential and absolute phases.",
			      "matisse.mat_cal_imarec",
			      "");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "vis_name");
  cpl_parameterlist_append(recipe->parameters, p);

  /* Return */
  return 0;
}


/**
   @brief Execute the plugin instance given by the interface
   @param plugin the plugin
   @return 0 if everything is ok
*/
static int mat_est_aphase_exec(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_est_aphase(recipe->parameters, recipe->frames);
}

/**
   @brief Destroy what has been created by the create function
   @param plugin the plugin
   @return 0 if everything is ok
*/
static int mat_est_aphase_destroy(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}

/*--------------------------------------------------------------------*/
/* mat_est_aphase_info related functions (init, destroy, ...)         */
/*--------------------------------------------------------------------*/
/**
   @brief Initializes the data structure holding all global variables.
   @param info   This data structure contains the image reconstruction context (a replacement for global variables).
*/
static void mat_info_init(mat_est_aphase_info *info)
{
  int w, r;

  info->kwd_lambda_min = NULL;
  info->kwd_lambda_max = NULL;
  info->kwd_lambda_center = NULL;
  info->kwd_lambda_width = NULL;
  info->lambda_list = NULL;
  info->lambda_window = NULL;
  info->lambda_image = NULL;
  info->pixel_scale = NULL;
  for (r = 0; r < MAT_NB_REFERENCE; r++)
    {
      info->refs[r].img  = NULL;
      info->refs[r].fimg = NULL;
    }
  for (w = 0; w < MAT_NB_WINDOW; w++)
    {
      info->wins[w].n2pi      = 0.0;
      //info->wins[w].n2pi_best = 0.0;
    }
  info->fname = NULL;
  info->oifits = NULL;
  info->vis_name[0] = '\0';
  info->phim  = NULL; /*!< Phase difference for baseline 1 (u). */
  info->phie  = NULL; /*!< Phase error for baseline 1 (u). */
  info->phir  = NULL; /*!< Phase difference for baseline 1 (u). */
  info->phid  = NULL; /*!< Phase difference for baseline 1 (u). */
  info->wn    = NULL; /*!< Wave number 1/lambda. */
}

/**
   @brief This function frees all allocated memory from an mat_est_aphase_info data structure.
   @param info   This data structure contains the image reconstruction context (a replacement for global variables).
*/
static void mat_info_delete(mat_est_aphase_info *info)
{
  int r;

  if (info->kwd_lambda_min != NULL)
    {
      cpl_free(info->kwd_lambda_min);
      info->kwd_lambda_min = NULL;
    }
  if (info->kwd_lambda_max != NULL)
    {
      cpl_free(info->kwd_lambda_max);
      info->kwd_lambda_max = NULL;
    }
  if (info->kwd_lambda_center != NULL)
    {
      cpl_free(info->kwd_lambda_center);
      info->kwd_lambda_center = NULL;
    }
  if (info->kwd_lambda_width != NULL)
    {
      cpl_free(info->kwd_lambda_width);
      info->kwd_lambda_width = NULL;
    }
  if (info->lambda_list != NULL)
    {
      cpl_vector_delete(info->lambda_list);
      info->lambda_list = NULL;
    }
  if (info->lambda_window != NULL)
    {
      cpl_vector_delete(info->lambda_window);
      info->lambda_window = NULL;
    }
  if (info->lambda_image != NULL)
    {
      cpl_vector_delete(info->lambda_image);
      info->lambda_image = NULL;
    }
  if (info->pixel_scale != NULL)
    {
      cpl_vector_delete(info->pixel_scale);
      info->pixel_scale = NULL;
    }
  for (r = 0; r < MAT_NB_REFERENCE; r++)
    {
      if (info->refs[r].img != NULL)
	{
	  cpl_image_delete(info->refs[r].img);
	  info->refs[r].img = NULL;
	}
      if (info->refs[r].fimg != NULL)
	{
	  cpl_image_delete(info->refs[r].fimg);
	  info->refs[r].fimg = NULL;
	}
    }
  if (info->fname != NULL)
    {
      cpl_free(info->fname);
      info->fname = NULL;
    }
  if (info->oifits != NULL)
    {
      mat_oifits_delete(info->oifits);
      info->oifits = NULL;
    }
  if (info->phim != NULL)
    {
      cpl_vector_delete(info->phim);
      info->phim = NULL;
    }
  if (info->phie != NULL)
    {
      cpl_vector_delete(info->phie);
      info->phie = NULL;
    }
  if (info->phir != NULL)
    {
      cpl_vector_delete(info->phir);
      info->phir = NULL;
    }
  if (info->phid != NULL)
    {
      cpl_vector_delete(info->phid);
      info->phid = NULL;
    }
  if (info->wn != NULL)
    {
      cpl_vector_delete(info->wn);
      info->wn = NULL;
    }
}

/**
   @brief Gets the lambda list from a parameter string
   @param info  A mat_est_aphase_info data structure as a substitute for global variables.
   @param parlist A cpl_parameterlist containing the plugin parameters.
   @return 1 if at least ine lambda filter range was read from the --lambda_list parameter.
*/
static int mat_get_lambda_list(mat_est_aphase_info  *info,
			       cpl_parameterlist *parlist)
{
  cpl_parameter    *param = NULL;
  const char       *arg = NULL;
  int               i, j;
  char              str[32];

  info->nref = 0;
  if (parlist == NULL)
    {
      return 0;
    }
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.lambda_list");
  if (param == NULL)
    {
      return 0;
    }
  arg = cpl_parameter_get_string(param);
  if (strcmp(arg, "none") == 0)
    {
      // no lambda list available, use the normal --lambda_from and --lambda_to parameters
      return 1;
    }
  i = 0;
  while (arg[i] != '\0')
    {
      /* 1. Extract the lower lambda value from the range list */
      j = 0;
      while ((arg[i] != '\0') && (arg[i] != ','))
	{
	  if (j < 30)
	    {
	      str[j++] = arg[i];
	    }
	  i++;
	}
      str[j] = '\0';
      if (str[0] == '\0')
	{
	  // no floating point number read
	  break;
	}
      info->refs[info->nref].lfrom = info->lambda_scale*atof(str);
      /* 2. Skip the comma character between two lambda values */
      if (arg[i] == ',')
	{
	  i++;
	}
      /* 3. Extract the upper lambda value from the range list */
      j = 0;
      while ((arg[i] != '\0') && (arg[i] != ','))
	{
	  if (j < 30)
	    {
	      str[j++] = arg[i];
	    }
	  i++;
	}
      str[j] = '\0';
      if (str[0] == '\0')
	{
	  // no floating point number read
	  break;
	}
      info->refs[info->nref].lto = info->lambda_scale*atof(str);
      cpl_msg_info(cpl_func, "lambda_range[%d] =%f .. %f",
		   info->nref,
		   info->refs[info->nref].lfrom,
		   info->refs[info->nref].lto);
      /* 4. Skip a comma if present and go to the next wavelength range. */
      if (arg[i] == ',')
	{
	  i++;
	}
      info->nref++;
      if (info->nref == MAT_NB_REFERENCE) break;
    }
  return (info->nref != 0);
}

/**
   @brief Gets all plugin parameters from a cpl_parameterlist and stores them in a mat_est_aphase_info data structure.
   @param info  A mat_est_aphase_info data structure as a substitute for global variables.
   @param parlist A cpl_parameterlist containing the plugin parameters.
   @return 0 if everything is ok
*/
static int mat_get_parameters(mat_est_aphase_info  *info,
			      cpl_parameterlist *parlist)
{
  cpl_parameter    *param = NULL;

  /* retrieve all parameters */
  /* --fov */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.fov");
  info->FOV = cpl_parameter_get_double(param);
  /* --npix */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.npix");
  info->npix = cpl_parameter_get_int(param);
  /* --kwd_lambda_min */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.kwd_lambda_min");
  if (param != NULL)
    {
      const char *arg = cpl_parameter_get_string(param);
      if (strcmp(arg, "") != 0)
	{
	  info->kwd_lambda_min = cpl_strdup(arg);
	}
    }
  /* --kwd_lambda_max */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.kwd_lambda_max");
  if (param != NULL)
    {
      const char *arg = cpl_parameter_get_string(param);
      if (strcmp(arg, "") != 0)
	{
	  info->kwd_lambda_max = cpl_strdup(arg);
	}
    }
  /* --kwd_lambda_center */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.kwd_lambda_center");
  if (param != NULL)
    {
      const char *arg = cpl_parameter_get_string(param);
      if (strcmp(arg, "") != 0)
	{
	  info->kwd_lambda_center = cpl_strdup(arg);
	}
    }
  /* --kwd_lambda_width */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.kwd_lambda_width");
  if (param != NULL)
    {
      const char *arg = cpl_parameter_get_string(param);
      if (strcmp(arg, "") != 0)
	{
	  info->kwd_lambda_width = cpl_strdup(arg);
	}
    }
  /* --lambda_unit */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.lambda_unit");
  if (param != NULL)
    {
      const char *arg = cpl_parameter_get_string(param);
      if (strcmp(arg, "m") == 0)
	{
	  info->lambda_scale = 1.0;
	}
      else if (strcmp(arg, "mm") == 0)
	{
	  info->lambda_scale = 1.0e-3;
	}
      else if (strcmp(arg, "um") == 0)
	{
	  info->lambda_scale = 1.0e-6;
	}
      else if (strcmp(arg, "nm") == 0)
	{
	  info->lambda_scale = 1.0e-9;
	}
    }
  /* --lambda_list */
  mat_get_lambda_list(info, parlist);
  /* --ref_mode */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.ref_mode");
  info->ref_mode = cpl_parameter_get_int(param);
  /* --ref_param */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.ref_param");
  info->ref_param = cpl_parameter_get_double(param);
  /* --phi_error */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.phi_error");
  info->phi_error = cpl_parameter_get_double(param);
  /* --vis_name */
  param = cpl_parameterlist_find(parlist, "matisse.mat_est_aphase.vis_name");
  strncpy(info->vis_name, cpl_parameter_get_string(param), 256);
  info->vis_name[255] = '\0';
  /* calculate some derived values */
  info->dxrek = info->FOV/(double)(info->npix);
  return 0;
}

static int mat_compare_reference(const void *e1, const void *e2)
{
  mat_reference_image  *r1 = (mat_reference_image*)e1;
  mat_reference_image  *r2 = (mat_reference_image*)e2;

  if (0.5*(r1->lfrom + r1->lto) < 0.5*(r2->lfrom + r2->lto)) return -1;
  if (0.5*(r1->lfrom + r1->lto) > 0.5*(r2->lfrom + r2->lto)) return +1;
  return 0;
}

/*--------------------------------------------------------------------*/
/* cpl_image related functions                                        */
/* The images used in this plugin have coordinates                    */
/* between -N/2 and N/2-1 or between 0 and N-1.                       */
/*--------------------------------------------------------------------*/
/*
  static cpl_image *mat_read_image(mat_est_aphase_info *info,
  cpl_frameset *frameset,
  const char *tag,
  int mode,
  double param)
  {
  int        fcount;
  cpl_image *img = NULL;

  cpl_msg_info(cpl_func, "load/create an image (%s), mode=%d, param=%f, dxrek=%f",
  tag, mode, param, info->dxrek);
  img = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (img == 0)
  {
  cpl_msg_error(cpl_func, "can't allocate image");
  return NULL;
  }
  mat_image_fill(img, 0.0);
  switch (mode)
  {
  case MAT_MODE_IMAGE:  // read an image from a file given by the tag
  fcount = cpl_frameset_count_tags(frameset, tag);
  if (fcount == 1)
  { // we found an image, try to load it
  cpl_frame *imgframe = NULL;
  cpl_image *oimg = NULL;
  imgframe = cpl_frameset_find(frameset, tag); // already checked previously!
  cpl_frame_set_group(imgframe, CPL_FRAME_GROUP_RAW);
  cpl_msg_info(cpl_func, "  reading image from %s", cpl_frame_get_filename(imgframe));
  oimg = cpl_image_load(cpl_frame_get_filename(imgframe), CPL_TYPE_DOUBLE,0,0);
  if(oimg != NULL)
  {
  //scale the image
  cpl_msg_info(cpl_func, "  scale image with dxrek=%f, scale=%f", info->dxrek, param);
  mat_image_swap(oimg);
  mat_image_scale(img, info->dxrek, oimg, param);
  cpl_image_delete(oimg);
  }
  else
  { // the mandatory image could not be loaded  -> return an empty image
  cpl_msg_error(cpl_func, "can't load image from %s", cpl_frame_get_filename(imgframe));
  }
  }
  else if (fcount == 0)
  { // the mandatory image does not exist -> return an empty image
  cpl_msg_error(cpl_func, "the mandatory image does not exist");
  }
  else
  {
  cpl_msg_error(cpl_func, "only zero or one image (tag=%s) is allowed in the SOF", tag);
  }
  break;
  case MAT_MODE_STAR: // create a single star
  cpl_msg_info(cpl_func, "  create a single star");
  mat_image_set_double(img, 0, 0, 1.0);
  break;
  default:;
  }
  {
  double total = mat_image_get_total(img);
  cpl_msg_info(cpl_func, "  total flux %g", total);
  }
  mat_image_normalize(img);
  return img;
  }
*/

static cpl_error_code mat_add_reference(mat_est_aphase_info *info, int iref)
{
  // allocate memory for the internal reference images and their FFT
  info->refs[iref].img = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE);
  if (info->refs[iref].img == 0)
    {
      cpl_msg_error(cpl_func, "can't allocate image");
      return CPL_ERROR_NULL_INPUT;
    }
  info->refs[iref].fimg = cpl_image_new(info->npix, info->npix, CPL_TYPE_DOUBLE_COMPLEX);
  if (info->refs[iref].fimg == 0)
    {
      cpl_msg_error(cpl_func, "can't allocate complex image");
      return CPL_ERROR_NULL_INPUT;
    }
  mat_image_fill(info->refs[iref].img, 0.0);
  if (iref >= info->nref)
    {
      info->nref = iref + 1;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_read_images(mat_est_aphase_info *info,
				      cpl_frameset *frameset,
				      const char *tag,
				      int mode,
				      double param)
{
  cpl_size               iref, iimg;
  cpl_frameset_iterator *it;
  cpl_frame             *frame;

  // read all imput images or create point sources
  switch (mode)
    {
    case MAT_MODE_IMAGE:  // read an image from a file given by the tag
      iref = 0;
      it = cpl_frameset_iterator_new(frameset);
      frame = cpl_frameset_iterator_get(it);
      while ((frame != NULL) && (iref < MAT_NB_REFERENCE))
	{
	  // we use only frames with the given tag
	  if (strcmp(cpl_frame_get_tag(frame), tag) == 0)
	    {
	      cpl_imagelist    *hlist;
	      cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
	      cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(frame));
	      if (iref >= info->nref)
		{ // we have no entry from the --lambda_list option
		  // -> try to read the wavelength window from the primary header by using the provided keywords
		  cpl_propertylist *plist;
		  cpl_msg_info(cpl_func, "not enough lambda_list entries, trying to use keywords");
		  plist = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
		  if (plist == NULL)
		    {
		      cpl_msg_error(cpl_func, "Cannot read the primary header keywords from that file.");
		    }
		  else
		    {
		      int ok = 0;
		      if ((info->kwd_lambda_min != NULL) && (info->kwd_lambda_max != NULL))
			{
			  double lmin = info->lambda_scale*cpl_propertylist_get_double(plist, info->kwd_lambda_min);
			  double lmax = info->lambda_scale*cpl_propertylist_get_double(plist, info->kwd_lambda_max);
			  if ((lmin == 0.0) || (lmax == 0.0))
			    {
			      cpl_msg_warning(cpl_func, "keywords %s and/or %s do not exist in this file", info->kwd_lambda_min, info->kwd_lambda_max);
			    }
			  else
			    {
			      info->refs[iref].lfrom = lmin;
			      info->refs[iref].lto   = lmax;
			      ok = 1;
			    }
			}
		      if (!ok && (info->kwd_lambda_center != NULL) && (info->kwd_lambda_width != NULL))
			{
			  double lcenter = info->lambda_scale*cpl_propertylist_get_double(plist, info->kwd_lambda_center);
			  double lwidth = info->lambda_scale*cpl_propertylist_get_double(plist, info->kwd_lambda_width);
			  if ((lcenter == 0.0) || (lwidth == 0.0))
			    {
			      cpl_msg_warning(cpl_func, "keywords %s and/or %s do not exist in this file", info->kwd_lambda_center, info->kwd_lambda_width);
			    }
			  else
			    {
			      info->refs[iref].lfrom = lcenter - 0.5*lwidth;
			      info->refs[iref].lto   = lcenter + 0.5*lwidth;
			      ok = 1;
			    }
			}
		      if (!ok)
			{
			  cpl_msg_error(cpl_func, "more reference images than wavelength windows given with --lambda_list and no valid keywords specified");
			}
		      cpl_propertylist_delete(plist);
		    }
		}
	      hlist = cpl_imagelist_load(cpl_frame_get_filename(frame), CPL_TYPE_DOUBLE, 0);
	      if (hlist == NULL)
		{
		  cpl_msg_error(cpl_func, "can't load image from file %s", cpl_frame_get_filename(frame));
		}
	      else
		{
		  cpl_msg_info(cpl_func, "  scale %d images with dxrek=%f, scale=%f", (int)cpl_imagelist_get_size(hlist), info->dxrek, param);
		  for (iimg = 0; (iimg < cpl_imagelist_get_size(hlist)) && (iref < MAT_NB_REFERENCE); iimg++)
		    {
		      cpl_image *himg = cpl_imagelist_get(hlist, iimg);
		      mat_add_reference(info, iref);
		      // rescale the reference image and put it into the internal image space
		      mat_image_swap(himg);
		      mat_image_scale(info->refs[iref].img, info->dxrek, himg, param);
		      iref++;
		    }
		  cpl_imagelist_delete(hlist);
		}
	    }
	  cpl_frameset_iterator_advance(it, 1);
	  frame = cpl_frameset_iterator_get(it);
	}
      cpl_frameset_iterator_delete(it);
      if (iref < (info->nref - 1))
	{ // not enough images
	  cpl_msg_error(cpl_func, "not enough images, %d loadad, %d needed, using point sources", (int)iref, info->nref);
	  while (iref < info->nref)
	    {
	      mat_add_reference(info, iref);
	      mat_image_set_double(info->refs[iref].img, 0, 0, 1.0);
	      iref++;
	    }
	}
      break;
    case MAT_MODE_STAR: // create a single star
      cpl_msg_info(cpl_func, "  create a single star");
      for (iref = 0; iref < info->nref; iref++)
	{
	  mat_add_reference(info, iref);
	  mat_image_set_double(info->refs[iref].img, 0, 0, 1.0);
	}
      break;
    default:;
    }
  // calculate for each image the FFT
  for (iref = 0; iref < info->nref; iref++)
    {
      mat_image_fft_forward(info->refs[iref].fimg, info->refs[iref].img);
    }
  // reorder the reference images to get increasing wavelength
  qsort(&(info->refs[0]), info->nref, sizeof(info->refs[0]), mat_compare_reference);
  return CPL_ERROR_NONE;
}

/**
   @brief Calculates scale factors for mapping uv coordinates into FFT array indices.
   @param info      A mat_est_aphase_info data structure as a substitute for global variables.
   @param datfile   ....
   @param oiwave    A data structure containing one OI_WAVELENGTH binary table.
   @returns An error code

   This function calculates fr each wavelength in an OI_WAVELENGTH binary table a scale
   factor for the uv coordinate mapping. If a wavelength is outside of the lambda filter,
   the scale factor for this wavelength is set to 0.0. The scale factor is calculated using
   f := pi/180*FOV/lambda (beware of the units for the field of view and the wavelength).
*/
static cpl_error_code mat_calc_baseline_scale(mat_est_aphase_info *info,
					      FILE                *datfile,
					      mat_oiwavelength    *oiwave)
{
  int ichan, iref;

  if ((info == NULL) || (oiwave == NULL))
    {
      return CPL_ERROR_NULL_INPUT;
    }
  // Create/Resize the vector containing the wavelength in [m]
  if (info->lambda_list == NULL)
    {
      info->lambda_list = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->lambda_list, oiwave->nbchannel);
    }
  // Create/Resize the vector containing the wavelength -> wavelength filter window mapping
  if (info->lambda_window == NULL)
    {
      info->lambda_window = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->lambda_window, oiwave->nbchannel);
    }
  // Create/Resize the vector containing the wavelength -> reference image mapping
  if (info->lambda_image == NULL)
    {
      info->lambda_image = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->lambda_image, oiwave->nbchannel);
    }
  // Create/Resize the vector containing the scaling m -> pixel for the uv mapping
  if (info->pixel_scale == NULL)
    {
      info->pixel_scale = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->pixel_scale, oiwave->nbchannel);
    }
  if (info->pixel_scale == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate memory for the baseline scaling");
      return CPL_ERROR_NULL_INPUT;
    }
  if (info->wn == NULL)
    {
      info->wn = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->wn, oiwave->nbchannel);
    }
  // initialize all mappings
  for (ichan = 0; ichan < oiwave->nbchannel; ichan++)
    {
      cpl_vector_set(info->lambda_list, ichan, oiwave->effwave[ichan]);
      cpl_vector_set(info->wn, ichan, 1.0e-6/oiwave->effwave[ichan]); // wave number = 1/lambda , lambda in [um]
      cpl_vector_set(info->lambda_window, ichan, (double)-1);
      cpl_vector_set(info->lambda_image, ichan, (double)-1);
      cpl_vector_set(info->pixel_scale, ichan, info->FOV*CPL_MATH_PI/(oiwave->effwave[ichan]*18.0*6.0*6.0*1.0e6));
    }
  // determine the mapping between wavelength and reference image
  for (ichan = 0; ichan < oiwave->nbchannel; ichan++)
    {
      for (iref = 0; iref < info->nref; iref++)
	{
	  if ((oiwave->effwave[ichan] >= info->refs[iref].lfrom) && (oiwave->effwave[ichan] <= info->refs[iref].lto))
	    {
	      cpl_vector_set(info->lambda_image, ichan, (double)iref);
	      break;
	    }
	}
    }
  // determine the mapping between wavelength and window
  info->nwin = 0;
  ichan = 0;
  while (ichan < oiwave->nbchannel)
    {
      // skip all wavelengths outside a reference image
      while ((ichan < oiwave->nbchannel) && (cpl_vector_get(info->lambda_image, ichan) < 0.0))
	{
	  ichan++;
	}
      if (ichan == oiwave->nbchannel) break;
      iref = (int)cpl_vector_get(info->lambda_image, ichan);
      info->wins[info->nwin].lfrom = info->refs[iref].lfrom;
      // add all succeeding wavelength to this window until we are outside of an reference image
      while ((ichan < oiwave->nbchannel) && (cpl_vector_get(info->lambda_image, ichan) >= 0.0))
	{
	  iref = (int)cpl_vector_get(info->lambda_image, ichan);
	  cpl_vector_set(info->lambda_window, ichan, (double)(info->nwin));
	  info->wins[info->nwin].lto = info->refs[iref].lto;
	  ichan++;
	}
      info->nwin++;
    }
  // print useful informations about the lambda, reference and window relation
  if (datfile != NULL)
    {
      for (ichan = 0; ichan < oiwave->nbchannel; ichan++)
	{
	  fprintf(datfile, "lambda %d  = %.6E reference = %d window = %d scale m -> pixel = %g\n",
		  ichan,
		  oiwave->effwave[ichan],
		  (int)cpl_vector_get(info->lambda_image, ichan),
		  (int)cpl_vector_get(info->lambda_window, ichan),
		  cpl_vector_get(info->pixel_scale, ichan));
	}
    }
  // Create/Resize the temporary vectors
  if (info->phim == NULL)
    {
      info->phim = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->phim, oiwave->nbchannel);
    }
  if (info->phie == NULL)
    {
      info->phie = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->phie, oiwave->nbchannel);
    }
  if (info->phir == NULL)
    {
      info->phir = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->phir, oiwave->nbchannel);
    }
  if (info->phid == NULL)
    {
      info->phid = cpl_vector_new(oiwave->nbchannel);
    }
  else
    {
      cpl_vector_set_size(info->phid, oiwave->nbchannel);
    }
  return CPL_ERROR_NONE;
}

static int mat_unwrap_phase(mat_est_aphase_info *info, cpl_vector *phase)
{
  int    i, n;
  double phase_prev = 0.0;
  int    dir = 0;

  n = cpl_vector_get_size(phase);
  // get the starting value for the phase unwrap (first wavelength of the first wavelength filter window)
  for (i = 0; i < n; i++)
    {
      if (cpl_vector_get(info->lambda_window, i) != -1.0)
	{
	  phase_prev = cpl_vector_get(phase, i);
	  break;
	}
    }
  for (i = 1; i < n; i++)
    {
      double phase_curr = cpl_vector_get(phase, i);
      // check that we are inside a valid wavelength filter window
      if (cpl_vector_get(info->lambda_window, i) == -1.0) continue;
      if (phase_curr > (phase_prev + CPL_MATH_PI))
	{
	  int twopi = floor((phase_curr - (phase_prev - CPL_MATH_PI))/(2.0*CPL_MATH_PI));
	  phase_curr -= 2.0*CPL_MATH_PI*(double)twopi;
	  dir--;
	}
      else if (phase_curr < (phase_prev - CPL_MATH_PI))
	{
	  int twopi = floor(((phase_prev + CPL_MATH_PI) - phase_curr)/(2.0*CPL_MATH_PI));
	  phase_curr += 2.0*CPL_MATH_PI*(double)twopi;
	  dir++;
	}
      cpl_vector_set(phase, i, phase_curr);
      phase_prev = phase_curr;
    }
  return dir;
}

static int mat_fit_straight_line(mat_est_aphase_info *info,
				 cpl_vector *ph,
				 cpl_vector *phe,
				 int         count,
				 double     *offset,
				 double     *slope,
				 double     *chi2)
{
  int i, n;
  double A = 0.0;
  double B = 0.0;
  double C = 0.0;
  double E1 = 0.0;
  double E2 = 0.0;
  double det;
  double sum0 = 0.0;
  double cnt  = 0.0;

  n = cpl_vector_get_size(ph);
  // calculate the 2x2 matrix
  for (i = 0; (i < count) && (i < n); i++)
    {
      int    iwin = (int)cpl_vector_get(info->lambda_window, i);
      double x    = cpl_vector_get(info->wn, i);
      double y    = cpl_vector_get(ph, i) + info->wins[iwin].n2pi*2.0*CPL_MATH_PI;
      double ye2  = cpl_vector_get(phe, i)*cpl_vector_get(phe, i);
      if (iwin < 0.0) continue;
      A   += x*x/ye2;
      B   += x/ye2;
      C   += 1.0/ye2;
      E1  += y*x/ye2;
      E2  += y/ye2;
      cnt += 1.0;
    }
  // calculate offset and slope
  det = A*C - B*B;
  // cpl_msg_info(cpl_func, "fit A = %g B = %g C = %g E1 = %g E2 = %g cnt = %g det = %g", A, B, C, E1, E2, cnt, det);
  if (fabs(det) < 1e-6) return 0;
  *slope  = (C*E1 - B*E2)/det;
  *offset = (A*E2 - B*E1)/det;
  // calculate chi squared of the fit
  for (i = 0; (i < count) && (i < n); i++)
    {
      int    iwin = (int)cpl_vector_get(info->lambda_window, i);
      double x    = cpl_vector_get(info->wn, i);
      double y    = cpl_vector_get(ph, i) + info->wins[iwin].n2pi*2.0*CPL_MATH_PI;
      double ye2  = cpl_vector_get(phe, i)*cpl_vector_get(phe, i);
      double yf   = (*offset) + (*slope)*x;
      if (iwin < 0.0) continue;
      sum0 += (y - yf)*(y - yf)/ye2;
    }
  *chi2 = sum0/cnt;
  return 1;
}

/*
  static int mat_fit_straight_line_incremental(mat_est_aphase_info *info,
  cpl_vector *ph,
  cpl_vector *phe,
  double *offset,
  double *slope,
  double *chi2)
  {
  int    i, n, w, count;
  
  n = cpl_vector_get_size(ph);
  // calculate the number of wavelength in the first window;
  count = 0;
  for (i = 0; i < n; i++)
  {
  if (cpl_vector_get(info->lambda_window, i) == 0.0) count = i+1;
  }
  if (count == 0) return 0;
  info->lambda_phoff[0] = 0.0; // we start without any 2PI phase offset
  for (w = 1; w < info->lambda_count; w++)
  {
  info->lambda_phoff[w] = 0.0; // we start without any 2PI phase offset
  // calculate the last index of 
  count = 0;
  for (i = 0; i < n; i++)
  {
  if (cpl_vector_get(info->lambda_window, i) == 0.0) count = i+1;
  }
  if (count == 0) continue;
  do {
  double roffset, rslope, rchi2; // reference/current
  double hoffset, hslope, hchi2; // fit at +2PI
  double loffset, lslope, lchi2; // fit at -2PI
  // calculate a straight line fit without shifting by 2PI
  mat_fit_straight_line(info, ph, phe, count, &roffset, &rslope, &rchi2);
  // shift the values in the current window by -2PI and calculate a fit
  info->lambda_phoff[w] -= 1.0;
  mat_fit_straight_line(info, ph, phe, count, &loffset, &lslope, &lchi2);
  // shift the values in the current window by +2PI and calculate a fit
  info->lambda_phoff[w] += 2.0; // we are at -2PI
  mat_fit_straight_line(info, ph, phe, count, &hoffset, &hslope, &hchi2);
  info->lambda_phoff[w] -= 1.0; // go back to reference
  if (lchi2 < rchi2)
  { // by subtracting 2PI the fit is better
  // -> use this as a new reference point
  info->lambda_phoff[w] -= 1.0;
  }
  else if (hchi2 < rchi2)
  { // by adding 2PI the fit is better
  // -> use this as a new reference point
  info->lambda_phoff[w] += 1.0;
  }
  else
  { // the current reference has the lowest chi2
  // -> keep the current reference point
  break;
  }
  } while (1);
  }
  // apply the 2PI offset to all phases inside the wavelength windows
  for (i = 0; i < n; i++)
  {
  lw = (int)(cpl_vector_get(info->lambda_window, i));
  if (lw < 0) continue;
  cpl_vector_set(ph, i, cpl_vector_get(ph, i) + info->lambda_phoff[lw]*2.0*CPL_MATH_PI);
  }
  for (lw = 0; lw < MAT_NB_LAMBDA_WINDOW; w++)
  {
  info->lambda_phoff[lw] = 0.0;
  }
  // make the final straight line fit
  mat_fit_straight_line(info, ph, phe, n, offset, slope, chi2);
  return 1;
  }
*/

/*
  static void mat_fit_straight_line_brute_force_step(mat_est_aphase_info *info,
  cpl_vector *ph,
  cpl_vector *phe,
  int win,
  int cur2pis,
  int max2pis)
  {
  if (win == info->lambda_count)
  { // we have modified all wavelength windows, try a straight line fit
  int i;
  int n = cpl_vector_get_size(ph);
  double offset, slope, chi2;
  mat_fit_straight_line(info, ph, phe, n, &offset, &slope, &chi2);
  if ((info->best_valid == 0) || (chi2 < info->best_chi2))
  { // this is the first straight line fit at all
  // store the current fit as best fit
  info->best_valid  = 1;
  info->best_offset = offset;
  info->best_slope  = slope;
  info->best_chi2   = chi2;
  for (i = 0; i < info->lambda_count; i++)
  {
  info->best_phoff[i] = info->lambda_phoff[i];
  }
  }
  }
  else
  {
  int i;
  for (i = -max2pis; i <= max2pis; i++)
  {
  info->lambda_phoff[win] = cur2pis + i;
  mat_fit_straight_line_brute_force_step(info, ph, phe, win+1, cur2pis + i, max2pis - abs(i));
  }
  }
  }
*/

/*
  static int mat_fit_straight_line_brute_force(mat_est_aphase_info *info,
  cpl_vector *ph,
  cpl_vector *phe,
  double *offset,
  double *slope,
  double *chi2)
  {
  int     i, n, w;
  double  aslope = 0.0;
  double  count  = 0.0;
  int     found = 0;
  double  min_phase, max_phase;
  int     max2pis;
  
  n = cpl_vector_get_size(ph);
  // Calculate an estimate of the slope by using the average slope from pairs of data points
  for (i = 0; i < n-1; i++)
  {
  if (cpl_vector_get(info->lambda_window, i) == 0.0) continue; // outside a wavelength window
  if (found)
  {
  if (cpl_vector_get(ph, i) < min_phase) min_phase = cpl_vector_get(ph, i);
  if (cpl_vector_get(ph, i) > max_phase) max_phase = cpl_vector_get(ph, i);
  }
  else
  {
  found = 1;
  min_phase = cpl_vector_get(ph, i);
  max_phase = cpl_vector_get(ph, i);
  }
  if (cpl_vector_get(info->lambda_window, i) != cpl_vector_get(info->lambda_window, i+1)) continue; // not the same window
  // wave number decreases with increasing wave length!
  aslope += (cpl_vector_get(ph, i + 1) - cpl_vector_get(ph, i))/(cpl_vector_get(info->wn, i) - cpl_vector_get(info->wn, i+1));
  count += 1.0;
  }
  if (count < 5.0) return 0; // we need at least 5 differences to estimate the slope
  aslope /= count;
  // the total slope (first to last wavelength) in multiples of 2PI
  // this gives the maximum _total_ 2PI shift over all wavelength windows
  max2pis = ceil((cpl_vector_get(info->wn, 0) - cpl_vector_get(info->wn, n-1) - (max_phase - min_phase))*aslope/2.0/CPL_MATH_PI);
  info->best_valid = 0;
  info->lambda_phoff[0] = 0.0;
  mat_fit_straight_line_brute_force_step(info, ph, phe, 1, 0, max2pis + 1);
  // apply the 2PI offset from the best fit to all phases inside the wavelength windows
  for (i = 0; i < n; i++)
  {
  w = (int)(cpl_vector_get(info->lambda_window, i));
  if (w < 0) continue;
  cpl_vector_set(ph, i, cpl_vector_get(ph, i) + info->best_phoff[w]*2.0*CPL_MATH_PI);
  }
  for (i = 0; i < MAT_NB_LAMBDA_WINDOW; i++)
  {
  info->lambda_phoff[i] = 0.0;
  }
  // make the final straight line fit
  mat_fit_straight_line(info, ph, phe, n, offset, slope, chi2);
  return 1;
  }
*/

static int mat_fit_straight_line_global(mat_est_aphase_info *info,
					cpl_vector *ph,
					cpl_vector *phe,
					double *offset,
					double *slope,
					double *chi2)
{
  int     ichan, nchan;
  int     iwin;
  double  aslope = 0.0;
  double  count  = 0.0;
  double  wc[MAT_NB_WINDOW];
  double  wx[MAT_NB_WINDOW];
  double  wy[MAT_NB_WINDOW];
  
  *offset = 0.0;
  *slope  = 0.0;
  *chi2   = 0.0;
  for (iwin = 0; iwin < MAT_NB_WINDOW; iwin++)
    {
      info->wins[iwin].n2pi = 0.0;
    }
  nchan = cpl_vector_get_size(ph);
  // Calculate an estimate of the slope by using the average slope from pairs of data points
  for (ichan = 0; ichan < nchan - 1; ichan++)
    {
      // check if we are inside the same wavelength window
      if (cpl_vector_get(info->lambda_window, ichan) == -1.0) continue;
      if (cpl_vector_get(info->lambda_window, ichan) != cpl_vector_get(info->lambda_window, ichan + 1)) continue;
      // wave number decreases with increasing wave length!
      aslope += (cpl_vector_get(ph, ichan + 1) - cpl_vector_get(ph, ichan))/(cpl_vector_get(info->wn, ichan) - cpl_vector_get(info->wn, ichan + 1));
      count += 1.0;
    }
  if (count < 5.0) return 0; // we need at least 5 differences to estimate the slope
  aslope /= count;
  // calculate for each wavelength window the mean x and y coordinate
  for (iwin = 0; iwin < MAT_NB_WINDOW; iwin++)
    {
      wc[iwin] = 0.0; wx[iwin] = 0.0; wy[iwin] = 0.0;
    }
  for (ichan = 0; ichan < nchan; ichan++)
    {
      double x, y;
      iwin = (int)(cpl_vector_get(info->lambda_window, ichan));
      x = cpl_vector_get(info->wn, ichan);
      y = cpl_vector_get(ph, ichan) + info->wins[iwin].n2pi*2.0*CPL_MATH_PI;
      if (iwin >= 0)
	{
	  wc[iwin] += 1.0;
	  wx[iwin] += x;
	  wy[iwin] += y;
	}
    }
  for (iwin = 0; iwin < info->nwin; iwin++)
    {
      if (wc[iwin] != 0.0)
	{
	  wx[iwin] /= wc[iwin];
	  wy[iwin] /= wc[iwin];
	}
      // cpl_msg_info(cpl_func, "win[%d] x = %g y = %g c = %g", iwin, wx[iwin], wy[iwin], wc[iwin]);
    }
  // incrementally adjust the phase offsets for the wavelength windows
  for (iwin = 1; iwin < info->nwin; iwin++)
    {
      double dx, mdy, sdy, phoff;
      dx = wx[iwin - 1] - wx[iwin]; // x decrements with increasing lambda (x = 1/lambda)
      mdy = wy[iwin] - wy[iwin - 1];
      sdy = dx*aslope;
      phoff = round((sdy - mdy)/2.0/CPL_MATH_PI);
      info->wins[iwin].n2pi = phoff + info->wins[iwin - 1].n2pi;
      // cpl_msg_info(cpl_func, "offset[%d] = %g dx = %g mdy = %g sdy = %g", iwin, phoff, dx, mdy, sdy);
    }
  // make the final straight line fit
  mat_fit_straight_line(info, ph, phe, nchan, offset, slope, chi2);
  // apply the 2PI offset to all phases inside the wavelength windows
  // cpl_msg_info(cpl_func, "# fit y = %g + %g * x", (*offset), (*slope));
  // cpl_msg_info(cpl_func, "# index x y 2pi fit");
  /*
    for (ichan = 0; ichan < nchan; ichan++)
    {
    double x, y, y_original, y_offset, y_fit;
    iwin = (int)(cpl_vector_get(info->lambda_window, ichan));
    if (iwin < 0) continue;
    x = cpl_vector_get(info->wn, ichan);
    y_original = cpl_vector_get(ph, ichan);
    y_offset   = info->wins[iwin].n2pi*2.0*CPL_MATH_PI;
    y = y_original + y_offset;
    y_fit = (*offset) + (*slope)*x;
    cpl_vector_set(ph, ichan, y);
    // cpl_msg_info(cpl_func, "%d %g %g %g %g", ichan, x, y_original, y_offset, y_fit);
    }
    for (iwin = 0; iwin < MAT_NB_WINDOW; iwin++)
    {
    info->wins[iwin].n2pi = 0.0;
    }
  */
  return 1;
}

static int mat_est_phase_elem(mat_est_aphase_info *info,
			      FILE                *datfile,
			      mat_oivis           *vis,
			      mat_viselem         *el)
{
  int      ichan;
  double   offset, slope, chi2;

  // Calculate: phase difference = measured - reference
  if (datfile != NULL)
    {
      fprintf(datfile, "# ref mjd u v lambda phi_r (phi_m - phi_r)\n");
    }
  for (ichan = 0; ichan < vis->nbchannel; ichan++)
    {
      double complex vism, visr; // F: ftu, ftuorig, ...
      int            iref = (int)cpl_vector_get(info->lambda_image, ichan);
      double         phim = CPL_MATH_PI*el->visphi[ichan]/180.0;     // degree -> radian
      double         u    = cpl_vector_get(info->pixel_scale, ichan)*el->ucoord;
      double         v    = cpl_vector_get(info->pixel_scale, ichan)*el->vcoord;
      double         l    = cpl_vector_get(info->lambda_list, ichan);
      // measured phase
      vism  = (cos(phim) + sin(phim)*I);
      cpl_vector_set(info->phim, ichan, phim);
      if (info->phi_error == 0.0)
	{
	  double phie = CPL_MATH_PI*el->visphierr[ichan]/180.0;  // degree -> radian
	  cpl_vector_set(info->phie, ichan, phie);
	}
      else
	{
	  cpl_vector_set(info->phie, ichan, 1.0);
	}
      if (iref == -1)
	{
	  continue;
	}
      // reconstructed phase
      visr  = mat_image_get_complex_interpolated(info->refs[iref].fimg, u, v);
      cpl_vector_set(info->phir, ichan, carg(visr));
      // calculate and store the phase difference
      cpl_vector_set(info->phid,  ichan, carg(vism*conj(visr)));
      if (datfile != NULL)
	{
	  fprintf(datfile, "ref %g %g %g %g %g %g\n",
		  el->dateobsmjd, el->ucoord, el->vcoord, l,
		  carg(visr), carg(vism*conj(visr)));
	}
    }
  // Unwrap the phases (produce a rising or falling straight line)
  mat_unwrap_phase(info, info->phid);
  // Fit a straight line to the phase differences
  mat_fit_straight_line_global(info, info->phid, info->phie, &offset, &slope, &chi2);
  if (datfile != NULL)
    {
      fprintf(datfile, "# fit %g %g %g offset %g slope %g chi2 %g\n",
	      el->dateobsmjd, el->ucoord, el->vcoord,
	      offset, slope, chi2);
    }
  if (datfile != NULL)
    {
      fprintf(datfile, "# phase mjd u v lambda phi_m phi_me fit\n");
    }
  for (ichan = 0; ichan < vis->nbchannel; ichan++)
    {
      double l    = cpl_vector_get(info->lambda_list, ichan);
      double x    = cpl_vector_get(info->wn, ichan);
      double phim = CPL_MATH_PI*el->visphi[ichan]/180.0;     // degree -> radian
      double phie = CPL_MATH_PI*el->visphierr[ichan]/180.0;  // degree -> radian
      double phid = offset + slope*x;
      double phia;
      double complex  vism, visd;
      vism = (cos(phim) + sin(phim)*I);
      visd = (cos(phid) + sin(phid)*I);
      phia = 180.0*carg(vism*conj(visd))/CPL_MATH_PI;
      el->visphi[ichan] = phia;
      if (datfile != NULL)
	{
	  fprintf(datfile, "phase %g %g %g %g %g %g %g\n",
		  el->dateobsmjd, el->ucoord, el->vcoord, l,
		  phim, phie, phid);
	}
    }
  return 0;
}

static int mat_est_phase(mat_est_aphase_info *info, int itupel)
{
  mat_oifits         *oifits = info->oifits;
  mat_oiwavelength   *wave   = oifits->oiwave;
  mat_oivis          *vis    = oifits->oivis;
  int                 i;
  FILE               *datfile = NULL;

  if (info->vis_name[0] != '\0')
    {
      char fname[1024];
      snprintf(fname, 1024, "%s_%d.dat", info->vis_name, itupel + 1);
      datfile = fopen(fname, "w");
    }
  // calculate the baseline to pixel scale for the selected wavelengths
  mat_calc_baseline_scale(info, datfile, wave);
  for (i = 0; i < vis->nbvis; i++)
    {
      mat_viselem *el = vis->list_vis[i];
      mat_est_phase_elem(info, datfile, vis, el);
    }
  if (datfile != NULL)
    {
      fclose(datfile);
    }
  return 0;
}

/*--------------------------------------------------------------------*/
/* Main plugin function                                               */
/*--------------------------------------------------------------------*/

/**
   @brief Interpret the command line options and execute the data processing
   @param parlist the parameters list
   @param frameset the frames list
   @return 0 if everything is ok
*/
static int mat_est_aphase(cpl_parameterlist *parlist,
			  cpl_frameset *frameset)
{
  mat_est_aphase_info    info;
  cpl_frameset_iterator *it;
  cpl_frame             *frame;
  int                    i;

  // initialize and get all command line parameters
  mat_info_init(&info);
  info.parlist = parlist;
  info.frameset = frameset;
  mat_get_parameters(&info, parlist);
  // Load/Create the reference image and calculate the FFT
  /* Old version (only one image)
     info.cont_image = mat_read_image(&info, frameset, MAT_DO_REFERENCE_IMAGE, info.cont_mode, info.cont_param);
     info.fcont_image = cpl_image_new(info.npix, info.npix, CPL_TYPE_DOUBLE_COMPLEX);
     if (info.fcont_image == NULL)
     {
     cpl_msg_error(cpl_func, "can't allocate memory for reference image (FFT)");
     mat_info_delete(&info);
     // Return
     if (cpl_error_get_code())
     return -1;
     else
     return 0;
     }
     mat_image_fft_forward(info.fcont_image, info.cont_image);
  */
  // New version (one image for each lambda window)
  if (mat_read_images(&info, frameset, MAT_DO_REFERENCE_IMAGE, info.ref_mode, info.ref_param) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "can't load enough images or allocate enough memory");
      mat_info_delete(&info);
      // Return
      if (cpl_error_get_code())
	return -1;
      else
	return 0;
    }
  // load all OI-FITS files (squared visibilities and bispectra)
  it = cpl_frameset_iterator_new(info.frameset);
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      // we use only calibrated target interferometric data
      if (strcmp(cpl_frame_get_tag(frame), MAT_DO_TARGET_CAL_INT) == 0)
	{
	  cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
	  cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(frame));
	  info.oifits = mat_oifits_load(frame);
	  if (info.oifits != NULL)
	    {
	      info.fname  = cpl_strdup(cpl_frame_get_filename(frame));
	      for (i = 0; i < info.oifits->nbtupel; i++)
		{
		  mat_oifits_select_tupel(info.oifits, i);
		  cpl_msg_info(cpl_func, "  processing data for ARRNAME=%s, INSNAME=%s", info.oifits->oivis->arrayname, info.oifits->oivis->insname);
		  mat_est_phase(&info, i);
		}
	      break;
	    }
	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  if (frame != NULL)
    { // We have a frame containing calibrated intreferometric values
      // -> we can store them as a new file
      //cpl_frame        *rframe = NULL;
      char              fname[256];
      snprintf(fname, 256, "aphase.fits");
      /*
	rframe = cpl_frame_new();
	cpl_frame_set_filename(rframe, fname);
	//cpl_frame_set_tag(rframe, MATISSE_REC_PROCATG);
	cpl_frame_set_type(rframe, CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(rframe, CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(rframe, CPL_FRAME_LEVEL_FINAL);
	if (cpl_error_get_code()) {
	cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
	cpl_error_get_code(),
	cpl_error_get_message());
	cpl_frame_delete(rframe);
	return CPL_ERROR_UNSPECIFIED;
	}
      */
      mat_oifits_save(info.oifits, frameset, frameset, parlist, "mat_est_aphase", fname, 9);
      /* Log the saved file in the input frameset */
      //cpl_frameset_insert(frameset, rframe);
    }

  mat_info_delete(&info);
  /* Return */
  if (cpl_error_get_code())
    {
      cpl_msg_error(cpl_func, "where %s message %s",
		    cpl_error_get_where(), cpl_error_get_message());
      return -1;
    }
  else
    return 0;
}
