/* $Id: mat_est_corr.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_corr.c $
 */
/*------------------------------------------------------------------------------
                                   Includes
 ------------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_est_corr_lib.h"
/*------------------------------------------------------------------------------
                                   Define
 ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_est_corr"



/* Plugin detailed description */
static const char *
mat_est_corr_help = "This recipe derives the uncontaminated complex correlated flux from \n"
  "each recorded interferogram. The uncotaminated complex correlated flux is the Fourier\n"
  " Transform of the interferogram without its huge low frequency part which contamintes\n"
  "the fringes peaks. This contamination could be compensated by applying the OPD modulation algorithm\n"
  "(useOpdMod=TRUE). In addition, the user can use to apply a coherent integration of the frames  \n"
  " voer a certain duration defined by the parameter coherentIntegTime.\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    TARGET_CAL             Cal data (if target)        Yes (if target)\n"
  "    CALIB_CAL              Cal data (if calibrator     Yes (if calibrator or lab data \n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    OBJ_CORR_FLUX          Correlated flux\n"
  ;
static int mat_est_corr_create(cpl_plugin *);
static int mat_est_corr_exec(cpl_plugin *);
static int mat_est_corr_destroy(cpl_plugin *);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = (cpl_plugin *)recipe;
    cpl_plugin_init(plugin,
                     CPL_PLUGIN_API,
                     MATISSE_BINARY_VERSION,
                     CPL_PLUGIN_TYPE_RECIPE,
                     "mat_est_corr",
                     "Estimation of the Correlated Flux",
                     mat_est_corr_help,
                     "Philippe Berio",
                     PACKAGE_BUGREPORT,
                     "GPL",
                     mat_est_corr_create,
                     mat_est_corr_exec,
                     mat_est_corr_destroy);
    cpl_pluginlist_append(list, plugin);
    return 0;
    }

static int mat_est_corr_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameter *p;
    recipe->parameters = cpl_parameterlist_new();

    p = cpl_parameter_new_value("matisse.mat_est_corr.useOpdMod",
				CPL_TYPE_BOOL, 
				"useOpdMod option. If TRUE, OPD modulation algorithm applied.", 
				"matisse.mat_est_corr",FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "useOpdMod") ;
    cpl_parameterlist_append(recipe->parameters, p) ;

    p = cpl_parameter_new_value("matisse.mat_est_corr.coherentIntegTime",
				CPL_TYPE_DOUBLE, "Specify a coherent integration time (in s)", 
				"matisse.mat_est_corr",0.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "coherentIntegTime") ;
    cpl_parameterlist_append(recipe->parameters, p) ;

    p = cpl_parameter_new_value("matisse.mat_raw_estimates.spectralBinning",
    				CPL_TYPE_INT,
    				 "Spectral Binning in pixels. Should be an odd integer number. If aven even integer is given, the pipeline will select the next odd integer.",
    				"matisse.mat_raw_estimates",1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "spectralBinning") ;
    cpl_parameterlist_append(recipe->parameters, p) ;

    
    cpl_msg_info("mat_est_corr_create", "Parameters created.");

    return 0;
}
static int mat_est_corr_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
    cpl_parameterlist_delete(recipe->parameters);
    return 0;
}


static int mat_est_corr_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe = (cpl_recipe *)plugin;
     
    if (mat_est_corr_lib(recipe->frames,recipe->parameters,RECIPE_NAME)) {
      cpl_msg_warning(cpl_func,"Error in mat_est_corr_lib");
      return CPL_ERROR_ILLEGAL_OUTPUT;
    }

    return 0;
}

