/* $Id: mat_est_shift.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_shift.c $
 */
/*------------------------------------------------------------------------------
  Includes
  ------------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_imagingdetector.h"
#include "mat_apply_staticcalib.h"
#include "mat_imagingdata.h"
#include "mat_shift.h"
#include "mat_badpixel.h"
#include "mat_nonlinearity.h"
#include "mat_obsflat.h"
#include "mat_const.h"
/*------------------------------------------------------------------------------
  Define
  ------------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_est_shift"
#define MATISSE_DO_OBSFLAT         "OBS_FLATFIELD"
#define MATISSE_DO_BADPIXEL         "BADPIX"
#define MATISSE_DO_NONLIN         "NONLINEARITY"

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static const char *mat_est_shift_help = 
  "This plugin calculates the distorsion coefficients of the photometric \n"
  "and interferometric channels, and it determines the relative shifts \n"
  "between each photometric channel and the interferometric one. The input \n"
  "data are: a series of images containing the laboratory background for \n"
  "each pinehole and for the slit (DISTOR_HOTDARK and SPECTRA_HOTDARK), a \n"
  "series of images of the three pinholes (DISTOR_IMAGES), a series of \n"
  "spectral images (SPECTRA_IMAGES), the static bad pixel map (BADPIX), \n"
  "the static nonlinearity map (NONLINEARITY) and the observation specific \n"
  "flatfield map (OBS_FLATFIELD). This plugin produces the SHIFT map and \n"
  "computes the QC DET<i> PICOV<l> (the relative coverage of the \n"
  "interferometric channel by the photometric channels).\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    DISTOR_HOTDARK         data Hole/Background        Yes\n"
  "    DISTOR_IMAGES          data Hole/Lamp              Yes\n"
  "    SPECTRA_HOTDARK        data Slit/Foil/Background   Yes\n"
  "    SPECTRA_IMAGES         data Slit/Foil/Lamp         Yes\n"
  "    OBS_FLATFIELD          Observing Flat Field        Yes\n"
  "    BADPIX                 Bad Pixel Map               Yes\n"
  "    NONLINEARITY           Nonlinearity Map            Yes\n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    SHIFT_MAP              Distorsion Map with Spectral Dispersion Law\n";

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_est_shift_create(cpl_plugin *);
static int mat_est_shift_exec(cpl_plugin *);
static int mat_est_shift_destroy(cpl_plugin *);
static int mat_est_shift(cpl_parameterlist *parlist, cpl_frameset *frames);

/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_est_shift",
		  "Estimation of the Shift Map",
		  mat_est_shift_help,
		  "Philippe Berio",
		  PACKAGE_BUGREPORT,
		  "GPL",
		  mat_est_shift_create,
		  mat_est_shift_exec,
		  mat_est_shift_destroy);
  cpl_pluginlist_append(list, plugin);

  return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_est_shift_create(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameter *p, *p1;

  recipe->parameters = cpl_parameterlist_new();
  p = cpl_parameter_new_value("matisse.mat_est_shift.debug",
			      CPL_TYPE_BOOL, 
			      "This parameter allows to produce debugging informations", 
			      "matisse.mat_est_shift",FALSE);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug");
  cpl_parameterlist_append(recipe->parameters, p);

  p1 = cpl_parameter_new_value("matisse.mat_est_shift.obsCorrection",
			      CPL_TYPE_BOOL, 
			      "This parameter allows to correct the spectral dispersion law and \
the shifts between photometric and interferometric channels \
from observations of Be stars and sky", 
			      "matisse.mat_est_shift",TRUE);
  cpl_parameter_set_alias(p1, CPL_PARAMETER_MODE_CLI, "obsCorrection");
  cpl_parameterlist_append(recipe->parameters, p1);
 
  return 0;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the ’create’ function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_est_shift_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface. This plugin
   calculates the distorsion coefficients of the photometric and interferometric
   channels, and it determines the relative shifts between each photometric 
   channel and the interferometric one. The input data are: a series of images 
   containing the laboratory background for each pinehole and for the slit 
   (DISTOR_HOTDARK and SPECTRA_HOTDARK), a series of images of the three pinholes
   (DISTOR_IMAGES), a series of spectral images (SPECTRA_IMAGES), the static 
   bad pixel map (BADPIX), the static nonlinearity map (NONLINEARITY) and the 
   observation specific flatfield map (OBS_FLATFIELD). This plugin produces the 
   SHIFT map and computes the QC DETi PICOVl (the relative coverage of the 
   interferometric channel by the photometric channels).
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_est_shift_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe=(cpl_recipe *)plugin;
  } else {
    return -1;
  }
  return mat_est_shift(recipe->parameters,recipe->frames);
}

int mat_est_shift(cpl_parameterlist *parlist, cpl_frameset *frames) {
  cpl_parameter *debug=NULL;
  cpl_parameter *obsCorrection=NULL;
  cpl_frame *cur_frame=NULL;
  cpl_frame *first_spectra=NULL;
  cpl_propertylist *plist=NULL;
  cpl_propertylist *plist_shift=NULL;
  /* cpl_propertylist *plist_tmp=NULL; */
  cpl_table *table=NULL;

  cpl_frameset *usedframes=NULL;
  cpl_frame *sel_frame=NULL;

  char *pszFileName=NULL;
  char *pszFileTag=NULL;
  char *pszChipName=NULL;
  char *pszHole=NULL;
  char *keyExtname=NULL;
  char filenameFits[]="SHIFT_MAP.fits";

  int i=0;
  /* int j=0; */
  /* int k=0; */
  int position=0;
  int nbframes[4] = {0, 0, 0, 0};
  int nbdistorMax_h1=0;
  int nbdistor_hotdarkMax=0;
  int nbdistorMax_h2=0;
  int nbdistorMax_h3=0;
  int nbdistor_h1=0;
  int nbdistor_h2=0;
  int nbdistor_h3=0;
  int nbdistor_hotdark=0;
  int nbspectra=0;
  int nbspectra_hotdark=0;
  int nbExtent=0;


  mat_gendata **distor=NULL;
  mat_gendata **distor_hotdark=NULL;
  mat_gendata **spectra=NULL;
  mat_gendata **spectra_hotdark=NULL;
  mat_gendata *gendata_tmp=NULL;
  mat_gendata *gendata_tmp_calib=NULL;

  /* mat_imagingdata *imgdata_tmp=NULL; */

  mat_shiftmap *shiftmap=NULL;
  mat_badpixel *bpm=NULL;
  mat_nonlinearity *nonlin=NULL;
  mat_obsflat *obsflat=NULL;
  cpl_frame *bpmframe = NULL;
  int bpmcount=0;
  cpl_frame *nonlinframe = NULL;
  int nonlincount=0;
  cpl_frame *obsflatframe = NULL;
  int obsflatcount=0;
  cpl_frameset_iterator *it=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  /* float ditH1=0.; */
  /* float ditH2=0.; */
  /* float ditH3=0.; */

  debug=cpl_parameterlist_find(parlist,
				     "matisse.mat_est_shift.debug");
  obsCorrection=cpl_parameterlist_find(parlist,
				     "matisse.mat_est_shift.obsCorrection");
  usedframes=cpl_frameset_new();
  // LOAD STATIC CALIB MAPS
  bpmcount = cpl_frameset_count_tags(frames, MATISSE_DO_BADPIXEL);
  if (bpmcount > 1) {
    cpl_msg_error(cpl_func, "only zero or one badpixel map is allowed in the SOF");
    cpl_frameset_delete(usedframes);
    return -1;
  }
  /* load the optional badpixel map */
  if (bpmcount == 1) {
    bpmframe = cpl_frameset_find(frames, MATISSE_DO_BADPIXEL); // already checked previously!
    cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_CALIB);
    sel_frame=cpl_frame_duplicate(bpmframe);
    cpl_frameset_insert(usedframes,sel_frame);
    bpm = mat_badpixel_load(bpmframe);
  }
  nonlincount = cpl_frameset_count_tags(frames, MATISSE_DO_NONLIN);
  if (nonlincount > 1) {
    cpl_msg_error(cpl_func, "only zero or one nonlin map is allowed in the SOF");
    return -1;
  }
  /* load the optional nonlin map */
  if (nonlincount == 1) {
    nonlinframe = cpl_frameset_find(frames, MATISSE_DO_NONLIN); // already checked previously!
    cpl_frame_set_group(nonlinframe, CPL_FRAME_GROUP_CALIB);
    sel_frame=cpl_frame_duplicate(nonlinframe);
    cpl_frameset_insert(usedframes,sel_frame);
    nonlin = mat_nonlinearity_load(nonlinframe);
  }
  obsflatcount = cpl_frameset_count_tags(frames, MATISSE_DO_OBSFLAT);
  if (obsflatcount > 1) {
    cpl_msg_error(cpl_func, "only one flatfield map is allowed in the SOF");
    return -1;
  }
  /* load the optional flatfield map */
  if (obsflatcount == 1) {
    obsflatframe = cpl_frameset_find(frames, MATISSE_DO_OBSFLAT); // already checked previously!
    cpl_frame_set_group(obsflatframe, CPL_FRAME_GROUP_CALIB);
    sel_frame=cpl_frame_duplicate(obsflatframe);
    cpl_frameset_insert(usedframes,sel_frame);
    obsflat = mat_obsflat_load(obsflatframe);
  }

  // 1. Check the data type of the input frames. 

  // Find the number of each type of file
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      // Get Classification Tag
      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
      pszHole = NULL;
      plist=cpl_propertylist_load(pszFileName,0);

      // Increment the number of files var which contains this tag
      // a. Case DISTOR_IMAGES
      if (strcmp(pszFileTag,"DISTOR_IMAGES") == 0 )
	{
	  pszChipName= 
	    (char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
	  if (pszChipName == NULL) {
	    cpl_msg_error(cpl_func,"No ESO DET CHIP NAME keyword in the primary header of %s",pszFileName);
	    return CPL_ERROR_UNSPECIFIED;
	  }
	  // Get the hole number depending on the band (N or L)
	  if(strcmp(pszChipName, "AQUARIUS") == 0) {
	    if (cpl_propertylist_has(plist,"ESO INS SFN POS") == 1) {
	      pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFN POS");
	    } else {
	      pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFN NAME");
	    }
	  }
	  else if(strcmp(pszChipName, "HAWAII2RG") == 0 || 
		  strcmp(pszChipName, "HAWAII-2RG") == 0) {
	    if (cpl_propertylist_has(plist,"ESO INS SFL POS") == 1) {
	      pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFL POS");
	    } else {
	      pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFL NAME");
	    }
	  }
	  else {
	    cpl_msg_warning(cpl_func,
			    "Wrong chip name found in %s", pszFileName);
	    return CPL_ERROR_ILLEGAL_INPUT;
	  }
	  if(pszHole == NULL) {
	    cpl_msg_warning(cpl_func,
			    "No hole number found in %s", pszFileName);
	    return CPL_ERROR_ILLEGAL_INPUT;
	  }
	  // Increment the number of DISTOR_IMAGES according to the hole number
	  if(strcmp(pszHole, "H1") == 0 || strcmp(pszHole, "DIST1") == 0) {
	    /* ditH1=cpl_propertylist_get_double(plist,"ESO DET SEQ1 DIT"); */
	    nbdistorMax_h1++;
	  } else if(strcmp(pszHole, "H2") == 0 || strcmp(pszHole, "HOLE1") == 0
		    || strcmp(pszHole, "1.00") == 0 || strcmp(pszHole, "1.50") == 0) { 
	    /* ditH2=cpl_propertylist_get_double(plist,"ESO DET SEQ1 DIT"); */
	    nbdistorMax_h2++;
	  } else if(strcmp(pszHole, "H3") == 0 || strcmp(pszHole, "DIST2") == 0) {
	    /* ditH3=cpl_propertylist_get_double(plist,"ESO DET SEQ1 DIT"); */
	    nbdistorMax_h3++;
	  }
	}
      // c. Case SPECTRA_IMAGES
      else if (strcmp(pszFileTag,"SPECTRA_IMAGES") == 0 ) {
	nbframes[2]++;
	if (nbframes[2] == 1) {
	  first_spectra=cpl_frame_duplicate(cur_frame);
	  if (first_spectra == NULL) {
	    cpl_msg_error(cpl_func, "could not duplicate frame");
	    cpl_propertylist_delete(plist);
	    return CPL_ERROR_UNSPECIFIED;
	  }
	}
      }
      // d. Case SPECTRA_HOTDARK
      else if (strcmp(pszFileTag,"SPECTRA_HOTDARK") == 0 ) {
	nbframes[3]++;
      }
      // Get next frame from SOF
      cpl_propertylist_delete(plist);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }

  // DISTOR_HOTDARK now
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      // Get Classification Tag
      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
      pszHole = NULL;
      plist=cpl_propertylist_load(pszFileName,0);
      // b. Case DISTOR_HOTDARK
      if (strcmp(pszFileTag,"DISTOR_HOTDARK") == 0 ) 
	{
	  nbdistor_hotdarkMax++;
	}
       // Get next frame from SOF
      cpl_propertylist_delete(plist);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }

  // Verify the number of files is equal for each type
  if(nbdistorMax_h1 != nbdistorMax_h2 || 
     nbdistorMax_h1 != nbdistorMax_h3 || 
     nbdistor_hotdarkMax == 0 || nbframes[2] == 0 ||
     nbframes[3] == 0) {
    cpl_msg_warning(cpl_func, "Files are missing");
    mat_badpixel_delete(bpm);
    mat_nonlinearity_delete(nonlin);
    mat_obsflat_delete(obsflat);
    cpl_frameset_delete(usedframes);
    cpl_frame_delete(first_spectra);
    return CPL_ERROR_FILE_NOT_FOUND;
  }
  nbframes[0]=nbdistorMax_h1;
  nbframes[1]=nbdistor_hotdarkMax;



  // 2. Load input frames in DRL structures. 
  // Allocate mat_gendata structures according to the number of files
  distor = cpl_calloc(nbframes[0] * 3, sizeof(mat_gendata*));
  if (distor == NULL) {
    cpl_msg_error(cpl_func, "could not allocate memory for mat_gendata");
    return CPL_ERROR_UNSPECIFIED;
  }
  distor_hotdark = cpl_calloc(nbframes[1], sizeof(mat_gendata*));
  if (distor_hotdark == NULL) {
    cpl_msg_error(cpl_func, "could not allocate memory for mat_gendata");
    cpl_free(distor);
    return CPL_ERROR_UNSPECIFIED;
  }
  spectra = cpl_calloc(nbframes[2], sizeof(mat_gendata*));
  if (spectra == NULL) {
    cpl_msg_error(cpl_func, "could not allocate memory for mat_gendata");
    cpl_free(distor);
    cpl_free(distor_hotdark);
    return CPL_ERROR_UNSPECIFIED;
  }
  spectra_hotdark = cpl_calloc(nbframes[3], sizeof(mat_gendata*));
  if (spectra_hotdark == NULL) {
    cpl_msg_error(cpl_func, "could not allocate memory for mat_gendata");
    cpl_free(distor);
    cpl_free(distor_hotdark);
    cpl_free(spectra);
    return CPL_ERROR_UNSPECIFIED;
  }

  nbdistor_h1 = nbdistorMax_h1;
  nbdistor_h2 = nbdistorMax_h2;
  nbdistor_h3 = nbdistorMax_h3;
  nbdistor_hotdark = nbdistor_hotdarkMax;
  nbspectra = nbframes[2];
  nbspectra_hotdark = nbframes[3];

  // Load each input frame in the right array of structures

  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileName= (char *)cpl_frame_get_filename( cur_frame );
      nbExtent=cpl_frame_get_nextensions(cur_frame);
      plist=cpl_propertylist_load(pszFileName,0); /* already tested */
      pszFileTag= (char *)cpl_frame_get_tag( cur_frame );

      pszChipName=
	(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
      if (pszChipName == NULL) {
	cpl_msg_error(cpl_func,"No ESO DET CHIP NAME keyword in the primary header of %s",pszFileName);
	cpl_free(distor);
	cpl_free(distor_hotdark);
	cpl_free(spectra);
	cpl_free(spectra_hotdark);
	return CPL_ERROR_UNSPECIFIED;
      }
      if (!strcmp(pszFileTag,"DISTOR_IMAGES") || 
	  !strcmp(pszFileTag,"DISTOR_HOTDARK")) {
	if(strcmp(pszChipName, "AQUARIUS") == 0) {
	  if (cpl_propertylist_has(plist,"ESO INS SFN POS") == 1) {
	    pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFN POS");
	  } else {
	    pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFN NAME");
	  }
	}
	else if(strcmp(pszChipName, "HAWAII2RG") == 0 || 
		strcmp(pszChipName, "HAWAII-2RG") == 0) {
	  if (cpl_propertylist_has(plist,"ESO INS SFL POS") == 1) {
	    pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFL POS");
	  } else {
	    pszHole= (char *)cpl_propertylist_get_string(plist,"ESO INS SFL NAME");
	  }
	}
      }
      else pszHole= NULL;




      // a. Case DISTOR_IMAGES
      if (!strcmp(pszFileTag,"DISTOR_IMAGES")) {
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);

	if(strcmp(pszHole, "H1") == 0 || strcmp(pszHole, "DIST1") == 0) {
	  position = nbdistorMax_h1 - nbdistor_h1; 
	  nbdistor_h1 --;
	}
	else if(strcmp(pszHole, "H2") == 0 || strcmp(pszHole, "HOLE1") == 0
		|| strcmp(pszHole, "1.00") == 0 || strcmp(pszHole, "1.50") == 0) {
	  position = 2 * nbdistorMax_h2 - nbdistor_h2; 
	  nbdistor_h2 --;
	}
	else {
	  position = 3 * nbdistorMax_h3 - nbdistor_h3; 
	  nbdistor_h3 --;
	}

	// Load FITS files in data structurse taking care of the size of 
	// the binary tables
	// Load the whole FITS file
	gendata_tmp = mat_gendata_load(cur_frame, CPL_TYPE_FLOAT);
	if (gendata_tmp == NULL) {
	  cpl_msg_error(cpl_func,"could not load a mat_gendata structure");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}




	// Apply the static calibration 
	gendata_tmp_calib = mat_calibration_detector(gendata_tmp, bpm, NULL, nonlin, obsflat, NULL,
						     FULL_COMPENSATION & ~CHANNEL_BIAS_COMPENSATION, 1, 0);
	/* gendata_tmp_calib = mat_apply_staticcalib(gendata_tmp, bpm, nonlin, obsflat); */


	/* cpl_image_save(gendata_tmp_calib->imgdata->list_frame[0] */
	/* 		   ->list_subwin[0]->imgreg[0], "tata.fits",  */
	/* 		   CPL_BPP_IEEE_FLOAT, gendata_tmp_calib->keywords,  */
	/* 		   CPL_IO_CREATE);  */
	/* exit(1); */
	if (gendata_tmp_calib == NULL) {
	  cpl_msg_error(cpl_func,"could not apply the static calibration");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp);
	// Sum all frames of each sub-window in one frame 
	distor[position] = cpl_calloc(1,sizeof(mat_gendata));
	if (distor[position] == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	distor[position]->keywords= cpl_propertylist_duplicate(gendata_tmp_calib->keywords);
	if (distor[position]->keywords == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a cpl_propertylist");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	distor[position]->imgdet= mat_imagingdetector_duplicate(gendata_tmp_calib->imgdet);
	if (distor[position]->imgdet == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a mat_imagngdetector");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	distor[position]->array= mat_array_duplicate(gendata_tmp_calib->array);
	if (distor[position]->array == NULL) {
	  cpl_msg_warning(cpl_func,"could not duplicate a mat_array");
	  /* return CPL_ERROR_UNSPECIFIED; */
	}
	distor[position]->imgdata=mat_imagingdata_sum(gendata_tmp_calib->imgdata, CPL_TYPE_FLOAT);
	if (distor[position]->imgdata == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a mat_arraysum all frames of a mat_imagingdata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp_calib);


	cpl_msg_info(cpl_func, "distor structure loaded.");
      }
      // b. Case DISTOR_HOTDARK
      else if (!strcmp(pszFileTag,"DISTOR_HOTDARK")) {
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);
	position = nbdistor_hotdarkMax - nbdistor_hotdark; 
	nbdistor_hotdark --;

	// Load the whole FITS file
	gendata_tmp = mat_gendata_load(cur_frame, CPL_TYPE_FLOAT);
	if (gendata_tmp == NULL) {
	  cpl_msg_error(cpl_func,"could not load a mat_gendata structure");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	// Apply the static calibration 
	gendata_tmp_calib = mat_calibration_detector(gendata_tmp, bpm, NULL, nonlin, obsflat, NULL,
						     FULL_COMPENSATION & ~CHANNEL_BIAS_COMPENSATION, 1, 0);
	/* gendata_tmp_calib = mat_apply_staticcalib(gendata_tmp, bpm, nonlin, obsflat); */
	if (gendata_tmp_calib == NULL) {

	  cpl_msg_error(cpl_func,"could not apply the static calibration");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp);
	// Sum all frames of each sub-window in one frame 
	distor_hotdark[position] = cpl_calloc(1,sizeof(mat_gendata));
	if (distor_hotdark[position] == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*distor_hotdark[position]).keywords= cpl_propertylist_duplicate((*gendata_tmp_calib).keywords);
	if (distor_hotdark[position]->keywords == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a cpl_propertylist");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*distor_hotdark[position]).imgdet= mat_imagingdetector_duplicate((*gendata_tmp_calib).imgdet);
	if (distor_hotdark[position]->imgdet == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a mat_imagingdetector");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*distor_hotdark[position]).array= mat_array_duplicate((*gendata_tmp_calib).array);
	if (distor_hotdark[position]->array == NULL) {
	  cpl_msg_warning(cpl_func,"could not duplicate a mat_array");
	  /* return CPL_ERROR_UNSPECIFIED; */
	}
	(*distor_hotdark[position]).imgdata= mat_imagingdata_sum((*gendata_tmp_calib).imgdata, CPL_TYPE_FLOAT);
	if (distor_hotdark[position]->imgdata == NULL) {
	  cpl_msg_error(cpl_func,"could not sum all frames of a mat_imagingdata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp_calib);
	cpl_msg_info(cpl_func, "distor_hotdark structure loaded.");
	
      }

      // c. Case SPECTRA_IMAGES
      else if (!strcmp(pszFileTag,"SPECTRA_IMAGES")) {
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);
	position = nbframes[2] - nbspectra; 

	// Load the whole FITS file
	gendata_tmp = mat_gendata_load(cur_frame, CPL_TYPE_FLOAT);
	if (gendata_tmp == NULL) {
	  cpl_msg_error(cpl_func,"could not load a mat_gendata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	// Apply the static calibration 
	gendata_tmp_calib = mat_calibration_detector(gendata_tmp, bpm, NULL, nonlin, obsflat, NULL,
						     FULL_COMPENSATION & ~CHANNEL_BIAS_COMPENSATION, 1, 0);
	/* gendata_tmp_calib = mat_apply_staticcalib(gendata_tmp, bpm,nonlin,obsflat);	 */

	/* cpl_image_save(gendata_tmp->imgdata->list_frame[0]->list_subwin[10]->imgreg[0], "image10.fits", */
	/* 	       CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE); */
	/* cpl_image_save(gendata_tmp_calib->imgdata->list_frame[0]->list_subwin[2]->imgreg[0], "image11.fits", */
	/* 	       CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE); */



	if (gendata_tmp_calib == NULL) {
	  cpl_msg_error(cpl_func,"could not apply the static calibration");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp);
	// Sum all frames of each sub-window in one frame 
	spectra[position] = cpl_calloc(1,sizeof(mat_gendata));
	if (spectra[position] == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*spectra[position]).keywords= cpl_propertylist_duplicate((*gendata_tmp_calib).keywords);
	if (spectra[position]->keywords == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a cpl_propertylist");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*spectra[position]).imgdet= mat_imagingdetector_duplicate((*gendata_tmp_calib).imgdet);
	if (spectra[position]->imgdet == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a mat_imagingdetector");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*spectra[position]).array= mat_array_duplicate((*gendata_tmp_calib).array);
	if (spectra[position]->array == NULL) {
	  cpl_msg_warning(cpl_func,"could not duplicate a mat_array");
	  /* return CPL_ERROR_UNSPECIFIED; */
	}
	(*spectra[position]).imgdata= mat_imagingdata_sum((*gendata_tmp_calib).imgdata, CPL_TYPE_FLOAT);
	if (spectra[position]->imgdata == NULL) {
	  cpl_msg_error(cpl_func,"could not sum all frames of a mat_imagingdata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp_calib);
	nbspectra--;
	cpl_msg_info(cpl_func, "spectra structure loaded.");
      }
      // d. Case SPECTRA_HOTDARK
      else if (!strcmp(pszFileTag,"SPECTRA_HOTDARK")) {
	sel_frame=cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	cpl_frameset_insert(usedframes,sel_frame);

	position = nbframes[3] - nbspectra_hotdark; 
	// Load the whole FITS file
	gendata_tmp = mat_gendata_load(cur_frame, CPL_TYPE_FLOAT);
	if (gendata_tmp == NULL) {
	  cpl_msg_error(cpl_func,"could not load a mat_gendata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	// Apply the static calibration 
	gendata_tmp_calib = mat_calibration_detector(gendata_tmp, bpm, NULL, nonlin, obsflat, NULL,
						     FULL_COMPENSATION & ~CHANNEL_BIAS_COMPENSATION
						     , 1, 0);
	/* gendata_tmp_calib = mat_apply_staticcalib(gendata_tmp, bpm,nonlin,obsflat);	 */
	if (gendata_tmp_calib == NULL) {
	  cpl_msg_error(cpl_func,"could not apply the static calibration");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp);
	// Sum all frames of each sub-window in one frame 
	spectra_hotdark[position] = cpl_calloc(1,sizeof(mat_gendata));
	if (spectra_hotdark[position] == NULL) {
	  cpl_msg_error(cpl_func,"could not allocate memory for mat_gendata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*spectra_hotdark[position]).keywords= cpl_propertylist_duplicate((*gendata_tmp_calib).keywords);
	if (spectra_hotdark[position]->keywords == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a cpl_propertylist");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*spectra_hotdark[position]).imgdet= mat_imagingdetector_duplicate((*gendata_tmp_calib).imgdet);
	if (spectra_hotdark[position]->imgdet == NULL) {
	  cpl_msg_error(cpl_func,"could not duplicate a mat_imagingdetector");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	(*spectra_hotdark[position]).array= mat_array_duplicate((*gendata_tmp_calib).array);
	if (spectra_hotdark[position]->array == NULL) {
	  cpl_msg_warning(cpl_func,"could not duplicate a mat_array");
	  /* return CPL_ERROR_UNSPECIFIED; */
	}
	(*spectra_hotdark[position]).imgdata= mat_imagingdata_sum((*gendata_tmp_calib).imgdata, CPL_TYPE_FLOAT);
	if (spectra_hotdark[position]->imgdata == NULL) {
	  cpl_msg_error(cpl_func,"could not sum all frames of a mat_imagingdata");
	  cpl_free(distor);
	  cpl_free(distor_hotdark);
	  cpl_free(spectra);
	  cpl_free(spectra_hotdark);
	  return CPL_ERROR_UNSPECIFIED;
	}
	mat_gendata_delete(gendata_tmp_calib);
	nbspectra_hotdark--;
	cpl_msg_info(cpl_func, "spectra_hotdark structure loaded.");
      }
      cpl_propertylist_delete(plist);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }


  // 4. Compute the distorsion coefﬁcients, spectral shifts and QC1 parameters. 
  shiftmap = mat_compute_shift(distor_hotdark, spectra_hotdark,
			       distor, spectra, nbframes,debug,obsCorrection);

  if(shiftmap == NULL) {
    cpl_msg_error(cpl_func, "NULL pointer returned by mat_compute_shift");
    mat_badpixel_delete(bpm);
    mat_nonlinearity_delete(nonlin);
    mat_obsflat_delete(obsflat);


    cpl_frame_delete(first_spectra);
    for(i=0;i<nbdistorMax_h1 * 3;i++) {
      mat_gendata_delete(distor[i]); 
    }
    cpl_free(distor);
  
    for(i=0;i<nbdistor_hotdarkMax;i++) {
      mat_gendata_delete(distor_hotdark[i]); 
    }
    cpl_free(distor_hotdark);

    for(i=0;i<nbframes[2];i++) {
      mat_gendata_delete(spectra[i]); 
    }
    cpl_free(spectra);
    for(i=0;i<nbframes[3];i++) {
      mat_gendata_delete(spectra_hotdark[i]); 
    }
    cpl_free(spectra_hotdark);

    mat_shiftmap_free(shiftmap);
    return CPL_ERROR_ILLEGAL_OUTPUT;
  }

  // 5. Save the mat_shiftmap structure in a FITS file 
  // 5.1 Create the file with primary header and SHIFT_TABLE
  nbExtent=cpl_frame_get_nextensions(first_spectra);

  // Find the table IMAGING_DATA
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(first_spectra),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (!strcmp(keyExtname,"IMAGING_DATA")) {
      plist_shift= cpl_propertylist_duplicate(plist);
      // Rename the table IMAGING_DATA to SHIFT_MAP 
      cpl_propertylist_update_string(plist_shift, "EXTNAME", "SHIFT_MAP");
      // Substract the number of calibration regions to the number of regions
      cpl_propertylist_update_int(plist_shift, "NREGION", shiftmap->imgdet->nbregion);
      // Save a new FITS file using the modified header and SHIFT_MAP table
      //mat_shiftmap_save(shiftmap, plist_shift, "shift.fits");
      mat_shiftmap_save(shiftmap,plist_shift,frames,usedframes,parlist,
			RECIPE_NAME,"SHIFT_MAP","IMAGE","F",filenameFits);
      cpl_propertylist_delete(plist_shift);
    }
    cpl_propertylist_delete(plist);
  }
  cpl_frameset_delete(usedframes);

  // 5.2 Append the others tables from the raw file except IMAGING_DATA  
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(first_spectra),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(first_spectra),i+1,0);
      // Case IMAGING_DETECTOR : Append the table without calibration regions
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
	mat_imagingdetector_save(shiftmap->imgdet, NULL, filenameFits);
      }
      // Other cases : append the table from the raw file
      if (!strcmp(keyExtname,"ARRAY_DESCRIPTION") ||
	  !strcmp(keyExtname,"ARRAY_GEOMETRY") ||
	  !strcmp(keyExtname,"OPTICAL_TRAIN")) {
	cpl_table_save(table, NULL, plist, filenameFits, CPL_IO_EXTEND);
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(plist);
  }
  

  // Free memory
  mat_badpixel_delete(bpm);
  mat_nonlinearity_delete(nonlin);
  mat_obsflat_delete(obsflat);


  cpl_frame_delete(first_spectra);
  for(i=0;i<nbdistorMax_h1 * 3;i++) {
    mat_gendata_delete(distor[i]); 
  }
  cpl_free(distor);
  
  for(i=0;i<nbdistor_hotdarkMax;i++) {
    mat_gendata_delete(distor_hotdark[i]); 
  }
  cpl_free(distor_hotdark);

  for(i=0;i<nbframes[2];i++) {
    mat_gendata_delete(spectra[i]); 
  }
  cpl_free(spectra);
  for(i=0;i<nbframes[3];i++) {
    mat_gendata_delete(spectra_hotdark[i]); 
  }
  cpl_free(spectra_hotdark);

  mat_shiftmap_free(shiftmap);

  return 0;
}
