/* $Id: mat_est_tfqual.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_tfqual.c $
 */
/*------------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/
#include <cpl.h>
#include "mat_drl.h"
#include "mat_oivis2.h"
#include "mat_utils.h"
#include <math.h>

/*------------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
#define TEL_NUMBER 4
#define RECIPE_NAME "mat_est_tfqual"

/* Plugin detailed description */
static const char *
mat_est_tfqual_help = 
  "This plugin estimates the quality of the OBs of a night based on the \n"
  "statistics of the absolute visibility transfer function. It uses a \n"
  "series of RAW_TF2 files. These files come from the observations of \n"
  "calibrators with the same instrumental configuration. Each OB is flagged: \n"
  "1=within 3 sigma of the median, 2=between 3 and 5 sigma and 3=outside \n"
  "5 sigma range. The results are saved in a FITS file as binary table \n"
  "(CAL_QUAL_FITS). No QC1 parameter created.\n"
  "\n"
  "Input Files:\n"
  "\n"
  "    DO category:           Explanation:                Required:\n"
  "    RAW_TF2                Transfer Functions          Yes \n"
  "\n"
  "Output Files:\n"
  "\n"
  "    DO category:           Explanation:\n"
  "    TF2_QUAL               Transfer function quality\n";
static int mat_est_tfqual_create(cpl_plugin *);
static int mat_est_tfqual_exec(cpl_plugin *);
static int mat_est_tfqual_destroy(cpl_plugin *);

static int mat_est_tfqual(cpl_parameterlist *parlist, cpl_frameset *frames);

int cpl_plugin_get_info(cpl_pluginlist *list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = (cpl_plugin *)recipe;
  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_est_tfqual",
		  "Estimation of the Transfer Function Quality",
		  mat_est_tfqual_help,
		  "Philippe Berio",
		  PACKAGE_BUGREPORT,
		  "GPL",
		  mat_est_tfqual_create,
		  mat_est_tfqual_exec,
		  mat_est_tfqual_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}

static int mat_est_tfqual_create(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  recipe->parameters = cpl_parameterlist_new();
  return 0;
}
static int mat_est_tfqual_destroy(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Estimation of the quality of the OBs of the night based on the 
   statistics of the absolute visibility transfer function
   @return int
*/
/*----------------------------------------------------------------------------*/

static int mat_est_tfqual_exec(cpl_plugin *plugin)
{
  cpl_recipe *recipe = (cpl_recipe *)plugin;
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe=(cpl_recipe *)plugin;
  } else {
    return -1;
  }
  return mat_est_tfqual(recipe->parameters,recipe->frames);
}


int mat_est_tfqual(cpl_parameterlist *parlist, cpl_frameset *frames) {
  cpl_frame *cur_frame = NULL;                                      
  cpl_frame *sel_frame = NULL;                                      
  cpl_frame *sel2_frame = NULL;                                     
  cpl_frameset *used_frames = NULL;                                                              
  cpl_frameset *good_frames = NULL;                                                              
  cpl_propertylist *p_list = NULL;                                                                   
  cpl_propertylist *p_header = NULL;                                                                                 
  cpl_table *table = NULL;
  cpl_table *tf_qual = NULL;                                                                     
  cpl_vector **median_vec = NULL;                                  
  cpl_vector *vector_aux = NULL;                                 
  mat_oifits *p_oifits = NULL;                           
  char *p_filename = NULL;
  char *p_filetag = NULL;
  char *keyExtname = NULL;
  /* char *insMod_ref = NULL; */
  /* char *insMod_aux = NULL; */
  char *file_name = NULL;
  char *str=NULL;
  char nameRef[80];
  char nameAux[80];
  /* int expert = 0; */
  int i = 0;
  int j = 0;
  int k = 0;
  int z = 0;
  int nb_extent = 0;
  int flag = 0;
  int nbbase = 0;
  int nbfile = 0;
  double *median = NULL;                                                                                
  double *deviation = NULL;                                         
  double **median_file = NULL;                                     
  cpl_array **quality_flag = NULL;                                  
  cpl_array *filename = NULL;                                       
  cpl_frameset_iterator *it=NULL;
  cpl_errorstate prestate= cpl_errorstate_get();

  file_name=cpl_malloc(20*sizeof(char));
  sprintf(file_name,"TF_QAUL.fits");
    
  used_frames = cpl_frameset_new();
  /* 1. Check the data type of the input frames. */ 
  /* prestate=cpl_errorstate_get(); */
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
  /* cur_frame = cpl_frameset_get_first(frames); */
  /* while(cur_frame) { */
    /*Get Classification Tag*/
      p_filename = (char *)cpl_frame_get_filename(cur_frame);
      p_filetag  = (char *)cpl_frame_get_tag(cur_frame);
      if (strcmp(p_filetag,"RAW_TF2") == 0) {
	/* if (mat_check_tag(p_filename, p_filetag, expert) == TRUE) { */
	cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	sel_frame = cpl_frame_duplicate(cur_frame);
	cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_CALIB);
	cpl_frameset_insert(used_frames,sel_frame);
	/* } */
      }
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }
  /*   cur_frame = cpl_frameset_get_next(frames); */
  /* } */

  /* 2. check telescopes */
  /*first file, references values*/
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(used_frames);
  cur_frame = cpl_frameset_iterator_get(it);
  /* cur_frame = cpl_frameset_get_first(used_frames); */
  good_frames = cpl_frameset_new();
  sel2_frame = cpl_frame_duplicate(cur_frame);
  cpl_frameset_insert(good_frames,sel2_frame);
  nb_extent = cpl_frame_get_nextensions(cur_frame);
  p_filename = (char *)cpl_frame_get_filename(cur_frame);  
  /*INS MODE value*/
  p_header = cpl_propertylist_load(p_filename,0);
  /* insMod_ref = (char *)cpl_propertylist_get_string(p_header, "ESO INS MODE"); */
  for(i=0;i<nb_extent;i++) {
    p_list = cpl_propertylist_load(cpl_frame_get_filename(cur_frame), i+1);
    keyExtname = (char *)cpl_propertylist_get_string(p_list, "EXTNAME");
    if (keyExtname != NULL) {
      if (!strcmp(keyExtname,"OI_ARRAY")){
	/*telescops names*/
	table = cpl_table_load(p_filename,i+1,0);
	for(j=0;j<cpl_table_get_nrow(table);j++) {
	  str=cpl_strdup(cpl_table_get_string(table, "STA_NAME",j));
	  if (j==0) {
	    strcpy(nameRef,str);
	  } else {
	    strcat(nameRef,str);
	  }
	  remove_spaces(nameRef);
	  cpl_free(str);
	}
	cpl_table_delete(table);
      }
    }
    cpl_propertylist_delete(p_list);
  }
  /*
    3. compare to first file
    *if INS MODE equals and telescops name equal: file ok
    *else file ko
    */
  if (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE) {
    do {
      cur_frame = cpl_frameset_iterator_get(it);
      if (cur_frame != NULL) {
	/* cur_frame = cpl_frameset_get_next(used_frames); */
	/* while(cur_frame) { */
	p_filename = (char *)cpl_frame_get_filename(cur_frame);
	p_list = cpl_propertylist_load(p_filename, 0);
	nb_extent = cpl_frame_get_nextensions(cur_frame);
	/*INS MODE value*/
	/* insMod_aux = */
	/*     (char *)cpl_propertylist_get_string(p_list, "ESO INS MODE"); */
	/* flag = 0; */
	/* if (!strcmp(insMod_aux,insMod_ref)) { */
	flag = 1;
	/* } */
	cpl_propertylist_delete(p_list);
	for(i = 0; i<nb_extent; i++) {
	  p_list =
	    cpl_propertylist_load(cpl_frame_get_filename(cur_frame),i+1);
	  keyExtname = (char *)cpl_propertylist_get_string(p_list,"EXTNAME");
	  if (keyExtname != NULL) {
	    if (!strcmp(keyExtname,"OI_ARRAY")) {
	      /*telescops names*/
	      table = cpl_table_load(p_filename,i+1,0);
	      for(j=0;j<cpl_table_get_nrow(table);j++) {
		str=cpl_strdup(cpl_table_get_string(table, 
						    "STA_NAME",j));
		if (j==0) {
		  strcpy(nameAux,str);
		} else {
		  strcat(nameAux,str);
		}
		cpl_free(str);
	      }
	      remove_spaces(nameAux);
	      if(strcmp(nameRef,nameAux)||  flag != 1) {
		cpl_msg_info(cpl_func, "bad file: %s\n",p_filename);
		cpl_msg_info(cpl_func,"We continue anyway...");
	      } else {
		sel2_frame = cpl_frame_duplicate(cur_frame);
		cpl_frameset_insert(good_frames,sel2_frame);
	      }
	      cpl_table_delete(table);
	    }
	  }
	  cpl_propertylist_delete(p_list);
	}
      }
    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  }
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }
  /*   cur_frame = cpl_frameset_get_next(used_frames); */
  /* } */

  cpl_frameset_dump(good_frames,NULL);

  /* 4. compute median */
  /*initialize array, vectors... */
  nbfile = cpl_frameset_get_size (good_frames);
  filename = cpl_array_new(nbfile, CPL_TYPE_STRING);
  it= cpl_frameset_iterator_new(good_frames);
  cur_frame = cpl_frameset_iterator_get(it);
  cpl_frameset_iterator_delete(it);
  /* cur_frame = cpl_frameset_get_first(good_frames); */
  cpl_array_set_string (filename, 0,
			(char *)cpl_frame_get_filename(cur_frame));
  p_oifits = mat_oifits_load(cur_frame);

  nbbase = (p_oifits->array->nbstation)*
    (p_oifits->array->nbstation-1)/2;

  //for each base, median of all files
  median = cpl_malloc(nbbase*sizeof(double));
  //for each base, deviation of all files
  deviation = cpl_malloc(nbbase*sizeof(double));
  //for each base, data of all files
  median_vec = cpl_malloc(nbbase*sizeof(cpl_vector *));
  /*for each file and each base, median*/
  median_file = cpl_malloc(nbfile*sizeof(double *));
  quality_flag = cpl_malloc(nbfile*sizeof(cpl_array *));
  for(i = 0; i<nbbase; i++) {
    median_vec[i] = cpl_vector_new(nbfile*p_oifits->oiwave->nbchannel);
  }
  for(i = 0; i<nbfile; i++) {
    median_file[i] = cpl_malloc(nbbase*sizeof(double));
    quality_flag[i] = cpl_array_new(nbbase,CPL_TYPE_INT);
  }
  //for each base concatenate data from mat_oifits
  mat_oifits_delete(p_oifits);
    
  z=0;
  k=0;
  it= cpl_frameset_iterator_new(frames);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
  /* cur_frame = cpl_frameset_get_first(good_frames); */
  /* while(cur_frame)  */
  /*   { */
      cpl_array_set_string (filename, z,
			    (char *)cpl_frame_get_filename(cur_frame));
      p_oifits = mat_oifits_load(cur_frame);
      for(j = 0; j<nbbase; j++) 
	{
	  vector_aux =
	    cpl_vector_wrap(p_oifits->oiwave->nbchannel,
			    p_oifits->oitf2->list_tf2[j]->tf2);
	  median_file[k][j] = cpl_vector_get_median(vector_aux);
	  for (i = 0; i<p_oifits->oiwave->nbchannel; i++) 
	    {
	      cpl_vector_set(median_vec[j],
			     i+k*p_oifits->oiwave->nbchannel,
			     cpl_vector_get(vector_aux,i));
	    }
	  cpl_vector_unwrap(vector_aux);
	}
      /*number of files*/
      k++;
      z++;
      mat_oifits_delete(p_oifits);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }
    /*   cur_frame = cpl_frameset_get_next(good_frames); */
    /* } */
  /*for each base, compute median and deviation of all files*/
  for (i = 0; i<nbbase; i++) {
    median[i] = cpl_vector_get_median(median_vec[i]);
    cpl_vector_add_scalar(median_vec[i],median[i]*(-1));
    for (j = 0; j<cpl_vector_get_size(median_vec[i]); j++) {
      cpl_vector_set(median_vec[i],j,
		     fabs(cpl_vector_get(median_vec[i],j)));
    }
    deviation[i] = cpl_vector_get_median(median_vec[i]);
    cpl_vector_delete(median_vec[i]);
  }
  cpl_free(median_vec);
  k = 0;
  for(i = 0; i<nbfile; i++) {
    for(j = 0; j<nbbase; j++) {
      k = fabs((median_file[i][j]-median[j])/deviation[j]);
      if (k<3) {
	cpl_array_set_int(quality_flag[i],j,1);
      } else if (k>=3 && k<5) {
	cpl_array_set_int(quality_flag[i],j,2);
      } else {
	cpl_array_set_int(quality_flag[i],j,3);
      }
    }
    cpl_free(median_file[i]);
  }
  cpl_free(median_file);
  cpl_free(median);
  cpl_free(deviation);

  /* *5. save in fits file */
  /* Prepare the keywords of the TF_QUAL binary table */
  tf_qual = cpl_table_new(nbfile);
  cpl_table_new_column(tf_qual, "OB_Name", CPL_TYPE_STRING);
  cpl_table_new_column_array(tf_qual, "Quality_Flag", CPL_TYPE_INT,nbbase);
  /* Fill the columns with the data */
  for(i = 0; i<nbfile; i++) {
    cpl_table_set_string(tf_qual, "OB_Name", i,
			 cpl_array_get_string(filename,i));
    cpl_table_set_array(tf_qual, "Quality_Flag", i,quality_flag[i]);
    cpl_array_delete(quality_flag[i]);
  }
  cpl_free(quality_flag);
  cpl_array_delete(filename);
    
  /*prepare the header*/
  cpl_propertylist_erase_regexp(p_header, "DPR", 0);
  cpl_propertylist_erase_regexp(p_header, "PRO", 0);
  cpl_propertylist_append_string(p_header, CPL_DFS_PRO_CATG, "CAL_QUAL");
  //  cpl_propertylist_append_string(p_header, CPL_DFS_PRO_SCIENCE, "F");
  cpl_propertylist_append_string(p_header, CPL_DFS_PRO_TECH, "");
    
  cpl_propertylist_update_string(p_header, "EXTNAME","TF_QUAL");
    
 


  cpl_dfs_save_table(frames, NULL, parlist,
		     good_frames, NULL,
		     tf_qual, p_header, "mat_est_tfqual",
		     p_header, NULL, PACKAGE "/" PACKAGE_VERSION, file_name);
    
  cpl_propertylist_delete(p_header);
  cpl_table_delete(tf_qual);
  cpl_frameset_delete(used_frames);
  cpl_frameset_delete(good_frames);
  cpl_free(file_name);
    
  return 0;
}
