/* $Id: mat_im_basic.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_im_basic.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MATISSE_LICENCE         "GPL"
#define MAX_FILES               16

#define MATISSE_KW_EXPTIME      "EXPTIME"
/* #define MATISSE_KW_EXPTIME      "ESO DET SEQ1 DIT" */

#define MATISSE_DO_COLD            "IM_COLD"
#define MATISSE_DO_BADPIXEL        "BADPIX"
#define MATISSE_DO_FLATFIELD       "FLATFIELD"
#define MATISSE_STATISTICS_PROCATG "STAT"
#define MEDIAN_KERNEL_SIZE 7

#define MAT_MIN_FLAT_INTENSITY  100.0

/*
 esorex --recipe-dir=/iri/irisrv/home/mhein/lib/matisse/plugins/matisse-0.0.1 mat_cal_det -darklimit=7.0 -flatlimit=7.0 -nbcoeff=3 TC_14_01_SIMa.sof
 esorex --recipe-dir=/iri/irisrv/home/mhein/lib/matisse/plugins/matisse-0.0.1 mat_cal_det -darklimit=7.0 -flatlimit=7.0 -nbcoeff=3 TC_14_01_SIM.sof
 esorex --recipe-dir=/iri/irisrv/home/mhein/lib/matisse/plugins/matisse-0.0.1 mat_cal_det -darklimit=7.0 -flatlimit=7.0 -nditskip=10 -nbcoeff=3 TC_14_01_AMB.sof
 */

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <strings.h>
#include <ctype.h>

#include <cpl.h>
#include "mat_drl.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_frame.h"
#include "mat_statistics.h"
#include "mat_badpixel.h"
#include "mat_flatfield.h"
#include "mat_detector.h"
#include "mat_imagingdetector.h"
#include "mat_imbasic.h"

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/
typedef struct {
  int                  nditskip;
  int                  cosmics;
  double               gain;
  int                  window;
  double               horizontal_freq;
  double               vertical_freq;
  int                  expert;
  mat_detector        *det; /* this the detector specification from the first file */
  mat_badpixel        *bpm;
  mat_flatfield       *ffm;
  mat_imagingdetector *imgdet;
  int                  nbfiles;
  double               exptime;
  cpl_propertylist    *keywords;
  int                  nbtel;
  int                  maxins;
  mat_frame           *darkmedian;
  mat_frame           *darkvariance;
  int                  swi; /* index of the sub-window which covers all detector channels */
  mat_imbasic         *imb;
} mat_im_basic_info;
/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_im_basic_create(cpl_plugin *);
static int mat_im_basic_exec(cpl_plugin *);
static int mat_im_basic_destroy(cpl_plugin *);
static int mat_im_basic(cpl_parameterlist *, cpl_frameset *);
/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static char mat_im_basic_description[] =
  "This plugin calculates a small set of detector monitoring QC1 parameters. A specialized template (MATISSE_gen_cal_imbasic) creates one series of cold dark frames (100 raw frames in a FITS file). This file is then used to calculate the detector channel offsets, detector channel noise and power spectrum peaks. Since the template is used before each observing night, the QC1 parameter give a simple but quick status of an instrument detector.\n"
"This plugin uses a FITS file with 100 cold dark frames (IM_COLD) for a specific detector. In addition, a bad pixel (PRO.CATG = BADPIX) and/or a flatfield (PRO.CATG = FLATFIELD) static calibration map can be specified.\n"
"This plugin creates an output file (PRO.CATG = IM_BASIC) which contains:\n"
" 1. an empty HDU containing the QC1 parameters\n"
" 2. a binary table (IM_STATISTICS) containing the pixel statistics for each detector channel in a row\n"
" 3. a binary table (IM_FAST_POWERSPECTRUM) containing the fast 1-d power spectrum for each detector channel in a row\n"
" 4. a binary table (IM_SLOW_POWERSPECTRUM) containing the slow 1-d power spectrum for each detector channel in a row\n"
  "The following QC1 parameters will be calculated and stored in a cpl_propertylist and the primary HDU of the result FITS file:\n"
  " - QC DET<i> CHANNEL<j> OFFSET<p> : The offset for the detector channels.\n"
  " - QC DET<i> CHANNEL<j> RON<p> : The read-out noise is given for each detector channel.\n"
  " - QC DET<i> PS FAST PEAK<k> FREQ : The frequency for the k-th strongest peak in the fast (horizontal) 1-d power spectrum.\n"
  " - QC DET<i> PS FAST PEAK<k> POWER : The power for the k-th strongest peak in the fast (horizontal) 1-d power spectrum.\n"
  " - QC DET<i> PS SLOW PEAK<k> FREQ : The frequency for the k-th strongest peak in the slow (vertical) 1-d power spectrum.\n"
  " - QC DET<i> PS SLOW PEAK<k> POWER : The power for the k-th strongest peak in the slow (vertical) 1-d power spectrum.\n"
  "Where i is the detector number (1 = L/M-Band, 2 = N-Band), j is the detector channel number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band), and p is the readout mode (HAWAII-2RG: 1 = slow readout, 2 = fast readout, Aquarius: 1 = low gain mode, 2 = high gain mode). The detector number (used for DET<i>) will be always 1 for the L/M-Band detector and 2 for the N-Band detector. In addition, the DET<i> CHIP1 ID keyword allows to distinguish between different detector chips. This is necessary, to deal with a detector change after, for example, an instrument repair. This will also result in a basic detector monitoring map which is only valid for a specific detector chip.\n"
  "\n"
  "Input files:\n"
  "\n"
  "  DO category:      Explanation:            Required:\n"
  "  IM_COLD           Raw cold frames         Yes\n"
  "  BADPIX            Static bad pixel map    No\n"
  "  FLATFIELD         Static flatfield map    No\n"
  "\n"
  "Output files:\n"
  "\n"
  "  DO category:      Explanation:\n"
  "  IM_BASIC          Basic instrument monitoring map"
  ;
/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist * list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
  cpl_plugin *plugin = &recipe->interface;

  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_im_basic",
		  "Calculates a basic detector monitoring map",
		  mat_im_basic_description,
		  "Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  MATISSE_LICENCE,
		  mat_im_basic_create,
		  mat_im_basic_exec,
		  mat_im_basic_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_im_basic_create(cpl_plugin * plugin)
{
  cpl_recipe    *recipe;
  cpl_parameter *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */
  /* --nditskip */
  p = cpl_parameter_new_range("matisse.mat_im_basic.nditskip",
			      CPL_TYPE_INT,
			      "number of skipped frames",
			      "matisse.mat_im_basic",
			      0, 0, 64);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nditskip");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --cosmics */
  p = cpl_parameter_new_range("matisse.mat_im_basic.cosmics",
			      CPL_TYPE_INT,
			      "flag if cosmics should be detected",
			      "matisse.mat_im_basic",
			      1, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cosmics");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --gain */
  p = cpl_parameter_new_range("matisse.mat_im_basic.gain",
			      CPL_TYPE_DOUBLE,
			      "global getector gain [e-/DU]",
			      "matisse.mat_im_basic",
			      1.0, 0.001, 1000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "gain");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --window */
  p = cpl_parameter_new_range("matisse.mat_im_basic.window",
			      CPL_TYPE_INT,
			      "window function for 1-d power spectrum",
			      "matisse.mat_im_basic",
			      3, 0, 3);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "window");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --hfreq */
  p = cpl_parameter_new_range("matisse.mat_im_basic.hfreq",
			      CPL_TYPE_DOUBLE,
			      "horizontal base frequency",
			      "matisse.mat_im_basic",
			      0.0, 0.0, 1000000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "hfreq");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --vfreq */
  p = cpl_parameter_new_range("matisse.mat_im_basic.vfreq",
			      CPL_TYPE_DOUBLE,
			      "vertical base frequency",
			      "matisse.mat_im_basic",
			      0.0, 0.0, 1000000.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "vfreq");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --expert */
  p = cpl_parameter_new_range("matisse.mat_im_basic.expert",
			      CPL_TYPE_INT,
			      "expert flag",
			      "matisse.mat_im_basic",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "expert");
  cpl_parameterlist_append(recipe->parameters, p);
  /* Return */
  return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_basic_exec(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_im_basic(recipe->parameters, recipe->frames);
}
/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the ’create’ function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_basic_destroy(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}

static void mat_info_init(mat_im_basic_info *info)
{
  /* initialize the info data structure */
  info->det = mat_detector_new();
  if (info->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for a mat_detector");
      return;
    }
  info->bpm = NULL;
  info->ffm = NULL;
  info->imgdet = NULL;
  info->keywords = NULL;
  info->exptime = 0.0;
  info->darkmedian = NULL;
  info->darkvariance = NULL;
  info->imb = NULL;
}

static void mat_info_delete(mat_im_basic_info *info)
{
  /* delete the info data structure */
  if (info->det != NULL)
    {
      mat_detector_delete(info->det);
      info->det = NULL;
    }
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->ffm != NULL)
    {
      mat_flatfield_delete(info->ffm);
      info->ffm = NULL;
    }
  if (info->imgdet != NULL)
    {
      mat_imagingdetector_delete(info->imgdet);
      info->imgdet = NULL;
    }
  info->nbfiles = 0;
  if (info->keywords != NULL)
    {
      cpl_propertylist_delete(info->keywords);
      info->keywords = NULL;
    }
  info->exptime = 0.0;
  if (info->darkmedian != NULL)
    {
      mat_frame_delete(info->darkmedian);
      info->darkmedian = NULL;
    }
  if (info->darkvariance != NULL)
    {
      mat_frame_delete(info->darkvariance);
      info->darkvariance = NULL;
    }
  if (info->imb != NULL)
    {
      mat_imbasic_delete(info->imb);
      info->imb = NULL;
    }
}

static cpl_error_code mat_info_store_statistics(mat_im_basic_info *info,
						const char *fname,
						cpl_parameterlist *parlist,
						cpl_frameset *frameset)
{
  cpl_imagelist    *imglist;
  cpl_frame        *statframe;
  cpl_propertylist *plist = NULL;
  int               swi;

  cpl_error_reset();
  cpl_ensure_code((info != NULL), CPL_ERROR_NULL_INPUT);
  swi = info->swi;
  /* create an empty image list and add all statistical data from the info */
  imglist = cpl_imagelist_new();
  cpl_imagelist_set(imglist, info->darkmedian->list_subwin[swi]->imgreg[0], cpl_imagelist_get_size(imglist));
  cpl_imagelist_set(imglist, info->darkvariance->list_subwin[swi]->imgreg[0], cpl_imagelist_get_size(imglist));
  /* Create product frame */
  statframe = cpl_frame_new();
  cpl_frame_set_filename(statframe, fname);
  cpl_frame_set_tag(statframe, MATISSE_STATISTICS_PROCATG);
  cpl_frame_set_type(statframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(statframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(statframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    /* remove (not delete!) all frames from the image list */
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, statframe, frameset, parlist,
				   "mat_info_store_statistics", "MATISSE", "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* Add QC parameters in the header */
  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_imagelist_save(imglist, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_imagelist_unwrap(imglist);
  cpl_frame_delete(statframe);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calc_raw_statistics(mat_im_basic_info *info,
					      cpl_frame *frame)
{
  mat_frame     *median = NULL;
  mat_frame     *variance = NULL;
  double         exptime;
  int            nz, z;
  cpl_imagelist *imglist = NULL;
  cpl_mask      *bpm = NULL;
  mat_imbasic   *imb = NULL;
  cpl_vector    *powerspectrum = NULL;

  mat_gendata *raw = mat_gendata_load_skip(frame, info->nditskip, CPL_TYPE_FLOAT);
  if (raw == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot read the FITS file");
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    {
      cpl_msg_info(cpl_func, "raw data file successfully loaded");
    }
  /* 
     This method works only if the one and only region covers the whole detector!
     If the type of the detector is either a HAWAII-2RG or an Aquarius, the size
     of the sub-window must be 2048x2048 for the HAWAII-2RG or 1024x1024 for the Aquarius.
     If the detector type is unknown, the sub-window size of the first file is used.
  */
  /* Get the detector specification and check it. */
  mat_detector_decode_raw(info->det, raw);
  if ((info->det->nx != info->det->data_nx))
    {
      cpl_msg_error(cpl_func,
			"the file %s does not cover the whole detector width",
		    cpl_frame_get_filename(frame));
      mat_gendata_delete(raw);
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  // -> load and keep the keywords from the primary header
  // -> load the imaging detector table for later use
  info->keywords = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
  info->imgdet = mat_imagingdetector_duplicate(raw->imgdet);
  /* if a badpixel map is already loaded (info->bpm) it must be from the same detector */
  if (info->bpm != NULL)
    {
      if (mat_detector_check(info->det, info->bpm->det) != CPL_ERROR_NONE)
	{
	  cpl_msg_error(cpl_func,
			"the file %s contains data which is not compatible with the badpixel map, ignoring the bpm",
			cpl_frame_get_filename(frame));
	  mat_badpixel_delete(info->bpm);
	  info->bpm = NULL;
	}
      else
	{
	  /* the badpixel map is ok, now map it to the sub-windows of the raw data */
	  mat_badpixel *nbpm = mat_badpixel_map(info->bpm, raw->imgdet);
	  mat_badpixel_delete(info->bpm);
	  info->bpm = nbpm;
	    }
    }
  /* if a flatfield map is already loaded (info->ffm) it must be from the same detector */
  if (info->ffm != NULL)
    {
      if (mat_detector_check(info->det, info->ffm->det) != CPL_ERROR_NONE)
	{
	  cpl_msg_error(cpl_func,
			"the file %s contains data which is not compatible with the flatfield map, ignoring the ffm",
			cpl_frame_get_filename(frame));
	  mat_flatfield_delete(info->ffm);
	  info->ffm = NULL;
	}
    }

  /* we have to find the sub-window which covers all detector channels (info->swi) */
  info->swi = mat_find_central_region(info->det, info->imgdet);
  if (info->swi == -1)
    {
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->nbtel = raw->imgdata->nbtel;
  info->maxins = raw->imgdata->maxins;
  exptime = cpl_propertylist_get_double(raw->keywords, MATISSE_KW_EXPTIME);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      // normally this FITS file is broken, but:
      // try to extract the exposure time from the IMAGING_DATA table (column EXPTIME)
      exptime = raw->imgdata->list_frame[0]->exptime;
      cpl_error_reset();
    }
  cpl_msg_info(cpl_func, "exptime=%lf", exptime);
  median = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  variance = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  if (mat_calc_statistics_cosmics(raw, median, variance, info->cosmics) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Cannot calculate the pixel statistics for the file %s",
		    cpl_frame_get_filename(frame));
      mat_gendata_delete(raw);
      return CPL_ERROR_UNSPECIFIED;
    }
  /* create an image list which contains only the region which is identified by info->swi */
  imglist = cpl_imagelist_new();
  if (imglist == NULL)
    {
      cpl_error_code ecode = cpl_error_get_code();
      cpl_msg_error(cpl_func, "could not allocate the image list, code = %d, message = %s",
		    ecode,
		    cpl_error_get_message());
      mat_gendata_delete(raw);
      return ecode;
    }
  nz = raw->imgdata->nbframe;
  for (z = 0; z < nz; z++)
    {
      cpl_imagelist_set(imglist, raw->imgdata->list_frame[z]->list_subwin[info->swi]->imgreg[0], z);
    }
  if (info->bpm != NULL)
    {
      bpm = info->bpm->list_badpixel[info->swi];
    }
  info->darkmedian = median;
  info->darkvariance = variance;
  /* calculate the fast and the slow 1-d power spectrum for each detector channel */
  imb = mat_imbasic_new(info->det, info->imgdet);
  mat_assert_not_null(imb, "could not create the basic detector monitoring map");
  info->imb = imb;
  mat_calc_spectra1d(info->darkmedian->list_subwin[info->swi]->imgreg[0],
		     imglist,
		     bpm,
		     info->det->channel_ncolumns,
		     info->det->channel_nrows,
		     info->window,
		     info->imb->list_fastpowerspectra,
		     info->imb->list_slowpowerspectra);
  /* delete the image list */
  cpl_imagelist_unwrap(imglist);
  /* calculate the average fast powerspectrum and try to find the strongest peaks */
  powerspectrum = cpl_vector_new(cpl_vector_get_size(info->imb->list_fastpowerspectra[0]));
  if (powerspectrum == NULL)
    {
      cpl_msg_error(cpl_func, "cannot allocate a cpl_vector for the average fast powerspectrum");
    }
  else
    {
      double frequency;
      int    nx = cpl_image_get_size_x(info->darkmedian->list_subwin[info->swi]->imgreg[0]);
      int    ny = cpl_image_get_size_y(info->darkmedian->list_subwin[info->swi]->imgreg[0]);
      int    np = info->det->channel_ncolumns*info->det->channel_nrows;
      int    p;
      cpl_vector_copy(powerspectrum, info->imb->list_fastpowerspectra[0]);
      for (p = 1; p < np; p++)
	{
	  cpl_vector_add(powerspectrum, info->imb->list_fastpowerspectra[p]);
	}
      cpl_vector_divide_scalar(powerspectrum, (double)np);
      if (info->horizontal_freq != 0.0)
	{
	  frequency = info->horizontal_freq;
	}
      else
	{
	  frequency = 1e6/(info->det->horizontal_timing[0]
			   + nx*info->det->horizontal_timing[1]
			   + ny*info->det->horizontal_timing[2]
			   + nx*ny*info->det->horizontal_timing[3]);
	}
      mat_find_peaks(frequency, powerspectrum, info->imb->fastpowerspectrapeaks);
      cpl_vector_set_size(powerspectrum, cpl_vector_get_size(info->imb->list_slowpowerspectra[0]));
      cpl_vector_copy(powerspectrum, info->imb->list_slowpowerspectra[0]);
      for (p = 1; p < np; p++)
	{
	  cpl_vector_add(powerspectrum, info->imb->list_slowpowerspectra[p]);
	}
      cpl_vector_divide_scalar(powerspectrum, (double)np);
      if (info->vertical_freq != 0.0)
	{
	  frequency = info->vertical_freq;
	}
      else
	{
	  frequency = 1e6/(info->det->vertical_timing[0]
			   + nx*info->det->vertical_timing[1]
			   + ny*info->det->vertical_timing[2]
			   + nx*ny*info->det->vertical_timing[3]);
	}
      mat_find_peaks(frequency, powerspectrum, info->imb->slowpowerspectrapeaks);
      cpl_vector_delete(powerspectrum);
    }
  mat_gendata_delete(raw);
  return CPL_ERROR_NONE;
}

/*
  For the calculation of the flatfield map, a smoothed median image will be used
  to calculate the individual pixel gain. This approach covers only the small scale
  variations. The large scale variations could only be preserved in the flatfield
  map if an illumination model is available. That model could then be used to cover
  the large scale gain variations.

  Remark:

  The algorithm is changed compared to the document. The mat_update_sumweight
  function is now included.

  This implementation works only if there is only one sub-window and it
  covers the whole detector!

  Mathematical description:

<pre>BEGIN
   CALL mat_cal_gain using the data from the longest exposure time;
   Create an empty flatfield map;
   Create an empty weight array (same dimensions as the flatfield map);
   FOR EACH series of flatfield frames DO
      FOR EACH pixel DO
         IF enough light on that pixel THEN
            ffm[pixel] += intensity[pixel]*intensity[pixel]/smooth[pixel];
	    weight[pixel] += intensity[pixel];
         ENDIF
      ENDFOR
   ENDFOR
   FOR EACH pixel DO
      IF weight[pixel] == 0.0 THEN
         ffm[pixel] = 1.0;
      ELSE
         ffm[pixel] /= weight[pixel];
      ENDIF
   ENDFOR
END</pre>
*/
static cpl_error_code mat_im_basic_map(mat_im_basic_info *info)
{
  cpl_mask       *bpm = NULL;
  cpl_image      *noise = NULL;
  int             ndc;

  mat_assert_not_null(info, "no valid mat_im_basic_info (info) given as argument");
  mat_assert_not_null(info->imb, "no basic detector monitoring map");
  ndc = info->det->channel_nrows*info->det->channel_ncolumns;
  if (info->bpm != NULL)
    {
      bpm = info->bpm->list_badpixel[info->swi];
    }
  mat_calc_median_dev(info->darkmedian->list_subwin[info->swi]->imgreg[0],
		      bpm,
		      info->det->channel_ncolumns,
		      info->det->channel_nrows,
		      info->imb->list_intensity,
		      info->imb->intensitymedian,
		      info->imb->intensitystdev);
  mat_calc_median_dev(info->darkvariance->list_subwin[info->swi]->imgreg[0],
		      bpm,
		      info->det->channel_ncolumns,
		      info->det->channel_nrows,
		      info->imb->list_variance,
		      info->imb->variancemedian,
		      info->imb->variancestdev);
  noise = cpl_image_power_create(info->darkvariance->list_subwin[info->swi]->imgreg[0], 0.5);
  if (noise != NULL)
    {
      int dc;
      mat_calc_median_dev(noise,
			  bpm,
			  info->det->channel_ncolumns,
			  info->det->channel_nrows,
			  info->imb->list_ron,
			  info->imb->ronmedian,
			  info->imb->ronstdev);
      cpl_image_delete(noise);
      /*
       * Scale the noise images and the noise values according to a given global gain or
       * the detector channel specific values given in an optional flatfield map.
       */
      for (dc = 0; dc < ndc; dc++)
	{
	  double gain = info->gain;
	  if (info->ffm != NULL)
	    {
	      /* we use the detector channel specific gain */
	      gain = cpl_vector_get(info->ffm->channelgain, dc);
	    }
	  noise = cpl_imagelist_get(info->imb->list_ron, dc);
	  if (noise != NULL)
	    {
	      cpl_image_multiply_scalar(noise, gain);
	    }
	  else
	    {
	      cpl_msg_error(cpl_func, "cannot get the noise image for the detector channel %d", dc);
	    }
	  cpl_vector_set(info->imb->ronmedian,
			 dc,
			 gain*cpl_vector_get(info->imb->ronmedian, dc));
	  cpl_vector_set(info->imb->ronstdev,
			 dc,
			 gain*cpl_vector_get(info->imb->ronstdev, dc));
	}
    }
  else
    {
      cpl_msg_error(cpl_func, "cannot calculate the noise from the variance");
    }
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Interpret the command line options and execute the data processing
   @param parlist the parameters list
   @param frameset the frames list
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_basic(cpl_parameterlist *parlist,
		       cpl_frameset *frameset)
{
  cpl_parameter    *param;
  int               darkcount;
  int               bpmcount;
  int               ffmcount;
  mat_im_basic_info   info;
  cpl_frame        *bpmframe = NULL;
  cpl_frame        *ffmframe = NULL;
  cpl_frame        *rawframe = NULL;

  mat_info_init(&info);
  /* HOW TO RETRIEVE INPUT PARAMETERS */
  /* --nditskip */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.nditskip");
  info.nditskip = cpl_parameter_get_int(param);
  /* --cosmics */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.cosmics");
  info.cosmics = cpl_parameter_get_int(param);
  /* --gain */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.gain");
  info.gain = cpl_parameter_get_double(param);
  /* --window */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.window");
  info.window = cpl_parameter_get_int(param);
  /* --hfreq */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.hfreq");
  info.horizontal_freq = cpl_parameter_get_double(param);
  /* --vfreq */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.vfreq");
  info.vertical_freq = cpl_parameter_get_double(param);
  /* --expert */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_basic.expert");
  info.expert = cpl_parameter_get_int(param);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Failed to retrieve the input parameters");
    return -1;
  }
  /* Identify the RAW and CALIB frames in the input frameset */
  /*
  if (matisse_dfs_set_groups(frameset)) {
    cpl_msg_error(cpl_func, "Cannot identify RAW and CALIB frames");
    return -1;
  }
  */
  /* The input data consists of dark and flatfield files (pairs!) */
  darkcount = cpl_frameset_count_tags(frameset, MATISSE_DO_COLD);
  if (darkcount == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not have any cold dark files");
      return -1;
    }
  bpmcount = cpl_frameset_count_tags(frameset, MATISSE_DO_BADPIXEL);
  if (bpmcount > 1)
    {
      cpl_msg_error(cpl_func, "only zero or one badpixel map is allowed in the SOF");
      mat_info_delete(&info);
      return -1;
    }
  ffmcount = cpl_frameset_count_tags(frameset, MATISSE_DO_FLATFIELD);
  if (ffmcount > 1)
    {
      cpl_msg_error(cpl_func, "only zero or one flatfield map is allowed in the SOF");
      mat_info_delete(&info);
      return -1;
    }
  /* load the optional badpixel map */
  if (bpmcount == 1)
    {
      bpmframe = cpl_frameset_find(frameset, MATISSE_DO_BADPIXEL); // already checked previously!
      cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_CALIB);
      info.bpm = mat_badpixel_load(bpmframe);
    }
  /* load the optional flatfield map */
  if (ffmcount == 1)
    {
      ffmframe = cpl_frameset_find(frameset, MATISSE_DO_FLATFIELD); // already checked previously!
      cpl_frame_set_group(ffmframe, CPL_FRAME_GROUP_CALIB);
      info.ffm = mat_flatfield_load(ffmframe);
    }
  /* Calculate the pixel statistics (modified due to the frameset processing schema) */
  /* CALL mat_calc_statistics for the first cold dark series */
  rawframe = cpl_frameset_find(frameset, MATISSE_DO_COLD); // already checked previously!
  cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(rawframe));
  cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
  if (mat_calc_raw_statistics(&info, rawframe) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      return -1;
    }
  if (mat_im_basic_map(&info) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      return -1;
    }
  if (info.expert != 0)
    {
      mat_info_store_statistics(&info, "mat_im_basic_stat.fits",
				parlist,
				frameset);
    }
  mat_imbasic_save(info.imb,
		   mat_detector_get_imb_name(info.det),
		   "mat_im_basic",
		   parlist,
		   frameset);
  mat_info_delete(&info);
  /* Return */
  if (cpl_error_get_code())
    return -1;
  else
    return 0;
}
