/* $Id: mat_im_extended.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_im_extended.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MATISSE_LICENCE         "GPL"
#define MAX_FILES               16

#define MATISSE_KW_EXPTIME      "EXPTIME"
/* #define MATISSE_KW_EXPTIME      "ESO DET SEQ1 DIT" */
#define MATISSE_KW_PERIOD       "ESO DET SEQ1 PERIOD"

#define MATISSE_DO_DARK            "IM_DARK"
#define MATISSE_DO_FLAT            "IM_FLAT"
#define MATISSE_DO_BADPIXEL        "BADPIX"
#define MATISSE_STATISTICS_PROCATG "STAT"
#define MEDIAN_KERNEL_SIZE 7

#define MAT_MIN_FLAT_INTENSITY  100.0

/*
 esorex --recipe-dir=/iri/irisrv/home/mhein/lib/matisse/plugins/matisse-0.0.1 mat_cal_det -darklimit=7.0 -flatlimit=7.0 -nbcoeff=3 TC_14_01_SIMa.sof
 esorex --recipe-dir=/iri/irisrv/home/mhein/lib/matisse/plugins/matisse-0.0.1 mat_cal_det -darklimit=7.0 -flatlimit=7.0 -nbcoeff=3 TC_14_01_SIM.sof
 esorex --recipe-dir=/iri/irisrv/home/mhein/lib/matisse/plugins/matisse-0.0.1 mat_cal_det -darklimit=7.0 -flatlimit=7.0 -nditskip=10 -nbcoeff=3 TC_14_01_AMB.sof
 */

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <strings.h>
#include <ctype.h>

#include <cpl.h>
#include "mat_drl.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_frame.h"
#include "mat_statistics.h"
#include "mat_badpixel.h"
#include "mat_detector.h"
#include "mat_imagingdetector.h"
#include "mat_imextended.h"

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/
typedef struct {
  int                  nditskip;
  int                  cosmics;
  int                  roix;
  int                  roiy;
  int                  roinx;
  int                  roiny;
  int                  expert;
  mat_detector        *det; /* this the detector specification from the first file */
  mat_badpixel        *bpm;
  mat_imagingdetector *imgdet;
  int                  nbfiles;
  double               exptime;
  cpl_propertylist    *keywords;
  int                  nbtel;
  int                  maxins;
  mat_frame           *darkmedian;
  mat_frame           *darkvariance;
  mat_frame           *flatmedian;
  mat_frame           *flatvariance;
  int                  swi; /* index if the sub-window which covers all detector channels */
  double               horizontalcorrelation; /* horizontal autocorr (QC1)     */
  double               verticalcorrelation;   /* vertical autocorr (QC1)       */
  double               allanvariancemin;      /* allan variance minimum (QC1)  */
  mat_imextended      *ime;
} mat_im_ext_info;
/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_im_extended_create(cpl_plugin *);
static int mat_im_extended_exec(cpl_plugin *);
static int mat_im_extended_destroy(cpl_plugin *);
static int mat_im_extended(cpl_parameterlist *, cpl_frameset *);
/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static char mat_im_extended_description[] =
  "This plugin uses one series of cold dark and one series of flatfield frames to calculate extended detector monitoring information:\n"
  " - The detector and detector channel specific gain which can be compared with the gain values calculated by the mat_cal_det plugin (comparison of the the QC1 values).\n"
  " - The detector channel specific offset and noise.\n"
  " - The Allan Variance give a hint about the long term stability of the detector and the detector related instrument part.\n"
  " - The autocorrelation specifies the inter-pixel dependencies which influences the spatial and spectral resolution of the raw data.\n"
  "Since the Allan Variance should cover a time span larger than a science observation, the series of the cold dark and flatfield frames will cover 10 minutes (3000 raw frames for each detector, both detectors will be slowed down to 5 frames per second):\n"
  "One series of cold darks (IM_DARK) classified by:\n"
  "  DPR.CATG = CALIB\n"
  "  DPR.TYPE = DARK,IME\n"
  "  DPR.TECH = IMAGE\n"
  "  TPL.ID   = MATISSE_gen_cal_imext_(L_FAST|N_HIGH)\n"
  "One series of flatfield frames (IM_FLAT) classified by:\n"
  "  DPR.CATG = CALIB\n"
  "  DPR.TYPE = FLAT,IME\n"
  "  DPR.TECH = IMAGE\n"
  "  TPL.ID   = MATISSE_gen_cal_imext_(L_FAST|N_HIGH)\n"
  "In addition, the bad pixel (PRO.CATG = BADPIX) static calibration map is needed.\n"
  "The pixel statistics for the cold dark frames contains the median (not mean!) intensity and the variance. The values for the flatfield frames are corrected by subtracting the median and variance values for the corresponding cold dark series. This means, that the flatfield values should represent directly the photon statistics. Since some parts of the detector are not illuminated, the variance values for these parts could be negative because the cold dark and flatfield frames show nearly the same median and variance. This function is equivalent to the corresponding function in the mat_cal_det plugin except, that only one series of cold dark and one series of flatfield frames are used.\n"
  "The offset and noise of a detector channel can indicate the health status of the detector and the electronics. The function is equivalent to the function in the mat_im_basic plugin.\n"
  "Since the pixel statistics for the flatfield series does not contain the cold dark statistics, the division of the median by the variance directly gives the conversion gain. As many pixels as possible were used to calculate the detector and detector channel specific gain. This function is equivalent to the corresponding function in the mat_cal_det plugin.\n"
  "This plugin creates an output file (PRO.CATG = IM_EXTENDED) which contains an empty HDU with all QC1 parameters.\n"
  "QC1 parameters:\n"
  "The following QC1 parameters will be calculated and stored in the primary HDU of the result FITS file:\n"
  " - QC DET<i> GAIN<p> : The global detector gain (in electrons per ADU) will be used for scientific purposes.\n"
  " - QC DET<i> RON<p> : The global read-out noise (in electrons).\n"
  " - QC DET<i> CHANNEL<j> GAIN<p> : The detector channel specific gain (in electrons per ADU) will be used to detect detector and read-out electronics problems.\n"
  " - QC DET<i> CHANNEL<j> OFFSET<p> : The offset for the detector channels.\n"
  " - QC DET<i> CHANNEL<j> RON<p> : The read-out noise is given for each detector channel.\n"
  " - GC DET<i> HACORR<p> : The Autocorrelation coefficient between two horizontally adjacent pixels.\n"
  " - GC DET<i> VACORR<p> : The Autocorrelation coefficient between two vertically adjacent pixels.\n"
  " - GC DET<i> AVAR MIN : Position of the turning point of the Allan variance plot.\n"
  "Where i is the detector number (1 = L/M-Band, 2 = N-Band), j is the detector channel number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band), and p is the readout mode (HAWAII-2RG: 1 = slow readout, 2 = fast readout, Aquarius: 1 = low gain mode, 2 = high gain mode). The detector number (used for DET<i>) will be always 1 for the L/M-Band detector and 2 for the N-Band detector. In addition, the DET<i> CHIP1 ID keyword allows to distinguish between different detector chips. This is necessary, to deal with a detector change after, for example, an instrument repair.\n"
  "\n"
  "Input files:\n"
  "\n"
  "  DO category:      Explanation:            Required:\n"
  "  IM_DARK           Raw cold frames         Yes\n"
  "  IM_FLAT           Raw exposed frames      Yes\n"
  "  BADPIX            Static bad pixel map    No\n"
  "\n"
  "Output files:\n"
  "\n"
  "  DO category:      Explanation:\n"
  "  IM_EXTENDED       Extended instrument monitoring map"
;
/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist * list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
  cpl_plugin *plugin = &recipe->interface;

  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_im_extended",
		  "Calculates a extended detector monitoring map",
		  mat_im_extended_description,
		  "Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  MATISSE_LICENCE,
		  mat_im_extended_create,
		  mat_im_extended_exec,
		  mat_im_extended_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_im_extended_create(cpl_plugin * plugin)
{
  cpl_recipe    *recipe;
  cpl_parameter *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */
  /* --nditskip */
  p = cpl_parameter_new_range("matisse.mat_im_extended.nditskip",
			      CPL_TYPE_INT,
			      "number of skipped frames",
			      "matisse.mat_im_extended",
			      0, 0, 20);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nditskip");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --cosmics */
  p = cpl_parameter_new_range("matisse.mat_im_extended.cosmics",
			      CPL_TYPE_INT,
			      "flag is cosmics should be detected",
			      "matisse.mat_im_extended",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cosmics");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --roi */
  p = cpl_parameter_new_value("matisse.mat_im_extended.roi",
			      CPL_TYPE_STRING,
			      "region of interest: <x>,<y>,<nx>,<ny>",
			      "matisse.mat_im_extended",
			      "-1,-1,-1,-1");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "roi");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --expert */
  p = cpl_parameter_new_range("matisse.mat_im_extended.expert",
			      CPL_TYPE_INT,
			      "expert flag",
			      "matisse.mat_im_extended",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "expert");
  cpl_parameterlist_append(recipe->parameters, p);
  /* Return */
  return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_extended_exec(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_im_extended(recipe->parameters, recipe->frames);
}
/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the ’create’ function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_extended_destroy(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}

static void mat_info_init(mat_im_ext_info *info)
{
  /* initialize the info data structure */
  info->det = mat_detector_new();
  if (info->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for a mat_detector");
      return;
    }
  info->bpm = NULL;
  info->imgdet = NULL;
  info->keywords = NULL;
  info->exptime = 0.0;
  info->darkmedian = NULL;
  info->darkvariance = NULL;
  info->flatmedian = NULL;
  info->flatvariance = NULL;
  info->ime = NULL;
}

static void mat_info_delete(mat_im_ext_info *info)
{
  /* delete the info data structure */
  if (info->det != NULL)
    {
      mat_detector_delete(info->det);
      info->det = NULL;
    }
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->imgdet != NULL)
    {
      mat_imagingdetector_delete(info->imgdet);
      info->imgdet = NULL;
    }
  info->nbfiles = 0;
  if (info->keywords != NULL)
    {
      cpl_propertylist_delete(info->keywords);
      info->keywords = NULL;
    }
  info->exptime = 0.0;
  if (info->darkmedian != NULL)
    {
      mat_frame_delete(info->darkmedian);
      info->darkmedian = NULL;
    }
  if (info->darkvariance != NULL)
    {
      mat_frame_delete(info->darkvariance);
      info->darkvariance = NULL;
    }
  if (info->flatmedian != NULL)
    {
      mat_frame_delete(info->flatmedian);
      info->flatmedian = NULL;
    }
  if (info->flatvariance != NULL)
    {
      mat_frame_delete(info->flatvariance);
      info->flatvariance = NULL;
    }
  if (info->ime != NULL)
    {
      mat_imextended_delete(info->ime);
      info->ime = NULL;
    }
}

static cpl_error_code mat_info_store_statistics(mat_im_ext_info *info,
						const char *fname,
						cpl_parameterlist *parlist,
						cpl_frameset *frameset)
{
  cpl_imagelist    *imglist;
  cpl_frame        *statframe;
  cpl_propertylist *plist = NULL;

  cpl_error_reset();
  cpl_ensure_code((info != NULL), CPL_ERROR_NULL_INPUT);
  /* create an empty image list and add all statistical data from the info */
  imglist = cpl_imagelist_new();
  cpl_imagelist_set(imglist, info->darkmedian->list_subwin[0]->imgreg[0], cpl_imagelist_get_size(imglist));
  cpl_imagelist_set(imglist, info->darkvariance->list_subwin[0]->imgreg[0], cpl_imagelist_get_size(imglist));
  cpl_imagelist_set(imglist, info->flatmedian->list_subwin[0]->imgreg[0], cpl_imagelist_get_size(imglist));
  cpl_imagelist_set(imglist, info->flatvariance->list_subwin[0]->imgreg[0], cpl_imagelist_get_size(imglist));
  /* Create product frame */
  statframe = cpl_frame_new();
  cpl_frame_set_filename(statframe, fname);
  cpl_frame_set_tag(statframe, MATISSE_STATISTICS_PROCATG);
  cpl_frame_set_type(statframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(statframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(statframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    /* remove (not delete!) all frames from the image list */
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, statframe, frameset, parlist,
				   "mat_info_store_statistics", "MATISSE", "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* Add QC parameters in the header */
  /*
  cpl_propertylist_append_double(plist, "ESO QC QCPARAM", qc_param);
  */
  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_imagelist_save(imglist, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_imagelist_unwrap(imglist);
    cpl_frame_delete(statframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_imagelist_unwrap(imglist);
  cpl_frame_delete(statframe);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_calc_raw_statistics(mat_im_ext_info *info,
					      int isdark,
					      cpl_frame *frame)
{
  mat_frame     *median = NULL;
  mat_frame     *variance = NULL;
  double         exptime;
  double         mindit, tpixel, tline, tframe;
  int            nz, z;
  cpl_imagelist *imglist = NULL;
  cpl_mask      *bpm = NULL;

  mat_gendata *raw = mat_gendata_load_skip(frame, info->nditskip, CPL_TYPE_FLOAT);
  if (raw == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot read the FITS file");
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    {
      cpl_msg_info(cpl_func, "raw data file successfully loaded");
    }
  /* 
     This method works only if the one and only region covers the whole detector!
     If the type of the detector is either a HAWAII-2RG or an Aquarius, the size
     of the sub-window must be 2048x2048 for the HAWAII-2RG or 1024x1024 for the Aquarius.
     If the detector type is unknown, the sub-window size of the first file is used.
  */
  if (isdark)
    {
      /* It is the first file -> get the detector specification and check it. */
      mat_detector_decode_raw(info->det, raw);
      if ((info->det->nx != info->det->data_nx))
	{
	  cpl_msg_error(cpl_func,
			"the file %s does not cover the whole detector width",
			cpl_frame_get_filename(frame));
	  mat_gendata_delete(raw);
	  return CPL_ERROR_INCOMPATIBLE_INPUT;
	}
      // it is the first file
      // -> load and keep the keywords from the primary header
      // -> load the imaging detector table for later use
      info->keywords = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
      info->imgdet = mat_imagingdetector_duplicate(raw->imgdet);
      /* if a badpixel map is already loaded (info->bpm) it must be from the same detector */
      if (info->bpm != NULL)
	{
	  if (mat_detector_check(info->det, info->bpm->det) != CPL_ERROR_NONE)
	    {
	      cpl_msg_error(cpl_func,
			    "the file %s contains data which is not compatible with the badpixel map, ignoring the bpm",
			    cpl_frame_get_filename(frame));
	      mat_badpixel_delete(info->bpm);
	      info->bpm = NULL;
	    }
	  else
	    {
	      /* the badpixel map is ok, now map it to the sub-windows of the raw data */
	      mat_badpixel *nbpm = mat_badpixel_map(info->bpm, raw->imgdet);
	      mat_badpixel_delete(info->bpm);
	      info->bpm = nbpm;
	    }
	}
    }
  /* Either we have a valid first file already used (info->nbfiles > 0) or this is a valid first file. */
  if (mat_detector_check_raw(info->det, raw) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func,
		    "the file %s contains data which is not compatible with the previously used data",
		    cpl_frame_get_filename(frame));
      mat_gendata_delete(raw);
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  /* we have to find the sub-window which covers all detector channels (info->swi) */
  info->swi = mat_find_central_region(info->det, info->imgdet);
  if (info->swi == -1)
    {
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  info->nbtel = raw->imgdata->nbtel;
  info->maxins = raw->imgdata->maxins;
  exptime = cpl_propertylist_get_double(raw->keywords, MATISSE_KW_EXPTIME);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      // normally this FITS file is broken, but:
      // try to extract the exposure time from the IMAGING_DATA table (column EXPTIME)
      exptime = raw->imgdata->list_frame[0]->exptime;
      cpl_error_reset();
    }
  cpl_msg_info(cpl_func, "exptime=%lf", exptime);
  mat_detector_calc_timing(info->det, raw->keywords, &mindit, &tpixel, &tline, &tframe);
  cpl_msg_info(cpl_func, "mindit %g tpixel %g tline %g tframe %g", mindit, tpixel, tline, tframe);
  median = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  variance = mat_frame_duplicate(raw->imgdata->list_frame[0], CPL_TYPE_DOUBLE);
  if (mat_calc_statistics_cosmics(raw, median, variance, info->cosmics) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "Cannot calculate the pixel statistics for the file %s",
		    cpl_frame_get_filename(frame));
      mat_gendata_delete(raw);
      return CPL_ERROR_UNSPECIFIED;
    }
  /* create an image list which contains only the region which is identified by info->swi */
  imglist = cpl_imagelist_new();
  if (imglist == NULL)
    {
      cpl_error_code ecode = cpl_error_get_code();
      cpl_msg_error(cpl_func, "could not allocate the image list, code = %d, message = %s",
		    ecode,
		    cpl_error_get_message());
      mat_gendata_delete(raw);
      return ecode;
    }
  nz = raw->imgdata->nbframe;
  for (z = 0; z < nz; z++)
    {
      cpl_imagelist_set(imglist, raw->imgdata->list_frame[z]->list_subwin[info->swi]->imgreg[0], z);
    }
  if (info->bpm != NULL)
    {
      bpm = info->bpm->list_badpixel[info->swi];
    }
  if (isdark)
    {
      info->darkmedian = median;
      info->darkvariance = variance;
      /* calculate the allan variance */
      info->allanvariancemin = mat_calc_avar(tframe, imglist, bpm);
    }
  else
    {
      info->flatmedian = median;
      info->flatvariance = variance;
      mat_calc_avar(tframe, imglist, bpm);
      /* calculate the autocorrelation */
      if ((info->roix > 0) && (info->roiy > 0) && (info->roinx > 0) && (info->roiny > 0))
	{ /* a region is specified, use it for the autocorrelation */
	  mat_calc_autocorr_region(info->flatmedian->list_subwin[info->swi]->imgreg[0],
				   imglist,
				   bpm,
				   info->roix, info->roiy, info->roinx, info->roiny,
				   &(info->horizontalcorrelation),
				   &(info->verticalcorrelation));
	  cpl_msg_info(cpl_func,
		       "region %dx%d @ (%d,%d): hcorr = %.3f, vcorr = %.3f",
		       info->roinx, info->roiny, info->roix, info->roiy,
		       info->horizontalcorrelation, info->verticalcorrelation);
	}
      else
	{
	  mat_calc_autocorr(info->flatmedian->list_subwin[info->swi]->imgreg[0],
			    imglist,
			    bpm,
			    info->det->channel_ncolumns,
			    info->det->channel_nrows,
			    &(info->horizontalcorrelation),
			    &(info->verticalcorrelation),
			    NULL,
			    NULL);
	}
    }
  /* delete the image list */
  cpl_imagelist_unwrap(imglist);
  mat_gendata_delete(raw);
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_compensate_flat_statistics(mat_im_ext_info *info)
{
  mat_assert_not_null(info, "no valid mat_im_ext_info (info) given as argument");
  mat_assert_not_null(info->darkmedian, "dark median missing");
  mat_assert_not_null(info->darkvariance, "dark variance missing");
  mat_assert_not_null(info->flatmedian, "flat median missing");
  mat_assert_not_null(info->flatvariance, "flat variance missing");
  mat_frame_subtract(info->flatmedian,   info->darkmedian);
  mat_frame_subtract(info->flatvariance, info->darkvariance);
  return CPL_ERROR_NONE;
}

/*
  For the calculation of the flatfield map, a smoothed median image will be used
  to calculate the individual pixel gain. This approach covers only the small scale
  variations. The large scale variations could only be preserved in the flatfield
  map if an illumination model is available. That model could then be used to cover
  the large scale gain variations.

  Remark:

  The algorithm is changed compared to the document. The mat_update_sumweight
  function is now included.

  This implementation works only if there is only one sub-window and it
  covers the whole detector!

  Mathematical description:

<pre>BEGIN
   CALL mat_cal_gain using the data from the longest exposure time;
   Create an empty flatfield map;
   Create an empty weight array (same dimensions as the flatfield map);
   FOR EACH series of flatfield frames DO
      FOR EACH pixel DO
         IF enough light on that pixel THEN
            ffm[pixel] += intensity[pixel]*intensity[pixel]/smooth[pixel];
	    weight[pixel] += intensity[pixel];
         ENDIF
      ENDFOR
   ENDFOR
   FOR EACH pixel DO
      IF weight[pixel] == 0.0 THEN
         ffm[pixel] = 1.0;
      ELSE
         ffm[pixel] /= weight[pixel];
      ENDIF
   ENDFOR
END</pre>
*/
static cpl_error_code mat_im_extended_map(mat_im_ext_info *info)
{
  mat_imextended *ime;
  cpl_mask       *bpm = NULL;
  int             cc, cr;

  mat_assert_not_null(info, "no valid mat_im_ext_info (info) given as argument");
  ime = mat_imextended_new(info->det);
  mat_assert_not_null(ime, "could not create the extended detector monitoring map");
  info->ime = ime;
  if (info->bpm != NULL)
    {
      bpm = info->bpm->list_badpixel[info->swi];
    }
  mat_calc_gain(info->flatmedian->list_subwin[info->swi]->imgreg[0],
		info->flatvariance->list_subwin[info->swi]->imgreg[0],
		NULL,
		bpm,
		info->det->channel_ncolumns,
		info->det->channel_nrows,
		&(ime->detectorgain),
		info->ime->channelgain);
  mat_calc_offset(info->darkmedian->list_subwin[info->swi]->imgreg[0],
		  bpm,
		  info->det->channel_ncolumns,
		  info->det->channel_nrows,
		  info->ime->channeloffset);
  mat_calc_noise(info->darkvariance->list_subwin[info->swi]->imgreg[0],
		 bpm,
		 info->det->channel_ncolumns,
		 info->det->channel_nrows,
		 ime->detectorgain,
		 &(ime->detectornoise),
		 info->ime->channelgain,
		 info->ime->channelnoise,
		 1.0);
  cpl_msg_info(cpl_func, "detector: gain = %.1f e-/DU, noise = %.1f e-", ime->detectorgain, ime->detectornoise);
  for (cr = 0; cr < info->det->channel_nrows; cr++)
    {
      for (cc = 0; cc < info->det->channel_ncolumns; cc++)
	{
	  int dci = cr*info->det->channel_ncolumns + cc; /* detector channel index */
	  double rgain = cpl_vector_get(info->ime->channelgain, dci);
	  double roffset = cpl_vector_get(info->ime->channeloffset, dci);
	  double rnoise = cpl_vector_get(info->ime->channelnoise, dci);
	  cpl_msg_info(cpl_func,
		       "channel[%d,%d]: gain = %.3f e-/DU, noise = %.3f e-, offset = %.3f DU",
		       cr, cc, rgain, rnoise, roffset);
	}
    }
  /* copy the prevoiusly calculated values into the extended instrument monitoring map */
  info->ime->horizontalcorrelation = info->horizontalcorrelation;
  info->ime->verticalcorrelation = info->verticalcorrelation;
  info->ime->allanvariancemin = info->allanvariancemin;
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Interpret the command line options and execute the data processing
   @param parlist the parameters list
   @param frameset the frames list
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_extended(cpl_parameterlist *parlist,
			   cpl_frameset *frameset)
{
  cpl_parameter    *param;
  int               darkcount;
  int               flatcount;
  int               bpmcount;
  mat_im_ext_info   info;
  cpl_frame        *bpmframe = NULL;
  cpl_frame        *rawframe = NULL;

  mat_info_init(&info);
  /* HOW TO RETRIEVE INPUT PARAMETERS */
  /* --nditskip */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_extended.nditskip");
  info.nditskip = cpl_parameter_get_int(param);
  /* --cosmics */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_extended.cosmics");
  info.cosmics = cpl_parameter_get_int(param);
  /* --roi */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_extended.roi");
  mat_parameter_get_int_quadruple(param, "roi", &(info.roix), &(info.roiy), &(info.roinx), &(info.roiny));
  /* --expert */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_extended.expert");
  info.expert = cpl_parameter_get_int(param);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Failed to retrieve the input parameters");
    return -1;
  }
  /* Identify the RAW and CALIB frames in the input frameset */
  /*
  if (matisse_dfs_set_groups(frameset)) {
    cpl_msg_error(cpl_func, "Cannot identify RAW and CALIB frames");
    return -1;
  }
  */
  /* The input data consists of dark and flatfield files (pairs!) */
  darkcount = cpl_frameset_count_tags(frameset, MATISSE_DO_DARK);
  if (darkcount == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not have any dark files");
      return -1;
    }
  flatcount = cpl_frameset_count_tags(frameset, MATISSE_DO_FLAT);
  if (flatcount == 0)
    {
      cpl_msg_error(cpl_func, "SOF does not have any flat field files ");
      return -1;
    }
  bpmcount = cpl_frameset_count_tags(frameset, MATISSE_DO_BADPIXEL);
  if (bpmcount > 1)
    {
      cpl_msg_error(cpl_func, "only zero or one badpixel map is allowed in the SOF");
      mat_info_delete(&info);
      return -1;
    }
  /* load the optional badpixel map */
  if (bpmcount == 1)
    {
      bpmframe = cpl_frameset_find(frameset, MATISSE_DO_BADPIXEL); // already checked previously!
      cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_CALIB);
      info.bpm = mat_badpixel_load(bpmframe);
    }
  /* Calculate the pixel statistics (modified due to the frameset processing schema) */
  /* CALL mat_calc_statistics for the first cold dark series */
  rawframe = cpl_frameset_find(frameset, MATISSE_DO_DARK); // already checked previously!
  cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(rawframe));
  cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
  if (mat_calc_raw_statistics(&info, 1, rawframe) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      return -1;
    }
  /* CALL mat_calc_statistics for the first flatfield series */
  rawframe = cpl_frameset_find(frameset, MATISSE_DO_FLAT); // already checked previously!
  cpl_msg_info(cpl_func, "loading and processing of %s", cpl_frame_get_filename(rawframe));
  cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
  if (mat_calc_raw_statistics(&info, 0, rawframe) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      return -1;
    }
  cpl_msg_info(cpl_func, "compensate the flat statistics using the dark statistics");
  if (mat_compensate_flat_statistics(&info) != CPL_ERROR_NONE)
    {
      cpl_msg_error(cpl_func, "could not compensate the flat statistics using the dark statistics");
      return -1;
    }
  if (mat_im_extended_map(&info) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      return -1;
    }
  if (info.expert != 0)
    {
      mat_info_store_statistics(&info, "mat_im_extended_stat.fits",
				parlist,
				frameset);
    }
  mat_imextended_save(info.ime,
		      mat_detector_get_ime_name(info.det),
		      "mat_im_extended",
		      parlist,
		      frameset);
  mat_info_delete(&info);
  /* Return */
  if (cpl_error_get_code())
    return -1;
  else
    return 0;
}
