/* $Id: mat_im_rem.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_im_rem.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MATISSE_LICENCE             "GPL"
#define MAX_FILES                   16

#define MATISSE_KW_EXPTIME          "EXPTIME"

#define MATISSE_DO_PERIODIC         "IM_PERIODIC"
#define MATISSE_DO_BADPIXEL         "BADPIX"
#define MATISSE_STATISTICS_PROCATG  "STAT"
#define MEDIAN_KERNEL_SIZE          7

#define MAT_MIN_INTENSITY_DIFFERENCE      200.0
#define MATISSE_MAX_SEQUENCE_COUNT 16


/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/
#include <strings.h>
#include <ctype.h>

#include <cpl.h>
#include "mat_drl.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_frame.h"
#include "mat_statistics.h"
#include "mat_detector.h"
#include "mat_imagingdetector.h"
#include "mat_badpixel.h"
#include "mat_imremanence.h"

/*-----------------------------------------------------------------------------
  Data types
  -----------------------------------------------------------------------------*/
typedef struct {
  int                  nditskip;
  int                  global_offset_compensation;
  int                  seqoff;
  int                  seqlen;
  int                  poir;
  int                  poix;
  int                  poiy;
  int                  roir;
  int                  roix;
  int                  roiy;
  int                  roinx;
  int                  roiny;
  int                  frame_left;
  int                  frame_right;
  int                  frame_bottom;
  int                  frame_top;
  int                  invert;
  int                  expert;
  mat_detector        *det; /* this the detector specification from the periodic file */
  mat_badpixel        *bpm;
  int                  nbregion;
  cpl_mask            *detector_frame;
  cpl_mask           **region_frames;
  cpl_vector          *intensities;
  double              *ovalues;
  double              *values;
  int                  count;
  int                  count_todark[MATISSE_MAX_SEQUENCE_COUNT];
  double               todark[MATISSE_MAX_SEQUENCE_COUNT];
  int                  count_tobright[MATISSE_MAX_SEQUENCE_COUNT];
  double               tobright[MATISSE_MAX_SEQUENCE_COUNT];
  mat_imremanence     *imr;
} mat_im_rem_info;
/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/
static int mat_im_rem_create(cpl_plugin *);
static int mat_im_rem_exec(cpl_plugin *);
static int mat_im_rem_destroy(cpl_plugin *);
static int mat_im_rem(cpl_parameterlist *, cpl_frameset *);
/*-----------------------------------------------------------------------------
  Static variables
  -----------------------------------------------------------------------------*/
static char mat_im_rem_description[] =
  "This plugin calculates the detector remanence. The corresponding MATISSE_gen_cal_imrem template creates the input FITS file. Since it is expected that the remanence will not change during the lifetime of the detector, this functionality is only needed during the detector characterization and later on once a year or after exchanging a detector. The template will create a FITS file containing interwoven dark and flatfield raw frames (DO classification IM_PERIODIC). The fast cold shutter of the instrument will be used to generate a FITS file where a fixed number of flatfield frames (open shutter) is followed by a fixed number of cold dark frames (closed shutter) and so on. In total at least 5 such pairs are needed. It must be noted, that these raw frames are taken as a continuous sequence and they are stored in one FITS file!\n"
  "In addition, the bad pixel (PRO.CATG = BADPIX) static calibration map is needed.\n"
  "Since the global illumination of the MATISSE detectors is non-uniform, all calculations are done on individual pixels which show a sufficient dark-to-bright difference. This means for each usable pixel all possible dark-to-bright and bright-to-dark remanence values (one value for each transition) are calculated and used to calculate a transition specific mean (we will have about five dark-to-bright and five bright-to-dark transitions in the input file). The smallest remanence value for a transition type is used to quantify the detector remanence, because if the shutter does not work perfectly, the remanence values are higher than for an ideal shutter.\n"
  "The sequence of pixel intensities should show a clear square wave form. These intensities are sorted (increasing value) and the middle value between the intensity at 5 percent of the values (5 percent quantil) and 95 percent is used as a threshold. All intensities (in the original order!) are classified as dark (below the threshold) or bright (above the threshold). For a sequence of at least 10 consecutive darks or brights the average is calculated by using the second half of these values (ignoring the last value, because it may be influenced by a slow shutter). For each transition, the first intensity (the first dark intensity after a sequence of bright intensities or the first bright intensity after a sequence of dark intensities) and the difference of the intensity averages is used to calculate the remanence for this transition.\n"
  "This plugin creates an output file (PRO.CATG = IM_REMANENCE) which contains an empty HDU with all QC1 parameters.\n"
  "The following QC1 parameters will be calculated and stored in the primary HDU of the result FITS file:\n"
  " - QC DET<i> REM TODARK : The remanence value for switching from a bright to a dark frame.\n"
  " - QC DET<i> REM TOBRIGHT : The remanence value for switching from dark to bright frames.\n"
  "Where i is the detector number (1 = L/M-Band, 2 = N-Band). The detector number (used for DET<i>) will be always 1 for the L/M-Band detector and 2 for the N-Band detector. In addition, the DET<i> CHIP1 ID keyword allows to distinguish between different detector chips. This is necessary, to deal with a detector change after, for example, an instrument repair.\n"
  "\n"
  "Input files:\n"
  "\n"
  "  DO category:      Explanation:              Required:\n"
  "  IM_PERIODIC       Raw dark/exposed frames   Yes\n"
  "  BADPIX            Static bad pixel map      No\n"
  "\n"
  "Output files:\n"
  "\n"
  "  DO category:      Explanation:\n"
  "  IM_REMANENCE      Detector remanence information"
;
/*-----------------------------------------------------------------------------
  Functions code
  -----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
   @brief Build the list of available plugins, for this module.
   @param list the plugin list
   @return 0 if everything is ok, -1 otherwise
   Create the recipe instance and make it available to the application using the
   interface. This function is exported.
*/
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist * list)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
  cpl_plugin *plugin = &recipe->interface;

  cpl_plugin_init(plugin,
		  CPL_PLUGIN_API,
		  MATISSE_BINARY_VERSION,
		  CPL_PLUGIN_TYPE_RECIPE,
		  "mat_im_rem",
		  "Calculates a detector remanence map",
		  mat_im_rem_description,
		  "Matthias Heininger",
		  "mhein@mpifr-bonn.mpg.de",
		  MATISSE_LICENCE,
		  mat_im_rem_create,
		  mat_im_rem_exec,
		  mat_im_rem_destroy);
  cpl_pluginlist_append(list, plugin);
  return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Setup the recipe options
   @param plugin the plugin
   @return 0 if everything is ok
   Defining the command-line/configuration parameters for the recipe.
*/
/*----------------------------------------------------------------------------*/
static int mat_im_rem_create(cpl_plugin * plugin)
{
  cpl_recipe    *recipe;
  cpl_parameter *p;

  /* Check that the plugin is part of a valid recipe */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  /* Create the parameters list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */
  /* --nditskip */
  p = cpl_parameter_new_range("matisse.mat_im_rem.nditskip",
			      CPL_TYPE_INT,
			      "number of skipped frames",
			      "matisse.mat_im_rem",
			      0, 0, 20);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "nditskip");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --global-offset */
  p = cpl_parameter_new_value("matisse.mat_im_rem.global-offset",
			      CPL_TYPE_INT,
			      "global offset compensation",
			      "matisse.mat_im_rem",
			      0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "global-offset");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --frame */
  p = cpl_parameter_new_value("matisse.mat_im_rem.frame",
			      CPL_TYPE_STRING,
			      "framesize: <left>,<right>,<bottom>,<top>",
			      "matisse.mat_im_rem",
			      "-1,-1,-1,-1");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "frame");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --poi */
  p = cpl_parameter_new_value("matisse.mat_im_rem.poi",
			      CPL_TYPE_STRING,
			      "pixel of interest: <region>,<x>,<y>",
			      "matisse.mat_im_rem",
			      "0,0,0");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "poi");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --roi */
  p = cpl_parameter_new_value("matisse.mat_im_rem.roi",
			      CPL_TYPE_STRING,
			      "region of interest: <r>,<x>,<y>,<nx>,<ny>",
			      "matisse.mat_im_rem",
			      "-1,-1,-1,-1,-1");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "roi");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --seq */
  p = cpl_parameter_new_value("matisse.mat_im_rem.seq",
			      CPL_TYPE_STRING,
			      "shutter sequence: <offset>,<length>",
			      "matisse.mat_im_rem",
			      "0,0");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "seq");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --invert */
  p = cpl_parameter_new_range("matisse.mat_im_rem.invert",
			      CPL_TYPE_INT,
			      "invert the pixel intensities value := (65535 - value)",
			      "matisse.mat_im_rem",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "invert");
  cpl_parameterlist_append(recipe->parameters, p);
  /* --expert */
  p = cpl_parameter_new_range("matisse.mat_im_rem.expert",
			      CPL_TYPE_INT,
			      "expert flag",
			      "matisse.mat_im_rem",
			      0, 0, 1);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "expert");
  cpl_parameterlist_append(recipe->parameters, p);
  /* Return */
  return 0;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Execute the plugin instance given by the interface
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_rem_exec(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  return mat_im_rem(recipe->parameters, recipe->frames);
}
/*----------------------------------------------------------------------------*/
/**
   @brief Destroy what has been created by the ’create’ function
   @param plugin the plugin
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_rem_destroy(cpl_plugin * plugin)
{
  cpl_recipe * recipe;

  /* Get the recipe out of the plugin */
  if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
    recipe = (cpl_recipe *)plugin;
  else
    return -1;
  cpl_parameterlist_delete(recipe->parameters);
  return 0;
}

static void mat_info_init(mat_im_rem_info *info)
{
  int   i;

  /* initialize the info data structure */
  info->det = mat_detector_new();
  if (info->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for a mat_detector");
      return;
    }
  info->bpm = NULL;
  info->nbregion = 0;
  info->detector_frame = NULL;
  info->region_frames = NULL;
  info->intensities = NULL;
  info->ovalues = NULL;
  info->values = NULL;
  info->count = 0;
  for (i = 0; i < MATISSE_MAX_SEQUENCE_COUNT; i++)
    {
      info->count_todark[i] = 0;
      info->todark[i] = 0.0;
      info->count_tobright[i] = 0;
      info->tobright[i] = 0.0;
    }
  info->imr = NULL;
}

static void mat_info_delete(mat_im_rem_info *info)
{
  /* delete the info data structure */
  if (info->det != NULL)
    {
      mat_detector_delete(info->det);
      info->det = NULL;
    }
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->detector_frame != NULL)
    {
      cpl_mask_delete(info->detector_frame);
      info->detector_frame = NULL;
    }
  if (info->region_frames != NULL)
    {
      int r;
      for (r = 0; r < info->nbregion; r++)
	{
	  if (info->region_frames[r] != NULL)
	    {
	      cpl_mask_delete(info->region_frames[r]);
	      info->region_frames[r] = NULL;
	    }
	}
      cpl_free(info->region_frames);
      info->region_frames = NULL;
    }
  if (info->intensities != NULL)
    {
      cpl_vector_delete(info->intensities);
      info->intensities = NULL;
    }
  if (info->ovalues != NULL)
    {
      cpl_free(info->ovalues);
      info->ovalues = NULL;
    }
  if (info->values != NULL)
    {
      cpl_free(info->values);
      info->values = NULL;
    }
  if (info->imr != NULL)
    {
      mat_imremanence_delete(info->imr);
      info->imr = NULL;
    }
}

static int mat_double_compare(const void *e1, const void *e2)
{
  double *x = (double*)e1;
  double *y = (double*)e2;
  
  if (*x > *y) return 1;
  if (*x == *y) return 0;
  return -1;
}

static void mat_calc_pixel_remanence(mat_im_rem_info *info,
				     mat_frame **list_frame,
				     int r,
				     int x,
				     int y,
				     int nz)
{
  int         z, zz;
  int         s;
  int         cdb;
  int         cbd;
  int         rejected;
  double      high = 0.0, low = 0.0, center = 0.0;
  double     *ovalues = info->ovalues;
  double     *values  = info->values;
  int         seqcount = 0;
  int         seqstarts[MATISSE_MAX_SEQUENCE_COUNT];
  int         seqlengths[MATISSE_MAX_SEQUENCE_COUNT];
  double      seqavgs[MATISSE_MAX_SEQUENCE_COUNT];

  /* 1. Extract the pixel intensities for a specific pixel and put them into a vector. */
  for (z = 0; z < nz; z++)
    {
      double v = cpl_image_get(list_frame[z]->list_subwin[r]->imgreg[0], x, y, &rejected);
      ovalues[z] = v;
    }
  memcpy(values, ovalues, nz*sizeof(double));
  if (info->seqlen == 0)
    { /* the user did not specify the shutter sequence => we have to extract it ourself */
      /* 2. Sort the intensities and use the lower 5 percent and upper 95 percent to calculate the middle intensity.
	 This procedure is used because for a periodic sequence of dark and flat frames the median or mean cannot be used. */
      qsort(values, nz, sizeof(double), mat_double_compare);
      low = values[nz/20];
      high = values[nz - nz/20];
      center = 0.5*(low + high);
      if ((r == info->poir) && (x == info->poix) && (y == info->poiy))
	{
	  cpl_msg_info(cpl_func, "  center=%f", center);
	}
      /* 3a. Separate the pixel intensities into sequences of dark (0.0) and bright (1.0) frames. */
      for (z = 0; z < nz; z++)
	{
	  if (ovalues[z] < center)
	    {
	      values[z] = 0.0;
	    }
	  else
	    {
	      values[z] = 1.0;
	    }
	}
      /* 3b. Try to deal with remanence up to 70 percent by reassigning the first dark image (bright to dark) and first bright (dark to bright) */
      for (z = 0; z < nz - 2; z++)
	{
	  if ((values[z] == 1.0) && (values[z + 1] == 1.0) && (values[z + 2] == 0.0))
	    { /* we have a falling edge, (bright, bright, dark) */
	      if (ovalues[z + 1] < (low + 0.7*(high - low)))
		{ /* the brightness is below 70 percent of the difference between dark and bright */
		  values[z + 1] = 0.0; /* we assume that this frame is already a dark frame */
		}
	    }
	  if ((values[z] == 0.0) && (values[z + 1] == 0.0) && (values[z + 2] == 1.0))
	    { /* we have a raising edge, (dark, dark, bright) */
	      if (ovalues[z + 1] > (low + 0.3*(high - low)))
		{ /* the brightness is above 30 percent of the difference between dark and bright */
		  values[z + 1] = 1.0; /* we assume that this frame is already a bright frame */
		}
	    }
	}
    }
  else if (info->seqlen == -1)
    { // use the target type code (B -> 1.0, D -> 0.0
      for (z = 0; z < nz; z++)
	{
	  if (list_frame[z]->tartype[0] == 'B')
	    {
	      values[z] = 1.0;
	    }
	  else
	    {
	      values[z] = 0.0;
	    }
	}
    }
  else
    { /* the user did specify the shutter sequence */
      /* seqoff : image number with the first bright frame) */
      /* seqlen : number of consecutive bright or dark frames */
      for (z = 0; z < nz; z++)
	{
	  if (((z - info->seqoff)/info->seqlen)%2 == 0)
	    {
	      values[z] = 1.0;
	    }
	  else
	    {
	      values[z] = 0.0;
	    }
	}
    }
  /* 3c. Show the optional pixel values and their classification */
  if ((r == info->poir) && (x == info->poix) && (y == info->poiy))
    {
      cpl_msg_info(cpl_func, "values for pixel %d,%d,%d", r, x, y);
      cpl_msg_info(cpl_func, "  limits %f %f %f", low, center, high);
      for (z = 0; z < nz; z++)
	{
	  cpl_msg_info(cpl_func, "  %d %f %f", z, ovalues[z], values[z]);
	}
    }
  /* 4. Extract the sequences of darks and brights and calculate their intensity level if possible. */
  seqcount = 0;
  seqstarts[seqcount] = 0;
  seqlengths[seqcount] = 0;
  for (z = 0; z < nz; z++)
    {
      /* Check if we are in the same dark or bright sequence */
      if (values[z] == values[seqstarts[seqcount]])
	{
	  seqlengths[seqcount]++;
	}
      else
	{
	  /* If we have at least 10 frames in this sequence, we can calculate an intensity average by using the second half of the values in that sequence. */
	  seqavgs[seqcount] = 0.0;
	  if (seqlengths[seqcount] >= 10)
	    {
	      // Beware: ignore the last frame in a sequence (matbe it is influenced by a slow shutter)
	      for (zz = 1; zz < seqlengths[seqcount]/2; zz++)
		{
		  seqavgs[seqcount] += ovalues[seqstarts[seqcount] + seqlengths[seqcount] - zz - 1];
		}
	      seqavgs[seqcount] /= (double)(seqlengths[seqcount]/2 - 1);
	    }
	  if ((r == info->poir) && (x == info->poix) && (y == info->poiy))
	    {
	      cpl_msg_info(cpl_func, "  sequence[%d] : start=%d, count=%d, avg=%f, first=%f",
			   seqcount, seqstarts[seqcount], seqlengths[seqcount], seqavgs[seqcount], ovalues[seqstarts[seqcount]]);
	    }
	  seqcount++;
	  if (seqcount == MATISSE_MAX_SEQUENCE_COUNT) break;
	  seqstarts[seqcount] = z;
	  seqlengths[seqcount] = 1;
	}
    }
  /* 5. Calculate the remanence going from dark to bright. Check that the intensity difference is large enough. */
  cdb = 0;
  for (s = 0; s < (seqcount - 1); s++)
    {
      double rem;
      if (values[seqstarts[s]] != 0.0) continue; /* the current sequence is not a dark sequence, skip */
      if (seqlengths[s] < 10) continue; /* the dark sequence is not long enough, skip */
      if (values[seqstarts[s + 1]] != 1.0) continue; /* the next sequence is not a bright sequence, skip (paranoia!) */
      if (seqlengths[s + 1] < 10) continue; /* the bright sequence is not long enough, skip */
      if (seqavgs[s + 1] - seqavgs[s] < MAT_MIN_INTENSITY_DIFFERENCE) continue; /* the intensity difference is too small, skip */
      /*
       * seqavgs[s]     : the average intensiy of the dark sequence, calculated from the end of the dark sequence
       * seqavgs[s + 1] : the average intensity for the bright sequence, calculated from the end of the bright sequence
       * ovalues[seqstarts[s + 1]] : the first bright intensity value
       * rem := (first - dark)/(bright - dark)
       */
      rem = (ovalues[seqstarts[s + 1]] - seqavgs[s])/(seqavgs[s + 1] - seqavgs[s]);
      if ((r == info->poir) && (x == info->poix) && (y == info->poiy))
	{
	  cpl_msg_info(cpl_func, "  rem tobright[%d]=%f", cdb, rem);
	}
      if (rem > 0.0)
	{
	  info->tobright[cdb] += rem;
	  info->count_tobright[cdb]++;
	}
      cdb++;
    }
  /* 6. Calculate the remanence going from bright to dark. Check that the intensity difference is large enough. */
  cbd = 0;
  for (s = 0; s < (seqcount - 1); s++)
    {
      double rem;
      if (values[seqstarts[s]] != 1.0) continue; /* the current sequence is not a bright sequence, skip */
      if (seqlengths[s] < 10) continue; /* the bright sequence is not long enough, skip */
      if (values[seqstarts[s + 1]] != 0.0) continue; /* the next sequence is not a dark sequence, skip (paranoia!) */
      if (seqlengths[s + 1] < 10) continue; /* the dark sequence is not long enough, skip */
      if (seqavgs[s] - seqavgs[s + 1] < MAT_MIN_INTENSITY_DIFFERENCE) continue; /* the intensity difference is too small, skip */
      /*
       * seqavgs[s]     : the average intensiy of the bright sequence, calculated from the end of the bright sequence
       * seqavgs[s + 1] : the average intensity for the dark sequence, calculated from the end of the dark sequence
       * ovalues[seqstarts[s + 1]] : the first dark intensity value
       * rem := (first - dark)/(bright - dark)
       */
      rem = (ovalues[seqstarts[s + 1]] - seqavgs[s + 1])/(seqavgs[s] - seqavgs[s + 1]);
      if ((r == info->poir) && (x == info->poix) && (y == info->poiy))
	{
	  cpl_msg_info(cpl_func, "  rem todark[%d]=%f", cbd, rem);
	}
      if (rem > 0.0)
	{
	  info->todark[cbd] += rem;
	  info->count_todark[cbd]++;
	}
      cbd++;
    }
  if ((cdb != 0) && (cbd != 0))
    {
      info->count++;
    }
}

/*
static cpl_error_code mat_compensate_globaloffset_amber(mat_im_rem_info *info, mat_gendata *raw)
{
  int          nz;
  int          x, y, r, z;
  int          rejected;
  int          count = 0;

  nz = raw->imgdata->nbframe;
  // 1. Create a mask representing the frame for the whole detector.
  info->detector_frame = cpl_mask_new(info->det->nx, info->det->ny);
  if (info->detector_frame == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the detector frame mask");
      return CPL_ERROR_UNSPECIFIED;
    }
  for (x = 1; x <= info->frame_left; x++)
    {
      for (y = 1; y <= info->det->ny; y++)
	{
	  cpl_mask_set(info->detector_frame, x, y, CPL_BINARY_1);
	}
    }
  for (x = info->det->nx - info->frame_right + 1; x <= info->det->nx; x++)
    {
      for (y = 1; y <= info->det->ny; y++)
	{
	  cpl_mask_set(info->detector_frame, x, y, CPL_BINARY_1);
	}
    }
  for (y = 1; y <= info->frame_bottom; y++)
    {
      for (x = 1; x <= info->det->nx; x++)
	{
	  cpl_mask_set(info->detector_frame, x, y, CPL_BINARY_1);
	}
    }
  for (y = info->det->ny - info->frame_top + 1; y <= info->det->ny; y++)
    {
      for (x = 1; x <= info->det->nx; x++)
	{
	  cpl_mask_set(info->detector_frame, x, y, CPL_BINARY_1);
	}
    }
  // 2. Create the frame masks for each region/sub-window which contains the dark pixels
  info->region_frames = (cpl_mask**)cpl_calloc(raw->imgdet->nbregion, sizeof(cpl_mask*));
  if (info->region_frames == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the frame masks");
      return CPL_ERROR_UNSPECIFIED;
    }
  count = 0;
  for (r = 0; r < info->nbregion; r++)
    {
      int x0 = raw->imgdet->list_region[r]->corner[0];
      int nx = raw->imgdet->list_region[r]->naxis[0];
      int y0 = raw->imgdet->list_region[r]->corner[1];
      int ny = raw->imgdet->list_region[r]->naxis[1];
      info->region_frames[r] = cpl_mask_extract(info->detector_frame, x0, y0, x0 + nx - 1, y0 + ny - 1);
      if (info->region_frames[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for the frame mask[%d]", r);
	  return CPL_ERROR_UNSPECIFIED;
	}
      count += cpl_mask_count(info->region_frames[r]);
    }
  if (count == 0)
    {
      // the sub-windows in the raw data do no have any dark pixels usable for the global offset
      cpl_msg_info(cpl_func, "the sub-windows in the raw data do no have any dark pixels");
      return CPL_ERROR_NONE;
    }
  info->intensities = cpl_vector_new(count);
  if (info->intensities == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the intensity vector");
      return CPL_ERROR_UNSPECIFIED;
    }
  // 3. Calculate and apply for each frame the global offset (valid only for the AMBER detector).
  // cpl_msg_info(cpl_func, "global offset for %d frames", nz);
  for (z = 0; z < nz; z++)
    {
      int    iPixel = 0;
      double mean;
      if (z%10 == 0)
	{
	  cpl_msg_info(cpl_func, "compensating frame %d", z);
	}
      for (r = 0; r < info->nbregion; r++)
	{
	  int nx = raw->imgdet->list_region[r]->naxis[0];
	  int ny = raw->imgdet->list_region[r]->naxis[1];
	  for (x = 1; x <= nx; x++)
	    {
	      for (y = 1; y <= ny; y++)
		{
		  if (cpl_mask_get(info->region_frames[r], x, y))
		    {
		      cpl_vector_set(info->intensities,
				     iPixel++,
				     cpl_image_get(raw->imgdata->list_frame[z]->list_subwin[r]->imgreg[0],
						   x,
						   y,
						   &rejected)
				     );
		    }
		}
	    }
	}
      mean   = cpl_vector_get_mean(info->intensities);
      // cpl_msg_info(cpl_func, "   %d %f %f", z, median, mean);
      for (r = 0; r < info->nbregion; r++)
	{
	  cpl_image_subtract_scalar(raw->imgdata->list_frame[z]->list_subwin[r]->imgreg[0], mean);
	}
    }
  return CPL_ERROR_NONE;
}
*/

static cpl_error_code mat_calc_det_remanence(mat_im_rem_info *info,
					     cpl_frame *frame)
{
  mat_gendata *raw;
  int         nz;
  int         x, y, z, r;
  int         i;

  cpl_msg_info(cpl_func, "load the raw data from file %s", cpl_frame_get_filename(frame));
  raw = mat_gendata_load_skip(frame, info->nditskip, CPL_TYPE_FLOAT);
  if (raw == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot read the FITS file");
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    {
      cpl_msg_info(cpl_func, "raw data file successfully loaded");
    }
  mat_detector_decode_raw(info->det, raw);
  /* if a badpixel map is already loaded (info->bpm) it must be from the same detector */
  if (info->bpm != NULL)
    {
      if (mat_detector_check(info->det, info->bpm->det) != CPL_ERROR_NONE)
	{
	  cpl_msg_error(cpl_func,
			"the file %s contains data which is not compatible with the badpixel map, ignoring the bpm",
			cpl_frame_get_filename(frame));
	  mat_badpixel_delete(info->bpm);
	  info->bpm = NULL;
	}
      else
	{
	  /* the badpixel map is ok, now map it to the sub-windows of the raw data */
	  mat_badpixel *nbpm = mat_badpixel_map(info->bpm, raw->imgdet);
	  mat_badpixel_delete(info->bpm);
	  info->bpm = nbpm;
	}
    }
  nz = raw->imgdata->nbframe;
  info->nbregion = raw->imgdet->nbregion;
  if (info->invert)
    { // invert the pixel intensities value := (65535 - value)
      for (r = 0; r < raw->imgdet->nbregion; r++)
	{
	  mat_region *reg = raw->imgdet->list_region[r];
	  int nx = reg->naxis[0];
	  int ny = reg->naxis[1];
	  for (z = 0; z < nz; z++)
	    {
	      cpl_image *img = raw->imgdata->list_frame[z]->list_subwin[r]->imgreg[0];
	      for (y = 1; y <= ny; y++)
		{
		  for (x = 1; x <= nx; x++)
		    {
		      int rejected;
		      cpl_image_set(img, x, y, 65535.0 - cpl_image_get(img, x, y, &rejected));
		    }
		}
	    }
	}
    }
  /* if the frame parameter is -1,-1,-1,-1 use the values from the mat_detector structure */
  if (info->frame_left == -1)
    {
      info->frame_left = info->det->frame_left;
    }
  if (info->frame_right == -1)
    {
      info->frame_right = info->det->frame_right;
    }
  if (info->frame_bottom == -1)
    {
      info->frame_bottom = info->det->frame_bottom;
    }
  if (info->frame_top == -1)
    {
      info->frame_top = info->det->frame_top;
    }
  /* global offset compensation is only implemented for the AMBER detector */
  if (info->global_offset_compensation)
    {
      switch (info->det->type)
	{
	case MAT_UNKNOWN_DET:
	  cpl_msg_warning(cpl_func, "cannot calculate the global offset for a unknown detector type, ignoring this option");
	  break;
	case MAT_HAWAII2RG_DET:
	  cpl_msg_warning(cpl_func, "cannot calculate the global offset for a HAWAII-2RG detector, ignoring this option");
	  break;
	case MAT_AQUARIUS_DET:
	  cpl_msg_warning(cpl_func, "cannot calculate the global offset for a Aquarius detector, ignoring this option");
	  break;
	default:
	  cpl_msg_warning(cpl_func, "cannot calculate the global offset for a detector type %d, ignoring this option", info->det->type);
	}
    }
  info->ovalues = cpl_calloc(nz, sizeof(double));
  if (info->ovalues == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the original pixel intensities");
      mat_gendata_delete(raw);
      return CPL_ERROR_UNSPECIFIED;
    }
  info->values = cpl_calloc(nz, sizeof(double));
  if (info->values == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for the pixel intensities");
      mat_gendata_delete(raw);
      return CPL_ERROR_UNSPECIFIED;
    }
  if ((info->roir != -1) && (info->roix > 0) && (info->roiy > 0) && (info->roinx > 0) && (info->roiny > 0))
    { /* a region is specified, use it for the remanence measurements (beware: zero based x and y!) */
      for (y = 0; y < info->roiny; y++)
	{
	  if (y%16 == 0)
	    {
	      cpl_msg_info(cpl_func, "processing pixel line %d", y);
	    }
	  for (x = 0; x < info->roinx; x++)
	    {
	      if (info->bpm != NULL)
		{
		  if (mat_badpixel_get(info->bpm, info->roir, info->roix + x, info->roiy + y)) continue; /* ignore bad pixels */
		}
	      mat_calc_pixel_remanence(info, raw->imgdata->list_frame, info->roir, info->roix + x, info->roiy + y, nz);
	    }
	}
    }
  else
    {
      for (r = 0; r < info->nbregion; r++)
	{
	  int nx = raw->imgdet->list_region[r]->naxis[0];
	  int ny = raw->imgdet->list_region[r]->naxis[1];
	  for (y = 1; y <= ny; y++)
	    {
	      if ((y - 1)%16 == 0)
		{
		  cpl_msg_info(cpl_func, "processing pixel line %d", y - 1);
		}
	      for (x = 1; x <= nx; x++)
		{
		  if (info->bpm != NULL)
		    {
		      if (mat_badpixel_get(info->bpm, r, x, y)) continue; /* ignore bad pixels */
		    }
		  mat_calc_pixel_remanence(info, raw->imgdata->list_frame, r, x, y, nz);
		}
	    }
	}
    }
  if (info->count == 0)
    {
      mat_gendata_delete(raw);
      cpl_msg_error(cpl_func, "no pixels could be used to calculate the remanence (no periodic darks/flats or intensity too small)");
      return CPL_ERROR_UNSPECIFIED;
    }
  else
    {
      double todark = 1.0;
      double tobright = 0.0;
      cpl_msg_info(cpl_func, "%d pixels used to calculate the remanence", info->count);
      /* using the lowest remanence values due to a possible slow shutter which leads to an increased remanence */
      for (i = 0; i < MATISSE_MAX_SEQUENCE_COUNT; i++)
	{
	  if (info->count_todark[i] != 0)
	    {
	      info->todark[i] /= (double)info->count_todark[i];
	      cpl_msg_info(cpl_func, "  bright to dark[%d] = %f", i, info->todark[i]);
	      if ((info->todark[i] < todark) && (info->todark[i] > 0.0))
		{
		  todark = info->todark[i];
		}
	    }
	  if (info->count_tobright[i] != 0)
	    {
	      info->tobright[i] /= (double)info->count_tobright[i];
	      cpl_msg_info(cpl_func, "  dark to bright[%d] = %f", i, info->tobright[i]);
	      if ((info->tobright[i] > tobright) && (info->tobright[i] > 0.0))
		{
		  tobright = info->tobright[i];
		}
	    }
	}
      cpl_msg_info(cpl_func, "  bright to dark = %f", todark);
      cpl_msg_info(cpl_func, "  dark to bright = %f", tobright);
      info->imr = mat_imremanence_new(info->det);
      if (info->imr == NULL)
	{
	  mat_gendata_delete(raw);
	  return CPL_ERROR_UNSPECIFIED;
	}
      info->imr->todarkremanence = todark;
      info->imr->tobrightremanence = tobright;
      mat_gendata_delete(raw);
      return CPL_ERROR_NONE;
    }
}

/*----------------------------------------------------------------------------*/
/**
   @brief Interpret the command line options and execute the data processing
   @param parlist the parameters list
   @param frameset the frames list
   @return 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/
static int mat_im_rem(cpl_parameterlist *parlist,
		       cpl_frameset *frameset)
{
  cpl_parameter    *param;
  int               periodiccount;
  int               bpmcount;
  mat_im_rem_info   info;
  cpl_frame        *bpmframe = NULL;
  cpl_frame        *rawframe = NULL;

  mat_info_init(&info);
  /* HOW TO RETRIEVE INPUT PARAMETERS */
  /* --nditskip */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.nditskip");
  info.nditskip = cpl_parameter_get_int(param);
  /* --global-offset */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.global-offset");
  info.global_offset_compensation = cpl_parameter_get_int(param);
  /* --frame */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.frame");
  mat_parameter_get_int_quadruple(param, "frame", &(info.frame_left), &(info.frame_right), &(info.frame_bottom), &(info.frame_top));
  /* --poi */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.poi");
  mat_parameter_get_int_triple(param, "poi", &(info.poir), &(info.poix), &(info.poiy));
  /* --roi */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.roi");
  mat_parameter_get_int_quintuple(param, "roi", &(info.roir), &(info.roix), &(info.roiy), &(info.roinx), &(info.roiny));
  /* --seq */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.seq");
  mat_parameter_get_int_double(param, "seq", &(info.seqoff), &(info.seqlen));
  /* --invert */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.invert");
  info.invert = cpl_parameter_get_int(param);
  /* --expert */
  param = cpl_parameterlist_find(parlist, "matisse.mat_im_rem.expert");
  info.expert = cpl_parameter_get_int(param);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Failed to retrieve the input parameters");
    return -1;
  }
  /* Identify the RAW and CALIB frames in the input frameset */
  /*
  if (matisse_dfs_set_groups(frameset)) {
    cpl_msg_error(cpl_func, "Cannot identify RAW and CALIB frames");
    return -1;
  }
  */
  /* The input data consists of dark and flatfield files (pairs!) */
  periodiccount = cpl_frameset_count_tags(frameset, MATISSE_DO_PERIODIC);
  if (periodiccount != 1)
    {
      cpl_msg_error(cpl_func, "SOF does not contain exactly one file with periodic darks/flats");
      mat_info_delete(&info);
      return -1;
    }
  bpmcount = cpl_frameset_count_tags(frameset, MATISSE_DO_BADPIXEL);
  if (bpmcount > 1)
    {
      cpl_msg_error(cpl_func, "only zero or one badpixel map is allowed in the SOF");
      mat_info_delete(&info);
      return -1;
    }
  /* load the optional badpixel map */
  if (bpmcount == 1)
    {
      bpmframe = cpl_frameset_find(frameset, MATISSE_DO_BADPIXEL); // already checked previously!
      cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_CALIB);
      info.bpm = mat_badpixel_load(bpmframe);
    }
  /* load the raw frames and calculate the remanence */
  rawframe = cpl_frameset_find(frameset, MATISSE_DO_PERIODIC); // already checked previously!
  cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
  if (mat_calc_det_remanence(&info, rawframe) != CPL_ERROR_NONE)
    {
      mat_info_delete(&info);
      return -1;
    }
  mat_imremanence_save(info.imr,
		       mat_detector_get_imr_name(info.det),
		       "mat_im_rem",
		       parlist,
		       frameset);
  mat_info_delete(&info);
  /* Return */
  if (cpl_error_get_code())
    return -1;
  else
    return 0;
}
