import os
import numpy as np
from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase
from adari_core.plots.points import LinePlot
from adari_core.plots.images import ImagePlot
from adari_core.plots.panel import Panel
from adari_core.utils.utils import fetch_kw_or_default

from .matisse_utils import MatisseSetupInfo, MatisseReportMixin


class MatisseDetmonReport(MatisseReportMixin, AdariReportBase):
    def __init__(self):
        super().__init__("matisse_detmon")

    def parse_sof(self):
        badpix = None
        flat = None
        nonlin = None

        for filename, catg in self.inputs:
            if catg == "BADPIX" and badpix is None:
                badpix = filename
            elif catg == "FLATFIELD" and flat is None:
                flat = filename
            elif catg == "NONLINEARITY" and nonlin is None:
                nonlin = filename

        # Build and return the file name list
        file_lists = []
        if badpix is not None and flat is not None and nonlin is not None:
            file_lists.append(
                {
                    "badpix": badpix,
                    "flat": flat,
                    "nonlin": nonlin,
                }
            )

        return file_lists

    def generate_panels(self, **kwargs):

        panels = {}
        p = Panel(x=3, y=2, height_ratios=[1, 3])

        badpix = self.hdus[0]["badpix"]
        flat = self.hdus[0]["flat"]
        nonlin = self.hdus[0]["nonlin"]
        ext = "PRIMARY"

        vspace = 0.2

        t1 = TextPlot(columns=1, v_space=vspace)
        fname = badpix.filename()
        hdr = badpix[ext].header
        col1 = (
            "INSTRUME: " + str(hdr.get("INSTRUME")),
            "EXTNAME: " + ext,
            "FILE NAME: " + os.path.basename(fname),
            "PRO.CATG: " + str(hdr.get("HIERARCH ESO PRO CATG")),
            "RAW1.NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
        )

        t1.add_data(col1)
        p.assign_plot(t1, 0, 0, xext=2)

        metadata = MatisseSetupInfo.detmon(badpix)
        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        col2 = metadata
        t2.add_data(col2)
        p.assign_plot(t2, 1, 0, xext=1)

        im1 = ImagePlot(badpix[ext].data, title="Badpix map")
        p.assign_plot(im1, 0, 1)
        im2 = ImagePlot(flat[ext].data, title="Flatfield")
        p.assign_plot(im2, 1, 1)

        det = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO DET NAME")
        fun = None
        if det == "MATISSE-LM":
            x = np.linspace(-5, 5, num=50, endpoint=True)
            a = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET1 NL A")
            b = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET1 NL B")
            c = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET1 NL C")
            d = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET1 NL D")
            fun = a * x + b * x * np.exp((x - c) * d)

        elif det == "MATISSE-N":
            x = np.linspace(0, 1024, num=50, endpoint=True)
            a = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET2 NL A")
            b = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET2 NL B")
            c = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET2 NL C")
            d = float(fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET2 NL D"))
            e = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET2 NL E")
            f = fetch_kw_or_default(nonlin[ext], "HIERARCH ESO QC DET2 NL F")
            fun = (
                a * x
                + b * (1.0 - np.exp(-c * np.power(x, d)))
                + e * (np.exp(f * x) - 1.0)
            )

        lin_plot = LinePlot()
        lin_plot.add_data((x, fun), label="Non linearity")
        p.assign_plot(lin_plot, 2, 1)

        addme = {
            "report_name": "matisse_detmon",
            "report_description": "MATISSE - detector monitoring",
            "report_tags": [],
        }

        panels[p] = addme

        return panels


rep = MatisseDetmonReport()
