import os
import numpy as np
from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase
from adari_core.plots.points import ScatterPlot, LinePlot
from adari_core.plots.combined import CombinedPlot
from adari_core.plots.panel import Panel
from adari_core.plots.images import ImagePlot
from .matisse_utils import MatisseSetupInfo, MatisseReportMixin

class MatisseKappaReport(MatisseReportMixin, AdariReportBase):
    def __init__(self):
        super().__init__("matisse_kappa")

    def parse_sof(self):

        master_kappa = None
        raw_kappa = None

        for filename, catg in self.inputs:
            if catg == "KAPPA_MATRIX" and master_kappa is None:
                master_kappa = filename

            if catg == "KAPPA_SRC" and raw_kappa is None:
                raw_kappa = filename

        # Build and return the file name list
        file_lists = []
        if master_kappa is not None and raw_kappa is not None:
            file_lists.append({"master_kappa": master_kappa, "raw_kappa": raw_kappa})

        return file_lists

    def get_data(self, data, ext):
        for i in range(len(data)):
            if str(data[i].header.get("EXTNAME")) == ext:
                out_data = data[i].data
        return out_data

    def first_panel(self):
        master_im = self.hdus[0]["master_kappa"]
        mk = master_im[1].data
        s = int(0.01 * np.shape(mk["MATRIX"][0])[0])
        ml = int(np.shape(mk["MATRIX"][0])[0] / 3)
        order = [j for j in range(len(mk["MATRIX"][0][:ml]))]
        p = Panel(3, 4, height_ratios=[1, 4, 4, 4])
        plot1 = LinePlot(legend=True, x_label="96% center", y_label="intensity")
        plot2 = LinePlot(legend=True, x_label="96% center", y_label="zoom")
        plot3 = LinePlot(legend=True, x_label="96% center", y_label="shift")
        for i in range(len(mk["MATRIX"][:, :0])):
            int_tel = mk["MATRIX"][i][:ml]
            plot1.add_data(
                d=[order[s:], int_tel[s:]],
                label="tel: " + str(i + 1),
            )

            zoom_tel = mk["MATRIX"][i][ml : 2 * ml]
            plot2.add_data(d=[order[s:], zoom_tel[s:]], label="tel: " + str(i + 1))

            shift_tel = mk["MATRIX"][i][2 * ml : 3 * ml]
            plot3.add_data(d=[order[s:], shift_tel[s:]], label="tel: " + str(i + 1))

        p.assign_plot(plot1, 0, 1, xext=3)
        p.assign_plot(plot2, 0, 2, xext=3)
        p.assign_plot(plot3, 0, 3, xext=3)

        return p

    def second_panel(self):

        p = Panel(3, 2, height_ratios=[1, 4])
        raw_kappa = self.hdus[0]["raw_kappa"]
        top = raw_kappa[0].header.get("HIERARCH ESO DET WIN MTRH2") - 10
        ext_info = "IMAGING_DETECTOR"
        ext_data = "IMAGING_DATA"

        raw_data = self.get_data(raw_kappa, ext_data)
        det_data = self.get_data(raw_kappa, ext_info)

        # Extracting the row index where REGNAME is 'INTERF_1'
        row_index = np.where(det_data["REGNAME"] == "PHOT1_1")[0]

        right_im = ImagePlot(raw_data[0][1:][row_index[0]], title="Raw KAPPA_SRC image")

        ny = np.shape(raw_data[0][1:][row_index[0]][top - 15 : top, :])[1]
        bottom_raw_im = raw_data[0][1:][row_index[0]][5:20, :]
        top_raw_im = raw_data[0][1:][row_index[0]][top - 15 : top, :]

        med_top_cut = []
        for i in range(ny):
            med = np.median(top_raw_im[:, i])
            med_top_cut.append(med)

        med_bottom_cut = []
        for i in range(ny):
            med = np.median(bottom_raw_im[:, i])
            med_bottom_cut.append(med)

        plot_left_l = LinePlot(
            legend=True, x_label="pixel (spatial)", y_label="intensity (ADU)"
        )
        plot_left_l.add_data(
            d=[range(ny), med_top_cut],
            color="blue",
            label="median of rows " + str(top - 15) + " to " + str(top),
        )
        plot_left_l.add_data(
            d=[range(ny), med_bottom_cut], color="red", label="median of rows 5 to 20"
        )

        plot_left_s = ScatterPlot(
            legend=False, markersize=2, title="Medians of raw image"
        )
        plot_left_s.add_data(
            d=[range(ny), med_top_cut],
            color="blue",
            label="median of rows " + str(top - 15) + " to " + str(top),
        )
        plot_left_s.add_data(
            d=[range(ny), med_bottom_cut], color="red", label="median of rows 5 to 20"
        )

        plot_1_combined = CombinedPlot()
        plot_1_combined.add_data(plot_left_l, z_order=0)
        plot_1_combined.add_data(plot_left_s, z_order=1)

        p.assign_plot(plot_1_combined, 0, 1, xext=2)

        p.assign_plot(right_im, 2, 1)

        return p

    def generate_panels(self, **kwargs):
        """Create both panels."""

        panels = {}
        master_kappa = self.hdus[0]["master_kappa"]
        hdr = master_kappa[0].header
        ext = "MATRIX"
        fname = master_kappa.filename()

        master_kappa_procatg = hdr.get("HIERARCH ESO PRO CATG")

        vspace = 0.2

        col1 = (
            "INSTRUME: " + str(hdr.get("INSTRUME")),
            "EXTNAME: " + ext,
            "PRO.CATG: " + str(master_kappa_procatg),
            "FILE NAME: " + os.path.basename(fname),
            "RAW1.NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
        )

        col2 = MatisseSetupInfo.kappa(master_kappa)

        t1 = TextPlot(columns=1, v_space=vspace, xext=1)
        t1.add_data(col1)

        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        t2.add_data(col2)

        p1 = self.first_panel()
        p1.assign_plot(t1, 0, 0, xext=2)
        p1.assign_plot(t2, 2, 0)

        p2 = self.second_panel()
        p2.assign_plot(t1, 0, 0, xext=2)
        p2.assign_plot(t2, 2, 0)

        addme1 = {
            "panel": "first panel",
            "ext": ext,
            "report_name": f"matisse_{master_kappa_procatg}_{str(ext)}",
            "report_description": f"MATISSE kappa report panel - " f"{ext}",
            "report_tags": [],
        }

        addme2 = {
            "panel": "second panel",
            "ext": ext,
            "report_name": f"matisse_{master_kappa_procatg}_{str(ext)}"
            + "_kappa_per_channel",
            "report_description": f"MATISSE kappa report panel - "
            f"{ext}" + "_kappa_per_channel",
            "report_tags": [],
        }

        panels[p1] = addme1

        panels[p2] = addme2

        return panels


rep = MatisseKappaReport()
