from edps import subworkflow, task, science, copy_upstream

from .matisse_datasources import *
from .matisse_task_functions import *


@subworkflow("calibrate_visibilities", "")
def calibrate_visibilities(merged_science, merged_calibrator, transfer_function):
    # This subworkflow produces calibrated visibilities for the scientific target. It uses
    # only calibrator data from which it was possible to compute the transfer function.

    match_calibrator = (match_rules()
                        .with_match_function(rules.is_assoc_calibrators, time_range=ONE_DAY, level=0))

    match_sky = (match_rules()
                 .with_match_keywords([kwd.obs_id, kwd.det_name, kwd.tpl_start], time_range=ONE_DAY, level=0))

    same_raw_file = (match_rules()
                     .with_match_keywords([kwd.mjd_obs], time_range=ONE_DAY, level=0))

    # The recipe mat_est_tf creates the transfer function only if the calibrator is listed in the input catalog, which
    # is a static calibration file.
    # The recipe mat_merge_results always creates the uncalibrated Visibility final products: pro.catg = TARGET_RAW_INT
    # and pro.catg = CALIB_RAW_INT, which also includes the transfer function if it is available.
    #
    # This filter selects only the CALIB_RAW_INT containing the transfer function and passes them to the next
    # step of the reduction chain.
    calibrators_with_transfer_function = (task("filter_calibrators")
                                          .with_function(copy_upstream)
                                          .with_main_input(transfer_function)
                                          .with_associated_input(merged_calibrator, match_rules=same_raw_file)
                                          .build())

    # - Task to generate calibrated visibilities for the target. Only calibrators with a transfer function
    #   are used in the process.
    # The workflow parameter $associate_calibrators_kwd  ("night" / "obs.container.id") specifies whether calibrators
    # from the same night or the same container.id as the target have to be used.
    calibrated_visibilities = (task("calibrate_visibilities")
                               .with_recipe("mat_cal_oifits")
                               .with_main_input(merged_science)
                               .with_associated_input(calibrators_with_transfer_function,
                                                      [reduced_calibrator_outout_class, reduced_calibrator_inin_class,
                                                       reduced_calibrator_inout_class, reduced_calibrator_outin_class],
                                                      max_ret=20, match_rules=match_calibrator)
                               .with_associated_input(raw_rmnrec, min_ret=0, max_ret=20)
                               .with_associated_input(raw_sky, min_ret=0, max_ret=20, match_rules=match_sky)
                               .with_meta_targets([science])
                               .build())

    return calibrated_visibilities

#
