from edps import qc1calib, ReportInput
from edps import task

from .matisse_calibrate_visibilities import calibrate_visibilities
from .matisse_datasources import *
from .matisse_distortion_coefficients import compute_distortion_coefficients
from .matisse_merge_results import merge_results
from .matisse_process_calibrations import process_observations
from .matisse_quality_control import quality_control
from .matisse_task_functions import *

__title__ = "MATISSE workflow"
# --- Processing tasks ----------------------------------------------------------------------------

# Task to generate Bad Pixel map, flat fielding, linearity
detector_calibration = (task('detector_calibration')
                        .with_recipe('mat_cal_det')
                        .with_report("matisse_detmon", ReportInput.RECIPE_INPUTS_OUTPUTS)
                        .with_main_input(raw_dark)
                        .with_associated_input(raw_flat, max_ret=1000)
                        .with_job_processing(setup_dark)
                        .with_meta_targets([qc1calib])
                        .build())

# Flat fielding for science.
lamp_flat = (task("lamp_flat")
             .with_recipe("mat_est_flat")
             .with_report("matisse_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
             .with_main_input(raw_flat_on)
             .with_associated_input(detector_calibration, [BADPIX, FLATFIELD, NONLINEARITY])
             .with_meta_targets([qc1calib])
             .build())

# Computation of the distortion coefficients of the photometric and interferometric channels, and it determines
# the relative shifts between each photometric channel and the interferometric one.
distortion = compute_distortion_coefficients(detector_calibration, lamp_flat)
distortion_coefficients = distortion.distortion_coefficients
calibrate_wave = distortion.calibrate_wave
distortion_coefficients_very_high_resolution = distortion.distortion_coefficients_very_high_resolution
distortion_calibrations = distortion.distortion_calibrations
distortion_calibrations_optional = distortion.distortion_calibrations_optional

# Correction for intensity ratio, the relative shift and zoom between each photometric and
# interferometric channels for the spectral dispersion.
kappa_matrix = (task('kappa_matrix')
                .with_recipe('mat_est_kappa')
                .with_dynamic_parameter("which_resolution", which_resolution)
                .with_report("matisse_kappa", ReportInput.RECIPE_INPUTS_OUTPUTS)
                .with_main_input(raw_kappa)
                .with_associated_input(detector_calibration, [BADPIX, NONLINEARITY])
                .with_associated_input(lamp_flat, [OBS_FLATFIELD], match_rules=assoc_flats_to_kappa)
                .with_alternative_associated_inputs(distortion.distortion_calibrations)
                .with_associated_input(raw_sky, min_ret=0, max_ret=100)
                .with_meta_targets([qc1calib])
                .build())

# Monitoring tasks
quality_control = quality_control(detector_calibration, lamp_flat, distortion.distortion_calibrations_optional,
                                  kappa_matrix)
# will be removed in future edps versions, when tasks will be "discovered" also inside classes
rmnrec = quality_control.rmnrec
test_calibrations = quality_control.test_calibrations
acquisition = quality_control.acquisition
cophasing = quality_control.cophasing
basic_detector_map = quality_control.basic_detector_map
extended_detector_map = quality_control.extended_detector_map
remanence_detector_map = quality_control.remanence_detector_map
# -

# --- Subworkflows to reduce monitoring (internal lamps), calibrators and science targets ---------
internal_lamp_tasks = process_observations(raw_monitoring_shutter_open, raw_monitoring_shutter_closed,
                                           detector_calibration, lamp_flat,
                                           distortion.distortion_calibrations,
                                           kappa_matrix, "monitoring")

# - will be removed in future edps versions, when tasks will be "discovered" also inside classes
calibration_monitoring = internal_lamp_tasks.calibration
extraction_monitoring = internal_lamp_tasks.extraction
correction_monitoring = internal_lamp_tasks.correction
optical_path_difference_monitoring = internal_lamp_tasks.optical_path_difference
coherent_processing_monitoring = internal_lamp_tasks.coherent_processing
incoherent_processing_monitoring = internal_lamp_tasks.incoherent_processing
# -
calibrator_tasks = process_observations(raw_calibrator_shutter_open, raw_calibrator_shutter_closed,
                                        detector_calibration, lamp_flat,
                                        distortion.distortion_calibrations, kappa_matrix, "calibrator")

# will be removed in future edps versions, when tasks will be "discovered" also inside classes
calibration_calibrator = calibrator_tasks.calibration
extraction_calibrator = calibrator_tasks.extraction
correction_calibrator = calibrator_tasks.correction
optical_path_difference_calibrator = calibrator_tasks.optical_path_difference
coherent_processing_calibrator = calibrator_tasks.coherent_processing
incoherent_processing_calibrator = calibrator_tasks.incoherent_processing

# Task to create the transfer function
transfer_function = (task("transfer_function")
                     .with_recipe("mat_est_tf")
                     .with_main_input(calibrator_tasks.coherent_processing)
                     .with_associated_input(calibrator_tasks.incoherent_processing)
                     .with_associated_input(jsdc_cat)
                     .with_meta_targets([qc1calib])
                     .build())

science_tasks = process_observations(raw_science_shutter_open, raw_science_shutter_closed, detector_calibration,
                                     lamp_flat, distortion.distortion_calibrations,
                                     kappa_matrix, "science")

# will be removed in future edps versions, when tasks will be "discovered" also inside classes
calibration_science = science_tasks.calibration
extraction_science = science_tasks.extraction
correction_science = science_tasks.correction
optical_path_difference_science = science_tasks.optical_path_difference
coherent_processing_science = science_tasks.coherent_processing
incoherent_processing_science = science_tasks.incoherent_processing

# --------------------------------------------------------------------------------------------------

# --- Subworkflows to merge the exposures of the 4 beam configurations (internal lamps, calibrator -
# --- and target)                                                                              -----
# The 4 beams configurations are OUT-OUT, IN-IN, IN-OUT, and OUT-IN, they are obtained
# by running the recipe mat_merge_results with the parameter bcdMode set to 0, 1, 2, or 3, respectively.

# Processing all internal lamp frames
merged_internal = merge_results(internal_lamp_tasks.incoherent_processing, internal_lamp_tasks.coherent_processing,
                                "internal_lamp")

# Processing all calibrators
merged_calibrator = merge_results(calibrator_tasks.incoherent_processing, calibrator_tasks.coherent_processing,
                                  "calibrator", transfer_function=transfer_function)

# Processing science targets
merged_science = merge_results(science_tasks.incoherent_processing, science_tasks.coherent_processing, "science")

# --------------------------------------------------------------------------------------------------

# --- Subworkflow to produce calibrated visibilities ----------------------------------------------------

calibrated_visibilities = calibrate_visibilities(merged_science, merged_calibrator, transfer_function)
