/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_fits.h"
#include "moo_params.h"
#include "moo_ext.h"
#include "moo_ext_single.h"
#include "moo_badpix.h"
#include "moo_fibres_table.h"
#include "moo_apply_flat.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Apply transmission to given EXT_SINGLE
  @param    ext EXT_SINGLE of frame which is divided by transmission
  @param    trans transmission vector (one value per fibre)
  @return   cpl_error_code or CPL_ERROR_NONE

 * This function multiply each fibre by her relative transmission
 *
Bad pixel map update:
  - NONE

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moo_apply_trans_single(moo_ext_single *ext,
                        cpl_vector *trans,
                        cpl_array *idxtab,
                        cpl_table *fibre_table,
                        const char *transcolname)
{
    cpl_ensure_code(ext != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trans != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();

    hdrl_image *ext_himage = moo_ext_single_get_image(ext);
    cpl_image *data = hdrl_image_get_image(ext_himage);
    cpl_image *err = hdrl_image_get_error(ext_himage);
    int ny = hdrl_image_get_size_y(ext_himage);
    int nx = hdrl_image_get_size_x(ext_himage);

    for (int j = 0; j < ny; j++) {
        double tv = cpl_vector_get(trans, j);
        int idx = cpl_array_get_cplsize(idxtab, j, NULL);
        cpl_table_set_float(fibre_table, transcolname, idx, tv);
        for (int i = 1; i <= nx; i++) {
            int rej1, rej2;
            double v = cpl_image_get(data, i, j + 1, &rej1);
            double ev = cpl_image_get(err, i, j + 1, &rej2);

            cpl_image_set(data, i, j + 1, v / tv);
            cpl_image_set(err, i, j + 1, ev / tv);
        }
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in apply_flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Divide EXT_SINGLE by the flat field EXT_SINGLE
  @param    ext EXT_SINGLE of frame which is divided
  @param    flat flat field EXT_SINGLE
  @return   cpl_error_code or CPL_ERROR_NONE

 * This function performs a simple division by a Flat field

Bad pixel map update:
   - BADPIX_CALIB_DEFECT where division failed

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moo_apply_flat_single(moo_ext_single *ext, moo_ext_single *flat)
{
    cpl_ensure_code(ext != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(flat != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();

    hdrl_image *ext_himage = moo_ext_single_get_image(ext);
    hdrl_image *flat_himage = moo_ext_single_get_image(flat);

    cpl_mask *mask1 = hdrl_image_get_mask(ext_himage);
    cpl_mask *mask2 = hdrl_image_get_mask(flat_himage);

    cpl_mask *orig = cpl_mask_duplicate(mask1);
    cpl_mask_or(orig, mask2);

    hdrl_image_div_image(ext_himage, flat_himage);

    cpl_mask_not(orig);
    cpl_mask_and(orig, mask1);

    moo_mask_to_badpix(ext->qual, orig, MOO_BADPIX_CALIB_DEFECT);

    moo_badpix_merge(ext->qual, flat->qual);

    cpl_mask_delete(orig);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in apply_flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Divide spectra by 1D flat-field
  @param    ext _EXT_ extracted to flatfield
  @param    flat _EXT_ extracted flatfield
  @param    f2f fibre-to-fibre relative transmission table
  @return   cpl_error_code or CPL_ERROR_NONE

 * This function performs a simple division by a flat field.

 _Bad pixels flags_:
  - BADPIX_CALIB_DEFECT where division failed.

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_apply_flat(moo_ext *ext, moo_ext *flat, moo_f2f *f2f)
{
    cpl_ensure_code(ext != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(flat != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_msg_info(__func__, "Apply flat-field");

    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_LOW_QE;
    //badpix_level = MOO_BADPIX_GOOD;
    const char *transcol[] = { MOO_FIBRES_TABLE_TRANS_RI,
                               MOO_FIBRES_TABLE_TRANS_YJ,
                               MOO_FIBRES_TABLE_TRANS_H };
    cpl_msg_indent_more();
    if (f2f != NULL) {
        moo_f2f_order_by_index(f2f);
    }
    cpl_table *fibres_table = moo_ext_get_fibre_table(ext);
    cpl_ensure_code(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT);
    moo_try_check(moo_fibres_table_add_f2f_cols(fibres_table), " ");

    for (int i = 1; i <= 2; i++) {
        cpl_array *idxtab = moo_fibres_table_get_spectro_index(fibres_table, i);
        for (int j = 0; j < 3; j++) {
            moo_ext_single *ext_single =
                moo_ext_load_single(ext, j, i, badpix_level);
            moo_ext_single *flat_single =
                moo_ext_load_single(flat, j, i, badpix_level);
            const char *transcolname = transcol[j];
            if (ext_single != NULL && flat_single != NULL) {
                cpl_msg_info(__func__, "Apply flat-field for extension %s",
                             moo_detector_get_extname(j, i));
                _moo_apply_flat_single(ext_single, flat_single);
            }
            if (f2f != NULL) {
                cpl_vector *trans = moo_f2f_get_trans(f2f, j, idxtab);
                if (ext_single != NULL) {
                    cpl_msg_info(__func__,
                                 "Apply F2F transmission for extension %s",
                                 moo_detector_get_extname(j, i));

                    _moo_apply_trans_single(ext_single, trans, idxtab,
                                            fibres_table, transcolname);
                }
                else {
                    int size = cpl_array_get_size(idxtab);
                    for (int k = 0; k < size; k++) {
                        double tv = cpl_vector_get(trans, k);
                        int idx = cpl_array_get_cplsize(idxtab, k, NULL);
                        cpl_table_set_float(fibres_table, transcolname, idx,
                                            tv);
                    }
                }
                cpl_vector_delete(trans);
            }
        }
        cpl_array_delete(idxtab);
    }
    cpl_msg_indent_less();

moo_try_cleanup:

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in apply_flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}
/**@}*/
