/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_fits.h"
#include "moo_params.h"
#include "moo_single.h"
#include "moo_psf_single.h"
#include "moo_badpix.h"
#include "moo_apply_p2p.h"
#include "moo_qc.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Divide DET_SINGLE by the pixel-to-pixel variation DET_SINGLE
  @param    flat DET_SINGLE of frame which is divided
  @param    p2pmap pixel to pixel variation DET_SINGLE  
  @return   cpl_error_code or CPL_ERROR_NONE
  
 * This function performs a simple division by a P2P map.
 
Bad pixel map update:
  - BADPIX_CALIB_DEFECT where division failed

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moo_apply_p2p_single(moo_single *flat, moo_single *p2pmap)
{
    cpl_ensure_code(flat != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(p2pmap != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_msg_indent_more();

    cpl_mask *mask1 = hdrl_image_get_mask(flat->image);
    cpl_mask *mask2 = hdrl_image_get_mask(p2pmap->image);
    cpl_mask *orig = cpl_mask_duplicate(mask1);

    cpl_mask_or(orig, mask2);

    hdrl_image_div_image(flat->image, p2pmap->image);

    cpl_mask_not(orig);
    cpl_mask_and(orig, mask1);

    moo_mask_to_badpix(flat->qual, orig, MOO_BADPIX_CALIB_DEFECT);
    moo_badpix_merge(flat->qual, p2pmap->qual);
    cpl_mask_delete(orig);

    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in apply_p2p");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Divide _DET_ by the pixel-to-pixel variation map
  @param    flat _DET_ which is divided
  @param    p2pmap _DET_ pixel to pixel variation map
  @return   cpl_error_code or CPL_ERROR_NONE
  
 * This function performs a simple division by a P2P map.
 
 _Bad pixels flags_:
  - BADPIX_CALIB_DEFECT where division failed.
 
 - - -  
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 * 

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_apply_p2p(moo_det *flat, moo_det *p2pmap)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    unsigned int badpix_level = MOO_BADPIX_GOOD;

    cpl_ensure_code(flat != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(p2pmap != NULL, CPL_ERROR_NULL_INPUT);

    cpl_msg_info(__func__, "Apply pixel-to-pixel variation map");

    cpl_propertylist *primary_header = moo_det_get_primary_header(flat);
    moo_qc_set_is_p2pcor(primary_header, CPL_TRUE);
    cpl_msg_indent_more();

    for (int i = 1; i <= 2; i++) {
        for (int j = 0; j < 3; j++) {
            moo_single *det_single =
                moo_det_load_single(flat, j, i, badpix_level);

            if (det_single != NULL) {
                moo_single *p2p_single =
                    moo_det_load_single(p2pmap, j, i, badpix_level);

                if (p2p_single != NULL) {
                    cpl_msg_info(__func__, "Apply p2p for extension %s",
                                 moo_detector_get_extname(j, i));
                    _moo_apply_p2p_single(det_single, p2p_single);
                }
            }
        }
    }
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in apply_p2p");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}
/**@}*/
