/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <ctype.h>
#include <string.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_atm.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_atm ATM format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_atm
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_atm.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_atm
  @return   1 newly allocated moo_atm or NULL in case of an error

  The returned object must be deallocated using moo_atm_delete().

 */
/*----------------------------------------------------------------------------*/
moo_atm* moo_atm_new(void)
{
    moo_atm* res = cpl_calloc(1,sizeof(moo_atm));
    return res;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load a ATM frame and create a moo_atm
  @return   1 newly allocated moo_flx or NULL in case of an error

  The returned object must be deallocated using moo_flx_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_atm* moo_atm_load(const cpl_frame* frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char* filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT,
               NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_atm* res = moo_atm_new();
    res->filename = cpl_strdup(filename);
    res->primary_header = cpl_propertylist_load( filename,0);

    cpl_size qnum = cpl_fits_find_extension(filename,
                MOO_ATM_EXTNAME);
    if ( qnum > 0){
        res->table = cpl_table_load(res->filename,qnum,0);
    }

    if ( !cpl_errorstate_is_equal(prev_state)){
        cpl_msg_debug(__func__,"Error in loading atm : %d",cpl_error_get_code());
        moo_atm_delete(res);
        res = NULL;
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Get the extinction table with additonal column FLUX
  @param self the ATM object
  @param airm the AIRMASS value
  @return the extinction table or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_table* moo_atm_convert_mag_to_flux(moo_atm* self,double airm)
{
    cpl_ensure(self!=NULL,CPL_ERROR_NULL_INPUT,NULL);

    cpl_table* res = cpl_table_duplicate(self->table);
    cpl_table_new_column(res, MOO_ATM_FLUX, CPL_TYPE_DOUBLE);
    int nrow = cpl_table_get_nrow(res);
    const double* extinction = cpl_table_get_data_double_const(
        res,MOO_ATM_EXTINCTION);
  
    for(int i=0; i<nrow; i++){
        double e = extinction[i];
        double flux = pow(10,-airm*e*0.4);
        cpl_table_set_double(res,MOO_ATM_FLUX,i,flux);
    }

    #if MOO_DEBUG_ATM
    cpl_table_save(res,NULL,NULL,"extinction.fits",CPL_IO_CREATE);
    #endif
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_atm
  @param    self    moo_atm to delete
  @return   void

  If the moo_atm @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void moo_atm_delete(moo_atm* self)
{
    if ( self!= NULL){
        if ( self->filename != NULL){
            cpl_free(self->filename);
        }
        if ( self->primary_header!= NULL){
            cpl_propertylist_delete(self->primary_header);
        }
        if ( self->table != NULL){
            cpl_table_delete(self->table);
        }
        cpl_free(self);
    }
}

/**@}*/
