/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_badpix.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_badpix  Bad pixels map functions
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_badpix
  @brief    Apply the badpix map on the given mask
  @param    badpix Bad pixel map
  @param    mask  the bad pixel mask
  @param    level the level of good pixel in badpix
  @return   CPL_ERROR_NONE or the relevant cpl_error_code on error

  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
  - CPL_ERROR_ILLEGAL_INPUT the type of BADPIX image in not CPL_TYPE_INT
  - CPL_ERROR_ILLEGAL_INPUT if image and mask have not same size
 *   
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_badpix_to_mask(cpl_image *badpix, cpl_mask *mask, unsigned int level)
{
    cpl_ensure_code(badpix != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mask != NULL, CPL_ERROR_NULL_INPUT);

    cpl_type type = cpl_image_get_type(badpix);
    cpl_ensure_code(type == CPL_TYPE_INT, CPL_ERROR_ILLEGAL_INPUT);

    int nx = cpl_image_get_size_x(badpix);
    int ny = cpl_image_get_size_y(badpix);
    int mask_nx = cpl_mask_get_size_x(mask);
    int mask_ny = cpl_mask_get_size_y(mask);
    cpl_ensure_code(nx == mask_nx, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ny == mask_ny, CPL_ERROR_ILLEGAL_INPUT);

    int *data = cpl_image_get_data_int(badpix);
    cpl_binary *mdata = cpl_mask_get_data(mask);

    int i;
    for (i = 0; i < nx * ny; i++) {
        if ((data[i] & level) > 0) {
            mdata[i] = 1;
        }
        else {
            mdata[i] = 0;
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_badpix
  @brief    Add the mask of the badpix level to the badpix map
  @param    badpix Bad pixel map
  @param    mask  the bad pixel mask
  @param    level the level of badpixel in mask
  @return   cpl error code

  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
  - CPL_ERROR_ILLEGAL_INPUT the type of BADPIX image in not CPL_TYPE_INT
  - CPL_ERROR_ILLEGAL_INPUT if image and mask have not same size
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_mask_to_badpix(cpl_image *badpix, cpl_mask *mask, unsigned int level)
{
    cpl_ensure_code(badpix != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mask != NULL, CPL_ERROR_NULL_INPUT);

    cpl_type type = cpl_image_get_type(badpix);
    cpl_ensure_code(type == CPL_TYPE_INT, CPL_ERROR_ILLEGAL_INPUT);

    int nx = cpl_image_get_size_x(badpix);
    int ny = cpl_image_get_size_y(badpix);
    int mask_nx = cpl_mask_get_size_x(mask);
    int mask_ny = cpl_mask_get_size_y(mask);
    cpl_ensure_code(nx == mask_nx, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ny == mask_ny, CPL_ERROR_ILLEGAL_INPUT);

    int *data = cpl_image_get_data_int(badpix);
    cpl_binary *mdata = cpl_mask_get_data(mask);
    int i;

    for (i = 0; i < nx * ny; i++) {
        if (mdata[i] > 0) {
            data[i] |= level;
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_badpix
  @brief    Merge to bad pixel map
  @param    badpix1 Bad pixel map to update
  @param    badpix2 Bad pixel map to merge
  @return   cpl error code

   - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_badpix_merge(cpl_image *badpix1, cpl_image *badpix2)
{
    cpl_ensure_code(badpix1 != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(badpix2 != NULL, CPL_ERROR_NULL_INPUT);

    cpl_type type1 = cpl_image_get_type(badpix1);
    cpl_ensure_code(type1 == CPL_TYPE_INT, CPL_ERROR_ILLEGAL_INPUT);
    cpl_type type2 = cpl_image_get_type(badpix2);
    cpl_ensure_code(type2 == CPL_TYPE_INT, CPL_ERROR_ILLEGAL_INPUT);

    int nx = cpl_image_get_size_x(badpix1);
    int ny = cpl_image_get_size_y(badpix1);
    int nx2 = cpl_image_get_size_x(badpix2);
    int ny2 = cpl_image_get_size_y(badpix2);

    cpl_ensure_code(nx == nx2, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ny == ny2, CPL_ERROR_ILLEGAL_INPUT);
    int *data1 = cpl_image_get_data_int(badpix1);
    int *data2 = cpl_image_get_data_int(badpix2);
    int i;

    for (i = 0; i < nx * ny; i++) {
        data1[i] |= data2[i];
    }
    return CPL_ERROR_NONE;
}
/**@}*/
