/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_bpm.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_bpm BPM format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_bpm
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_bpm.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_bpm
  @return   1 newly allocated moo_bpm or NULL in case of an error

  The returned object must be deallocated using moo_bpm_delete().

 */
/*----------------------------------------------------------------------------*/
moo_bpm *
moo_bpm_new(void)
{
    moo_bpm *res = cpl_calloc(1, sizeof(moo_bpm));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    set bpm data for relevant extension
  @param self
  @param type the detector type
  @param ntas number of TAS
  @param data the bpm data
  @param header the bpm header
  @return   error code

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_bpm_set_data(moo_bpm *self,
                 moo_detector_type type,
                 int ntas,
                 cpl_image *data,
                 cpl_propertylist *header)
{
    cpl_ensure_code(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);

    self->data[(ntas - 1) * 3 + type] = data;
    self->data_header[(ntas - 1) * 3 + type] = header;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Add CPL_IMAGE extension to BPM filename and update moo_bpm structure
  @param self moo_bpm structure to update
  @param img CPL_IMAGE extension to store
  @param type type of detector
  @param ntas number of spectrographs

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_bpm_add_data(moo_bpm *self,
                 cpl_image *img,
                 moo_detector_type type,
                 int ntas)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    cpl_propertylist *header = cpl_propertylist_new();
    char *extname = NULL;

    extname = cpl_sprintf("QUAL_%s", moo_detector_get_extname(type, ntas));
    cpl_propertylist_append_string(header, MOO_PFITS_EXTNAME, extname);
    cpl_free(extname);

    if (img != NULL) {
        cpl_image_save(img, self->filename, CPL_TYPE_INT, header,
                       CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(header, self->filename, CPL_IO_EXTEND);
    }
    moo_bpm_set_data(self, type, ntas, img, header);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_bpm", "Error for adding data to %s %s (%s)",
                      self->filename, extname,
                      cpl_error_get_message_default(cpl_error_get_code()));
        status = cpl_error_get_code();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
@brief Merge to moo_bpm structure in the first
@param a moo_bpm structure which contains the result
@param b moo_bpm structure to merge

Possible _cpl_error_code_ set in this function:
- CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_bpm_merge(moo_bpm *a, moo_bpm *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    for (int i = 0; i < 6; i++) {
        cpl_image *im1 = a->data[i];
        cpl_image *im2 = b->data[i];
        if (im1 != NULL && im2 != NULL) {
            status = cpl_image_or(im1, NULL, im2);
        }
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_bpm to a FITS file
  @param    self moo_bpm to write to disk or NULL
  @param    filename Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_bpm to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_bpm_save(moo_bpm *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        for (int i = 0; i < 2; i++) {
            for (int j = 0; j < 3; j++) {
                const char *extname = moo_detector_get_extname(j, i + 1);
                moo_fits_write_extension_image(self->data[i * 3 + j], filename,
                                               "QUAL", extname, CPL_TYPE_INT,
                                               self->data_header[i * 3 + j]);
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_bpm
  @param    self    moo_bpm to delete
  @return   void

  If the moo_map @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_bpm_delete(moo_bpm *self)
{
    if (self != NULL) {
        int i;
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 6; i++) {
            if (self->data[i] != NULL) {
                cpl_image_delete(self->data[i]);
            }
            if (self->data_header[i] != NULL) {
                cpl_propertylist_delete(self->data_header[i]);
            }
        }
        cpl_free(self->filename);
        cpl_free(self);
    }
}
/**@}*/
