/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_coadd.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
typedef struct
{
    const char *name;
    int *indexes;
} _moo_target;


static moo_target_table *
_moo_coadd_target_table(moo_scilist *scilist)
{
    moo_target_table *res = NULL;

    cpl_ensure(scilist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int size = moo_scilist_get_size(scilist);
    cpl_ensure(size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_errorstate prestate = cpl_errorstate_get();

    moo_sci *first = moo_scilist_get(scilist, 0);

    res = moo_target_table_duplicate(moo_sci_get_target_table(first));

    for (int i = 1; i < size; i++) {
        moo_sci *sci = moo_scilist_get(scilist, i);
        moo_target_table *ttable = moo_sci_get_target_table(sci);
        moo_try_check(moo_target_table_merge(res, ttable), " ");
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_target_table_delete(res);
        res = NULL;
    }
    return res;
}


static cpl_error_code
_moo_create_pheader(cpl_propertylist *header, moo_scilist *scilist)
{
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(scilist != NULL, CPL_ERROR_NULL_INPUT);

    int size = (int)moo_scilist_get_size(scilist);
    double exptime_sum = 0.0;

    for (int i = 0; i < size; i++) {
        moo_sci *sci = moo_scilist_get(scilist, i);

        double exptime = moo_pfits_get_exptime(sci->primary_header);
        exptime_sum += exptime;
    }

    double pexptime = exptime_sum;

    cpl_propertylist_append_double(header, MOO_PFITS_EXPTIME, pexptime);
    cpl_propertylist_set_comment(header, MOO_PFITS_EXPTIME,
                                 MOO_PFITS_EXPTIME_C);
    moo_sci *sci = moo_scilist_get(scilist, 0);
    cpl_propertylist *sci_primary_header = sci->primary_header;
    cpl_propertylist_copy_property_regexp(header, sci_primary_header,
                                          "ESO QC IS*", 0);
    cpl_propertylist_copy_property(header, sci_primary_header,
                                   MOO_PFITS_FLUXCAL);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Co-add reduced science frames in SCI format
  @param    scilist List of SCI files
  @param    params coadd params
  @param    filename name of result file
  @return   the _SCI_ result

 * _Flags considered as bad : BADPIX_COSMETIC | BADPIX_HOT
 *
 * _Bad pixels flags_:
  - BADPIX_COSMIC_UNREMOVED is set in SIGCLIP mode

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the SCI list have a size
 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_coadd(moo_scilist *scilist, moo_coadd_params *params, const char *filename)
{
    moo_sci *sci = NULL;
    int ntarget = 0;
    _moo_target *targets = NULL;

    cpl_ensure(scilist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;

    int size = (int)moo_scilist_get_size(scilist);
    cpl_ensure(size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_msg_info(__func__, "Coadding %d frames with method %s", size,
                 params->method);

    if (size == 1) {
        cpl_frame *frame = cpl_frame_new();
        cpl_frame_set_filename(frame, filename);

        moo_sci *scif = moo_scilist_get(scilist, 0);
        moo_sci_load(scif, badpix_level);
        moo_try_check(moo_target_table_remove_coadd_cols(scif->target_table),
                      " ");
        moo_sci_save(scif, filename);
        sci = moo_sci_create(frame);

        moo_sci_load(sci, badpix_level);

        cpl_frame_delete(frame);
    }
    else {
        const char **targnames = NULL;
        moo_try_check(moo_fits_create(filename), " ");
        moo_try_check(sci = moo_sci_new(), " ");
        sci->filename = filename;
        sci->primary_header = cpl_propertylist_new();
        moo_try_check(_moo_create_pheader(sci->primary_header, scilist), " ");

        moo_target_table *sci_target_table = NULL;
        moo_try_check(sci_target_table = _moo_coadd_target_table(scilist), " ");
        sci->target_table = sci_target_table;

        moo_try_check(moo_target_table_remove_coadd_cols(sci_target_table),
                      " ");
        ntarget = cpl_table_get_nrow(sci_target_table->table);

        moo_try_check(targnames = cpl_table_get_data_string_const(
                          sci_target_table->table, MOO_TARGET_TABLE_TARGNAME),
                      " ");

        targets = cpl_calloc(ntarget, sizeof(_moo_target));
        int ftarget = 0;
        for (int i = 0; i < ntarget; i++) {
            const char *targname = targnames[i];

            targets[i].indexes = cpl_calloc(size, sizeof(int));
            int nb = 0;
            double exptimeri = 0;
            double exptimeyj = 0;
            double exptimeh = 0;

            for (int j = 0; j < size; j++) {
                moo_sci *scif = moo_scilist_get(scilist, j);
                moo_target_table *ttable = moo_sci_get_target_table(scif);
                int indextarg = moo_target_table_find_target(ttable, targname);
                int tindex = moo_target_table_find_index(ttable, targname);
                targets[i].name = targname;
                targets[i].indexes[j] = indextarg;
                if (tindex >= 0) {
                    nb++;
                    ftarget = j;
                    exptimeri +=
                        cpl_table_get_double(ttable->table,
                                             MOO_TARGET_TABLE_EXPTIMERI, tindex,
                                             NULL);
                    exptimeyj +=
                        cpl_table_get_double(ttable->table,
                                             MOO_TARGET_TABLE_EXPTIMEYJ, tindex,
                                             NULL);
                    exptimeh += cpl_table_get_double(ttable->table,
                                                     MOO_TARGET_TABLE_EXPTIMEH,
                                                     tindex, NULL);
                }
            }
            moo_try_assure(nb > 0, CPL_ERROR_UNSPECIFIED, "Target %s not found",
                           targname);

            cpl_table_set_double(sci_target_table->table,
                                 MOO_TARGET_TABLE_EXPTIMERI, i, exptimeri);
            cpl_table_set_double(sci_target_table->table,
                                 MOO_TARGET_TABLE_EXPTIMEYJ, i, exptimeyj);
            cpl_table_set_double(sci_target_table->table,
                                 MOO_TARGET_TABLE_EXPTIMEH, i, exptimeh);
        }

        cpl_msg_indent_more();
        moo_sci *first_sci = moo_scilist_get(scilist, ftarget);

        for (int i = 0; i < 3; i++) {
            moo_sci_single *sci_single = NULL;
            moo_sci_single *first_sci_single = moo_sci_get_single(first_sci, i);

            if (first_sci_single != NULL) {
                sci_single = moo_sci_single_new(i);
                cpl_propertylist *first_header =
                    moo_sci_single_get_header(first_sci_single);

                moo_scilist_load_single(scilist, i, badpix_level);
                hdrl_image *fimage = moo_sci_single_get_image(first_sci_single);

                int nx = hdrl_image_get_size_x(fimage);

                sci_single->image = hdrl_image_new(nx, ntarget);
                cpl_propertylist_append(sci_single->header, first_header);
                sci_single->sky = cpl_image_new(nx, ntarget, CPL_TYPE_DOUBLE);
                sci_single->qual = cpl_image_new(nx, ntarget, CPL_TYPE_INT);

                for (int j = 0; j < ntarget; j++) {
                    _moo_target target = targets[j];
                    hdrl_imagelist *list = hdrl_imagelist_new();
                    cpl_imagelist *qlist = cpl_imagelist_new();
                    cpl_imagelist *slist = cpl_imagelist_new();

                    for (int k = 0; k < size; k++) {
                        int idx = target.indexes[k];
                        moo_sci_single *current_single =
                            moo_scilist_get_single(scilist, k, i);
                        if (idx > 0) {
                            hdrl_image *current =
                                moo_sci_single_get_image(current_single);
                            cpl_image *qcurrent =
                                moo_sci_single_get_qual(current_single);
                            cpl_image *scurrent =
                                moo_sci_single_get_sky(current_single);

                            int cnx = hdrl_image_get_size_x(current);
                            hdrl_image *sub =
                                hdrl_image_extract(current, 1, idx, cnx, idx);
                            cpl_image *qsub =
                                cpl_image_extract(qcurrent, 1, idx, cnx, idx);
                            cpl_image *ssub =
                                cpl_image_extract(scurrent, 1, idx, cnx, idx);

                            int pos = hdrl_imagelist_get_size(list);

                            hdrl_imagelist_set(list, sub, pos);
                            cpl_imagelist_set(qlist, qsub, pos);
                            cpl_imagelist_set(slist, ssub, pos);
                        }
                    }
                    hdrl_image *out = NULL;
                    cpl_image *contrib = NULL;
                    cpl_image *skyout = NULL;
                    cpl_image *qout = NULL;


                    if (strcmp(params->method, MOO_COADD_METHOD_MEDIAN) == 0) {
                        hdrl_imagelist_collapse_median(list, &out, &contrib);
                        skyout = cpl_imagelist_collapse_median_create(slist);
                        qout = moo_imagelist_collapse_bitwiseor(qlist, list);
                    }
                    else if (strcmp(params->method, MOO_COADD_METHOD_MEAN) ==
                             0) {
                        hdrl_imagelist_collapse_mean(list, &out, &contrib);
                        skyout = cpl_imagelist_collapse_create(slist);
                        qout = moo_imagelist_collapse_bitwiseor(qlist, list);
                    }
                    else {
                        hdrl_imagelist_collapse_sigclip(list,
                                                        params->clip_kappa,
                                                        params->clip_kappa,
                                                        params->clip_niter,
                                                        &out, &contrib, NULL,
                                                        NULL);
                        skyout = cpl_imagelist_collapse_sigclip_create(
                            slist, params->clip_kappa, params->clip_kappa, 0.7,
                            CPL_COLLAPSE_MEAN, NULL);

                        qout = moo_imagelist_collapse_bitwiseor(qlist, list);
                    }

                    hdrl_image_copy(sci_single->image, out, 1, j + 1);
                    cpl_image_copy(sci_single->sky, skyout, 1, j + 1);
                    cpl_image_copy(sci_single->qual, qout, 1, j + 1);
                    hdrl_image_delete(out);
                    cpl_image_delete(qout);
                    cpl_image_delete(skyout);
                    cpl_image_delete(contrib);

                    hdrl_imagelist_delete(list);
                    cpl_imagelist_delete(qlist);
                    cpl_imagelist_delete(slist);
                }
            }
            moo_sci_add_single(sci, i, sci_single);
        }

        moo_sci_add_target_table(sci, sci_target_table);
        cpl_msg_indent_less();
    }

    moo_pfits_update_ncoadd(sci->primary_header, size);

moo_try_cleanup:
    if (targets != NULL) {
        for (int i = 0; i < ntarget; i++) {
            cpl_free(targets[i].indexes);
        }
        cpl_free(targets);
    }
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in coadding frames");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        moo_sci_delete(sci);
        sci = NULL;
    }
    return sci;
}
/**@}*/
