/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_dfs.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_badpix.h"
#include "moo_combine_sky.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute qual from input QUAL list of frames
  @param    list QUAL list
  @return   error code

 The function checks taht all images are equals
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if list size is lower or equal to zero
 */
/*----------------------------------------------------------------------------*/
static cpl_image *
_moo_qual_fromlist(cpl_imagelist *list)
{
    cpl_ensure(list, CPL_ERROR_NULL_INPUT, NULL);

    int size = cpl_imagelist_get_size(list);
    cpl_ensure(size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    const cpl_image *qimg = cpl_imagelist_get_const(list, 0);

    if (size > 1) {
        int k;

        int nx = cpl_image_get_size_x(qimg);
        int ny = cpl_image_get_size_y(qimg);
        const int *qdata = cpl_image_get_data_int_const(qimg);

        for (k = 1; k < size; k++) {
            cpl_image *qimg2 = cpl_imagelist_get(list, k);
            int *qdata2 = cpl_image_get_data_int(qimg2);
            for (int i = 0; i < nx * ny; i++) {
                if (qdata[i] != qdata2[i]) {
                    cpl_msg_error(__func__,
                                  "QUAL of first frame and %d are not equal",
                                  k);
                }
            }
        }
    }
    return cpl_image_duplicate(qimg);
}

static moo_rbn_single *
_moo_collapse_median(hdrl_imagelist *list,
                     cpl_imagelist *quallist,
                     moo_detector_type type)
{
    moo_rbn_single *single = NULL;
    int j;
    cpl_ensure(list != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int size = hdrl_imagelist_get_size(list);

    if (size > 0) {
        hdrl_image *res = NULL;
        cpl_image *contrib = NULL;
        hdrl_imagelist_collapse_median(list, &res, &contrib);

        for (j = 0; j < size; j++) {
            hdrl_imagelist_unset(list, 0);
        }

        cpl_image *qual = _moo_qual_fromlist(quallist);

        single = moo_rbn_single_new(type);
        single->qual = qual;
        single->image = res;

        for (j = 0; j < size; j++) {
            cpl_imagelist_unset(quallist, 0);
        }

        cpl_image_delete(contrib);
    }

    return single;
}

static cpl_error_code
_moo_copy_header_kw(cpl_propertylist *res_header,
                    const cpl_propertylist *orig_header)
{
    cpl_ensure_code(res_header != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(orig_header != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist_copy_property_regexp(res_header, orig_header, "ESO DET *",
                                          0);
    cpl_propertylist_copy_property_regexp(res_header, orig_header, "HDU*", 0);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_BUNIT);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_ERRDATA);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_QUALDATA);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CRVAL2);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CD1_2);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CUNIT1);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CD2_1);
    cpl_propertylist_copy_property(res_header, orig_header, MOO_PFITS_CD2_2);

    return cpl_error_get_code();
}

static cpl_error_code
_moo_combine_sky_single(moo_rbnlist *rbnlist,
                        moo_rbn *orig_rbn,
                        moo_detector_type type,
                        unsigned int badpix_level,
                        moo_rbn *res)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_rbn_single *single = NULL;
    moo_rbn_single *orig_single = NULL;

    moo_rbnlist_load_single(rbnlist, type, badpix_level);
    hdrl_imagelist *list = moo_rbnlist_get_image(rbnlist, type);
    cpl_imagelist *qlist = moo_rbnlist_get_single_qual(rbnlist, type);

    orig_single = moo_rbn_get_single(orig_rbn, type);
    single = _moo_collapse_median(list, qlist, type);

    if (single != NULL) {
        cpl_propertylist *header = NULL;
        moo_try_check(header = moo_rbn_single_get_header(orig_single), " ");
        moo_try_check(_moo_copy_header_kw(single->header, header), " ");
    }
moo_try_cleanup:
    moo_rbn_set_single(res, single);
    cpl_imagelist_unwrap(qlist);
    hdrl_imagelist_unwrap(list);
    moo_rbnlist_free_single(rbnlist, type);
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Combine RBN sky frames
  @param    rbnlist list of _RBN_ frames
  @return   the _RBN_ result

 - - -
 * _Flags considered as bad : BADPIX_COSMETIC
 *
 * _Bad pixels flags_:
  - BADPIX_COSMIC_UNREMOVED is set in SIGCLIP mode

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if _RBN_ list size is lower or equal to zero
 */
/*----------------------------------------------------------------------------*/
moo_rbn *
moo_combine_sky(moo_rbnlist *rbnlist)
{
    moo_rbn *res = NULL;
    cpl_ensure(rbnlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rbnlist->size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_msg_info(__func__, "Combine sky of %d frames", (int)rbnlist->size);

    int badpix_level = MOO_BADPIX_GOOD;

    moo_rbn *orig_rbn = moo_rbnlist_get(rbnlist, 0);
    res = moo_rbn_new();
    res->primary_header = cpl_propertylist_duplicate(orig_rbn->primary_header);
    moo_qc_set_ncom(res->primary_header, rbnlist->size);
    cpl_errorstate prev_state = cpl_errorstate_get();
    res->fibre_table = cpl_table_duplicate(moo_rbn_get_fibre_table(orig_rbn));

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_info(__func__, "DET input files do not have a fibre table.");
        cpl_errorstate_set(prev_state);
    }

    for (int i = 0; i < 3; i++) {
        moo_try_check(_moo_combine_sky_single(rbnlist, orig_rbn, i,
                                              badpix_level, res),
                      " ");
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in remove CRH");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        moo_rbn_delete(res);
        res = NULL;
    }
    return res;
}
/**@}*/
