/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_params.h"
#include "moo_badpix.h"
#include "moo_ext.h"
#include "moo_ext_single.h"
#include "moo_compute_fibtrans.h"
#include "moo_utils.h"
#include "moo_fibres_table.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static double
_moo_compute_fibtrans_fibre(hdrl_image *himage, int ifib, cpl_image *qual)
{
    double integrated_flux = 0.0;
    double thresh = MOO_COMPUTE_FIBTRANS_NORMALISE_THRESH;
    int nx = hdrl_image_get_size_x(himage);
    int nbgood = 0;
    cpl_image *idata = hdrl_image_get_image(himage);
    cpl_image *ierrs = hdrl_image_get_error(himage);
    int nbrej = 0;
    for (int i = 1; i <= nx; i++) {
        int isrejected = cpl_image_is_rejected(idata, i, ifib);
        if (isrejected) {
            nbrej++;
        }
    }
    nbgood = nx - nbrej;
    if (nbgood > 0) {
        //double mean_flux = cpl_image_get_mean_window(idata, 1, ifib, nx, ifib);
        double median_flux =
            cpl_image_get_median_window(idata, 1, ifib, nx, ifib);
        integrated_flux = median_flux;

        for (int i = 1; i <= nx; i++) {
            int rej;
            double flux = cpl_image_get(idata, i, ifib, &rej);
            double tflux = flux / integrated_flux;
            double err = cpl_image_get(ierrs, i, ifib, &rej);

            cpl_image_set(idata, i, ifib, tflux);
            cpl_image_set(ierrs, i, ifib, err / integrated_flux);
            if (tflux < thresh) {
                int q = cpl_image_get(qual, i, ifib, &rej);
                q |= MOO_BADPIX_LOW_QE;
                cpl_image_set(qual, i, ifib, q);
            }
        }
    }
    return integrated_flux;
}

static cpl_vector *
_moo_compute_fibtrans_single(moo_ext_single *ext,
                             const int *health,
                             const int *index,
                             int fibref)
{
    cpl_vector *res = NULL;
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(health != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(index != NULL, CPL_ERROR_NULL_INPUT, NULL);

    hdrl_image *himage = moo_ext_single_get_image(ext);
    cpl_image *qual = moo_ext_single_get_qual(ext);

    if (himage != NULL) {
        int nb_fibres = hdrl_image_get_size_y(himage);
        res = cpl_vector_new(nb_fibres);
/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions where
 * these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_fibres, health, himage, res, qual)
#else
#pragma omp parallel default(none) shared(nb_fibres, health, himage, res, qual)
#endif
        {
#pragma omp for
#endif
            for (int i = 1; i <= nb_fibres; i++) {
                int h = health[i - 1];
                double iflux = 0.0;
                if (h == 1) {
                    iflux = _moo_compute_fibtrans_fibre(himage, i, qual);
                }
                cpl_vector_set(res, i - 1, iflux);
            }
#ifdef _OPENMP
        }
#endif
        double ref_flux = NAN;

        for (int i = 0; i < nb_fibres; i++) {
            int ref_idx = index[i];
            if (ref_idx == fibref) {
                ref_flux = cpl_vector_get(res, i);
                break;
            }
        }
        if (!isnan(ref_flux) && ref_flux > 0) {
            cpl_msg_info("compute_fibtrans", "Use reference flux %f", ref_flux);
            cpl_vector_divide_scalar(res, ref_flux);
        }
        else {
            cpl_error_set_message("compute_fibtrans", CPL_ERROR_ILLEGAL_INPUT,
                                  "Reference fibre %d not found", fibref);
            cpl_vector_delete(res);
            res = NULL;
        }
    }
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in extraction");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Computes fibre-to-fibre relative transmission table
  @param    ext _EXT_ spectra from which computes transmission
  @param    params fibre transmission parameters
  @param    filename name of the result F2F
  @return   the result F2F

 * The function expects as input flat-field data obtained with a uniformly
 * illuminated field of view. Then it decouples the relative intensity and
 * the shape of the extracted flat-field spectra into a fibre-to-fibre relative
 * transmission table and a normalised extracted Flat-field respectively.
 *
 * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_f2f *
moo_compute_fibtrans(moo_ext *ext,
                     moo_compute_fibtrans_params *params,
                     const char *filename)
{
    moo_f2f *res = NULL;

    cpl_ensure(ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_msg_info(__func__, "Compute fibre transmission");

    cpl_table *fibres_table = moo_ext_get_fibre_table(ext);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int nrows = cpl_table_get_nrow(fibres_table);

    unsigned int badpix_level = MOO_BADPIX_CALIB_DEFECT | MOO_BADPIX_HOT |
                                MOO_BADPIX_COSMETIC |
                                MOO_BADPIX_OUTSIDE_DATA_RANGE;

    res = moo_f2f_create(nrows, fibres_table);
    cpl_table_save(res->table, NULL, NULL, filename, CPL_IO_CREATE);

    cpl_msg_indent_more();

    for (int i = 1; i <= 2; i++) {
        int fibref = params->fibref[i - 1];
        cpl_table_unselect_all(fibres_table);
        int fibname = i;

        cpl_table_or_selected_int(fibres_table, MOO_FIBRES_TABLE_SPECTRO,
                                  CPL_EQUAL_TO, fibname);
        cpl_table *selected = cpl_table_extract_selected(fibres_table);
        const int *health =
            cpl_table_get_data_int_const(selected, MOO_FIBRES_TABLE_HEALTH);
        const int *index =
            cpl_table_get_data_int_const(selected, MOO_FIBRES_TABLE_INDEX);
        cpl_array *selected_idx = cpl_table_where_selected(fibres_table);

        if (health != NULL) {
            for (int j = 0; j < 3; j++) {
                moo_ext_single *ext_single =
                    moo_ext_load_single(ext, j, i, badpix_level);
                if (ext_single != NULL) {
                    cpl_msg_info(__func__,
                                 "Computing transmission for extension %s "
                                 "using fibref %d",
                                 moo_detector_get_extname(j, i), fibref);
                    cpl_vector *fluxes =
                        _moo_compute_fibtrans_single(ext_single, health, index,
                                                     fibref);
                    if (fluxes != NULL) {
                        moo_f2f_set_trans(res, j, selected_idx, fluxes);
                        cpl_vector_delete(fluxes);
                    }
                }
            }
        }
        cpl_array_delete(selected_idx);
        cpl_table_delete(selected);
    }

    cpl_msg_indent_less();

    cpl_table_unselect_all(fibres_table);
    cpl_table_or_selected_int(fibres_table, MOO_FIBRES_TABLE_HEALTH,
                              CPL_EQUAL_TO, 0);
    cpl_array *bad = cpl_table_where_selected(fibres_table);

    moo_f2f_compute_qc(res, params->fibref, bad);

    cpl_array_delete(bad);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in compute fibtrans");
        moo_f2f_delete(res);
        res = NULL;
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}
/**@}*/
