/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_fits.h"
#include "moo_params.h"
#include "moo_single.h"
#include "moo_psf_single.h"
#include "moo_badpix.h"
#include "moo_compute_p2p.h"
#include "moo_qc.h"
#include "moo_fibres_table.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static cpl_error_code
_moo_compute_p2p_qc(moo_single *single, int level)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);
    hdrl_image *himage = moo_single_get_image(single);

    cpl_mask *mask = hdrl_image_get_mask(single->image);
    moo_badpix_to_mask(single->qual, mask, level);

    cpl_image *data = hdrl_image_get_image(himage);
    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);

    double min = cpl_image_get_min(data);
    double max = cpl_image_get_max(data);
    double rms = cpl_image_get_stdev(data);
    double mad;
    double median = cpl_image_get_mad(data, &mad);
    double mean = cpl_image_get_mean(data);

    cpl_propertylist *header = moo_single_get_header(single);
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(moo_qc_set_p2p_min(header, min), status);
    moo_ensure_status(moo_qc_set_p2p_max(header, max), status);
    moo_ensure_status(moo_qc_set_p2p_avg(header, mean), status);
    moo_ensure_status(moo_qc_set_p2p_med(header, median), status);
    moo_ensure_status(moo_qc_set_p2p_mad(header, mad), status);
    moo_ensure_status(moo_qc_set_p2p_rms(header, rms), status);
moo_try_cleanup:
    return status;
}

static void
_moo_reproject_model_fibre(cpl_image *image,
                           int numfib,
                           cpl_image *centroids,
                           cpl_image *model,
                           double crpix2,
                           double cd2_2,
                           cpl_image *result,
                           cpl_image *contrib)
{
    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_y(model);

    double *data = cpl_image_get_data(result);
    int *contrib_data = cpl_image_get_data(contrib);

    for (int x = 1; x <= nx; x++) {
        int rej;
        double centroid = cpl_image_get(centroids, x, numfib, &rej);
        if (!isnan(centroid)) {
            double y_lo = centroid + (1 - crpix2) * cd2_2;
            double y_up = centroid + (ny - crpix2) * cd2_2;
            int y_start = (int)ceil(y_lo);
            int y_stop = (int)floor(y_up);
            int model_idx = 1;
            for (int y = y_start; y <= y_stop; y++) {
                double y_model = centroid + (model_idx - crpix2) * cd2_2;
                while (y_model < y) {
                    model_idx++;
                    y_model = centroid + (model_idx - crpix2) * cd2_2;
                }
                double y_model1 = centroid + (model_idx - 1 - crpix2) * cd2_2;
                ;
                double flux_model1 =
                    cpl_image_get(model, x, model_idx - 1, &rej);
                double y_model2 = y_model;
                double flux_model2 = cpl_image_get(model, x, model_idx, &rej);
                double y_flux = flux_model1 + (flux_model2 - flux_model1) /
                                                  (y_model2 - y_model1) *
                                                  (y - y_model1);
                data[(y - 1) * nx + x - 1] += y_flux;
                contrib_data[(y - 1) * nx + x - 1]++;
            }
        }
    }
}

static cpl_image *
_moo_reproject_model(moo_single *det,
                     moo_loc_single *loc,
                     moo_psf_single *psf_single,
                     const int *health)
{
    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(psf_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(health != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_image *res = NULL;
    cpl_image *fcentroids = moo_loc_single_get_f_centroids(loc);
    hdrl_image *himage = moo_single_get_image(det);
    cpl_image *image = hdrl_image_get_image(himage);

    int nx = hdrl_image_get_size_x(himage);
    int ny = hdrl_image_get_size_y(himage);
    int nb_fibres = cpl_image_get_size_y(fcentroids);

    cpl_imagelist *cube = moo_psf_single_get_cube(psf_single);
    double crpix2, cd2_2;
    moo_psf_single_get_cube_ref(psf_single, &crpix2, &cd2_2);
    res = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image *contrib = cpl_image_new(nx, ny, CPL_TYPE_INT);
    cpl_msg_debug(__func__, "start %d", cpl_error_get_code());

    /*REGDEBUG */
    // nb_fibres = 1;

/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions where
 * these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_fibres, health, cube, image, fcentroids, \
                                crpix2, cd2_2, res, contrib)
#else
#pragma omp parallel default(none)                                         \
    shared(nb_fibres, health, cube, image, fcentroids, crpix2, cd2_2, res, \
               contrib)
#endif
    {
#pragma omp for
#endif
        for (int i = 1; i <= nb_fibres; i++) {
            int h = health[i - 1];
            cpl_msg_debug(__func__, "reproject fibre %d", i);

            if (h != 0) {
                cpl_image *model = cpl_imagelist_get(cube, i - 1);
                _moo_reproject_model_fibre(image, i, fcentroids, model, crpix2,
                                           cd2_2, res, contrib);
            }
        }
#ifdef _OPENMP
    }
#endif
    double *data = cpl_image_get_data(res);

    int *contrib_data = cpl_image_get_data(contrib);
    for (int i = 0; i < nx * ny; i++) {
        if (contrib_data[i] != 0) {
            data[i] /= contrib_data[i];
        }
    }
    cpl_image_delete(contrib);

    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in modelize flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Reporject a model on detector
  @param    flat1 _DET_ first flat field
  @param    loc1 _LOC_ localisation of the first flat field
  @param    model_flat1 _PSF_ model of the first flat field
  @param    filename the name of the produce DET file
  @return   _DET_ the pixel to pixel variation map

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_reproject_model(moo_det *flat1,
                    moo_loc *loc1,
                    moo_psf *model_flat1,
                    const char *filename)
{
    cpl_ensure_code(flat1 != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(loc1 != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(model_flat1 != NULL, CPL_ERROR_NULL_INPUT);

    cpl_msg_info(__func__, "Reproject model");

    cpl_table *fibres_table = moo_loc_get_fibre_table(loc1);
    cpl_ensure_code(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT);

    unsigned int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE |
                                MOO_BADPIX_LOW_QE | MOO_BADPIX_CALIB_DEFECT;

    moo_fits_create(filename);

    cpl_msg_indent_more();

    for (int i = 1; i <= 2; i++) {
        cpl_table_unselect_all(fibres_table);
        int fibname = i;
        cpl_table_or_selected_int(fibres_table, MOO_FIBRES_TABLE_SPECTRO,
                                  CPL_EQUAL_TO, fibname);
        cpl_table *selected = cpl_table_extract_selected(fibres_table);
        const int *health =
            cpl_table_get_data_int_const(selected, MOO_FIBRES_TABLE_HEALTH);

        for (int j = 0; j < 3; j++) {
            moo_single *flat_single1 =
                moo_det_load_single(flat1, j, i, badpix_level);

            moo_loc_single *loc_single1 = moo_loc_get_single(loc1, j, i);

            moo_psf_single *psf_single1 = moo_psf_get_single(model_flat1, j, i);

            if (flat_single1 != NULL && loc_single1 != NULL && psf_single1) {
                cpl_image *model =
                    _moo_reproject_model(flat_single1, loc_single1, psf_single1,
                                         health);
                moo_fits_write_extension_image(model, filename, "MODEL_",
                                               flat_single1->extname,
                                               CPL_TYPE_FLOAT, NULL);
                cpl_image_delete(model);
            }
        }
        cpl_table_delete(selected);
    }
    cpl_msg_indent_less();
    return CPL_ERROR_NONE;
}

static moo_single *
_moo_compute_p2p_single(moo_single *flat1,
                        moo_loc_single *loc1,
                        moo_psf_single *model_flat1,
                        moo_single *flat2,
                        moo_loc_single *loc2,
                        moo_psf_single *model_flat2,
                        const int *health)
{
    moo_single *result = NULL;
    cpl_mask *mask_loc1 = NULL;
    cpl_mask *mask_loc2 = NULL;

    cpl_msg_indent_more();
    cpl_msg_info("moo_compute_p2p", "Reproject model");
    cpl_ensure(flat1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(model_flat1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(flat2 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc2 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(model_flat2 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(health != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_image *model1 = NULL;
    cpl_image *model2 = NULL;
    cpl_image *model_sum = NULL;
    hdrl_image *model = NULL;
    cpl_mask *orig = NULL;

    moo_try_check(model1 =
                      _moo_reproject_model(flat1, loc1, model_flat1, health),
                  " Reproject model1 failed");
    moo_try_check(model2 =
                      _moo_reproject_model(flat2, loc2, model_flat2, health),
                  "reproject model2 failed");

    moo_try_check(model_sum = cpl_image_add_create(model1, model2), " ");

#if MOO_DEBUG_P2P_MODEL
    moo_fits_write_extension_image(model1, "model.fits", "MODEL1_",
                                   flat1->extname, CPL_TYPE_FLOAT, NULL);
    moo_fits_write_extension_image(model2, "model.fits", "MODEL2_",
                                   flat1->extname, CPL_TYPE_FLOAT, NULL);
#endif
    result = moo_single_add_create(flat1, flat2);

    model = hdrl_image_create(model_sum, NULL);

    cpl_mask *mask = hdrl_image_get_mask(result->image);

    int size_y = hdrl_image_get_size_y(result->image);
    mask_loc1 = moo_loc_single_get_ODR(loc1, size_y);
    mask_loc2 = moo_loc_single_get_ODR(loc2, size_y);
    cpl_mask_and(mask_loc1, mask_loc2);
    moo_mask_to_badpix(result->qual, mask_loc1, MOO_BADPIX_OUTSIDE_DATA_RANGE);

    cpl_mask_or(mask, mask_loc1);

    orig = cpl_mask_duplicate(mask);
    hdrl_image_div_image(result->image, model);

    cpl_mask_not(orig);
    cpl_mask_and(orig, mask);
    cpl_msg_info("", "new bad pix %" CPL_SIZE_FORMAT, cpl_mask_count(orig));
    moo_mask_to_badpix(result->qual, orig, MOO_BADPIX_CALIB_DEFECT);

moo_try_cleanup:
    cpl_msg_indent_less();
    cpl_mask_delete(mask_loc1);
    cpl_mask_delete(mask_loc2);
    cpl_image_delete(model1);
    cpl_image_delete(model2);
    cpl_mask_delete(orig);
    hdrl_image_delete(model);
    cpl_image_delete(model_sum);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in model flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    To compute pixel-to-pixel variation map.
  @param    flat1 _DET_ first flat field
  @param    loc1 _LOC_ localisation of the first flat field
  @param    model_flat1 _PSF_ model of the first flat field
  @param    flat2 _DET_ second flat field
  @param    loc2 _LOC_ localisation of the second flat field
  @param    model_flat2 _PSF_ model of the second flat field
  @return   _DET_ the pixel to pixel variation map

 The function co-adds two preprocessed flat-field frames at two slit positions
 is order to increase the illumination coverage of the detectors, and it co-adds
 as well the corresponding modeled FFs. Then it derives the pixel-to-pixel
 variation map.

 * _Bad pixels flags_:
  - BADPIX_CALIB_DEFECT

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if first localisation has not a fibres table
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_compute_p2p(moo_det *flat1,
                moo_loc *loc1,
                moo_psf *model_flat1,
                moo_det *flat2,
                moo_loc *loc2,
                moo_psf *model_flat2)
{
    moo_det *result = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_table *selected = NULL;

    cpl_ensure(flat1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(model_flat1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(flat2 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc2 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(model_flat2 != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_msg_info(__func__, "Compute pixel-to-pixel variation map");

    cpl_table *fibres_table = moo_loc_get_fibre_table(loc1);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    unsigned int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE |
                                MOO_BADPIX_LOW_QE | MOO_BADPIX_CALIB_DEFECT;

#if MOO_DEBUG_P2P_MODEL
    moo_fits_create("model.fits");
#endif
    cpl_msg_indent_more();

    result = moo_det_new();

    result->primary_header = cpl_propertylist_new();

    for (int i = 1; i <= 2; i++) {
        cpl_table_unselect_all(fibres_table);
        int fibname = i;
        cpl_table_or_selected_int(fibres_table, MOO_FIBRES_TABLE_SPECTRO,
                                  CPL_EQUAL_TO, fibname);
        moo_try_check(selected = cpl_table_extract_selected(fibres_table), " ");
        const int *health =
            cpl_table_get_data_int_const(selected, MOO_FIBRES_TABLE_HEALTH);
        for (int j = 0; j < 3; j++) {
            moo_single *single = NULL;
            moo_single *det_single1 =
                moo_det_load_single(flat1, j, i, badpix_level);
            moo_loc_single *loc_single1 = moo_loc_get_single(loc1, j, i);
            moo_psf_single *psf_single1 = moo_psf_get_single(model_flat1, j, i);

            moo_single *det_single2 =
                moo_det_load_single(flat2, j, i, badpix_level);
            moo_loc_single *loc_single2 = moo_loc_get_single(loc2, j, i);
            moo_psf_single *psf_single2 = moo_psf_get_single(model_flat2, j, i);

            if (det_single1 != NULL && loc_single1 != NULL && psf_single1 &&
                det_single2 != NULL && loc_single2 != NULL && psf_single2) {
                cpl_msg_info(__func__, "Compute p2p for extension %s",
                             moo_detector_get_extname(j, i));

                moo_try_check(single = _moo_compute_p2p_single(
                                  det_single1, loc_single1, psf_single1,
                                  det_single2, loc_single2, psf_single2,
                                  health),
                              " ");
                moo_try_check(_moo_compute_p2p_qc(single, badpix_level), " ");
                moo_try_check(moo_det_set_single(result, j, i, single), " ");
            }
#if MOO_DEBUG_P2P_MODEL
            else {
                moo_fits_write_extension_image(NULL, "model.fits", "MODEL1_",
                                               moo_detector_get_extname(j, i),
                                               CPL_TYPE_FLOAT, NULL);
                moo_fits_write_extension_image(NULL, "model.fits", "MODEL2_",
                                               moo_detector_get_extname(j, i),
                                               CPL_TYPE_FLOAT, NULL);
            }
#endif
        }
        cpl_table_delete(selected);
        selected = NULL;
    }
    cpl_msg_indent_less();

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in compute_p2p");
        moo_det_delete(result);
        cpl_table_delete(selected);
        result = NULL;
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}
/**@}*/
