/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_compute_slitoffset.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
static double
_moo_compute_slitoffset_single(moo_single *det,
                               moo_loc_single *loc,
                               moo_compute_slitoffset_params *params)
{
    double res = NAN;
    cpl_vector *sumfluxv = NULL;
    cpl_matrix *xpos = NULL;
    cpl_vector *values = NULL;
    cpl_polynomial *poly = NULL;

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NAN);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NAN);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NAN);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_image *fcentroids = moo_loc_single_get_f_centroids(loc);
    int nb_fibres = cpl_image_get_size_y(fcentroids);
    int nx = cpl_image_get_size_x(fcentroids);


    hdrl_image *himage = moo_single_get_image(det);
    cpl_image *image = hdrl_image_get_image(himage);

    double min = params->min;
    double max = params->max;
    double step = params->step;

    int size = (int)((max - min) / step + 1);
    moo_try_check(sumfluxv = cpl_vector_new(size), " ");

    double *sumflux = cpl_vector_get_data(sumfluxv);
    for (int i = 0; i < size; i++) {
        sumflux[i] = 0;
    }
/* The following addresses an issue with the gcc9 compiler series prior to
  * gcc 9.3. These compiler versions require that the variable '__func__'
  * appears in the data sharing clause if default(none) is used. However
  * adding it to the data sharing clause breaks older compiler versions where
  * these variables are pre-determined as shared.
  * This was fixed in gcc 9.3 and OpenMP 5.1.
  */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(sumflux, size, min, max, step, nb_fibres, nx, \
                                fcentroids, image)
#else
#pragma omp parallel default(none) \
    shared(sumflux, size, min, max, step, nb_fibres, nx, fcentroids, image)
#endif
    {
#pragma omp for
#endif
        for (int s = 0; s < size; s++) {
            double delta = min + s * step;
            for (int j = 1; j <= nb_fibres; j++) {
                for (int i = 1; i <= nx; i++) {
                    int rej;
                    double yc = cpl_image_get(fcentroids, i, j, &rej);
                    if (!isnan(yc)) {
                        int nyc = (int)round(yc + delta);
                        double flux = cpl_image_get(image, i, nyc, &rej);
                        if (rej == 0) {
                            sumflux[s] += flux;
                        }
                    }
                }
            }
        }
#ifdef _OPENMP
    }
#endif
    double vmax = cpl_vector_get_max(sumfluxv);
    double lim = vmax * 0.9;
    int imin = 0;

    for (int i = 0; i < size; i++) {
        double v = sumflux[i];
        if (v >= lim) {
            imin = i;
            break;
        }
    }

    int imax = size - 1;
    for (int i = size - 1; i >= 0; i--) {
        double v = sumflux[i];
        if (v >= lim) {
            imax = i;
            break;
        }
    }

    int fitsize = imax - imin + 1;
    xpos = cpl_matrix_new(1, fitsize);
    values = cpl_vector_new(fitsize);

    for (int i = imin; i <= imax; i++) {
        double v = sumflux[i];
        double s = min + i * step;
        cpl_matrix_set(xpos, 0, i - imin, s);
        cpl_vector_set(values, i - imin, v);
    }
    poly = cpl_polynomial_new(1);
    const cpl_boolean sampsym = CPL_TRUE;
    const cpl_size maxdeg1d = 2;
    cpl_polynomial_fit(poly, xpos, &sampsym, values, NULL, CPL_FALSE, NULL,
                       &maxdeg1d);

    cpl_size power = 2;
    double a = cpl_polynomial_get_coeff(poly, &power);
    power = 1;
    double b = cpl_polynomial_get_coeff(poly, &power);
    res = b / (2 * a);

moo_try_cleanup:
    cpl_polynomial_delete(poly);
    cpl_matrix_delete(xpos);
    cpl_vector_delete(values);
    cpl_vector_delete(sumfluxv);
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in compute_slitoffset");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        res = NAN;
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Compute slit offset in LOC
  @param    det the DET frame
  @param    loc the LOC frame
  @param    params the compute slit offset parameters
  @return   the slit offset by detector

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
*/
/*----------------------------------------------------------------------------*/
float *
moo_compute_slitoffset(moo_det *det,
                       moo_loc *loc,
                       moo_compute_slitoffset_params *params)
{
    float *offsets = NULL;

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    unsigned int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE |
                                MOO_BADPIX_COSMETIC | MOO_BADPIX_CALIB_DEFECT;

    offsets = cpl_calloc(6, sizeof(float));
    cpl_msg_info("compute_slitoffset",
                 "Compute slit offset with [%f,%f] step %f", params->min,
                 params->max, params->step);

    for (int i = 1; i <= 2; i++) {
        for (int j = 0; j < 3; j++) {
            int idx = j + (i - 1) * 3;
            moo_single *det_single =
                moo_det_load_single(det, j, i, badpix_level);
            moo_loc_single *loc_single = moo_loc_get_single(loc, j, i);
            if (det_single != NULL && loc_single != NULL) {
                offsets[idx] =
                    (float)_moo_compute_slitoffset_single(det_single,
                                                          loc_single, params);
            }
            else {
                offsets[idx] = NAN;
            }
        }
    }

    return offsets;
}
/**@}*/
