/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_pfits.h"
#include "moo_single.h"
#include "moo_badpix.h"
#include "moo_det.h"
#include "moo_fibres_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_data DATA structure
 * @brief This module contains functions to handling MOONS data structure
 *
 * @defgroup moo_det  DET format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_det
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_det.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_det
  @return   1 newly allocated moo_det or NULL in case of an error

  The returned object must be deallocated using moo_det_delete().

 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_det_new(void)
{
    moo_det *res = cpl_calloc(1, sizeof(moo_det));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_det from the given DET frame
  @param frame the given DET frame
  @return   1 newly allocated moo_det or NULL in case of an error

  The returned object must be deallocated using moo_detlist_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the frame group is not CPL_FRAME_GROUP_PRODUCT
    or CPL_FRAME_GROUP_CALIB
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_det_create(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(frame);

    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_det *res = moo_det_new();

    res->primary_header = cpl_propertylist_load(filename, 0);
    res->filename = cpl_strdup(filename);

    cpl_errorstate prev_state = cpl_errorstate_get();

    res->ri[0] = moo_single_create(filename, MOO_TYPE_RI, 1);
    res->yj[0] = moo_single_create(filename, MOO_TYPE_YJ, 1);
    res->h[0] = moo_single_create(filename, MOO_TYPE_H, 1);
    res->ri[1] = moo_single_create(filename, MOO_TYPE_RI, 2);
    res->yj[1] = moo_single_create(filename, MOO_TYPE_YJ, 2);
    res->h[1] = moo_single_create(filename, MOO_TYPE_H, 2);
    moo_det_get_fibre_table(res);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_info("MOO_PREPARE", "DET: can't load %s : code %d", filename,
                     cpl_error_get_code());
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Load all parts in DET
  @param    self    the DET
  @param    level   the mask error level
  @return   error code or CPL_ERROR_NONE

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_load(moo_det *self, unsigned int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_det_load_ri(self, i, level);
        moo_det_load_yj(self, i, level);
        moo_det_load_h(self, i, level);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Load the RI part in DET
  @param    self    the DET
  @param    num     the TAS number
  @param    level   the mask error level
  @return   error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_load_ri(moo_det *self, int num, int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(num <= 1, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(level >= 0, CPL_ERROR_ILLEGAL_INPUT);

    if (self->ri[num] != NULL) {
        moo_single_load(self->ri[num], level);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Load the type part in DET and return it
  @param    self    the DET
  @param    type    the type of extension to load
  @param    num     the TAS number [1,2]
  @param    level   the mask error level (equal or greater than 0)

  @return  the moo_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is not in range [1,2]
 *
 */
/*----------------------------------------------------------------------------*/
moo_single *
moo_det_load_single(moo_det *self, moo_detector_type type, int num, int level)
{
    moo_single *res = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    res = moo_det_get_single(self, type, num);
    if (res != NULL) {
        moo_single_load(res, level);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Free the given type part in DET
  @param    self    the DET
  @param    type    the type of extension to free
  @param    num     the TAS number
  @return   error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_free_single(moo_det *self, moo_detector_type type, int num)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_error_code status = CPL_ERROR_NONE;

    moo_single *single = moo_det_get_single(self, type, num);
    if (single != NULL) {
        status = moo_single_free(single);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Load the YJ part in DET
  @param    self    the DET
  @param    num     the TAS number
  @param    level   the mask error level
  @return   error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_load_yj(moo_det *self, int num, int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(num <= 1, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(level >= 0, CPL_ERROR_ILLEGAL_INPUT);

    if (self->yj[num] != NULL) {
        moo_single_load(self->yj[num], level);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Load the H part in DET
  @param    self    the DET
  @param    num     the TAS number
  @param    level   the mask error level
  @return   error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
  - CPL_ERROR_ILLEGAL_INPUT if type is unknown
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_load_h(moo_det *self, int num, int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(num <= 1, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(level >= 0, CPL_ERROR_ILLEGAL_INPUT);

    if (self->h[num] != NULL) {
        moo_single_load(self->h[num], level);
    }
    return CPL_ERROR_NONE;
}

cpl_error_code
moo_det_set_single(moo_det *self,
                   moo_detector_type type,
                   int num,
                   moo_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    switch (type) {
        case MOO_TYPE_RI:
            moo_single_delete(self->ri[num - 1]);
            self->ri[num - 1] = single;
            break;
        case MOO_TYPE_YJ:
            moo_single_delete(self->yj[num - 1]);
            self->yj[num - 1] = single;
            break;
        case MOO_TYPE_H:
            moo_single_delete(self->h[num - 1]);
            self->h[num - 1] = single;
            break;
        default:
            status = CPL_ERROR_ILLEGAL_INPUT;
            cpl_error_set(__func__, status);
            break;
    }
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Get the type part in DET and return it
  @param    self    the DET
  @param    type    the type of extension to get
  @param    num     the TAS number [1,2]

  @return  the moo_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is not in range [1,2]
 *
 */
/*----------------------------------------------------------------------------*/
moo_single *
moo_det_get_single(moo_det *self, moo_detector_type type, int num)
{
    moo_single *result = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            result = self->ri[num - 1];
            break;
        case MOO_TYPE_YJ:
            result = self->yj[num - 1];
            break;
        case MOO_TYPE_H:
            result = self->h[num - 1];
            break;
        default:
            result = NULL;
            break;
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Get the RI part in DET
  @param    self    the DET
  @param    num     the TAS number
  @return   The RI part

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
moo_single *
moo_det_get_ri(moo_det *self, int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 1, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    return self->ri[num];
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Get the FIBRE TABLE in DET
  @param    self    the DET
  @return   The FIBRE TABLE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_det_get_fibre_table(moo_det *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->fibre_table == NULL && self->filename != NULL) {
        cpl_size qnum =
            cpl_fits_find_extension(self->filename, MOO_FIBRES_TABLE_EXTNAME);
        if (qnum > 0) {
            self->fibre_table = cpl_table_load(self->filename, qnum, 0);
        }
    }
    return self->fibre_table;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Get the PRIMARY HEADER in DET
  @param    self    the DET
  @return   The PRIMARY_HEADER

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist *
moo_det_get_primary_header(moo_det *self)
{
    if (self->primary_header == NULL && self->filename != NULL) {
        self->primary_header = cpl_propertylist_load(self->filename, 0);
    }
    return self->primary_header;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_det
  @param    self    moo_det to delete
  @return   void

  If the moo_det @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_det_delete(moo_det *self)
{
    if (self != NULL) {
        int i;
        if (self->filename != NULL) {
            cpl_free(self->filename);
        }

        for (i = 0; i < 2; i++) {
            moo_single_delete(self->ri[i]);
            moo_single_delete(self->yj[i]);
            moo_single_delete(self->h[i]);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        if (self->fibre_table != NULL) {
            cpl_table_delete(self->fibre_table);
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_det to a FITS file
  @param    self          moo_det to write to disk or NULL
  @param    filename      Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_det to a FITS file, using cfitsio.
  Only not NULL extensions are written.

*/
/*----------------------------------------------------------------------------*/
void
moo_det_save(moo_det *self, const char *filename)
{
    if (self != NULL) {
        if (self->primary_header != NULL) {
            cpl_propertylist_save(self->primary_header, filename,
                                  CPL_IO_CREATE);
        }
        else {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_save(h, filename, CPL_IO_CREATE);
            cpl_propertylist_delete(h);
        }
        int i;
        for (i = 1; i <= 2; i++) {
            moo_single_save(self->ri[i - 1], filename, MOO_TYPE_RI, i);
            moo_single_save(self->yj[i - 1], filename, MOO_TYPE_YJ, i);
            moo_single_save(self->h[i - 1], filename, MOO_TYPE_H, i);
        }

        if (self->fibre_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_FIBRES_TABLE_EXTNAME);
            cpl_table_save(self->fibre_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Dump structural information of DET
  @param    self    DET to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_dump(const moo_det *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_DET\n");
    fprintf(stream, "filename %s\n", self->filename);
    int i;
    for (i = 0; i < 2; i++) {
        if (self->ri[i] != NULL) {
            fprintf(stream, "---RI_%d %p\n", i, self->ri[i]);
            moo_single_dump(self->ri[i], stream);
        }

        if (self->yj[i] != NULL) {
            fprintf(stream, "---YJ_%d %p\n", i, self->yj[i]);
            moo_single_dump(self->yj[i], stream);
        }

        if (self->h[i] != NULL) {
            fprintf(stream, "---H_%d %p\n", i, self->h[i]);
            moo_single_dump(self->h[i], stream);
        }
    }

    return CPL_ERROR_NONE;
}

cpl_propertylist *
moo_det_get_single_header(moo_det *self, moo_detector_type type, int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    moo_single *single = moo_det_get_single(self, type, num);

    return moo_single_get_header(single);
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Sum DET structure
  @param    self DET result of the sum
  @param    det DET to sum
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_sum(moo_det *self, moo_det *det)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(det != NULL, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_single *ri = self->ri[i];
        moo_single *detri = det->ri[i];
        if (ri != NULL) {
            moo_single_sum(ri, detri);
        }
        moo_single *yj = self->yj[i];
        if (yj != NULL) {
            moo_single *detyj = det->yj[i];
            moo_single_sum(yj, detyj);
        }
        moo_single *h = self->h[i];
        if (h != NULL) {
            moo_single *deth = det->h[i];
            moo_single_sum(h, deth);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Subtract DET structure
  @param    self DET result of the sum
  @param    det DET to sum
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_subtract(moo_det *self, moo_det *det)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(det != NULL, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_single *ri = self->ri[i];
        moo_single *detri = det->ri[i];
        if (ri != NULL) {
            moo_single_sub(ri, detri);
        }
        moo_single *yj = self->yj[i];
        if (yj != NULL) {
            moo_single *detyj = det->yj[i];
            moo_single_sub(yj, detyj);
        }
        moo_single *h = self->h[i];
        if (h != NULL) {
            moo_single *deth = det->h[i];
            moo_single_sub(h, deth);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Divide DET structure
  @param    self DET result of the division
  @param    det DET to divide
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_divide(moo_det *self, moo_det *det)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(det != NULL, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_single *ri = self->ri[i];
        moo_single *detri = det->ri[i];
        if (ri != NULL) {
            moo_single_divide(ri, detri);
        }
        moo_single *yj = self->yj[i];
        if (yj != NULL) {
            moo_single *detyj = det->yj[i];
            moo_single_divide(yj, detyj);
        }
        moo_single *h = self->h[i];
        if (h != NULL) {
            moo_single *deth = det->h[i];
            moo_single_divide(h, deth);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Rescale using exptime factor
  @param    self DET result of the rescaling
  @param    det DET use to rescale
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_rescale_using_exptime(moo_det *self, moo_det *det)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(det != NULL, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_single *ri = self->ri[i];
        if (ri != NULL) {
            double exptimea = moo_pfits_get_exptime(self->primary_header);
            double exptimeb = moo_pfits_get_exptime(det->primary_header);
            moo_single_multiply_scalar(ri, exptimeb / exptimea);
        }
        moo_single *yj = self->yj[i];
        if (yj != NULL) {
            moo_single *detyj = det->yj[i];
            double exptimea =
                moo_pfits_get_dit(yj->header) * moo_pfits_get_ndit(yj->header);
            double exptimeb = moo_pfits_get_dit(detyj->header) *
                              moo_pfits_get_ndit(detyj->header);
            moo_single_multiply_scalar(yj, exptimeb / exptimea);
        }
        moo_single *h = self->h[i];
        if (h != NULL) {
            moo_single *deth = det->h[i];
            double exptimea =
                moo_pfits_get_dit(h->header) * moo_pfits_get_ndit(h->header);
            double exptimeb = moo_pfits_get_dit(deth->header) *
                              moo_pfits_get_ndit(deth->header);
            moo_single_multiply_scalar(h, exptimeb / exptimea);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Sum DET structure
  @param    self DET result of the mean
  @param    det DET to sum
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_mean(moo_det *self, moo_det *det)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(det != NULL, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_single *ri = self->ri[i];
        moo_single *detri = det->ri[i];
        if (ri != NULL) {
            moo_single_mean(ri, detri);
        }
        moo_single *yj = self->yj[i];
        if (yj != NULL) {
            moo_single *detyj = det->yj[i];
            moo_single_mean(yj, detyj);
        }
        moo_single *h = self->h[i];
        if (h != NULL) {
            moo_single *deth = det->h[i];
            moo_single_mean(h, deth);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Flag low snr pixels from DET structure
  @param    self DET to filter
  @param    min_snr_tab minimum snr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_det_filter_snr(moo_det *self, double *min_snr_tab)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);

    for (int i = 0; i < 2; i++) {
        moo_single *ri = self->ri[i];
        if (ri != NULL) {
            double min_snr = min_snr_tab[i * 3];
            moo_det_load_single(self, MOO_TYPE_RI, i + 1, MOO_BADPIX_GOOD);
            moo_single_filter_snr(ri, min_snr);
        }
        moo_single *yj = self->yj[i];
        if (yj != NULL) {
            double min_snr = min_snr_tab[i * 3 + 1];
            moo_det_load_single(self, MOO_TYPE_YJ, i + 1, MOO_BADPIX_GOOD);
            moo_single_filter_snr(yj, min_snr);
        }
        moo_single *h = self->h[i];
        if (h != NULL) {
            double min_snr = min_snr_tab[i * 3 + 2];
            moo_det_load_single(self, MOO_TYPE_H, i + 1, MOO_BADPIX_GOOD);
            moo_single_filter_snr(h, min_snr);
        }
    }
    return CPL_ERROR_NONE;
}
/**@}*/
