/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <assert.h>
#include "moo_detlist.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_detlist DET_LIST format
 * @ingroup moo_data
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_detlist
  @return   1 newly allocated moo_detlist or NULL in case of an error

  The returned object must be deallocated using moo_detlist_delete().

 */
moo_detlist *
moo_detlist_new(void)
{
    return (moo_detlist *)cpl_calloc(1, sizeof(moo_detlist));
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_detlist from the given DET frameset
  @return   1 newly allocated moo_detlist or NULL in case of an error

  The returned object must be deallocated using moo_detlist_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
moo_detlist *
moo_detlist_create(cpl_frameset *frameset)
{
    cpl_ensure(frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int i;
    moo_detlist *res = moo_detlist_new();

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame =
            cpl_frameset_get_position_const(frameset, i);
        moo_det *det = moo_det_create(current_frame);
        moo_detlist_push(res, det);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Load the type part for all DET in the detlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @param    level   Mask error level
  @return   The error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_detlist_load_single(const moo_detlist *self,
                        moo_detector_type type,
                        int num,
                        int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT);

    int i;

    for (i = 0; i < self->size; i++) {
        moo_det *det = self->list[i];
        moo_det_load_single(det, type, num, level);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Free the type part for all DET in the detlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @return   The error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_detlist_free_single(const moo_detlist *self,
                        moo_detector_type type,
                        int num)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < self->size; i++) {
        moo_det *det = self->list[i];
        cpl_error_code status = moo_det_free_single(det, type, num);
        if (status != CPL_ERROR_NONE) {
            return status;
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Get the number of DET in the detlist
  @param    self    the list of DET
  @return   The number of DET or -1 on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_size
moo_detlist_get_size(const moo_detlist *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    assert(self->size >= 0);

    return self->size;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Get the DET at the position i in the list
  @param    self    the list of DET
  @param    i       position in the list
  @return   DET element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_detlist_get(moo_detlist *self, int i)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(i >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int size = moo_detlist_get_size(self);
    cpl_ensure(i < size, CPL_ERROR_ILLEGAL_INPUT, NULL);

    return self->list[i];
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Get the all the images of the type part in the detlist
  @param    self    the list of DET
  @param    type     the type of extension to load
  @param    num     the TAS number
  @return   The  image list or NULL

  TAS number is 1 or 2.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is num smaller than 1
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 2
 *
 */
/*----------------------------------------------------------------------------*/
hdrl_imagelist *
moo_detlist_get_image(const moo_detlist *self, moo_detector_type type, int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    hdrl_imagelist *res = NULL;
    int i;

    res = hdrl_imagelist_new();

    for (i = 0; i < self->size; i++) {
        moo_det *det = self->list[i];
        moo_single *single = NULL;
        moo_try_check(single = moo_det_get_single(det, type, num), " ");

        if (single != NULL) {
            hdrl_image *img = moo_single_get_image(single);
            hdrl_imagelist_set(res, img, i);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in detlist");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        hdrl_imagelist_delete(res);
        res = NULL;
    }
    return res;
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Get the type data part for all DET in the detlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @return   The type data list

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_detlist_get_single_data(const moo_detlist *self,
                            moo_detector_type type,
                            int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    cpl_imagelist *res = cpl_imagelist_new();
    int i;

    for (i = 0; i < self->size; i++) {
        moo_det *det = self->list[i];
        moo_single *single = moo_det_get_single(det, type, num);
        if (single != NULL) {
            cpl_imagelist_set(res, moo_single_get_data(single), i);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Get the type QUAL part for all DET in the detlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @return   The type QUAL list

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_detlist_get_single_qual(const moo_detlist *self,
                            moo_detector_type type,
                            int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    cpl_imagelist *res = cpl_imagelist_new();
    int i;

    for (i = 0; i < self->size; i++) {
        moo_det *det = self->list[i];
        moo_single *single = moo_det_get_single(det, type, num);
        if (single != NULL) {
            cpl_imagelist_set(res, moo_single_get_qual(single), i);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Insert a DET into an moo_detlist
  @param    self    The moo_detlist
  @param    det     The DET to insert
  @param    pos     The list position (from 0 to number of DET)
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  It is allowed to specify the position equal to the number of DET in the
  list. This will increment the size of the detlist.

  No action occurs if a DET is inserted more than once into the same
  position. It is allowed to insert the same DET into two different
  positions in a list.

  The DET is inserted at the position pos in the DET list. If the DET
  already there is only present in that one location in the list, then the
  DET is deallocated.

  The added DET is owned by the moo_detlist object, which deallocates it
  moo_detlist_delete is called. Other option is to use moo_detlist_unset to
  recover ownership of the DET, in which case the moo_detlist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    DET in detlist
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_detlist_set(moo_detlist *self, moo_det *det, cpl_size pos)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(det, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pos >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(pos <= self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Do nothing if the det is already there */
    if (pos < self->size && det == self->list[pos])
        return CPL_ERROR_NONE;

    if (pos == self->size) {
        self->size++;
        self->list =
            cpl_realloc(self->list, (size_t)self->size * sizeof(moo_det *));
    }
    else {
        /* Check if the det at the position to be overwritten
           is present in only one position */
        int i;

        for (i = 0; i < self->size; i++) {
            if (i != pos && self->list[i] == self->list[pos])
                break;
        }
        if (i == self->size) {
            /* The image at the position to be overwritten
               is present in only one position, so delete it */
            moo_det_delete(self->list[pos]);
        }
    }

    self->list[pos] = det;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Insert a DET a the end of moo_detlist
  @param    self    The moo_detlist
  @param    det     The DET to insert
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
  @see  moo_detlist_set()

  No action occurs if a DET is inserted more than once into the same
  position. It is allowed to insert the same DET into two different
  positions in a list.

  The DET is inserted at the end in the DET list. If the DET
  already there is only present in that one location in the list, then the
  DET is deallocated.

  The added DET is owned by the moo_detlist object, which deallocates it
  moo_detlist_delete is called. Other option is to use moo_detlist_unset to
  recover ownership of the DET, in which case the moo_detlist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_detlist_push(moo_detlist *self, moo_det *det)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    return moo_detlist_set(self, det, self->size);
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup  moo_detlist
  @brief    Remove a DET from a DET list
  @param    self   The moo_detlist
  @param    pos      The list position (from 0 to number of DET-1)
  @return   The pointer to the removed DET or NULL in error case

  The specified DET is not deallocated, it is simply removed from the
  list. The pointer to the DET is returned to let the user decide to
  deallocate it or not.
  Eventually, the DET will have to be deallocated with moo_det_delete().

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    images in self
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_detlist_unset(moo_detlist *self, cpl_size pos)
{
    moo_det *out;
    cpl_size i;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pos >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(pos < self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    /* Get pointer to DET to be removed */
    out = self->list[pos];

    /* Move the following DET one position towards zero */
    for (i = pos + 1; i < self->size; i++) {
        self->list[i - 1] = self->list[i];
    }

    /* Decrement of the size */
    self->size--;

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Empty an moo_detlist and deallocate all its DET
  @param    self  The DET list or NULL
  @return   Nothing
  @see  moo_detlist_empty(), moo_detlist_delete()
  @note If @em self is @c NULL nothing is done and no error is set.

  After the call the moo_det list can be populated again. It must eventually
  be deallocted with a call to moo_detlist_delete().

 */
/*----------------------------------------------------------------------------*/
void
moo_detlist_empty(moo_detlist *self)
{
    if (self != NULL) {
        while (self->size > 0) { /* An iteration may unset more than 1 image */
            int i = self->size - 1;
            moo_det *del = moo_detlist_unset(self, i);

            moo_det_delete(del);

            /* If this image was inserted more than once into the list,
               the other insertions must be unset without a delete. */
            while (--i >= 0) {
                if (self->list[i] == del) {
                    /* This image was inserted more than once in the list */
                    (void)moo_detlist_unset(self, i);
                }
            }
        }
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Free memory used by a moo_detlist object, except the DET
  @param    self    The DET list or NULL
  @return   Nothing
  @see moo_detlist_empty()
  @note The caller must have pointers to all DET in the list and is
        reponsible for their deallocation. If @em self is @c NULL nothing is
        done and no error is set.
 */
/*----------------------------------------------------------------------------*/
void
moo_detlist_unwrap(moo_detlist *self)
{
    if (self != NULL) {
        cpl_free(self->list);
        cpl_free(self);
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free all memory used by a moo_detlist object including the DET
  @param    self    The DET list or NULL
  @return   void
  @see      moo_detlist_empty(), moo_detlist_unwrap()
  If the moo_detlist @em self is @c NULL, nothing is done and no error is set.

 */
void
moo_detlist_delete(moo_detlist *self)
{
    if (self != NULL) {
        moo_detlist_empty(self);
        moo_detlist_unwrap(self);
    }

    return;
}

/**@}*/
