/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_dfs.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_params.h"
#include "moo_badpix.h"
#include "moo_single.h"
#include "moo_det.h"
#include "moo_detlist.h"
#include "moo_cube.h"
#include "moo_prepare.h"
#include "moo_drl.h"
#include <gsl/gsl_poly.h>
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Subtracts the Master Bias frame from a _DET_ frame
  @param    det _DET_ frame to subtract
  @param    masterbias_frame the Master bias frame
  @param    params correct bias parameters
  @return   the error code

  This function applies to the RI extensions, it is applied to frames in basic
 * calibration recipes and to frames used in science reduction in stare mode.
 * The Master Bias extensions are simply subtracted from the corresponding
 * extensions in the input Frame, taking into account the associated Bad Pixel
 * Map extensions. The output frame is an image with associated bad pixel map
 * and statistical errors. The output bad pixel maps are the combination
 * of the input ones.
 *
 _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the frame is not tagged has MASTER BIAS
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_correct_bias(moo_det *det,
                 const cpl_frame *masterbias_frame,
                 moo_correct_bias_params *params)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
    int badpix_level = MOO_BADPIX_GOOD;

    if (masterbias_frame != NULL) {
        const char *filename = cpl_frame_get_filename(masterbias_frame);
        const char *tag = cpl_frame_get_tag(masterbias_frame);

        cpl_msg_info(__func__, "Subtracting MASTER_BIAS");
        cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
        cpl_ensure_code(strcmp(tag, MOONS_TAG_MASTER_BIAS) == 0,
                        CPL_ERROR_ILLEGAL_INPUT);

        moo_det *mbias = moo_det_create(masterbias_frame);
        int i;
        for (i = 1; i <= 2; i++) {
            moo_single *s1 =
                moo_det_load_single(det, MOO_TYPE_RI, i, badpix_level);
            if (s1 != NULL) {
                cpl_msg_info(__func__, "  Correcting RI_%d and ERR_RI_%d", i,
                             i);
                moo_single *s2 =
                    moo_det_load_single(mbias, MOO_TYPE_RI, i, badpix_level);

                hdrl_image *img1 = moo_single_get_image(s1);
                hdrl_image *img2 = moo_single_get_image(s2);

                if (strcmp(params->subtract_bias_method,
                           MOO_CORRECT_BIAS_METHOD_MASTER) == 0) {
                    hdrl_image_sub_image(img1, img2);
                }
                else {
                    cpl_propertylist *header = moo_single_get_header(s1);

                    moo_outputs *outputs = moo_outputs_load(header, s1->type);

                    for (int ri = 0; ri < outputs->nb; ri++) {
                        int llx = outputs->outputs[ri].x;
                        int lly = outputs->outputs[ri].y;
                        int urx = outputs->outputs[ri].x +
                                  outputs->outputs[ri].nx - 1;
                        int ury = outputs->outputs[ri].y +
                                  outputs->outputs[ri].ny - 1;

                        hdrl_image *data_im =
                            hdrl_image_extract(s1->image, llx, lly, urx, ury);
                        hdrl_image *bias_im =
                            hdrl_image_extract(s2->image, llx, lly, urx, ury);
                        hdrl_value median = hdrl_image_get_median(bias_im);
                        hdrl_image_sub_scalar(data_im, median);
                        cpl_image *data = hdrl_image_get_image(data_im);
                        cpl_image *err = hdrl_image_get_error(data_im);
                        hdrl_image_insert(s1->image, data, err, llx, lly);
                        hdrl_image_delete(data_im);
                        hdrl_image_delete(bias_im);
                    }
                    moo_outputs_delete(outputs);
                }
                cpl_msg_info(__func__, "  Updating QUAL_RI_%d", i);
                cpl_image *qual1 = moo_single_get_qual(s1);
                cpl_image *qual2 = moo_single_get_qual(s2);
                cpl_image_or(s1->qual, qual1, qual2);
            }
        }

        moo_det_delete(mbias);
    }
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in correct bias");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_correct_dark_using_dit(moo_det *det,
                            moo_det *detoff,
                            moo_det *mnir,
                            moo_detector_type type,
                            int i,
                            int badpix_level)
{
    double dit;
    double ndit;

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_single *single = moo_det_get_single(det, type, i);
    if (single != NULL) {
        cpl_ensure_code(detoff != NULL, CPL_ERROR_NULL_INPUT);
        moo_single *off = moo_det_get_single(detoff, type, i);

        if (off != NULL) {
            moo_det_load_single(det, type, i, badpix_level);
            moo_det_load_single(detoff, type, i, badpix_level);

            if (mnir != NULL) {
                moo_single *s = moo_det_get_single(mnir, type, i);
                if (s != NULL) {
                    moo_det_load_single(mnir, type, i, badpix_level);
                    moo_badpix_merge(single->qual, s->qual);
                }
            }
            moo_try_check(dit = moo_pfits_get_dit(single->header), " ");
            moo_try_check(ndit = moo_pfits_get_ndit(single->header), " ");
            double exptime = dit * ndit;
            moo_try_check(dit = moo_pfits_get_dit(off->header), " ");
            moo_try_check(ndit = moo_pfits_get_ndit(off->header), " ");
            double exptimeoff = dit * ndit;
            moo_try_assure(
                exptime == exptimeoff, CPL_ERROR_ILLEGAL_INPUT,
                "Exptime in ON and OFF frames (type=%d) are not equal %f %f",
                type, exptime, exptimeoff);
            moo_single_sub(single, off);
        }
        else {
            cpl_msg_error(__func__,
                          "No OFF frame (or MASTER_DARK_NIR) for extname %s",
                          single->extname);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_errorstate_dump(prev_state, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_errorstate_set(prev_state);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Subtracts the master dark frame from a frame after scaling for
exposure time (RI). Subtract OFF frame from ON frame (YJ, H)
  @param    det _DET_ structure to subtract
  @param    detoff corresponding _DET_ Off structure or NULL
  @param    masterDarkVis Master dark frame for optical detectors
  @param    masterDarkNir Master dark frame for IR detectors or NULL
  @return   the error code

  The input frame is corrected for dark current using the Master Dark frame and
  accounting for its associated Bad Pixel Mask darkbpixmask which identifies pixels
  where the dark current is far above the average.
  In RI extensions the Master Dark is scaled according to the exposure time and
  subtracted from the input frame. The bad pixel masks are combined and propagated.
  In YJ and H extensions, the OFF frame or the master  dark (IR) is subtracted from
  the ON frame and the bad pixel masks are combined and propagated.

  * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the frame is not tagged has MASTER DARK_VIS
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_correct_dark(moo_det *det,
                 moo_det *detoff,
                 const cpl_frame *masterDarkVis,
                 const cpl_frame *masterDarkNir)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();

    int i;
    int badpix_level = MOO_BADPIX_GOOD;
    moo_det *master_nir = NULL;
    moo_det *off = NULL;

    if (masterDarkVis != NULL) {
        const char *filename = cpl_frame_get_filename(masterDarkVis);
        const char *tag = cpl_frame_get_tag(masterDarkVis);
        cpl_ensure_code(strcmp(tag, MOONS_TAG_MASTER_DARK_VIS) == 0,
                        CPL_ERROR_ILLEGAL_INPUT);
        cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);

        moo_det *mdarkVis = moo_det_create(masterDarkVis);

        double exptime = moo_pfits_get_exptime(det->primary_header);
        double exptime2 = moo_pfits_get_exptime(mdarkVis->primary_header);
        for (i = 1; i <= 2; i++) {
            moo_single *s1 = moo_det_get_single(det, MOO_TYPE_RI, i);
            if (s1 != NULL) {
                cpl_msg_info(__func__,
                             "Subtract DARK on %s using MASTER_DARK_VIS: %s",
                             moo_detector_get_extname(MOO_TYPE_RI, i),
                             filename);
                moo_det_load_single(det, MOO_TYPE_RI, i, badpix_level);
                moo_det_load_single(mdarkVis, MOO_TYPE_RI, i, badpix_level);

                moo_single *s2 = moo_det_get_single(mdarkVis, MOO_TYPE_RI, i);
                if (s2 != NULL) {
                    hdrl_image *img2 = moo_single_get_image(s2);
                    hdrl_image_mul_scalar(img2, (hdrl_value){
                                                    exptime / exptime2, 0.0 });
                    moo_single_sub(s1, s2);
                }
                else {
                    cpl_msg_error(__func__, "No MASTER_DARK_VIS for RI_%d", i);
                }
            }
        }
        moo_det_delete(mdarkVis);
    }
    if (detoff != NULL) {
        off = detoff;
        cpl_msg_info(__func__, "Subtract DARK using OFF %s", off->filename);
        if (masterDarkNir != NULL) {
            cpl_msg_info(__func__, "Subtract DARK using MASTER_DARK_NIR %s",
                         cpl_frame_get_filename(masterDarkNir));
            master_nir = moo_det_create(masterDarkNir);
        }
    }
    else if (masterDarkNir != NULL) {
        moo_try_check(master_nir = moo_det_create(masterDarkNir),
                      "Can't create off frame from MASTER_DARK_NIR");
        cpl_msg_info(__func__, "Subtract DARK using MASTER_DARK_NIR %s",
                     cpl_frame_get_filename(masterDarkNir));
        off = master_nir;
    }
    if (off != NULL) {
        for (i = 1; i <= 2; i++) {
            _moo_correct_dark_using_dit(det, off, master_nir, MOO_TYPE_YJ, i,
                                        badpix_level);
            _moo_correct_dark_using_dit(det, off, master_nir, MOO_TYPE_H, i,
                                        badpix_level);
        }
    }
    else {
        cpl_msg_warning(__func__,
                        "No OFF or MASTER_DARK_NIR provided for %s, No DARK "
                        "correction done",
                        det->filename);
    }

moo_try_cleanup:
    moo_det_delete(master_nir);
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in Correct Dark");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_single_correct_detlin(moo_single *s, cpl_imagelist *cube)
{
    cpl_ensure_code(s != NULL, CPL_ERROR_NULL_INPUT);
    cpl_msg_indent_more();
    if (cube == NULL) {
        cpl_msg_warning("moo_correct_detlin",
                        "No linearity correction : coeffs cube is missing");
    }
    else if (cpl_imagelist_get_size(cube) == 3) {
        cpl_image *ima = cpl_imagelist_get(cube, 0);
        cpl_image *imb = cpl_imagelist_get(cube, 1);
        cpl_image *imc = cpl_imagelist_get(cube, 2);

        float *pima = cpl_image_get_data_float(ima);
        float *pimb = cpl_image_get_data_float(imb);
        float *pimc = cpl_image_get_data_float(imc);

        hdrl_image *himg = moo_single_get_image(s);
        cpl_image *img = hdrl_image_get_image(himg);

        double *data = cpl_image_get_data_double(img);
        int nx = cpl_image_get_size_x(img);
        int ny = cpl_image_get_size_x(img);
        for (int i = 0; i < nx * ny; i++) {
            double a = (double)pima[i];
            double b = (double)pimb[i];
            double c = (double)pimc[i];
            double influx = data[i];

            if (isnan(c) || isnan(b) || isnan(a)) {
                continue;
            }
            if (fabs(c) < 1e-5) {
                double val = data[i];
                data[i] = val - a;
            }
            else if (fabs(b) < 1e-3) {
                data[i] = 0;
            }
            else {
                // C<1e-5 signal_true = signal-A
                // B<1e-3 signal_true = 0
                //  C > 0 && B > 0
                //  (A-signal) + signal_true + C/B^2 *signal_true^2. = 0
                // signal_true = (-B^2 + B^2 * sqrt(1 - 4*C*(A-signal)/B^2)) / (2*C).
                double x0 = NAN, x1 = NAN;
                double sol = NAN;
                double val = data[i];
                gsl_poly_solve_quadratic(c / (b * b), 1, a - val, &x0, &x1);

                sol = x0;
                double dist = fabs(x0 - influx);
                double dist_x1 = fabs(x1 - influx);
                if (!isnan(x1) && dist > dist_x1) {
                    sol = x1;
                }

                if (isnan(sol)) {
                    sol = val - a;
                }

                double val2 = 2 * c / (b * b);
                double val3 = 1 - 2 * val2 * (a - val);
                if (val3 < 0.0) {
                    data[i] = val - a;
                }
                else {
                    data[i] = (sqrt(val3) - 1) / val2;
                }
                // cpl_msg_info("test","test i %d %f %f",i,influx,data[i]);
            }
        }
    }
    else {
        cpl_image *ima = cpl_imagelist_get(cube, 0);
        cpl_image *imb = cpl_imagelist_get(cube, 1);
        cpl_image *imc = cpl_imagelist_get(cube, 2);
        cpl_image *imd = cpl_imagelist_get(cube, 3);

        float *pima = cpl_image_get_data_float(ima);
        float *pimb = cpl_image_get_data_float(imb);
        float *pimc = cpl_image_get_data_float(imc);
        float *pimd = cpl_image_get_data_float(imd);

        hdrl_image *himg = moo_single_get_image(s);
        cpl_image *img = hdrl_image_get_image(himg);
        double *data = cpl_image_get_data_double(img);

        int nx = cpl_image_get_size_x(img);
        int ny = cpl_image_get_size_y(img);

        for (int i = 0; i < nx * ny; i++) {
            double a = (double)pima[i];
            double b = (double)pimb[i];
            double c = (double)pimc[i];
            double d = (double)pimd[i];

            double influx = data[i];
            if (isnan(d) || isnan(c) || isnan(b) || isnan(a)) {
                continue;
            }
            if (fabs(d) < 1e-5) {
                double val = data[i];
                data[i] = val - a;
            }
            else if (fabs(c) < 1e-5) {
                double val = data[i];
                data[i] = val - a;
            }
            else if (fabs(b) < 1e-3) {
                data[i] = 0;
            }
            else {
                double x0 = NAN, x1 = NAN, x2 = NAN;
                double sol = NAN;
                // (A-s) + st + C/B^2*st^2 + D/B^3*st^3 = 0
                gsl_poly_solve_cubic((c / (b * b)) / (d / (b * b * b)),
                                     1 / (d / (b * b * b)),
                                     (a - influx) / (d / (b * b * b)), &x0, &x1,
                                     &x2);
                double dist_x1 = fabs(x1 - influx);
                double dist_x2 = fabs(x2 - influx);
                sol = x0;
                double dist = fabs(x0 - influx);
                if (!isnan(x1) && dist > dist_x1) {
                    dist = dist_x1;
                    sol = x1;
                }
                if (!isnan(x2) && dist > dist_x2) {
                    dist = dist_x2;
                    sol = x2;
                }
                data[i] = sol;
                /*{
          int xt = i % nx+1;
          int yt = i / nx+1;
          if(xt==1001 && yt==1001){
            cpl_msg_info("test","test x=%d y=%d no=%f corr=%f x1=%f x2=%f x3=%f",xt,yt,influx,data[i],
              x0,x1,x2);
          }
        }*/
            }
        }
    }
    cpl_msg_indent_less();
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Apply the detector linearity correction
  @param    det _DET_ frame to correct
  @param    cube_frame cube coefficients to appply linearity correction by pixel
  @return   the error code

  * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the frame of COEFF is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_correct_detlin(moo_det *det, const cpl_frame *cube_frame)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    int badpix_level = MOO_BADPIX_GOOD;

    if (cube_frame != NULL) {
        const char *filename = cpl_frame_get_filename(cube_frame);
        cpl_ensure_code(filename != NULL, CPL_ERROR_ILLEGAL_INPUT);

        cpl_propertylist *primary_header = moo_det_get_primary_header(det);
        moo_qc_set_is_linearcor(primary_header, CPL_TRUE);

        cpl_msg_info("moo_correct_detlin", "Correct linearity with %s",
                     filename);
        moo_cube *cube = moo_cube_load(cube_frame);
        cpl_msg_indent_more();
        for (int i = 0; i < 2; i++) {
            for (int j = 0; j <= 3; j++) {
                moo_single *s1 =
                    moo_det_load_single(det, j, i + 1, badpix_level);
                if (s1 != NULL) {
                    int idx = j + i * 3;
                    cpl_msg_info(__func__, "  Correcting %s", s1->extname);
                    cpl_imagelist *cdata = cube->data[idx];
                    _moo_single_correct_detlin(s1, cdata);
                }
            }
        }
        cpl_msg_indent_less();
        moo_cube_delete(cube);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculates the noisey bad pixel mask for one extension
  @param    single  _DET_ single extension
  @param    params noise map parameters
  @param    list DARK list DATA images
  @param    qlist DARK list QUAL images
  @return   the error code

 * _Bad pixels flags_:
  - BADPIX_NOISY

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_compute_noise_map(moo_single *single,
                             moo_nos_params *params,
                             cpl_imagelist *list,
                             cpl_imagelist *qlist)
{
    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(list != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qlist != NULL, CPL_ERROR_NULL_INPUT);

    cpl_image *data = NULL;
    cpl_image *sigma = NULL;
    cpl_mask *mask = NULL;

    moo_try_check(data = moo_single_get_data(single), " ");
    moo_try_check(sigma = moo_compute_sigma_map(list, qlist, data), " ");

#if MOO_DEBUG_COMPUTE_NOISE_MAP
    {
        char *name = cpl_sprintf("sigma_%s.fits", single->extname);
        char *qname = cpl_sprintf("mask_sigma_%s.fits", single->extname);
        cpl_image_save(sigma, name, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        cpl_mask *msigma = cpl_image_get_bpm(sigma);
        cpl_mask_save(msigma, qname, NULL, CPL_IO_CREATE);
        cpl_free(name);
        cpl_free(qname);
    }
#endif
    moo_try_check(mask = moo_kappa_sigma_clipping(sigma, params->clip_niter,
                                                  params->clip_kappa,
                                                  params->clip_diff,
                                                  params->clip_frac),
                  " ");

    moo_try_check(moo_single_apply_mask(single, mask, MOO_BADPIX_NOISY), " ");

moo_try_cleanup:
    cpl_mask_delete(mask);
    cpl_image_delete(data);
    cpl_image_delete(sigma);
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Calculates the noisey bad pixel mask
  @param    det _DET_ Average dark frame
  @param    darkList series of individual _DET_ dark frames
  @param    cosmiclist series of individual _MASK_ of dark frames
  @param    params noise map parameters
  @return   the error code

 * This function searches for static bad pixels in stacks of dark frames taken
 * with equal exposure times. The noise in each pixel stack is computed and
 * compared to the mean noise and a pixel is declared bad if the deviation
 * exceeds nos-clip-sigma.
  The QUAL extension of det frame is updated.

 * _Flags considered as bad : BADPIX_COSMETIC | BADPIX_COSMIC_UNREMOVED | BADPIX_HOT
 *
 * _Bad pixels flags_:
  - BADPIX_NOISY

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the frame is darkList size < 3
  - CPL_ERROR_ILLEGAL_INPUT if the frame is not tagged has MASTER BIAS
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_compute_noise_map(moo_det *det,
                      moo_detlist *darkList,
                      moo_masklist *cosmiclist,
                      moo_nos_params *params)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(darkList != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);

    int size = moo_detlist_get_size(darkList);
    cpl_ensure_code(size >= 3, CPL_ERROR_ILLEGAL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();

    int badpix_level = MOO_BADPIX_COSMETIC | MOO_BADPIX_COSMIC_UNREMOVED |
                       MOO_BADPIX_HOT | MOO_BADPIX_OUTSIDE_DATA_RANGE;

    int i;

    cpl_msg_info("moo_compute_noise_map",
                 "Computing the noise map using %d dark files", size);
    cpl_msg_indent_more();
    for (i = 1; i <= 2; i++) {
        for (int extnum = 1; extnum <= 2; extnum++) {
            moo_detlist_load_single(darkList, extnum, i, badpix_level);
            cpl_imagelist *list =
                moo_detlist_get_single_data(darkList, extnum, i);
            cpl_imagelist *qlist =
                moo_detlist_get_single_qual(darkList, extnum, i);

            moo_single *single =
                moo_det_load_single(det, extnum, i, badpix_level);

            if (single != NULL) {
                if (cosmiclist != NULL) {
                    for (int k = 0; k < size; k++) {
                        moo_mask *mmask = moo_masklist_get(cosmiclist, k);

                        cpl_image *qimg = cpl_imagelist_get(qlist, k);
                        cpl_mask *mask = moo_mask_get(mmask, extnum, i);
                        moo_mask_to_badpix(qimg, mask,
                                           MOO_BADPIX_COSMIC_UNREMOVED);
                    }
                }
                moo_single_compute_noise_map(single, params, list, qlist);
            }
            cpl_imagelist_delete(list);
            cpl_imagelist_unwrap(qlist);
            moo_detlist_free_single(darkList, extnum, i);
        }
    }
    cpl_msg_indent_less();
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in compute noise map");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculates the hot bad pixel mask for a single DET extension
  @param    single single _DET_
  @param    params hot map parameters
  @return   the number of detected hot pixels

 * This function searches for hot bad pixels in a master dark frame by applying
 * an iterative kappa-sigma clipping algorithm.
  The QUAL extension of det frame is updated.
 *
 * _Bad pixels flags_:
  - BADPIX_HOT

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static int
_moo_single_compute_hot_map(moo_single *single, moo_hot_params *params)
{
    int nb_global_hot = -1;
    int nb_hot = -1;
    cpl_ensure(single != NULL, CPL_ERROR_NULL_INPUT, -1);

    cpl_image *data = NULL;
    cpl_mask *mask = NULL;

    moo_try_check(data = moo_single_get_data(single), " ");
    int nx = cpl_image_get_size_x(data);
    int ny = cpl_image_get_size_y(data);

    cpl_mask *data_bpm = cpl_mask_duplicate(cpl_image_get_bpm(data));
    moo_try_check(mask = moo_kappa_sigma_clipping(data, params->clip_niter,
                                                  params->clip_kappa,
                                                  params->clip_diff,
                                                  params->clip_frac),
                  " ");

    moo_try_check(nb_global_hot = cpl_mask_count(mask), " ");
    cpl_msg_info("moo_compute_hot_map",
                 "Global hot pixels detection : %d pixels found",
                 nb_global_hot);
    cpl_mask_delete(cpl_image_get_bpm(data));
    cpl_image_set_bpm(data, data_bpm);

    int winhsize = params->local_winhsize;
    if (winhsize != 0) {
        cpl_msg_info("moo_compute_hot_map",
                     "Local hot pixels detection with winhsize: %d pixels",
                     winhsize);
#ifdef _OPENMP
#pragma omp parallel default(none) \
    shared(nx, ny, data, data_bpm, mask, params, winhsize)
        {
#pragma omp for
#endif
            for (int j = 1; j <= ny; j++) {
                for (int i = 1; i <= nx; i++) {
                    int rej;
                    cpl_binary m = cpl_mask_get(mask, i, j);
                    if (m == CPL_BINARY_1) {
                        int llx = i - winhsize;
                        int lly = j - winhsize;
                        int urx = i + winhsize;
                        int ury = j + winhsize;
                        if (llx < 1) {
                            llx = 1;
                        }
                        if (lly < 1) {
                            lly = 1;
                        }
                        if (urx > nx) {
                            urx = nx;
                        }
                        if (ury > ny) {
                            ury = ny;
                        }
                        double flux = cpl_image_get(data, i, j, &rej);
                        cpl_stats *stats = cpl_stats_new_from_image_window(
                            data, CPL_STATS_MEDIAN | CPL_STATS_MEDIAN_DEV, llx,
                            lly, urx, ury);
                        double median = cpl_stats_get_median(stats);
                        double sigma = cpl_stats_get_median_dev(stats);

                        cpl_stats_delete(stats);

                        if (fabs(flux - median) <=
                            params->local_kappa * sigma) {
                            cpl_mask_set(mask, i, j, CPL_BINARY_0);
                        }
                    }
                }
            }
        }
#ifdef _OPENMP
    }
#endif
    for (int j = 1; j <= ny; j++) {
        for (int i = 1; i <= nx; i++) {
            cpl_binary m = cpl_mask_get(mask, i, j);
            if (m == CPL_BINARY_1) {
                cpl_mask_set(data_bpm, i, j, CPL_BINARY_1);
            }
        }
    }
    moo_try_check(moo_single_apply_mask(single, mask, MOO_BADPIX_HOT), " ");
    moo_try_check(nb_hot = cpl_mask_count(mask), " ");

    moo_try_check(moo_qc_set_mdark_nhot(single->header, nb_hot), " ");

moo_try_cleanup:
    cpl_mask_delete(mask);
    cpl_image_delete(data);

    return nb_hot;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Calculates the hot bad pixel mask
  @param    det _DET_ Average dark frame
  @param    params hot map parameters
  @return   the error code

 * This function searches for hot bad pixels in a master dark frame by applying
 * an iterative kappa-sigma clipping algorithm.
  The QUAL extension of det frame is updated.
 *
 * _Bad pixels flags_:
  - BADPIX_HOT

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_compute_hot_map(moo_det *det, moo_hot_params *params)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();
    int badpix_level = MOO_BADPIX_COSMETIC | MOO_BADPIX_COSMIC_UNREMOVED |
                       MOO_BADPIX_OUTSIDE_DATA_RANGE;
    int i;

    cpl_msg_info(__func__, "Computing the hot map");
    cpl_msg_indent_more();

    for (i = 1; i <= 2; i++) {
        for (int extnum = 0; extnum < 3; extnum++) {
            moo_single *single =
                moo_det_load_single(det, extnum, i, badpix_level);
            if (single != NULL) {
                int nb_hot = 0;
                moo_try_check(nb_hot =
                                  _moo_single_compute_hot_map(single, params),
                              " ");
                cpl_msg_info(__func__, "Extension %s found %d hot pixels",
                             moo_detector_get_extname(extnum, i), nb_hot);
            }
        }
    }

moo_try_cleanup:
    cpl_msg_indent_less();

    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in compute hot map");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }

    return CPL_ERROR_NONE;
}
/**@}*/
