/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_badpix.h"
#include "moo_qc.h"
#include "moo_utils.h"
#include "moo_ext.h"
#include "moo_fibres_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_ext EXT format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_ext
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_ext.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_ext
  @return   1 newly allocated moo_ext or NULL in case of an error

  The returned object must be deallocated using moo_ext_delete().

 */
/*----------------------------------------------------------------------------*/
moo_ext *
moo_ext_new(void)
{
    moo_ext *res = cpl_calloc(1, sizeof(moo_ext));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty EXT filename
  @return   1 newly allocated moo_ext or NULL in case of an error

  The returned object must be deallocated using moo_ext_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_ext *
moo_ext_create(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_ext *res = moo_ext_new();
    const char *filename = cpl_frame_get_filename(frame);
    res->filename = filename;
    res->primary_header = cpl_propertylist_load(filename, 0);

    res->ri[0] = moo_ext_single_create(filename, MOO_TYPE_RI, 1);
    res->yj[0] = moo_ext_single_create(filename, MOO_TYPE_YJ, 1);
    res->h[0] = moo_ext_single_create(filename, MOO_TYPE_H, 1);
    res->ri[1] = moo_ext_single_create(filename, MOO_TYPE_RI, 2);
    res->yj[1] = moo_ext_single_create(filename, MOO_TYPE_YJ, 2);
    res->h[1] = moo_ext_single_create(filename, MOO_TYPE_H, 2);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_errorstate_set(prev_state);
        cpl_msg_error("moo_ext", "Can't create EXT from file : %s", filename);
        moo_ext_delete(res);
        res = NULL;
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext
  @brief    Load all parts in EXT
  @param    self    the EXT
  @param    level   the bad pixel level
  @return   error code or CPL_ERROR_NONE

  The returned object must be deallocated using moo_ext_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_load(moo_ext *self, unsigned int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    for (int i = 0; i < 2; i++) {
        moo_ext_single_load(self->ri[i], level);
        moo_ext_single_load(self->yj[i], level);
        moo_ext_single_load(self->h[i], level);
    }

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext
  @brief    Load the type part in EXT and return it
  @param    self    the EXT
  @param    type    the type of extension to load
  @param    num     the TAS number [1,2]
  @param    level   the bad pixel level

  @return  the moo_ext_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is not in range [1,2]
 *
 */
/*----------------------------------------------------------------------------*/
moo_ext_single *
moo_ext_load_single(moo_ext *self,
                    moo_detector_type type,
                    int num,
                    unsigned int level)
{
    moo_ext_single *res = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            moo_ext_single_load(self->ri[num - 1], level);
            res = self->ri[num - 1];
            break;
        case MOO_TYPE_YJ:
            moo_ext_single_load(self->yj[num - 1], level);
            res = self->yj[num - 1];
            break;
        case MOO_TYPE_H:
            moo_ext_single_load(self->h[num - 1], level);
            res = self->h[num - 1];
            break;
        default:
            res = NULL;
            break;
    }
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get a EXT single from EXT
  @param self moo_ext structure to update
  @param type type of detector
  @param ntas number of spectrographs
  @return   EXT_SINGLE

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_ext_single *
moo_ext_get_single(moo_ext *self, moo_detector_type type, int ntas)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            return self->ri[ntas - 1];
            break;
        case MOO_TYPE_YJ:
            return self->yj[ntas - 1];
            break;
        case MOO_TYPE_H:
            return self->h[ntas - 1];
            break;
        default:
            return NULL;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief assign moo_ext_single structure in moo_ext structure
  @param self moo_ext structure to update
  @param type type of detector
  @param ntas number of spectrographs
  @param single moo_ext_single structure

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_set_single(moo_ext *self,
                   moo_detector_type type,
                   int ntas,
                   moo_ext_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    switch (type) {
        case MOO_TYPE_RI:
            self->ri[ntas - 1] = single;
            break;
        case MOO_TYPE_YJ:
            self->yj[ntas - 1] = single;
            break;
        case MOO_TYPE_H:
            self->h[ntas - 1] = single;
            break;
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_ext", "Error for adding ext to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Add EXT_SINGLE extension to EXT filename and update moo_ext structure
  @param self moo_ext structure to update
  @param type type of detector
  @param ntas number of spectrographs
  @param single EXT_SINGLE extension to store on LOC file

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_add_single(moo_ext *self,
                   moo_ext_single *single,
                   moo_detector_type type,
                   int ntas)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_ext_set_single(self, type, ntas, single);
    moo_ext_single_save(single, self->filename, type, ntas);
    if (!cpl_errorstate_is_equal(prev_state)) {
        const char *extname = moo_detector_get_extname(type, ntas);
        cpl_msg_error("moo_ext", "Error for adding single to %s %s (%s)",
                      self->filename, extname,
                      cpl_error_get_message_default(cpl_error_get_code()));
        status = cpl_error_get_code();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add fibre table to EXT filename and update moo_ext structure
  @param self moo_ext structure to update
  @param fibre_table fibre table to store on EXT file

  The returned object must be deallocated using moo_ext_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_add_fibre_table(moo_ext *self, cpl_table *fibre_table)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fibre_table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    self->fibre_table = fibre_table;

    cpl_propertylist *h = cpl_propertylist_new();
    cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                   MOO_FIBRES_TABLE_EXTNAME);
    cpl_table_save(self->fibre_table, h, h, self->filename, CPL_IO_EXTEND);
    cpl_propertylist_delete(h);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_loc", "Error for adding fibre table to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get the FIBRE TABLE in EXT
  @param    self    the EXT
  @return   The FIBRE TABLE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_ext_get_fibre_table(moo_ext *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->fibre_table == NULL && self->filename != NULL) {
        cpl_size qnum =
            cpl_fits_find_extension(self->filename, MOO_FIBRES_TABLE_EXTNAME);
        if (qnum > 0) {
            self->fibre_table = cpl_table_load(self->filename, qnum, 0);
        }
    }
    return self->fibre_table;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_ext
  @param    self    moo_ext to delete
  @return   void

  If the moo_ext @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_ext_delete(moo_ext *self)
{
    if (self != NULL) {
        int i;
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 2; i++) {
            if (self->ri[i] != NULL) {
                moo_ext_single_delete(self->ri[i]);
            }
            if (self->yj[i] != NULL) {
                moo_ext_single_delete(self->yj[i]);
            }
            if (self->h[i] != NULL) {
                moo_ext_single_delete(self->h[i]);
            }
        }
        if (self->fibre_table != NULL) {
            cpl_table_delete(self->fibre_table);
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_ext to a FITS file
  @param    self          moo_ext to write to disk or NULL
  @param    filename      Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_ext to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_ext_save(moo_ext *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        int i;
        for (i = 0; i < 2; i++) {
            moo_ext_single_save(self->ri[i], filename, MOO_TYPE_RI, i + 1);
            moo_ext_single_save(self->yj[i], filename, MOO_TYPE_YJ, i + 1);
            moo_ext_single_save(self->h[i], filename, MOO_TYPE_H, i + 1);
        }

        if (self->fibre_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_FIBRES_TABLE_EXTNAME);
            cpl_table_save(self->fibre_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext
  @brief    Dump structural information of EXT
  @param    self    LOC to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_dump(const moo_ext *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_EXT\n");
    fprintf(stream, "filename %s\n", self->filename);
    int i;
    for (i = 0; i < 2; i++) {
        if (self->ri[i] != NULL) {
            moo_ext_single_dump(self->ri[i], stream);
        }
        if (self->yj[i] != NULL) {
            moo_ext_single_dump(self->yj[i], stream);
        }
        if (self->h[i] != NULL) {
            moo_ext_single_dump(self->h[i], stream);
        }
    }
    if (self->fibre_table) {
        fprintf(stream, "fibre table %p\n", self->fibre_table);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_det
  @brief    Free the given type part in EXT
  @param    self    the EXT
  @param    type    the type of extension to free
  @param    num     the TAS number
  @return   error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_free_single(moo_ext *self, moo_detector_type type, int num)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_error_code status = CPL_ERROR_NONE;

    moo_ext_single *single = moo_ext_get_single(self, type, num);
    if (single != NULL) {
        status = moo_ext_single_free(single);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext
  @brief    Sum EXT structure
  @param    self EXT result of the sum
  @param    ext EXT to sum
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_sum(moo_ext *self, moo_ext *ext)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ext != NULL, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        moo_ext_single *ri = self->ri[i];
        moo_ext_single *extri = ext->ri[i];
        if (ri != NULL) {
            moo_ext_single_sum(ri, extri);
        }
        moo_ext_single *yj = self->yj[i];
        if (yj != NULL) {
            moo_ext_single *extyj = ext->yj[i];
            moo_ext_single_sum(yj, extyj);
        }
        moo_ext_single *h = self->h[i];
        if (h != NULL) {
            moo_ext_single *exth = ext->h[i];
            moo_ext_single_sum(h, exth);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute SNR for all fibres EXT

  @param    self [EXT] the ext
  @param    wmap [MAP] the wavemap
  @param    sformat the spectral format
  @param    skylines SKY_LINE_LIST
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_compute_snr(moo_ext *self,
                    moo_map *wmap,
                    moo_spectral_format *sformat,
                    moo_sky_lines_list *skylines)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wmap != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skylines != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table *fibre_table = moo_ext_get_fibre_table(self);
    moo_qc_set_snr_range(self->primary_header, MOONS_QC_SNR_RANGE_OHFREEZONE);
    const char *snrs[] = { MOO_FIBRES_TABLE_MEDSNR_RI_EXT,
                           MOO_FIBRES_TABLE_MEDSNR_YJ_EXT,
                           MOO_FIBRES_TABLE_MEDSNR_H_EXT };

    const char *dersnrs[] = { MOO_FIBRES_TABLE_DERSNR_RI_EXT,
                              MOO_FIBRES_TABLE_DERSNR_YJ_EXT,
                              MOO_FIBRES_TABLE_DERSNR_H_EXT };

    int nrow = cpl_table_get_nrow(fibre_table);

    for (int t = 0; t < nrow; t++) {
        int indexext =
            cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_INDEXEXT, t, NULL);
        int health =
            cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_HEALTH, t, NULL);
        const int spectro =
            cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_SPECTRO, t, NULL);
        int num = 1;
        if (spectro == MOO_FIBRES_TABLE_SPECTRO_2) {
            num = 2;
        }
        if (health > 0) {
            for (int i = 0; i < 3; i++) {
                moo_ext_single *single =
                    moo_ext_load_single(self, i, num, MOO_BADPIX_GOOD);
                moo_spectral_format_info *sinfo =
                    moo_spectral_format_get(sformat, i, num);
                cpl_image *smap = wmap->data[i + (num - 1) * 3];
                if (single != NULL) {
                    double snr = 0.0;
                    double der_snr = 0.0;
                    snr = moo_ext_single_compute_snr(single, indexext, smap,
                                                     sinfo, skylines, &der_snr);

                    moo_try_check(cpl_table_set_float(fibre_table, snrs[i], t,
                                                      snr),
                                  " ");
                    moo_try_check(cpl_table_set_float(fibre_table, dersnrs[i],
                                                      t, der_snr),
                                  " ");
                }
                moo_spectral_format_info_delete(sinfo);
                sinfo = NULL;
            }
        }
    }

moo_try_cleanup:
    return CPL_ERROR_NONE;
}
/**@}*/
