/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_EXT_SINGLE_H
#define MOO_EXT_SINGLE_H


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include "moo_detector.h"
#include "moo_spectral_format.h"
#include "moo_sky_lines_list.h"
/*-----------------------------------------------------------------------------
                                   Types
 -----------------------------------------------------------------------------*/
typedef struct
{
    const char *filename;
    const char *extname;
    cpl_propertylist *header;
    hdrl_image *image;
    cpl_image *qual;
    unsigned int badpix_mask;
    moo_detector_type type;
    int ntas;
} moo_ext_single;

#define MOO_EXT_SINGLE_ERR       "ERR"
#define MOO_EXT_SINGLE_ERR_TYPE  CPL_TYPE_FLOAT
#define MOO_EXT_SINGLE_QUAL      "QUAL"
#define MOO_EXT_SINGLE_QUAL_TYPE CPL_TYPE_INT

#define MOO_EXT_SINGLE_CRPIX2    1
#define MOO_EXT_SINGLE_CRVAL2    1
#define MOO_EXT_SINGLE_CDELT2    1.0
#define MOO_EXT_SINGLE_CTYPE2    "INDEX"
/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
moo_ext_single *moo_ext_single_new(moo_detector_type type, int ntas);
moo_ext_single *
moo_ext_single_create(const char *filename, moo_detector_type type, int ntas);
cpl_error_code moo_ext_single_load(moo_ext_single *self, unsigned int level);

cpl_error_code moo_ext_single_free(moo_ext_single *self);
void moo_ext_single_delete(moo_ext_single *self);

hdrl_image *moo_ext_single_get_image(moo_ext_single *self);
cpl_image *moo_ext_single_get_data(moo_ext_single *self);
cpl_image *moo_ext_single_get_errs(moo_ext_single *self);
cpl_image *moo_ext_single_get_qual(moo_ext_single *self);
cpl_propertylist *moo_ext_single_get_header(moo_ext_single *self);
cpl_error_code moo_ext_single_compute_qc(moo_ext_single *self,
                                         const char *colname,
                                         const char *dersnr_colname,
                                         cpl_table *fibres_table);
cpl_error_code moo_ext_single_set_wcs1(moo_ext_single *self,
                                       double crpix1,
                                       double crval1,
                                       double cd1_1,
                                       const char *ctype1,
                                       const char *cunit1);
void moo_ext_single_delete(moo_ext_single *self);

void moo_ext_single_save(const moo_ext_single *self,
                         const char *filename,
                         moo_detector_type type,
                         int ntas);

cpl_error_code moo_ext_single_dump(const moo_ext_single *self, FILE *stream);
cpl_error_code moo_ext_single_sum(moo_ext_single *a, moo_ext_single *b);

double moo_ext_single_compute_snr(moo_ext_single *self,
                                  int ext_idx,
                                  cpl_image *wmap,
                                  moo_spectral_format_info *sinfo,
                                  moo_sky_lines_list *skylines,
                                  double *dersnr);
#endif
