/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <assert.h>
#include "moo_extlist.h"
#include "moo_badpix.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_extlist EXT_LIST format
 * @ingroup moo_data
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_extlist
  @return   1 newly allocated moo_extlist or NULL in case of an error

  The returned object must be deallocated using moo_extlist_delete().

 */
moo_extlist *
moo_extlist_new(void)
{
    return (moo_extlist *)cpl_calloc(1, sizeof(moo_extlist));
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_extlist from the given DET frameset
  @return   1 newly allocated moo_extlist or NULL in case of an error

  The returned object must be deallocated using moo_extlist_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
moo_extlist *
moo_extlist_create(cpl_frameset *frameset)
{
    cpl_ensure(frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int i;
    moo_extlist *res = moo_extlist_new();

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame =
            cpl_frameset_get_position_const(frameset, i);
        moo_ext *ext = moo_ext_create(current_frame);
        moo_extlist_push(res, ext);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Load the type part for all DET in the extlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @param    level   Mask error level
  @return   The error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_extlist_load_single(const moo_extlist *self,
                        moo_detector_type type,
                        int num,
                        int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT);

    int i;

    for (i = 0; i < self->size; i++) {
        moo_ext *ext = self->list[i];
        moo_ext_load_single(ext, type, num, level);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Free the type part for all DET in the extlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @return   The error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_extlist_free_single(const moo_extlist *self,
                        moo_detector_type type,
                        int num)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num >= 1 && num <= 2, CPL_ERROR_ILLEGAL_INPUT);

    int i;
    for (i = 0; i < self->size; i++) {
        moo_ext *ext = self->list[i];
        cpl_error_code status = moo_ext_free_single(ext, type, num);
        if (status != CPL_ERROR_NONE) {
            return status;
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Get the number of DET in the extlist
  @param    self    the list of DET
  @return   The number of DET or -1 on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_size
moo_extlist_get_size(const moo_extlist *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    assert(self->size >= 0);

    return self->size;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Get the DET at the position i in the list
  @param    self    the list of DET
  @param    i       position in the list
  @return   DET element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
moo_ext *
moo_extlist_get(moo_extlist *self, int i)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(i >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int size = moo_extlist_get_size(self);
    cpl_ensure(i < size, CPL_ERROR_ILLEGAL_INPUT, NULL);

    return self->list[i];
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Get the all the images of the type part in the extlist
  @param    self    the list of DET
  @param    type     the type of extension to load
  @param    num     the TAS number
  @return   The  image list or NULL

  TAS number is 1 or 2.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is num smaller than 1
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 2
 *
 */
/*----------------------------------------------------------------------------*/
hdrl_imagelist *
moo_extlist_get_image(const moo_extlist *self, moo_detector_type type, int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    hdrl_imagelist *res = NULL;
    int i;

    res = hdrl_imagelist_new();

    for (i = 0; i < self->size; i++) {
        moo_ext *ext = self->list[i];
        moo_ext_single *single = NULL;
        moo_try_check(single = moo_ext_get_single(ext, type, num), " ");

        if (single != NULL) {
            hdrl_image *img = moo_ext_single_get_image(single);
            hdrl_imagelist_set(res, img, i);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in extlist");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        hdrl_imagelist_delete(res);
        res = NULL;
    }
    return res;
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Get the type data part for all DET in the extlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @return   The type data list

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_extlist_get_single_data(const moo_extlist *self,
                            moo_detector_type type,
                            int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    cpl_imagelist *res = cpl_imagelist_new();
    int i;

    for (i = 0; i < self->size; i++) {
        moo_ext *ext = self->list[i];
        moo_ext_single *single = moo_ext_get_single(ext, type, num);
        if (single != NULL) {
            cpl_imagelist_set(res, moo_ext_single_get_data(single), i);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Get the type QUAL part for all DET in the extlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @param    num     the TAS number
  @return   The type QUAL list

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_extlist_get_single_qual(const moo_extlist *self,
                            moo_detector_type type,
                            int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(num <= 2, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    cpl_imagelist *res = cpl_imagelist_new();
    int i;

    for (i = 0; i < self->size; i++) {
        moo_ext *ext = self->list[i];
        moo_ext_single *single = moo_ext_get_single(ext, type, num);
        if (single != NULL) {
            cpl_imagelist_set(res, moo_ext_single_get_qual(single), i);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Insert a DET into an moo_extlist
  @param    self    The moo_extlist
  @param    ext     The DET to insert
  @param    pos     The list position (from 0 to number of DET)
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  It is allowed to specify the position equal to the number of DET in the
  list. This will increment the size of the extlist.

  No action occurs if a DET is inserted more than once into the same
  position. It is allowed to insert the same DET into two different
  positions in a list.

  The DET is inserted at the position pos in the DET list. If the DET
  already there is only present in that one location in the list, then the
  DET is deallocated.

  The added DET is owned by the moo_extlist object, which deallocates it
  moo_extlist_delete is called. Other option is to use moo_extlist_unset to
  recover ownership of the DET, in which case the moo_extlist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    DET in extlist
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_extlist_set(moo_extlist *self, moo_ext *ext, cpl_size pos)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ext, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pos >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(pos <= self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Do nothing if the ext is already there */
    if (pos < self->size && ext == self->list[pos])
        return CPL_ERROR_NONE;

    if (pos == self->size) {
        self->size++;
        self->list =
            cpl_realloc(self->list, (size_t)self->size * sizeof(moo_ext *));
    }
    else {
        /* Check if the ext at the position to be overwritten
           is present in only one position */
        int i;

        for (i = 0; i < self->size; i++) {
            if (i != pos && self->list[i] == self->list[pos])
                break;
        }
        if (i == self->size) {
            /* The image at the position to be overwritten
               is present in only one position, so delete it */
            moo_ext_delete(self->list[pos]);
        }
    }

    self->list[pos] = ext;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Insert a DET a the end of moo_extlist
  @param    self    The moo_extlist
  @param    ext     The DET to insert
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
  @see  moo_extlist_set()

  No action occurs if a DET is inserted more than once into the same
  position. It is allowed to insert the same DET into two different
  positions in a list.

  The DET is inserted at the end in the DET list. If the DET
  already there is only present in that one location in the list, then the
  DET is deallocated.

  The added DET is owned by the moo_extlist object, which deallocates it
  moo_extlist_delete is called. Other option is to use moo_extlist_unset to
  recover ownership of the DET, in which case the moo_extlist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_extlist_push(moo_extlist *self, moo_ext *ext)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    moo_extlist_set(self, ext, self->size);
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup  moo_extlist
  @brief    Remove a DET from a DET list
  @param    self   The moo_extlist
  @param    pos      The list position (from 0 to number of DET-1)
  @return   The pointer to the removed DET or NULL in error case

  The specified DET is not deallocated, it is simply removed from the
  list. The pointer to the DET is returned to let the user decide to
  deallocate it or not.
  Eventually, the DET will have to be deallocated with moo_ext_delete().

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    images in self
 */
/*----------------------------------------------------------------------------*/
moo_ext *
moo_extlist_unset(moo_extlist *self, cpl_size pos)
{
    moo_ext *out;
    cpl_size i;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pos >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(pos < self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    /* Get pointer to DET to be removed */
    out = self->list[pos];

    /* Move the following DET one position towards zero */
    for (i = pos + 1; i < self->size; i++) {
        self->list[i - 1] = self->list[i];
    }

    /* Decrement of the size */
    self->size--;

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Empty an moo_extlist and deallocate all its DET
  @param    self  The DET list or NULL
  @return   Nothing
  @see  moo_extlist_empty(), moo_extlist_delete()
  @note If @em self is @c NULL nothing is done and no error is set.

  After the call the moo_ext list can be populated again. It must eventually
  be deallocted with a call to moo_extlist_delete().

 */
/*----------------------------------------------------------------------------*/
void
moo_extlist_empty(moo_extlist *self)
{
    if (self != NULL) {
        while (self->size > 0) { /* An iteration may unset more than 1 image */
            int i = self->size - 1;
            moo_ext *del = moo_extlist_unset(self, i);

            moo_ext_delete(del);

            /* If this image was inserted more than once into the list,
               the other insertions must be unset without a delete. */
            while (--i >= 0) {
                if (self->list[i] == del) {
                    /* This image was inserted more than once in the list */
                    (void)moo_extlist_unset(self, i);
                }
            }
        }
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_extlist
  @brief    Free memory used by a moo_extlist object, except the DET
  @param    self    The DET list or NULL
  @return   Nothing
  @see moo_extlist_empty()
  @note The caller must have pointers to all DET in the list and is
        reponsible for their deallocation. If @em self is @c NULL nothing is
        done and no error is set.
 */
/*----------------------------------------------------------------------------*/
void
moo_extlist_unwrap(moo_extlist *self)
{
    if (self != NULL) {
        cpl_free(self->list);
        cpl_free(self);
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free all memory used by a moo_extlist object including the DET
  @param    self    The DET list or NULL
  @return   void
  @see      moo_extlist_empty(), moo_extlist_unwrap()
  If the moo_extlist @em self is @c NULL, nothing is done and no error is set.

 */
void
moo_extlist_delete(moo_extlist *self)
{
    if (self != NULL) {
        moo_extlist_empty(self);
        moo_extlist_unwrap(self);
    }

    return;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Sum EXT element from the list and save result in filename
  @param    self    The EXT list to sum
  @param    filename filename to save the new EXT result
  @return   EXT result of the sum
  @see      moo_extlist_empty(), moo_extlist_unwrap()
  If the moo_extlist @em self is @c NULL, nothing is done and no error is set.

 */
moo_ext *
moo_extlist_sum(moo_extlist *self, const char *filename)
{
    moo_ext *result = NULL;
    cpl_frame *res_frame = NULL;
    int size = 0;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    moo_try_check(size = moo_extlist_get_size(self), " ");
    if (size > 0) {
        result = moo_extlist_get(self, 0);
        moo_ext_load(result, MOO_BADPIX_GOOD);
        moo_ext_get_fibre_table(result);
        moo_try_check(moo_ext_save(result, filename), " ");

        res_frame = cpl_frame_new();
        cpl_frame_set_filename(res_frame, filename);
        result = moo_ext_create(res_frame);
        moo_ext_load(result, MOO_BADPIX_GOOD);
        moo_ext_get_fibre_table(result);
        for (int i = 1; i < size; i++) {
            moo_ext *ext = moo_extlist_get(self, i);
            moo_ext_load(ext, MOO_BADPIX_GOOD);
            moo_ext_sum(result, ext);
        }
    }
moo_try_cleanup:
    cpl_frame_delete(res_frame);
    return result;
}
/**@}*/
