/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_FIBRES_TABLE_H
#define MOO_FIBRES_TABLE_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include <moo_detector.h>
/*-----------------------------------------------------------------------------
                                   Types
 -----------------------------------------------------------------------------*/
typedef struct
{
    cpl_table *table;
} moo_fibres_table;

#define MOO_FIBRES_TABLE_EXTNAME             "FIBRE_TABLE"
#define MOO_FIBRES_TABLE_INDEX               "INDEX"
#define MOO_FIBRES_TABLE_INDEXEXT            "INDEXEXT"
#define MOO_FIBRES_TABLE_INDEXRBN            "INDEXRBN"

#define MOO_FIBRES_TABLE_FIBRE               "FIBRE"
#define MOO_FIBRES_TABLE_PAIREDFIBRE         "PAIREDINDEX"
#define MOO_FIBRES_TABLE_HEALTH              "HEALTH"
#define MOO_FIBRES_TABLE_HEALTH_BROKEN       0
#define MOO_FIBRES_TABLE_HEALTH_GOOD         1
#define MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN 2
#define MOO_FIBRES_TABLE_SPECTRO             "SPECTRO"
#define MOO_FIBRES_TABLE_SLITLET             "SLITLET"
#define MOO_FIBRES_TABLE_TARGNAME            "TARGNAME"
#define MOO_FIBRES_TABLE_TARGALPHA           "TARGALPHA"
#define MOO_FIBRES_TABLE_TARGDELTA           "TARGDELTA"
#define MOO_FIBRES_TABLE_TYPE                "TYPE"


/*
    SCIENCE      -> T
    SKY          -> S
    SKY_CONTAM   -> C
    BLANK        -> V
    CHECK_OBJECT -> Q
    FLUX,STD     -> P
    TELL,STD     -> K
*/
#define MOO_FIBRES_TABLE_TYPE_CHECKOBJ   "Q"
#define MOO_FIBRES_TABLE_TYPE_SCIENCE    "T"
#define MOO_FIBRES_TABLE_TYPE_SKY        "S"
#define MOO_FIBRES_TABLE_TYPE_BLANK      "V"
#define MOO_FIBRES_TABLE_TYPE_SKYCONTAM  "C"
#define MOO_FIBRES_TABLE_TYPE_STD_FLUX   "P"
#define MOO_FIBRES_TABLE_TYPE_STD_TELL   "K"

#define MOO_FIBRES_TABLE_SPECTRO_1       1
#define MOO_FIBRES_TABLE_SPECTRO_2       2

#define MOO_FIBRES_TABLE_TYPE_NOD        "TYPE_NOD"
#define MOO_FIBRES_TABLE_TRANS_RI        "TRANS_RI"
#define MOO_FIBRES_TABLE_TRANS_YJ        "TRANS_YJ"
#define MOO_FIBRES_TABLE_TRANS_H         "TRANS_H"

#define MOO_FIBRES_TABLE_GOODPTSFRAC_RI  "GOODPTSFRAC_RI"
#define MOO_FIBRES_TABLE_GOODPTSFRAC_YJ  "GOODPTSFRAC_YJ"
#define MOO_FIBRES_TABLE_GOODPTSFRAC_H   "GOODPTSFRAC_H"

#define MOO_FIBRES_TABLE_MEDSNR_RI_EXT   "MEDIAN_SNR_RI_EXT"
#define MOO_FIBRES_TABLE_MEDSNR_YJ_EXT   "MEDIAN_SNR_YJ_EXT"
#define MOO_FIBRES_TABLE_MEDSNR_H_EXT    "MEDIAN_SNR_H_EXT"

#define MOO_FIBRES_TABLE_DERSNR_RI_EXT   "DERSNR_RI_EXT"
#define MOO_FIBRES_TABLE_DERSNR_YJ_EXT   "DERSNR_YJ_EXT"
#define MOO_FIBRES_TABLE_DERSNR_H_EXT    "DERSNR_H_EXT"

#define MOO_FIBRES_TABLE_MEDSNR_RI_RBN   "MEDIAN_SNR_RI_RBN"
#define MOO_FIBRES_TABLE_MEDSNR_YJ_RBN   "MEDIAN_SNR_YJ_RBN"
#define MOO_FIBRES_TABLE_MEDSNR_H_RBN    "MEDIAN_SNR_H_RBN"

#define MOO_FIBRES_TABLE_DETECTLINE_RI   "DETECTLINE_RI"
#define MOO_FIBRES_TABLE_DETECTLINE_YJ   "DETECTLINE_YJ"
#define MOO_FIBRES_TABLE_DETECTLINE_H    "DETECTLINE_H"

#define MOO_FIBRES_TABLE_FAILEDFIT_RI    "FAILEDFIT_RI"
#define MOO_FIBRES_TABLE_FAILEDFIT_YJ    "FAILEDFIT_YJ"
#define MOO_FIBRES_TABLE_FAILEDFIT_H     "FAILEDFIT_H"

#define MOO_FIBRES_TABLE_FITLINE_RI      "FITLINE_RI"
#define MOO_FIBRES_TABLE_FITLINE_YJ      "FITLINE_YJ"
#define MOO_FIBRES_TABLE_FITLINE_H       "FITLINE_H"

#define MOO_FIBRES_TABLE_MATCHLINE_RI    "MATCHLINE_RI"
#define MOO_FIBRES_TABLE_MATCHLINE_YJ    "MATCHLINE_YJ"
#define MOO_FIBRES_TABLE_MATCHLINE_H     "MATCHLINE_H"

#define MOO_FIBRES_TABLE_MODELLINE_RI    "MODELLINE_RI"
#define MOO_FIBRES_TABLE_MODELLINE_YJ    "MODELLINE_YJ"
#define MOO_FIBRES_TABLE_MODELLINE_H     "MODELLINE_H"

#define MOO_FIBRES_TABLE_REJECTLINE_RI   "REJECTLINE_RI"
#define MOO_FIBRES_TABLE_REJECTLINE_YJ   "REJECTLINE_YJ"
#define MOO_FIBRES_TABLE_REJECTLINE_H    "REJECTLINE_H"

#define MOO_FIBRES_TABLE_XDIFF_MEDIAN_RI "XDIFF_MEDIAN_RI"
#define MOO_FIBRES_TABLE_XDIFF_MEDIAN_YJ "XDIFF_MEDIAN_YJ"
#define MOO_FIBRES_TABLE_XDIFF_MEDIAN_H  "XDIFF_MEDIAN_H"

#define MOO_FIBRES_TABLE_MONOTONOUS_RI   "MONOTONOUS_RI"
#define MOO_FIBRES_TABLE_MONOTONOUS_YJ   "MONOTONOUS_YJ"
#define MOO_FIBRES_TABLE_MONOTONOUS_H    "MONOTONOUS_H"

#define MOO_FIBRES_TABLE_MAG             "MAG"


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
cpl_array *moo_fibres_table_get_spectro_index(cpl_table *table, int num);
cpl_array *moo_fibres_table_get_spectro_indexext(cpl_table *table, int num);
cpl_table *moo_fibres_table_get_spectro_table(cpl_table *table, int num);
int moo_fibres_table_get_spectro_nrow(cpl_table *table, int num);

cpl_error_code moo_fibres_table_add_locguess_cols(cpl_table *table);
cpl_error_code moo_fibres_table_add_extract_cols(cpl_table *table);
cpl_error_code moo_fibres_table_erase_extract_cols(cpl_table *table);
cpl_error_code moo_fibres_table_add_f2f_cols(cpl_table *table);
cpl_error_code moo_fibres_table_add_wavecal_cols(cpl_table *table,
                                                 cpl_table *guess,
                                                 const char *model);
cpl_error_code moo_fibres_table_add_rebin_cols(cpl_table *table);
cpl_error_code moo_fibres_table_add_wavecalguess_cols(cpl_table *table);
cpl_error_code moo_fibres_table_add_nod_cols(cpl_table *table);

float *
moo_fibres_table_get_median_snr(cpl_table *table, moo_detector_type type);

cpl_error_code moo_fibres_table_set_detectline(cpl_table *table,
                                               moo_detector_type type,
                                               cpl_size row,
                                               int value);
cpl_error_code moo_fibres_table_set_failedfit(cpl_table *table,
                                              moo_detector_type type,
                                              cpl_size row,
                                              int value);
cpl_error_code moo_fibres_table_set_fitline(cpl_table *table,
                                            moo_detector_type type,
                                            cpl_size row,
                                            int value);
cpl_error_code moo_fibres_table_set_matchline(cpl_table *table,
                                              moo_detector_type type,
                                              cpl_size row,
                                              int value);
cpl_error_code moo_fibres_table_set_modelline(cpl_table *table,
                                              moo_detector_type type,
                                              cpl_size row,
                                              int value);
cpl_error_code moo_fibres_table_set_rejectline(cpl_table *table,
                                               moo_detector_type type,
                                               cpl_size row,
                                               int value);
cpl_error_code moo_fibres_table_set_xdiff(cpl_table *table,
                                          moo_detector_type type,
                                          cpl_size row,
                                          double value);

int moo_fibres_table_get_matchline(cpl_table *table,
                                   moo_detector_type type,
                                   cpl_size row);

int moo_fibres_table_get_rejectline(cpl_table *table,
                                    moo_detector_type type,
                                    cpl_size row);
double moo_fibres_table_get_xdiff(cpl_table *table,
                                  moo_detector_type type,
                                  cpl_size row);
int moo_fibres_table_filter_health(cpl_table *table);

cpl_error_code moo_fibres_table_rbn_to_ext(cpl_table *table,
                                           int rbnindex,
                                           int *extindex,
                                           int *num);

cpl_table *moo_fibre_table_load(cpl_frame *frame);
int moo_fibres_table_get_spectro(int num);
const char *moo_fibres_table_get_std_name(cpl_table *table);
cpl_error_code moo_fibres_table_by_index(cpl_table *table);
cpl_error_code moo_fibres_table_by_indexrbn(cpl_table *table);
#endif
