/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <string.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_utils.h"
#include "moo_kernel.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_line_table LINE TABLE format
 * @ingroup moo_data
 * This module provides functions to use LINE TABLE
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_map.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_kernel
  @return   1 newly allocated moo_kernel or NULL in case of an error

  The returned object must be deallocated using moo_kernel_delete().

 */
/*----------------------------------------------------------------------------*/
moo_kernel* moo_kernel_new(void)
{
    moo_kernel* res = cpl_calloc(1,sizeof(moo_kernel));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a new moo_kernel
  @return   1 newly allocated moo_kernel or NULL in case of an error

  The returned object must be deallocated using moo_kernel_delete().
 */
/*----------------------------------------------------------------------------*/
moo_kernel* moo_kernel_load(const cpl_frame* frame)
{
  moo_kernel* res = NULL;
  
  if(frame!=NULL){
    const char* filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT,
      NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    
    res = moo_kernel_new();
    res->filename = cpl_strdup(filename);
    
    if ( !cpl_errorstate_is_equal(prev_state)){
      moo_kernel_delete(res);
      res = NULL;
    }
  }
  return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_kernel
  @param    self    moo_kernel to delete
  @return   void

  If the moo_kernel @em self is @c NULL, nothing is done and no error is set.

*/
/*----------------------------------------------------------------------------*/

void moo_kernel_delete(moo_kernel* self)
{
  if ( self!= NULL){        
    cpl_free(self->filename);
    cpl_free(self);
  }
}

static char* _moo_get_extname(moo_detector_type type, int ntas,int rbn)
{
  char* extname = NULL;
  const char* detector = moo_detector_get_name(type);  
  extname = cpl_sprintf(MOO_KERNEL_EXTNAME,detector,
    ntas,rbn);
  return extname;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get the header of the kernel for given idrbn or NULL
  @param    self    moo_kernel to delete
  @param    type    the detector type
  @param    ntas    the number of tas
  @param    rbn    the idrbn of the fibre
  @return   the HEADER of the given kernel or NULL

  If the moo_kernel @em self is @c NULL, nothing is done and no error is set.

*/
/*----------------------------------------------------------------------------*/
cpl_propertylist* moo_kernel_get_header(moo_kernel* self,
    moo_detector_type type, int ntas,int rbn)
{
  cpl_propertylist* header = NULL;

  if(self!=NULL){
    char* extname = _moo_get_extname(type,ntas,rbn);
    if(extname!=NULL){
      header = moo_fits_load_extension_header(self->filename,NULL,extname);  
    }
    cpl_free(extname);
  }  
  return header;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the mtarix of the kernel for given idrbn or NULL
  @param    self    moo_kernel to delete
  @param    type    the detector type
  @param    ntas    the number of tas
  @param    rbn    the idrbn of the fibre
  @return   the matrix of the given kernel or NULL

  If the moo_kernel @em self is @c NULL, nothing is done and no error is set.

*/
/*----------------------------------------------------------------------------*/
cpl_matrix* moo_kernel_get_matrix(moo_kernel* self,
    moo_detector_type type, int ntas,int rbn)
{
  cpl_image* img = NULL;
  cpl_matrix* matrix = NULL;
  
  if(self!=NULL){
    char* extname = _moo_get_extname(type,ntas,rbn);
        
    if(extname!=NULL){
      cpl_size extnum = cpl_fits_find_extension(self->filename, extname);
      if(extnum >0){
        img = cpl_image_load(self->filename, CPL_TYPE_DOUBLE, 0, extnum);  
        cpl_matrix *mat = cpl_matrix_wrap( 
          cpl_image_get_size_y(img),
          cpl_image_get_size_x(img),
          cpl_image_get_data_double(img));
        matrix = cpl_matrix_duplicate(mat);
        
        cpl_matrix_unwrap(mat);
        cpl_image_delete(img);
      }
    }                    
    cpl_free(extname);
  }  
  return matrix;
}
/**@}*/
