/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <string.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_utils.h"
#include "moo_line_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_line_table LINE TABLE format
 * @ingroup moo_data
 * This module provides functions to use LINE TABLE
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_map.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief create a line table  
  @param isGuess TRUE if is a guess line table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_line_table_new(int isGuess)
{
    cpl_table *table = cpl_table_new(0);

    cpl_table_new_column(table, MOO_LINE_TABLE_NAME, CPL_TYPE_STRING);
    cpl_table_new_column(table, MOO_LINE_TABLE_WAVE, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_LINE_TABLE_FLUX, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_LINE_TABLE_SPECTRO, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_LINE_TABLE_DETECTOR, CPL_TYPE_STRING);
    cpl_table_new_column(table, MOO_LINE_TABLE_SLITLET, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_LINE_TABLE_INDEXEXT, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_LINE_TABLE_INDEXRBN, CPL_TYPE_INT);

    cpl_errorstate prestate = cpl_errorstate_get();
    if (isGuess) {
        cpl_table_new_column(table, MOO_LINE_TABLE_INTENSITY, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_XGAUSS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_XBARY, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_YLOC, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_AMPLITUDE, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_BACKGROUND, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FWHM, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_FLUX, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_ERR, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_MSE, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_CHI2, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_MATCHED, CPL_TYPE_INT);
        cpl_table_new_column(table, MOO_LINE_TABLE_DISP, CPL_TYPE_DOUBLE);
    }
    else {
        cpl_table_new_column(table, MOO_LINE_TABLE_XGUESS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_INTENSITY, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_GFITTED, CPL_TYPE_INT);
        cpl_table_new_column(table, MOO_LINE_TABLE_XGAUSS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_XBARY, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_XDIFF, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_WAVEDIFF, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_YLOC, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_AMPLITUDE, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_BACKGROUND, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FWHM, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_FLUX, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_ERR, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_MSE, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FIT_CHI2, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, MOO_LINE_TABLE_FILTERED, CPL_TYPE_INT);
    }
    cpl_table_new_column(table, MOO_LINE_TABLE_MFITTED, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_LINE_TABLE_WAVEFIT, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_LINE_TABLE_RESOLUTION, CPL_TYPE_DOUBLE);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_table_delete(table);
        table = NULL;
    }
    return table;
}

cpl_table *
moo_slitlet_shift_table_new(int size)
{
    cpl_table *detected_table = cpl_table_new(size);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_table_new_column(detected_table, MOO_SLITLET_SHIFT_TABLE_SLITLET,
                         CPL_TYPE_INT);
    cpl_table_new_column(detected_table, MOO_SLITLET_SHIFT_TABLE_WAVE,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(detected_table, MOO_SLITLET_SHIFT_TABLE_XDIFF,
                         CPL_TYPE_DOUBLE);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_table_delete(detected_table);
        detected_table = NULL;
    }
    return detected_table;
}

cpl_table *
moo_detected_table_new(int size, int isGuess)
{
    cpl_table *detected_table = cpl_table_new(size);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_table_new_column(detected_table, MOO_LINE_TABLE_INDEXEXT, CPL_TYPE_INT);
    cpl_table_new_column(detected_table, MOO_LINE_TABLE_SLITLET, CPL_TYPE_INT);

    if (isGuess) {
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_MIN,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_MAX,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_WIN_CEN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_NGOOD,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_INTENSITY,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_BADFIT,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_FLUX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_ERR,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_MSE,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_CHI2,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_BACKGROUND,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_XGAUSS,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_AMPLITUDE,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_BACKGROUND,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_XBARY,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_FWHM,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_FILTERED,
                             CPL_TYPE_INT);
    }
    else {
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_WAVE,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_XGUESS,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_XDIFF,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_NOT_IN_RANGE,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_WIN_CEN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_WIN_MIN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_WIN_MAX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_INTENSITY,
                             CPL_TYPE_DOUBLE);

        cpl_table_new_column(detected_table, MOO_LINE_TABLE_GFITTED,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_XGAUSS,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_AMPLITUDE,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_BACKGROUND,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_FLUX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_ERR,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_MSE,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_FIT_CHI2,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_XBARY,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_FWHM,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detected_table, MOO_LINE_TABLE_FILTERED,
                             CPL_TYPE_INT);
        cpl_table_new_column(detected_table, MOO_DETECTED_TABLE_WAVETH,
                             CPL_TYPE_DOUBLE);
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_table_delete(detected_table);
        detected_table = NULL;
    }
    return detected_table;
}

cpl_error_code
moo_table_append(cpl_table *global_line_table, cpl_table *line_table)
{
    cpl_ensure_code(global_line_table != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(line_table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_size nrow = cpl_table_get_nrow(global_line_table);
    cpl_error_code status =
        cpl_table_insert(global_line_table, line_table, nrow);

    return status;
}
/**@}*/
