/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_pfits.h"
#include "moo_loc.h"
#include "moo_fibres_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_loc LOC format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_loc
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_loc.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_loc
  @return   1 newly allocated moo_loc or NULL in case of an error

  The returned object must be deallocated using moo_det_delete().

 */
/*----------------------------------------------------------------------------*/
moo_loc *
moo_loc_new(void)
{
    moo_loc *res = cpl_calloc(1, sizeof(moo_loc));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty LOC filename
  @return   1 newly allocated moo_loc or NULL in case of an error

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_loc *
moo_loc_create(const char *filename)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_loc *res = moo_loc_new();
    res->primary_header = cpl_propertylist_new();
    res->filename = cpl_strdup(filename);
    cpl_propertylist_save(res->primary_header, filename, CPL_IO_CREATE);
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_loc", "Can't load filename %s (%s)", filename,
                      cpl_error_get_message());
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a LOC frame and create a moo_loc
  @return   1 newly allocated moo_loc or NULL in case of an error

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_loc *
moo_loc_load(const cpl_frame *locframe)
{
    cpl_ensure(locframe != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(locframe);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_loc *res = moo_loc_new();

    res->filename = cpl_strdup(filename);
    res->primary_header = cpl_propertylist_load(filename, 0);

    for (int i = 1; i <= 2; i++) {
        res->ri[i - 1] =
            moo_loc_single_create(res->filename,
                                  moo_detector_get_extname(MOO_TYPE_RI, i));
        res->yj[i - 1] =
            moo_loc_single_create(res->filename,
                                  moo_detector_get_extname(MOO_TYPE_YJ, i));
        res->h[i - 1] =
            moo_loc_single_create(res->filename,
                                  moo_detector_get_extname(MOO_TYPE_H, i));
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        const char *tag = cpl_frame_get_tag(locframe);
        cpl_msg_error("moo_loc", "Cant' load %s (%s) error code : %d", filename,
                      tag, cpl_error_get_code());
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the type part in LOC and return it
  @param    self    the LOC
  @param    type    the type of detector to get
  @param    ntas     the TAS number [1,2]

  @return  the loc_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is not in range [1,2]
 */
/*----------------------------------------------------------------------------*/
moo_loc_single *
moo_loc_get_single(moo_loc *self, moo_detector_type type, int ntas)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            return self->ri[ntas - 1];
            break;
        case MOO_TYPE_YJ:
            return self->yj[ntas - 1];
            break;
        case MOO_TYPE_H:
            return self->h[ntas - 1];
            break;
        default:
            return NULL;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief assign moo_loc_single structure in moo_loc structure
  @param self moo_loc structure to update
  @param type type of detector
  @param ntas number of spectrographs
  @param single moo_loc_single structure

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_loc_set_single(moo_loc *self,
                   moo_detector_type type,
                   int ntas,
                   moo_loc_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    switch (type) {
        case MOO_TYPE_RI:
            self->ri[ntas - 1] = single;
            break;
        case MOO_TYPE_YJ:
            self->yj[ntas - 1] = single;
            break;
        case MOO_TYPE_H:
            self->h[ntas - 1] = single;
            break;
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_loc", "Error for adding fibre table to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Add LOC_SINGLE extension to LOC filename and update moo_loc structure
  @param self moo_loc structure to update
  @param type type of detector
  @param ntas number of spectrographs
  @param keep_points keep measured points
  @param single LOC_SINGLE extension to store on LOC file

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_loc_add_single(moo_loc *self,
                   moo_loc_single *single,
                   moo_detector_type type,
                   int ntas,
                   int keep_points)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_loc_set_single(self, type, ntas, single);
    moo_loc_single_save(single, self->filename, keep_points);
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_errorstate_dump(prev_state, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_msg_error("moo_loc", "Error for adding LOC single table to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add fibre table to LOC filename and update moo_loc structure
  @param self moo_loc structure to update
  @param fibre_table fibre table to store on LOC file

  The returned object must be deallocated using moo_loc_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_loc_add_fibre_table(moo_loc *self, cpl_table *fibre_table)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fibre_table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    self->fibre_table = fibre_table;

    cpl_propertylist *h = cpl_propertylist_new();
    cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                   MOO_FIBRES_TABLE_EXTNAME);
    cpl_table_save(self->fibre_table, NULL, h, self->filename, CPL_IO_EXTEND);
    cpl_propertylist_delete(h);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_loc", "Error for adding fibre table to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get the FIBRE TABLE in LOC
  @param    self    the LOC
  @return   The FIBRE TABLE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_loc_get_fibre_table(moo_loc *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->fibre_table == NULL && self->filename != NULL) {
        cpl_size qnum =
            cpl_fits_find_extension(self->filename, MOO_FIBRES_TABLE_EXTNAME);
        if (qnum > 0) {
            self->fibre_table = cpl_table_load(self->filename, qnum, 0);
        }
    }
    return self->fibre_table;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_loc
  @param    self    moo_loc to delete
  @return   void

  If the moo_loc @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_loc_delete(moo_loc *self)
{
    if (self != NULL) {
        int i;
        if (self->filename != NULL) {
            cpl_free(self->filename);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 2; i++) {
            if (self->ri[i] != NULL) {
                moo_loc_single_delete(self->ri[i]);
            }
            if (self->yj[i] != NULL) {
                moo_loc_single_delete(self->yj[i]);
            }
            if (self->h[i] != NULL) {
                moo_loc_single_delete(self->h[i]);
            }
        }
        if (self->fibre_table != NULL) {
            cpl_table_delete(self->fibre_table);
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_loc to a FITS file
  @param    self          moo_loc to write to disk or NULL
  @param    filename      Name of the file to write
  @param    keep_points   1 to save the localise points use for the fit or 0 if not
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_loc to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_loc_save(moo_loc *self, const char *filename, int keep_points)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);

        int i;
        for (i = 0; i < 2; i++) {
            if (self->ri[i] != NULL) {
                moo_loc_single_save(self->ri[i], filename, keep_points);
            }
            if (self->yj[i] != NULL) {
                moo_loc_single_save(self->yj[i], filename, keep_points);
            }
            if (self->h[i] != NULL) {
                moo_loc_single_save(self->h[i], filename, keep_points);
            }
        }

        if (self->fibre_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_FIBRES_TABLE_EXTNAME);
            cpl_table_save(self->fibre_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Dump structural information of LOC
  @param    self    LOC to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_loc_dump(const moo_loc *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_LOC\n");
    fprintf(stream, "filename %s\n", self->filename);
    int i;
    for (i = 0; i < 2; i++) {
        moo_loc_single_dump(self->ri[i], stream);
        moo_loc_single_dump(self->yj[i], stream);
        moo_loc_single_dump(self->h[i], stream);
    }

    return CPL_ERROR_NONE;
}

/**@}*/
