/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_dfs.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_params.h"
#include "moo_badpix.h"
#include "moo_single.h"
#include "moo_det.h"
#include "moo_localise.h"
#include "moo_loc_single.h"
#include "moo_detector.h"
#include "moo_fibres_table.h"
#include "moo_line_table.h"

#ifdef _OPENMP
#include <omp.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

typedef struct
{
    double center;
    double width;
    int ymin;
    int ymax;
    int num;
    int health;
    float goodfrac;
} fibre;
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the flux for given position
  @param    fluxes the vector containing flux values (NOT NULL)
  @param    posy Y position (FITS) to evaluate

  @return   the flux a posy position

 */
/*----------------------------------------------------------------------------*/
static double
_moo_get_flux(cpl_vector *fluxes, double posy)
{
    double flux;

    int nx = cpl_vector_get_size(fluxes);
    int pos1 = (int)floor(posy);
    int pos2 = pos1;

    double flux1 = cpl_vector_get(fluxes, pos1 - 1);
    while (isnan(flux1)) {
        pos1--;
        flux1 = cpl_vector_get(fluxes, pos1 - 1);
    }
    if (pos1 == nx) {
        flux = flux1;
    }
    else {
        double flux2 = NAN;
        while (isnan(flux2)) {
            pos2++;
            flux2 = cpl_vector_get(fluxes, pos2 - 1);
        }
        flux = flux1 + (posy - pos1) * (flux2 - flux1);
    }
    return flux;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the flux for given position
  @param    fluxes the image containing flux values (NOT NULL)
  @param    posy Y position (FITS) to evaluate

  @return   the flux a posy position

 */
/*----------------------------------------------------------------------------*/
static double
_moo_get_image_flux(const cpl_image *fluxes, int x, double posy)
{
    double flux;
    int rej;
    int ny = cpl_image_get_size_y(fluxes);
    int pos1 = (int)floor(posy);
    int pos2 = pos1;

    double flux1 = cpl_image_get(fluxes, x, pos1, &rej);
    while (rej == 1) {
        pos1--;
        flux1 = cpl_image_get(fluxes, x, pos1, &rej);
    }
    if (pos1 == ny) {
        flux = flux1;
    }
    else {
        double flux2;
        rej = 1;
        while (rej == 1) {
            pos2++;
            flux2 = cpl_image_get(fluxes, x, pos2, &rej);
        }
        double diff = pos2 - pos1;
        flux = (pos2 - posy) / diff * flux1 + (posy - pos1) / diff * flux2;
    }
    return flux;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get the lower integrate flux for given position
  @param    fluxes the vector containing flux values (NOT NULL)
  @param    posy Y position (FITS) to evaluate
  @param  cbin center of bin in Y position (FITS) to evaluate
  @return   the flux a posy position

 */
/*----------------------------------------------------------------------------*/
static double
_moo_get_flux_integrate_low(cpl_vector *fluxes, double posy, double *cbin)
{
    double flux;

    int ipos = (int)round(posy);

    flux = cpl_vector_get(fluxes, ipos - 1);
    double frac = ipos + 0.5 - posy;
    flux = frac * flux;
    *cbin = posy + frac / 2.;
    return flux;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the upper integrate flux for given position
  @param    fluxes the vector containing flux values (NOT NULL)
  @param    posy Y position (FITS) to evaluate
  @param  cbin center of bin in Y position (FITS) to evaluate
  @return   the flux a posy position

 */
/*----------------------------------------------------------------------------*/
static double
_moo_get_flux_integrate_up(cpl_vector *fluxes, double posy, double *cbin)
{
    double flux;

    int ipos = (int)round(posy);

    flux = cpl_vector_get(fluxes, ipos - 1);
    double frac = posy - (ipos - 0.5);
    flux = frac * flux;
    *cbin = posy - frac / 2.;
    return flux;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Extract a profile from fluxes
  @param    fluxes the vector containing flux values
  @param    ymin the minimum Y position of profile (FITS)
  @param    ymax the maximum Y position of profile (FITS)

  @return   1 newly allocated bivector contaning the profile (FITS)

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_bivector *
_moo_extract_profile(cpl_vector *fluxes, double ymin, double ymax)
{
    cpl_ensure(fluxes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure((ymax - ymin) > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int y1 = ceil(ymin);
    int y2 = floor(ymax);
    int size = y2 - y1 + 3;

    cpl_bivector *points = cpl_bivector_new(size);
    cpl_vector *y = cpl_bivector_get_x(points);
    cpl_vector *f = cpl_bivector_get_y(points);

    double fmin = _moo_get_flux(fluxes, ymin);
    double fmax = _moo_get_flux(fluxes, ymax);

    cpl_vector_set(y, 0, ymin);
    cpl_vector_set(f, 0, fmin);
    cpl_vector_set(y, size - 1, ymax);
    cpl_vector_set(f, size - 1, fmax);

    int i;
    for (i = y1; i <= y2; i++) {
        double flux = cpl_vector_get(fluxes, i - 1);
        cpl_vector_set(y, i - y1 + 1, i);
        cpl_vector_set(f, i - y1 + 1, flux);
    }

    return points;
}

static cpl_bivector *
_moo_extract_profile_integrate(cpl_vector *fluxes, double ymin, double ymax)
{
    cpl_ensure(fluxes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure((ymax - ymin) > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    double ycmin;
    double ycmax;
    double fmin = _moo_get_flux_integrate_low(fluxes, ymin, &ycmin);
    double fmax = _moo_get_flux_integrate_up(fluxes, ymax, &ycmax);
    int y1 = ceil(ycmin);
    int y2 = floor(ycmax);

    int size = y2 - y1 + 3;

    cpl_bivector *points = cpl_bivector_new(size);
    cpl_vector *y = cpl_bivector_get_x(points);
    cpl_vector *f = cpl_bivector_get_y(points);

    cpl_vector_set(y, 0, ycmin);
    cpl_vector_set(f, 0, fmin);
    cpl_vector_set(y, size - 1, ycmax);
    cpl_vector_set(f, size - 1, fmax);

    int i;

    for (i = y1; i <= y2; i++) {
        double flux = cpl_vector_get(fluxes, i - 1);
        cpl_vector_set(y, i - y1 + 1, i);
        cpl_vector_set(f, i - y1 + 1, flux);
    }

    return points;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Find y positions of thresholds on profile
  @param    fluxes the vector containing flux values
  @param    centroid_y y position of profile centroid
  @param    thresh_low threshold of lower edge
  @param    thresh_up threshold of upper edge
  @param    y_low y position of thresh_low
  @param    y_up y position of thresh_up

  @return (void)
 */
/*----------------------------------------------------------------------------*/
static void
_moo_profile_find_thresh_positions(cpl_vector *fluxes,
                                   double centroid_y,
                                   double thresh_low,
                                   double thresh_up,
                                   double *y_low,
                                   double *y_up)
{
    int size = cpl_vector_get_size(fluxes);

    int centroid_ylow = (int)floor(centroid_y);
    int centroid_yup = centroid_ylow + 1;

    double centroid_f = _moo_get_flux(fluxes, centroid_y);

    int i = 0;
    double f1 = centroid_f;
    double y1 = centroid_y;
    /*
    while(f1>thresh_low){
        y1 = centroid_ylow+i;
        f2 = f1;
        f1 = cpl_vector_get(fluxes,y1-1);
        //cpl_msg_info("test","update with %f at pos %f",f1,y1);
        i--;
    }
    cpl_msg_info("test","find LO f %f for t %f at pos %f and f2 %f",f1,thresh_low,y1,f2);
    i=0;
    f1 = centroid_f;
    f2 = centroid_f;
    y1 = centroid_y;

    while(f1>thresh_up){
        y1 = centroid_yup+i;
        f2 = f1;
        f1 = cpl_vector_get(fluxes,y1-1);
        //cpl_msg_info("test","update with %f at pos %f",f1,y1);
        i++;
    }
    cpl_msg_info("test","find UP f %f for t %f at pos %f and f2 %f",f1,thresh_up,y1,f2);

    f1 = centroid_f;
    f2 = centroid_f;
    y1 = centroid_y;*/
    for (i = centroid_yup; i > 0; i--) {
        double f2 = cpl_vector_get(fluxes, i - 1);
        if (!isnan(f2)) {
            if (f2 <= thresh_low) {
                *y_low = i + (thresh_low - f2) / (f1 - f2) * (y1 - i);
                if ((isnan(*y_low))) {
                    *y_low = (y1 + i) / 2.0;
                }
                break;
            }
            f1 = f2;
            y1 = i;
        }
    }
    f1 = centroid_f;
    y1 = centroid_y;

    for (i = centroid_ylow; i < size - 1; i++) {
        double f2 = cpl_vector_get(fluxes, i - 1);
        if (!isnan(f2)) {
            if (f2 <= thresh_up) {
                *y_up = i + (thresh_up - f2) / (f1 - f2) * (y1 - i);
                if ((isnan(*y_up))) {
                    *y_up = (y1 + i) / 2.0;
                }
                break;
            }
            f1 = f2;
            y1 = i;
        }
    }
    // cpl_msg_info("test","find OLD %f %f",*y_low, *y_up);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Find y positions of thresholds on profile
  @param    fluxes the image containing flux values
  @param    x image X position
  @param    centroid_y y position of profile centroid
  @param    thresh_low threshold of lower edge
  @param    thresh_up threshold of upper edge
  @param    y_low y position of thresh_low
  @param    y_up y position of thresh_up

  @return (void)
 */
/*----------------------------------------------------------------------------*/
static void
_moo_profile_find_image_thresh_positions(const cpl_image *fluxes,
                                         int x,
                                         double centroid_y,
                                         double thresh_low,
                                         double thresh_up,
                                         double *y_low,
                                         double *y_up)
{
    double f2 = NAN;

    int ny = cpl_image_get_size_y(fluxes);
    int centroid_ylow = (int)floor(centroid_y);
    int centroid_yup = centroid_ylow + 1;

    double centroid_f = _moo_get_image_flux(fluxes, x, centroid_y);
    int i = 0;
    double f1 = centroid_f;
    double y1 = centroid_y;

    for (i = centroid_yup; i > 0; i--) {
        int rej = 0;
        f2 = cpl_image_get(fluxes, x, i, &rej);
        if (rej == 0) {
            if (f2 <= thresh_low) {
                *y_low = i + (thresh_low - f2) / (f1 - f2) * (y1 - i);
                if ((isnan(*y_low))) {
                    *y_low = (y1 + i) / 2.0;
                }
                break;
            }
            f1 = f2;
            y1 = i;
        }
    }
    f1 = centroid_f;
    y1 = centroid_y;

    for (i = centroid_ylow; i < ny; i++) {
        int rej = 0;
        f2 = cpl_image_get(fluxes, x, i, &rej);
        if (rej == 0) {
            if (f2 <= thresh_up) {
                *y_up = i + (thresh_up - f2) / (f1 - f2) * (y1 - i);
                if ((isnan(*y_up))) {
                    *y_up = (y1 + i) / 2.0;
                }
                break;
            }
            f1 = f2;
            y1 = i;
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Check the value of center and width in TRACE GUESS
  @param    center value of center in trace guess
  @param    width value of width of the profile

  @return relevant flag
 */
/*----------------------------------------------------------------------------*/
static int
_check_trace_guess(double center, double width)
{
    int flag = MOONS_FLAG_GOOD;

    if (isnan(center) || isnan(width)) {
        flag = MOONS_FLAG_NAN_IN_TRACE_GUESS;
    }
    return flag;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Check the bad pixel map at column x around center
  @param    himg the image
  @param    x the column to test
  @param    center the center of the profile
  @param    width the width of the profile

  @return Number of rejected pixels in the profile
 */
/*----------------------------------------------------------------------------*/
static int
_check_bpm(hdrl_image *himg, int x, double center, double width)
{
    int flag = MOONS_FLAG_GOOD;
    const cpl_mask *bpm = hdrl_image_get_mask_const(himg);

    int lly = (int)(center - width / 2);
    int ury = (int)(center + width / 2);

    cpl_mask *extractm = cpl_mask_extract(bpm, x, lly, x, ury);

    int nbrej = cpl_mask_count(extractm);

    cpl_mask_delete(extractm);

    if (nbrej > 2) {
        flag = MOONS_FLAG_BADPIX;
    }

    return flag;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Check the snr of a profile
  @param    fluxes the fluxes vector
  @param    errs the error vector
  @param    centroid_y the centroid of the profile
  @param    b1_y the background of the lower edge
  @param    b2_y the background of the upper edge
  @return the flux in the profile is SIGNIFICANT to be considered as valid
 */
/*----------------------------------------------------------------------------*/
static int
_check_snr(cpl_vector *fluxes,
           cpl_vector *errs,
           double centroid_y,
           double b1_y,
           double b2_y,
           double ref_snr,
           double *snr)
{
    int flag = MOONS_FLAG_GOOD;

    double centroid_flux = _moo_get_flux(fluxes, centroid_y);
    double centroid_err = _moo_get_flux(errs, centroid_y);

    double b1_flux = _moo_get_flux(fluxes, b1_y);
    double b1_err = _moo_get_flux(errs, b1_y);

    double b2_flux = _moo_get_flux(fluxes, b2_y);
    double b2_err = _moo_get_flux(errs, b2_y);

    double b_flux = (b1_flux + b2_flux) / 2.0;
    if (b_flux <= 0) {
        b_flux = 0;
    }

    double A = centroid_flux - b_flux;
    double eA = sqrt(centroid_err * centroid_err +
                     (b1_err * b1_err + b2_err * b2_err) / 4);
    double snrA = A / eA;
    *snr = snrA;
    if (snrA <= ref_snr) {
        flag = MOONS_FLAG_NON_SIGNIFICANT_FLUX;
        // cpl_msg_info("test","A/eA = %f/%f = %f",A,eA,A/eA);
    }

    return flag;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Find positions of profile background
  @param    fluxes the vector containing flux values
  @param    centroid_y y position of profile centroid
  @param    width width of the profile
  @param    frac fraction to compute the threshold
  @param    b_ylow background positions of lower edge
  @param    b_yup background positions of upper edge
  @param    thresh_low threshold flux of lower edge
  @param    thresh_up threshold flux of upper edge
  @return (void)
 */
/*----------------------------------------------------------------------------*/
static void
_moo_profile_find_background_flux(cpl_vector *fluxes,
                                  double centroid,
                                  double width,
                                  double frac,
                                  double *b_ylow,
                                  double *b_yup,
                                  double *thresh_low,
                                  double *thresh_up)
{
    // cpl_msg_info(__func__,"estimate background for centroid %f",centroid);
    int size = cpl_vector_get_size(fluxes);

    int centroid_y1 = (int)floor(centroid);
    int centroid_y2 = centroid_y1 + 1;
    double centroid_f = _moo_get_flux(fluxes, centroid);
    int i;
    double b1 = centroid_f;
    *b_ylow = centroid;
    int end = centroid_y1 - width;

    if (end < 0) {
        end = 0;
    }

    for (i = centroid_y1; i > end; i--) {
        double v1 = cpl_vector_get(fluxes, i - 1);

        if (v1 < b1) {
            b1 = v1;
            *b_ylow = (double)i;
        }
    }

    double b2 = centroid_f;
    *b_yup = centroid;

    end = centroid_y2 + width;

    if (end > size) {
        end = size;
    }
    for (i = centroid_y2; i <= end; i++) {
        double v1 = cpl_vector_get(fluxes, i - 1);

        if (v1 < b2) {
            b2 = v1;
            *b_yup = (double)i;
        }
    }
    *thresh_low = b1 + (centroid_f - b1) * frac;
    *thresh_up = b2 + (centroid_f - b2) * frac;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Find positions of profile background
  @param    fluxes the image containing flux values
  @param    centroid_y y position of profile centroid
  @param    width width of the profile
  @param    frac fraction to compute the threshold
  @param    b_ylow background positions of lower edge
  @param    b_yup background positions of upper edge
  @param    thresh_low threshold flux of lower edge
  @param    thresh_up threshold flux of upper edge
  @return (void)
 */
/*----------------------------------------------------------------------------*/
static void
_moo_profile_find_image_background_flux(const cpl_image *fluxes,
                                        int x,
                                        double centroid,
                                        double width,
                                        double frac,
                                        double *b_ylow,
                                        double *b_yup,
                                        double *thresh_low,
                                        double *thresh_up)
{
    // cpl_msg_info(__func__,"estimate background for centroid %f",centroid);
    int ny = cpl_image_get_size_y(fluxes);

    int centroid_y1 = (int)floor(centroid);
    int centroid_y2 = centroid_y1 + 1;
    double centroid_f = _moo_get_image_flux(fluxes, x, centroid);

    int i;
    double b1 = centroid_f;
    *b_ylow = centroid;
    int end = centroid_y1 - width;

    if (end < 0) {
        end = 0;
    }

    for (i = centroid_y1; i > end; i--) {
        int rej = 0;
        double v1 = cpl_image_get(fluxes, x, i, &rej);
        if (v1 < b1 && rej == 0) {
            b1 = v1;
            *b_ylow = (double)i;
        }
    }

    double b2 = centroid_f;
    *b_yup = centroid;

    end = centroid_y2 + width;

    if (end > ny) {
        end = ny;
    }
    for (i = centroid_y2; i <= end; i++) {
        int rej = 0;
        double v1 = cpl_image_get(fluxes, x, i, &rej);

        if (v1 < b2 && rej == 0) {
            b2 = v1;
            *b_yup = (double)i;
        }
    }
    *thresh_low = b1 + (centroid_f - b1) * frac;
    *thresh_up = b2 + (centroid_f - b2) * frac;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Find X edges of valid localisation for a fibre
  @param    flags flag array of pts centroid
  @param    size size of X axis
  @param    hwidth half width of X running window
  @param    min_frac minimum fraction of good pixel
  @param    x_min [OUT] minimum x value of valid localisation or -1 if not found
  @param    x_max [OUT] maximum x value of valid localisation or -1 if not found
  @return (void)
 */
/*----------------------------------------------------------------------------*/
static void
_moo_detect_x_edges(cpl_table *tracking_table,
                    int hwidth,
                    double min_frac,
                    int *x_min,
                    int *x_max)
{
    *x_min = -1;
    *x_max = -1;
    int nbgood = 0;

    int size = cpl_table_get_nrow(tracking_table);

    for (int x = 0; x < hwidth; x++) {
        int flag = cpl_table_get_int(tracking_table,
                                     MOO_TRACKING_LOC_TABLE_FLAG, x, NULL);
        if (flag == 0) {
            nbgood++;
        }
    }
    double ratio = (double)nbgood / (double)hwidth;
    for (int x = 0; x < hwidth; x++) {
        cpl_table_set_double(tracking_table, MOO_TRACKING_LOC_TABLE_XGOOD_RATIO,
                             x, ratio);
    }
    if (ratio >= min_frac) {
        *x_min = 1;
    }

    if (*x_min == -1) {
        for (int x = hwidth; x < size - hwidth; x++) {
            int total = 2 * hwidth + 1;
            nbgood = 0;
            for (int j = x - hwidth; j <= x + hwidth; j++) {
                int flag =
                    cpl_table_get_int(tracking_table,
                                      MOO_TRACKING_LOC_TABLE_FLAG, j, NULL);
                if (flag == 0) {
                    nbgood++;
                }
            }
            ratio = (double)nbgood / (double)total;
            cpl_table_set_double(tracking_table,
                                 MOO_TRACKING_LOC_TABLE_XGOOD_RATIO, x, ratio);
            if (ratio >= min_frac) {
                *x_min = x + 1;
                break;
            }
        }
    }

    if (*x_min != -1) {
        nbgood = 0;
        for (int x = size - hwidth; x < size; x++) {
            int flag = cpl_table_get_int(tracking_table,
                                         MOO_TRACKING_LOC_TABLE_FLAG, x, NULL);
            if (flag == 0) {
                nbgood++;
            }
        }
        ratio = (double)nbgood / (double)hwidth;
        for (int x = size - hwidth; x < size; x++) {
            cpl_table_set_double(tracking_table,
                                 MOO_TRACKING_LOC_TABLE_XGOOD_RATIO, x, ratio);
        }

        if (ratio >= min_frac) {
            *x_max = size;
        }

        if (*x_max == -1) {
            for (int x = size - hwidth - 1; x >= hwidth; x--) {
                int total = 2 * hwidth + 1;
                nbgood = 0;

                for (int j = x - hwidth; j <= x + hwidth; j++) {
                    int flag =
                        cpl_table_get_int(tracking_table,
                                          MOO_TRACKING_LOC_TABLE_FLAG, j, NULL);

                    if (flag == 0) {
                        nbgood++;
                    }
                }
                ratio = (double)nbgood / (double)total;
                cpl_table_set_double(tracking_table,
                                     MOO_TRACKING_LOC_TABLE_XGOOD_RATIO, x,
                                     ratio);
                if (ratio >= min_frac) {
                    *x_max = x + 1;
                    break;
                }
            }
        }
    }
}

static int
_moo_localise_row(hdrl_image *himg,
                  int x,
                  double wdiff_lim,
                  double ydiff_lim,
                  float frac,
                  double ref_snr,
                  double *min,
                  double *max,
                  cpl_table *tracking_loc_table,
                  double *last_center,
                  cpl_vector *last_width_vect,
                  double *last_x)
{
    int is_invalid = 0;

    double y1, y2, center = NAN, width = NAN;
    int idx = x - 1;

    cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X, idx, x);
    double last_width = cpl_vector_get_median_const(last_width_vect);
    const cpl_image *img = hdrl_image_get_image_const(himg);
    const cpl_image *error = hdrl_image_get_error_const(himg);
    const cpl_mask *mask = cpl_image_get_bpm_const(img);
    cpl_vector *fluxes = cpl_vector_new_from_image_column(img, x);
    cpl_vector *errs = cpl_vector_new_from_image_column(error, x);

    int size = cpl_vector_get_size(fluxes);
    for (int i = 1; i < size; i++) {
        int rej = cpl_mask_get(mask, x, i);
        if (rej) {
            cpl_vector_set(fluxes, i - 1, NAN);
        }
    }
    int flag = _check_bpm(himg, x, *last_center, last_width);

    if (flag == MOONS_FLAG_GOOD) {
        double thresh_low, thresh_up, b1_y, b2_y, snr;
        _moo_profile_find_background_flux(fluxes, *last_center, last_width,
                                          frac, &b1_y, &b2_y, &thresh_low,
                                          &thresh_up);

        flag =
            _check_snr(fluxes, errs, *last_center, b1_y, b2_y, ref_snr, &snr);
        cpl_table_set_double(tracking_loc_table, MOO_TRACKING_LOC_TABLE_SNR,
                             idx, snr);

        if (flag == MOONS_FLAG_GOOD) {
            _moo_profile_find_thresh_positions(fluxes, *last_center, thresh_low,
                                               thresh_up, &y1, &y2);
            cpl_errorstate prev_state = cpl_errorstate_get();
            cpl_bivector *points = _moo_extract_profile(fluxes, y1, y2);
            if (points != NULL) {
                moo_barycenter_fit(points, &center, &width);
                cpl_bivector_delete(points);

                double wdiff = width - last_width;
                double ydiff =
                    (center - *last_center) / sqrt(fabs(x - *last_x));
                cpl_table_set_double(tracking_loc_table,
                                     MOO_TRACKING_LOC_TABLE_WDIFF, idx, wdiff);
                cpl_table_set_double(tracking_loc_table,
                                     MOO_TRACKING_LOC_TABLE_YDIFF, idx, ydiff);
                if (fabs(wdiff) < wdiff_lim) {
                    if (fabs(ydiff) < ydiff_lim) {
                        if (center < *min) {
                            *min = center;
                        }
                        if (center > *max) {
                            *max = center;
                        }
                        cpl_table_set_double(tracking_loc_table,
                                             MOO_TRACKING_LOC_TABLE_YCENTROID,
                                             idx, center);
                        cpl_table_set_double(tracking_loc_table,
                                             MOO_TRACKING_LOC_TABLE_YMIN, idx,
                                             y1);
                        cpl_table_set_double(tracking_loc_table,
                                             MOO_TRACKING_LOC_TABLE_YMAX, idx,
                                             y2);
                        cpl_table_set_int(tracking_loc_table,
                                          MOO_TRACKING_LOC_TABLE_FLAG, idx,
                                          MOONS_FLAG_GOOD);
                        *last_center = center;

                        int vsize = cpl_vector_get_size(last_width_vect);
                        for (int iv = 0; iv < vsize - 1; iv++) {
                            double lv = cpl_vector_get(last_width_vect, iv + 1);
                            cpl_vector_set(last_width_vect, iv, lv);
                        }
                        cpl_vector_set(last_width_vect, vsize - 1, width);
                        *last_x = x;
                    }
                    else {
                        cpl_table_set_int(tracking_loc_table,
                                          MOO_TRACKING_LOC_TABLE_FLAG, idx,
                                          MOONS_FLAG_YDIFF_OUTLIERS);

                        is_invalid = 1;
                    }
                }
                else {
                    cpl_table_set_int(tracking_loc_table,
                                      MOO_TRACKING_LOC_TABLE_FLAG, idx,
                                      MOONS_FLAG_WDIFF_OUTLIERS);

                    is_invalid = 1;
                }
            }
            else {
                cpl_errorstate_set(prev_state);
                cpl_table_set_int(tracking_loc_table,
                                  MOO_TRACKING_LOC_TABLE_FLAG, idx,
                                  MOONS_FLAG_NOPROFILE);
                is_invalid = 1;
            }
        }
        else {
            cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FLAG,
                              idx, MOONS_FLAG_NON_SIGNIFICANT_FLUX);
            is_invalid = 1;
        }
    }
    else {
        cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FLAG, idx,
                          MOONS_FLAG_BADPIX);
        is_invalid = 1;
    }

    cpl_vector_delete(fluxes);
    cpl_vector_delete(errs);
    return is_invalid;
}

static cpl_error_code
_moo_localise_tchebychev_fit(moo_loc_single *res,
                             int fibnum,
                             int deg_poly,
                             cpl_table *tracking_table,
                             int xlim_hwin,
                             double xlim_fracmin)
{
    cpl_error_code status = CPL_ERROR_NONE;

    int invalid_wlo = 0;
    int invalid_wup = 0;
    cpl_image *measured_centroids = res->m_centroids;
    cpl_image *fit_centroids = res->f_centroids;
    cpl_image *wlow = res->f_wlow;
    cpl_image *wup = res->f_wup;
    cpl_image *mlow = res->m_wlow;
    cpl_image *mup = res->m_wup;
    cpl_image *flags = res->flags;

    int nx = cpl_image_get_size_x(measured_centroids);

    for (int i = 1; i <= nx; i++) {
        double yc =
            cpl_table_get_double(tracking_table,
                                 MOO_TRACKING_LOC_TABLE_YCENTROID, i - 1, NULL);
        double yl =
            cpl_table_get_double(tracking_table, MOO_TRACKING_LOC_TABLE_YMIN,
                                 i - 1, NULL);
        double yu =
            cpl_table_get_double(tracking_table, MOO_TRACKING_LOC_TABLE_YMAX,
                                 i - 1, NULL);
        int flag = cpl_table_get_int(tracking_table,
                                     MOO_TRACKING_LOC_TABLE_FLAG, i - 1, NULL);
        cpl_image_set(measured_centroids, i, fibnum, yc);
        cpl_image_set(mlow, i, fibnum, yl);
        cpl_image_set(mup, i, fibnum, yu);
        cpl_image_set(flags, i, fibnum, flag);
        cpl_image_set(fit_centroids, i, fibnum, NAN);
        cpl_image_set(wlow, i, fibnum, NAN);
        cpl_image_set(wup, i, fibnum, NAN);
    }

    int xmin, xmax;

    _moo_detect_x_edges(tracking_table, xlim_hwin, xlim_fracmin, &xmin, &xmax);

    if (xmin != -1) {
        cpl_msg_debug("moo_localise",
                      "Fibre #%d detected on pixel range %d - %d", fibnum, xmin,
                      xmax);

        int *cflag =
            cpl_table_get_data_int(tracking_table, MOO_TRACKING_LOC_TABLE_FLAG);
        int *cposx =
            cpl_table_get_data_int(tracking_table, MOO_TRACKING_LOC_TABLE_X);
        double *cposy =
            cpl_table_get_data_double(tracking_table,
                                      MOO_TRACKING_LOC_TABLE_YCENTROID);

        cpl_bivector *cpos = cpl_bivector_new(nx);
        cpl_vector *vx = cpl_bivector_get_x(cpos);
        cpl_vector *vy = cpl_bivector_get_y(cpos);

        for (int i = 0; i < nx; i++) {
            cpl_vector_set(vx, i, (double)cposx[i]);
            cpl_vector_set(vy, i, cposy[i]);
        }

        double min = moo_vector_get_min(vy, cflag);
        double max = moo_vector_get_max(vy, cflag);

        moo_tchebychev_fit(cpos, cflag, deg_poly, xmin, xmax, min, max);

        for (int i = xmin; i <= xmax; i++) {
            double fyc = cpl_vector_get(vy, i - 1);
            cpl_table_set(tracking_table, MOO_TRACKING_LOC_TABLE_FITYCENTROID,
                          i - 1, fyc);
            cpl_image_set(fit_centroids, i, fibnum, fyc);
        }

        double *cposylow =
            cpl_table_get_data_double(tracking_table,
                                      MOO_TRACKING_LOC_TABLE_YMIN);

        for (int i = 0; i < nx; i++) {
            cpl_vector_set(vy, i, cposylow[i]);
        }
        min = moo_vector_get_min(vy, cflag);
        max = moo_vector_get_max(vy, cflag);

        moo_tchebychev_fit(cpos, cflag, deg_poly, xmin, xmax, min, max);

        for (int i = xmin; i <= xmax; i++) {
            double yl = cpl_vector_get(vy, i - 1);
            double yc =
                cpl_table_get(tracking_table,
                              MOO_TRACKING_LOC_TABLE_FITYCENTROID, i - 1, NULL);
            double val = yc - yl;
            if (val <= 0) {
                invalid_wlo++;
            }
            cpl_table_set(tracking_table, MOO_TRACKING_LOC_TABLE_FITYMIN, i - 1,
                          yl);
            cpl_image_set(wlow, i, fibnum, val);
        }

        double *cposyup =
            cpl_table_get_data_double(tracking_table,
                                      MOO_TRACKING_LOC_TABLE_YMAX);

        for (int i = 0; i < nx; i++) {
            cpl_vector_set(vy, i, cposyup[i]);
        }
        min = moo_vector_get_min(vy, cflag);
        max = moo_vector_get_max(vy, cflag);

        moo_tchebychev_fit(cpos, cflag, deg_poly, xmin, xmax, min, max);

        for (int i = xmin; i <= xmax; i++) {
            double yu = cpl_vector_get(vy, i - 1);
            double yc =
                cpl_table_get(tracking_table,
                              MOO_TRACKING_LOC_TABLE_FITYCENTROID, i - 1, NULL);
            double val = yu - yc;
            if (val <= 0) {
                invalid_wup++;
            }
            cpl_table_set(tracking_table, MOO_TRACKING_LOC_TABLE_FITYMAX, i - 1,
                          yu);
            cpl_image_set(wup, i, fibnum, val);
        }
        cpl_bivector_delete(cpos);
        if (invalid_wlo) {
            cpl_msg_error("moo_localise",
                          "Fibre #%d : localisation fitting (polynomial degree "
                          "%d) failed : negative lower width",
                          fibnum, deg_poly);
            status = CPL_ERROR_ILLEGAL_OUTPUT;
        }
        if (invalid_wup) {
            cpl_msg_error("moo_localise",
                          "Fibre #%d : localisation fitting (polynomial degree "
                          "%d) failed : negative upper width",
                          fibnum, deg_poly);
            status = CPL_ERROR_ILLEGAL_OUTPUT;
        }
    }
    else {
        cpl_msg_error("moo_localise",
                      "Fibre #%d : localisation tracking failed", fibnum);
        status = CPL_ERROR_ILLEGAL_OUTPUT;
    }
    return status;
}

static cpl_error_code
_moo_localise_fibre(fibre *f,
                    hdrl_image *himg,
                    int cx,
                    int deg_poly,
                    double wdiff_lim,
                    double ydiff_lim,
                    float frac,
                    double ref_snr,
                    int xgap_max,
                    double goodptsfrac_min,
                    int xlim_hwin,
                    double xlim_fracmin,
                    moo_loc_single *res,
                    cpl_table **all_tracked)
{
    cpl_table *tracking_loc_table = NULL;
    cpl_error_code status = CPL_ERROR_NONE;
    double max = 0;

    cpl_msg_indent_more();

    int nx = hdrl_image_get_size_x(himg);
    int ny = hdrl_image_get_size_y(himg);

    double min = (double)ny;

    tracking_loc_table = cpl_table_new(nx);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FIBNUM,
                         CPL_TYPE_INT);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X,
                         CPL_TYPE_INT);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YCENTROID,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YMIN,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YMAX,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_WDIFF,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YDIFF,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_SNR,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FLAG,
                         CPL_TYPE_INT);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_XGOOD_RATIO,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table,
                         MOO_TRACKING_LOC_TABLE_FITYCENTROID, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FITYMIN,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FITYMAX,
                         CPL_TYPE_DOUBLE);

    for (int i = 0; i < nx; i++) {
        cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FIBNUM, i,
                          f->num);
    }
    cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X, cx - 1, cx);
    cpl_table_set_double(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YCENTROID,
                         cx - 1, (f->center));
    cpl_table_set_double(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YMIN,
                         cx - 1, (f->ymin));
    cpl_table_set_double(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YMAX,
                         cx - 1, (f->ymax));
    cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FLAG, cx - 1,
                      MOONS_FLAG_GOOD);

    double last_center = f->center;
    cpl_vector *last_width = cpl_vector_new(3);
    cpl_vector_fill(last_width, f->width);

    double last_x = cx;
    int nb_invalid = 0;
    int nb_follow_invalid = 0;
    for (int i = cx + 1; i <= nx; i++) {
        int is_invalid =
            _moo_localise_row(himg, i, wdiff_lim, ydiff_lim, frac, ref_snr,
                              &min, &max, tracking_loc_table, &last_center,
                              last_width, &last_x);

        if (is_invalid) {
            nb_follow_invalid++;
            nb_invalid++;
        }
        else {
            nb_follow_invalid = 0;
        }
        if (nb_follow_invalid > xgap_max) {
            for (int j = i + 1; j <= nx; j++) {
                nb_invalid++;
                cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X,
                                  j - 1, j);
                cpl_table_set_int(tracking_loc_table,
                                  MOO_TRACKING_LOC_TABLE_FLAG, j - 1,
                                  MOONS_FLAG_XMAX_GAP);
            }
            break;
        }
    }
    last_center = f->center;
    cpl_vector_fill(last_width, f->width);
    last_x = cx;
    nb_follow_invalid = 0;

    for (int i = cx - 1; i > 0; i--) {
        int is_invalid =
            _moo_localise_row(himg, i, wdiff_lim, ydiff_lim, frac, ref_snr,
                              &min, &max, tracking_loc_table, &last_center,
                              last_width, &last_x);
        if (is_invalid) {
            nb_follow_invalid++;
            nb_invalid++;
        }
        else {
            nb_follow_invalid = 0;
        }
        if (nb_follow_invalid > xgap_max) {
            for (int j = i - 1; j > 0; j--) {
                nb_invalid++;
                cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X,
                                  j - 1, j);
                cpl_table_set_int(tracking_loc_table,
                                  MOO_TRACKING_LOC_TABLE_FLAG, j - 1,
                                  MOONS_FLAG_XMAX_GAP);
            }
            break;
        }
    }
    cpl_vector_delete(last_width);

    cpl_msg_debug("moo_localise", "Fraction of invalid detection %.2f %%",
                  (double)nb_invalid / (double)nx * 100.);
    f->goodfrac = (1. - nb_invalid / (double)nx) * 100.;
    cpl_msg_indent_less();
    if (f->goodfrac >= goodptsfrac_min) {
        status = _moo_localise_tchebychev_fit(res, f->num, deg_poly,
                                              tracking_loc_table, xlim_hwin,
                                              xlim_fracmin);
    }
    else {
        f->health = MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN;
    }
    *all_tracked = tracking_loc_table;
    return status;
}

static int
_moo_localise_col_flux_from_guess(hdrl_image *himg,
                                  cpl_table *tracking_loc_table,
                                  int x,
                                  double *yc,
                                  double *width,
                                  double *yl,
                                  double *yu,
                                  const char *method,
                                  double frac,
                                  double ref_snr)
{
    int flag = MOONS_FLAG_GOOD;
    int idx = x - 1;

    const cpl_image *img = hdrl_image_get_image_const(himg);
    const cpl_image *error = hdrl_image_get_error_const(himg);

    cpl_vector *fluxes = cpl_vector_new_from_image_column(img, x);
    cpl_vector *errs = cpl_vector_new_from_image_column(error, x);

    if (strcmp(method, MOO_LOCALISE_METHOD_BARYCENTER) == 0) {
        double thresh_low, thresh_up;
        double b1_y, b2_y;

        _moo_profile_find_image_background_flux(img, x, *yc, *width, frac,
                                                &b1_y, &b2_y, &thresh_low,
                                                &thresh_up);

        _moo_profile_find_image_thresh_positions(img, x, *yc, thresh_low,
                                                 thresh_up, yl, yu);

        cpl_errorstate prev_state = cpl_errorstate_get();
        cpl_bivector *points = _moo_extract_profile_integrate(fluxes, *yl, *yu);

        if (points != NULL) {
            double snr;
            moo_barycenter_fit(points, yc, width);
            cpl_bivector_delete(points);
            flag = _check_snr(fluxes, errs, *yc, b1_y, b2_y, ref_snr, &snr);
            cpl_table_set_double(tracking_loc_table, MOO_TRACKING_LOC_TABLE_SNR,
                                 idx, snr);
        }
        else {
            cpl_errorstate_set(prev_state);
            flag = MOONS_FLAG_NOPROFILE;
        }
    }
    else {
        cpl_errorstate prev_state = cpl_errorstate_get();
        double gc = *yc, gwidth, background = 0, area, sigma = *width / 4.0;
        cpl_bivector *points = _moo_extract_profile_integrate(fluxes, *yl, *yu);
        /*{
            char* name = cpl_sprintf("extractedflux_%d.txt",x);
            FILE* test = fopen(name,"w");
            cpl_bivector_dump(points,test);
            fclose(test);
            cpl_free(name);
        }*/
        cpl_fit_mode fit_pars = CPL_FIT_CENTROID | CPL_FIT_STDEV | CPL_FIT_AREA;
        moo_gaussian_fit(points, fit_pars, &gc, &sigma, &background, &area);

        if (!cpl_errorstate_is_equal(prev_state)) {
            cpl_errorstate_set(prev_state);
            flag = MOONS_FLAG_BADPROFILE;
        }
        else {
            double ycval = moo_gaussian_eval(gc, gc, sigma, background, area);
            /*{
                char* name = cpl_sprintf("modelflux_%d.txt",x);
                FILE* test = fopen(name,"w");
                fprintf(test,"#y flux\n");
                for(double t=-5; t<=5; t+=0.1){
                    double tycval = moo_gaussian_eval(gc+t,gc,sigma,background,area);
                    fprintf(test,"%f %f\n",gc+t,tycval);
                }
                fclose(test);
                cpl_free(name);
            }*/
            double thresh = background + (ycval - background) * frac;
            double gyl, gyu;
            moo_gaussian_eval_inv(thresh, gc, sigma, background, area, &gyl,
                                  &gyu);
            gwidth = gyu - gyl;
            /*{
                char* name = cpl_sprintf("params_%d.txt",x);
                FILE* test = fopen(name,"w");
                fprintf(test,"%d %f %f %f %f %f %f\n",x,thresh,gc,gyl,gyu,sigma,background);
                fclose(test);
                cpl_free(name);
            }*/
            if (gwidth > 2) {
                double thresh_low, thresh_up;
                double b1_y, b2_y, snr;

                _moo_profile_find_background_flux(fluxes, gc, *width, frac,
                                                  &b1_y, &b2_y, &thresh_low,
                                                  &thresh_up);

                flag = _check_snr(fluxes, errs, gc, b1_y, b2_y, ref_snr, &snr);
                *yc = gc;
                *width = gwidth;
                *yl = gyl;
                *yu = gyu;
            }
            else {
                flag = MOONS_FLAG_NARROWPROFILE;
            }
        }
        cpl_bivector_delete(points);
    }

    cpl_vector_delete(fluxes);
    cpl_vector_delete(errs);

    return flag;
}

static int
_moo_localise_col_from_guess(hdrl_image *himg,
                             int x,
                             double yc,
                             double wl,
                             double wu,
                             cpl_table *tracking_table,
                             const char *method,
                             double frac,
                             double ref_snr)
{
    double center = yc;
    double width = wl + wu;
    double yl = yc - wl;
    double yu = yc + wu;

    int fl = _check_trace_guess(center, width);
    if (fl != MOONS_FLAG_GOOD) {
        return fl;
    }

    fl = _check_bpm(himg, x, center, width);
    if (fl != MOONS_FLAG_GOOD) {
        return fl;
    }

    fl = _moo_localise_col_flux_from_guess(himg, tracking_table, x, &center,
                                           &width, &yl, &yu, method, frac,
                                           ref_snr);

    if (fl == MOONS_FLAG_GOOD) {
        cpl_table_set_double(tracking_table, MOO_TRACKING_LOC_TABLE_YCENTROID,
                             x - 1, center);
        cpl_table_set_double(tracking_table, MOO_TRACKING_LOC_TABLE_YMIN, x - 1,
                             yl);
        cpl_table_set_double(tracking_table, MOO_TRACKING_LOC_TABLE_YMAX, x - 1,
                             yu);
    }

    return fl;
}

static int
_moo_localise_fibre_from_guess(int fibnum,
                               hdrl_image *himg,
                               cpl_image *gcentroid,
                               cpl_image *gwlo,
                               cpl_image *gwup,
                               moo_localise_params *params,
                               int deg_poly,
                               double goodptsfrac_min,
                               double ref_snr,
                               moo_loc_single *res,
                               cpl_table **all_tracked)
{
    int detected = 0;
    cpl_table *tracking_loc_table = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_ensure(himg != NULL, CPL_ERROR_NULL_INPUT, 0);

    int xlim_hwin = params->loc_xlim_hwin;
    double xlim_fracmin = params->loc_xlim_fracmin;

    cpl_msg_indent_more();

    cpl_msg_debug("moo_localise", "Localise fibres %d using guess localisation",
                  fibnum);
    int nx = hdrl_image_get_size_x(himg);

    tracking_loc_table = cpl_table_new(nx);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FIBNUM,
                         CPL_TYPE_INT);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X,
                         CPL_TYPE_INT);
    cpl_table_new_column(tracking_loc_table,
                         MOO_TRACKING_LOC_TABLE_GUESS_YCENTROID,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_GUESS_WLO,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_GUESS_WUP,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YCENTROID,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YMIN,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YMAX,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_WDIFF,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_YDIFF,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_SNR,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FLAG,
                         CPL_TYPE_INT);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_XGOOD_RATIO,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table,
                         MOO_TRACKING_LOC_TABLE_FITYCENTROID, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FITYMIN,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FITYMAX,
                         CPL_TYPE_DOUBLE);

    for (int i = 0; i < nx; i++) {
        cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FIBNUM, i,
                          fibnum);
    }

    int nb_invalid = 0;

    for (int i = 1; i <= nx; i++) {
        int rej;
        double yc = cpl_image_get(gcentroid, i, fibnum, &rej);
        double wl = cpl_image_get(gwlo, i, fibnum, &rej);
        double wu = cpl_image_get(gwup, i, fibnum, &rej);
        cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_X, i - 1,
                          i);
        cpl_table_set_double(tracking_loc_table,
                             MOO_TRACKING_LOC_TABLE_GUESS_YCENTROID, i - 1, yc);
        cpl_table_set_double(tracking_loc_table,
                             MOO_TRACKING_LOC_TABLE_GUESS_WLO, i - 1, wl);
        cpl_table_set_double(tracking_loc_table,
                             MOO_TRACKING_LOC_TABLE_GUESS_WUP, i - 1, wu);
        int flag = MOONS_FLAG_GOOD;
        moo_try_check(flag = _moo_localise_col_from_guess(
                          himg, i, yc, wl, wu, tracking_loc_table,
                          params->method, params->relativethresh, ref_snr),
                      " ");
        cpl_table_set_int(tracking_loc_table, MOO_TRACKING_LOC_TABLE_FLAG,
                          i - 1, flag);

        if (flag != MOONS_FLAG_GOOD) {
            nb_invalid++;
        }
    }
    cpl_msg_debug("moo_localise", "Fraction of invalid detection %.1f %%",
                  nb_invalid / (double)nx * 100);
    double frac_good = (1 - nb_invalid / (double)nx) * 100;

    if (frac_good > goodptsfrac_min) {
        _moo_localise_tchebychev_fit(res, fibnum, deg_poly, tracking_loc_table,
                                     xlim_hwin, xlim_fracmin);
        detected = 1;
    }
    else {
        cpl_image *measured_centroids = res->m_centroids;
        cpl_image *fit_centroids = res->f_centroids;
        cpl_image *wlow = res->f_wlow;
        cpl_image *wup = res->f_wup;
        cpl_image *mlow = res->m_wlow;
        cpl_image *mup = res->m_wup;
        cpl_image *flags = res->flags;

        for (int i = 1; i <= nx; i++) {
            int flag =
                cpl_table_get_int(tracking_loc_table,
                                  MOO_TRACKING_LOC_TABLE_FLAG, i - 1, NULL);
            double yc = cpl_table_get_double(tracking_loc_table,
                                             MOO_TRACKING_LOC_TABLE_YCENTROID,
                                             i - 1, NULL);
            double yl =
                cpl_table_get_double(tracking_loc_table,
                                     MOO_TRACKING_LOC_TABLE_YMIN, i - 1, NULL);
            double yu =
                cpl_table_get_double(tracking_loc_table,
                                     MOO_TRACKING_LOC_TABLE_YMAX, i - 1, NULL);
            cpl_image_set(measured_centroids, i, fibnum, yc);
            cpl_image_set(mlow, i, fibnum, yl);
            cpl_image_set(mup, i, fibnum, yu);

            cpl_image_set(flags, i, fibnum, flag);
            cpl_image_set(fit_centroids, i, fibnum, NAN);
            cpl_image_set(wlow, i, fibnum, NAN);
            cpl_image_set(wup, i, fibnum, NAN);
        }
    }

moo_try_cleanup:
    *all_tracked = tracking_loc_table;
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return detected;
}

static moo_loc_single *
_moo_localise_single_from_guess(moo_single *single,
                                moo_loc_single *guess,
                                int nb_total,
                                int *health,
                                const char **names,
                                const int *indexext,
                                moo_localise_params *params,
                                moo_detector_type type,
                                int ntas,
                                const char *filename,
                                int degpoly,
                                double goodptsfrac_min,
                                double ref_snr,
                                cpl_array *indexes)
{
    int nmissing_fibres = 0;
    moo_loc_single *res = NULL;
    cpl_table **all_tracked = NULL;

    cpl_ensure(single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(guess != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_msg_info("moo_localise", "goodptsfrac-min %f", goodptsfrac_min);
    hdrl_image *himg = moo_single_get_image(single);
    int nx = hdrl_image_get_size_x(himg);

    const char *extname = moo_detector_get_extname(type, ntas);

    moo_try_check(res = moo_loc_single_create(filename, extname), " ");
    res->m_centroids = cpl_image_new(nx, nb_total, CPL_TYPE_DOUBLE);
    res->f_centroids = cpl_image_new(nx, nb_total, CPL_TYPE_DOUBLE);
    res->m_wlow = cpl_image_new(nx, nb_total, CPL_TYPE_DOUBLE);
    res->f_wlow = cpl_image_new(nx, nb_total, CPL_TYPE_DOUBLE);
    res->m_wup = cpl_image_new(nx, nb_total, CPL_TYPE_DOUBLE);
    res->f_wup = cpl_image_new(nx, nb_total, CPL_TYPE_DOUBLE);
    res->flags = cpl_image_new(nx, nb_total, CPL_TYPE_INT);
    res->header = cpl_propertylist_new();

    cpl_image *gcentroids = moo_loc_single_get_f_centroids(guess);
    cpl_image *gwlo = moo_loc_single_get_f_wlo(guess);
    cpl_image *gwup = moo_loc_single_get_f_wup(guess);

    all_tracked = cpl_calloc(nb_total, sizeof(cpl_table *));
#ifdef _OPENMP

#pragma omp parallel default(none)                                             \
    shared(all_tracked, nb_total, health, res, degpoly, goodptsfrac_min,       \
               ref_snr, params, gwup, gwlo, gcentroids, himg, nmissing_fibres, \
               names, nx, indexes, indexext, single)
    {
#pragma omp for
#endif
        for (int f = 1; f <= nb_total; f++) {
            //for (int f = 351; f <= 351; f++) {
            int idx = cpl_array_get_cplsize(indexes, f - 1, NULL);

            if (health[idx] == MOO_FIBRES_TABLE_HEALTH_GOOD) {
                int detected =
                    _moo_localise_fibre_from_guess(f, himg, gcentroids, gwlo,
                                                   gwup, params, degpoly,
                                                   goodptsfrac_min, ref_snr,
                                                   res, &all_tracked[f - 1]);

                if (detected == 0) {
                    cpl_msg_info("moo_localise",
                                 "Bad detection fibre indexext %d with name %s",
                                 indexext[idx], names[idx]);
                    health[idx] = MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN;
                }
            }
            else {
                cpl_msg_info("moo_localise",
                             "Bad health fibre indexext %d with name %s",
                             indexext[idx], names[idx]);
                for (int i = 1; i <= nx; i++) {
                    cpl_image_set(res->m_centroids, i, f, NAN);
                    cpl_image_set(res->m_wlow, i, f, NAN);
                    cpl_image_set(res->m_wup, i, f, NAN);
                    cpl_image_set(res->flags, i, f, MOONS_FLAG_BROKEN_FIBRE);
                    cpl_image_set(res->f_centroids, i, f, NAN);
                    cpl_image_set(res->f_wlow, i, f, NAN);
                    cpl_image_set(res->f_wup, i, f, NAN);
                }
            }
        }
#ifdef _OPENMP
    }
#endif
    for (int f = 0; f < nb_total; f++) {
        if (health[f] == MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN) {
            health[f] = MOO_FIBRES_TABLE_HEALTH_BROKEN;
            nmissing_fibres++;
            moo_qc_set_missingfib(res->header, nmissing_fibres, names[f]);
        }
    }
#if MOO_DEBUG_LOCALISE_TRACKING
    {
        char *name = cpl_sprintf("%s_TRACKING_LOC_TABLE.fits", res->extname);
        cpl_table *test = NULL;
        int first = 0;
        for (int i = 0; i < nb_total; i++) {
            cpl_table *t = all_tracked[i];
            if (t != NULL) {
                test = cpl_table_duplicate(t);
                first = i + 1;
                break;
            }
        }
        if (test != NULL) {
            for (int i = first; i < nb_total; i++) {
                cpl_table *t = all_tracked[i];
                if (t != NULL) {
                    moo_table_append(test, t);
                }
            }
            cpl_table_save(test, NULL, NULL, name, CPL_IO_CREATE);
            cpl_table_delete(test);
        }
        cpl_free(name);
    }
#endif
    moo_qc_set_nmissingfib(res->header, nmissing_fibres);
moo_try_cleanup:
    if (all_tracked != NULL) {
        for (int i = 0; i < nb_total; i++) {
            if (all_tracked[i] != NULL) {
                cpl_table_delete(all_tracked[i]);
            }
        }
        cpl_free(all_tracked);
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        moo_loc_single_delete(res);
        res = NULL;
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    To localise the centroid of fibre spectrum on frames
  @param    det _DET_ frame to localize
  @param    guess _LOC_ trace guess
  @param    params the localisation parameters
  @param    locname the file name of the result
  @return   the FF_TRACE or NULL in error case

 * This function is to find the localisation of the centroid of each fibre spectrum
 * on CCD Frame, then generate the FF_TRACE.
 * construct the FF_TRACE based on user input FF_TRACE_GUESS

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if _DET_ has not a fibre table
 */
/*----------------------------------------------------------------------------*/

static moo_loc *
_moo_localise_from_guess(moo_det *det,
                         moo_loc *guess,
                         moo_localise_params *params,
                         const char *locname,
                         int badpix_level)
{
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(guess != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_table *fibres_table = moo_loc_get_fibre_table(guess);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    const char *filename = locname;
    moo_loc *loc = NULL;
    int i, j;

    moo_try_check(loc = moo_loc_create(locname), " ");
    cpl_msg_info("moo_localise", "Create TRACE localisation");
    cpl_msg_indent_more();

    cpl_table *loc_fibre_table = NULL;
    int *health = NULL;
    const int *indexext = NULL;
    const char **names = NULL;
    cpl_array *indexes = NULL;

    loc_fibre_table = cpl_table_duplicate(fibres_table);
    moo_try_check(health = cpl_table_get_data_int(loc_fibre_table,
                                                  MOO_FIBRES_TABLE_HEALTH),
                  " ");

    moo_try_check(indexext = cpl_table_get_data_int(loc_fibre_table,
                                                    MOO_FIBRES_TABLE_INDEXEXT),
                  " ");

    moo_try_check(names =
                      cpl_table_get_data_string_const(loc_fibre_table,
                                                      MOO_FIBRES_TABLE_FIBRE),
                  " ");

    for (i = 1; i <= 2; i++) {
        indexes = moo_fibres_table_get_spectro_indexext(loc_fibre_table, i);

        int nb_total = cpl_array_get_size(indexes);

        for (j = 0; j < 3; j++) {
            int idx = (i - 1) * 3 + j;
            int polydeg = params->polydeg[idx];
            double ref_snr = params->ref_snr[idx];
            double goodptsfrac_min = params->goodptsfrac_min[idx] * 100.;
            moo_single *s1 = moo_det_load_single(det, j, i, badpix_level);
            moo_loc_single *lguess = moo_loc_get_single(guess, j, i);

            if (s1 != NULL && lguess != NULL) {
                cpl_msg_info("moo_localise", "Localising %d fibres in %s",
                             nb_total, moo_detector_get_extname(j, i));

                moo_loc_single *ls =
                    _moo_localise_single_from_guess(s1, lguess, nb_total,
                                                    health, names, indexext,
                                                    params, j, i, filename,
                                                    polydeg, goodptsfrac_min,
                                                    ref_snr, indexes);

                if (ls != NULL) {
                    moo_loc_single_compute_qc_trace(ls, polydeg, lguess);
                    moo_loc_add_single(loc, ls, j, i, params->keep_points);
                }
            }
        }
        cpl_array_delete(indexes);
        indexes = NULL;
    }

    moo_loc_add_fibre_table(loc, loc_fibre_table);

moo_try_cleanup:
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_array_delete(indexes);
        cpl_table_delete(loc_fibre_table);
        loc_fibre_table = NULL;
        moo_loc_delete(loc);
        loc = NULL;
    }
    return loc;
}
/******************************************************************************/

static double
_moo_bckg_poly_eval(cpl_polynomial *fit1d, double v)
{
    cpl_size p0 = 0;
    cpl_size p1 = 1;
    cpl_size p2 = 2;
    cpl_size p3 = 3;
    cpl_size p4 = 4;
    double c0, c1, c2, c3, c4;
    c0 = cpl_polynomial_get_coeff(fit1d, &p0);
    c1 = cpl_polynomial_get_coeff(fit1d, &p1);
    c2 = cpl_polynomial_get_coeff(fit1d, &p2);
    c3 = cpl_polynomial_get_coeff(fit1d, &p3);
    c4 = cpl_polynomial_get_coeff(fit1d, &p4);
    double b = c0 + v * c1 + v * v * c2 + v * v * v * c3 + v * v * v * v * c4;

    return b;
}

static cpl_error_code
_moo_localise_background_fit_rmin(cpl_table *fibre_loc_table, int winhsize)
{
    cpl_ensure_code(winhsize >= 0, CPL_ERROR_ILLEGAL_INPUT);

    double *fluxes =
        cpl_table_get_data_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX);
    double *errs = cpl_table_get_data_double(fibre_loc_table,
                                             MOO_FIBRE_LOC_TABLE_FLUX_ERR);
    int ny = cpl_table_get_nrow(fibre_loc_table);
    int winsize = 2 * winhsize + 1;

    int ib = 0;
    for (int i = 0; i < winsize; i++) {
        double b = fluxes[i];
        if (isnan(fluxes[ib])) {
            ib = i;
        }
        else if (!isnan(b) && fluxes[ib] > b) {
            ib = i;
        }
    }
    double b = fluxes[ib];
    double be = errs[ib];
    for (int i = 0; i < winhsize; i++) {
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BACKGROUND, i,
                             b);
        cpl_table_set_double(fibre_loc_table,
                             MOO_FIBRE_LOC_TABLE_BACKGROUND_ERR, i, be);
    }
    ib = ny - winsize;
    for (int i = ny - winsize; i < ny; i++) {
        b = fluxes[i];
        if (isnan(fluxes[ib])) {
            ib = i;
        }
        else if (!isnan(b) && fluxes[ib] > b) {
            ib = i;
        }
    }
    b = fluxes[ib];
    be = errs[ib];
    for (int i = ny - winhsize; i < ny; i++) {
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BACKGROUND, i,
                             b);
        cpl_table_set_double(fibre_loc_table,
                             MOO_FIBRE_LOC_TABLE_BACKGROUND_ERR, i, be);
    }
    for (int i = winhsize; i < ny - winhsize; i++) {
        ib = i - winhsize;
        for (int j = i - winhsize; j < i + winhsize; j++) {
            b = fluxes[j];
            if (isnan(fluxes[ib])) {
                ib = j;
            }
            else if (!isnan(b) && fluxes[ib] > b) {
                ib = j;
            }
        }
        b = fluxes[ib];
        be = errs[ib];
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BACKGROUND, i,
                             b);
        cpl_table_set_double(fibre_loc_table,
                             MOO_FIBRE_LOC_TABLE_BACKGROUND_ERR, i, be);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_localise_background_fit_poly(cpl_table *fibre_loc_table,
                                  int polydeg,
                                  double klo,
                                  double kup,
                                  int niter)
{
    cpl_error_code error1d = CPL_ERROR_NONE;
    cpl_polynomial *fit1d = NULL;
    cpl_polynomial *errfit1d = NULL;
    cpl_matrix *samppos1d = NULL;
    cpl_vector *fitvals = NULL;
    cpl_vector *errvals = NULL;
    const cpl_boolean sampsym = CPL_FALSE;
    const cpl_size maxdeg1d = polydeg;
    double min_thresh_flux = 0.12;
    double min_kappa_flux = 2;
    double *fluxes =
        cpl_table_get_data_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX);
    double *errs = cpl_table_get_data_double(fibre_loc_table,
                                             MOO_FIBRE_LOC_TABLE_FLUX_ERR);
    double *ytab =
        cpl_table_get_data_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_Y);
    int *fitted =
        cpl_table_get_data_int(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BCKGFITTED);
    int ny = cpl_table_get_nrow(fibre_loc_table);
    fitvals = cpl_vector_new(ny);
    for (int i = 0; i < ny; i++) {
        cpl_vector_set(fitvals, i, fluxes[i]);
    }
    double min_flux =
        moo_vector_get_percentile(fitvals, min_thresh_flux) * min_kappa_flux;
    cpl_vector_delete(fitvals);
    fitvals = NULL;

    int nb = 0;

    for (int i = 0; i < ny; i++) {
        if (fluxes[i] < min_flux) {
            fitted[i] = 0;
        }
        else {
            nb++;
        }
    }

    fit1d = cpl_polynomial_new(1);
    errfit1d = cpl_polynomial_new(1);
    samppos1d = cpl_matrix_new(1, nb);
    fitvals = cpl_vector_new(nb);
    errvals = cpl_vector_new(nb);
    nb = 0;
    for (int i = 0; i < ny; i++) {
        if (fitted[i] == 1) {
            cpl_matrix_set(samppos1d, 0, nb, ytab[i]);
            cpl_vector_set(fitvals, nb, fluxes[i]);
            cpl_vector_set(errvals, nb, errs[i]);
            nb++;
        }
    }
    error1d = cpl_polynomial_fit(fit1d, samppos1d, &sampsym, fitvals, NULL,
                                 CPL_FALSE, NULL, &maxdeg1d);
    error1d = cpl_polynomial_fit(errfit1d, samppos1d, &sampsym, errvals, NULL,
                                 CPL_FALSE, NULL, &maxdeg1d);

    cpl_vector *vdiff = cpl_vector_new(nb);

    for (int i = 0; i < niter; i++) {
        nb = 0;
        for (int j = 0; j < ny; j++) {
            int f = fitted[j];
            if (f == 1) {
                double v = ytab[j];
                double b = _moo_bckg_poly_eval(fit1d, v);
                double flux = fluxes[j];
                double diff = flux - b;
                cpl_vector_set(vdiff, nb, diff);
                nb++;
            }
        }
        nb = 0;
        double stdev = cpl_vector_get_stdev(vdiff);
        cpl_msg_info("test", "niter %d stdev %f", i + 1, stdev);

        for (int j = 0; j < ny; j++) {
            int f = fitted[j];
            if (f == 1) {
                double v = ytab[i];
                double b = _moo_bckg_poly_eval(fit1d, v);
                double flux = fluxes[j];
                double diff = flux - b;
                if ((diff > kup * stdev) || (diff < -klo * stdev)) {
                    fitted[j] = 0;
                }
                else {
                    nb++;
                }
            }
        }
        cpl_polynomial_delete(fit1d);
        cpl_polynomial_delete(errfit1d);
        cpl_matrix_delete(samppos1d);
        cpl_vector_delete(fitvals);
        cpl_vector_delete(errvals);
        cpl_vector_delete(vdiff);

        fit1d = NULL;
        errfit1d = NULL;
        vdiff = NULL;
        samppos1d = NULL;
        fitvals = NULL;
        errvals = NULL;

        vdiff = cpl_vector_new(nb);
        fit1d = cpl_polynomial_new(1);
        errfit1d = cpl_polynomial_new(1);
        samppos1d = cpl_matrix_new(1, nb);
        fitvals = cpl_vector_new(nb);
        errvals = cpl_vector_new(nb);
        nb = 0;

        for (int j = 0; j < ny; j++) {
            if (fitted[j] == 1) {
                cpl_matrix_set(samppos1d, 0, nb, ytab[j]);
                cpl_vector_set(fitvals, nb, fluxes[j]);
                cpl_vector_set(errvals, nb, errs[j]);
                nb++;
            }
        }
        error1d = cpl_polynomial_fit(fit1d, samppos1d, &sampsym, fitvals, NULL,
                                     CPL_FALSE, NULL, &maxdeg1d);
        error1d = cpl_polynomial_fit(errfit1d, samppos1d, &sampsym, errvals,
                                     NULL, CPL_FALSE, NULL, &maxdeg1d);
    }

    for (int i = 0; i < ny; i++) {
        double v = ytab[i];
        double b = _moo_bckg_poly_eval(fit1d, v);
        double be = _moo_bckg_poly_eval(errfit1d, v);
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BACKGROUND, i,
                             b);
        cpl_table_set_double(fibre_loc_table,
                             MOO_FIBRE_LOC_TABLE_BACKGROUND_ERR, i, be);
    }

    cpl_polynomial_delete(fit1d);
    cpl_polynomial_delete(errfit1d);
    cpl_matrix_delete(samppos1d);
    cpl_vector_delete(fitvals);
    cpl_vector_delete(errvals);
    cpl_vector_delete(vdiff);
    return error1d;
}

static cpl_array *
_moo_localise_flag_detected(cpl_table *table, double *max)
{
    cpl_array *flagged = NULL;

    cpl_table_select_all(table);
    cpl_table_and_selected_int(table, MOO_FIBRE_LOC_TABLE_FLAG, CPL_EQUAL_TO,
                               0);
    cpl_table *seltable = cpl_table_extract_selected(table);
    int size = cpl_table_get_nrow(seltable);
    if (size > 0) {
        double *fluxes =
            cpl_table_get_data_double(seltable, MOO_FIBRE_LOC_TABLE_FLUX);
        cpl_vector *vfluxes = cpl_vector_wrap(size, fluxes);
        *max = cpl_vector_get_max(vfluxes) * 0.5;

        cpl_table_and_selected_double(table, MOO_FIBRE_LOC_TABLE_FLUX,
                                      CPL_GREATER_THAN, *max);
        flagged = cpl_table_where_selected(table);
        cpl_vector_unwrap(vfluxes);
    }
    cpl_table_delete(seltable);

    return flagged;
}

static cpl_error_code
_moo_localise_detect_fibres(cpl_array *flagged,
                            cpl_table *fibre_loc_table,
                            cpl_table *detect_loc_table,
                            int n)
{
    cpl_ensure_code(flagged != NULL, CPL_ERROR_NULL_INPUT);

    int nb_flagged = cpl_array_get_size(flagged);
    int ymin = -1;
    int ymax = -1;
    int ysize = 0;
    int nb = 0;
    int nrow = cpl_table_get_nrow(fibre_loc_table);

    for (int i = 0; i < nb_flagged; i++) {
        int nbsel1 = 0;
        int nbsel2 = 0;

        int index = cpl_array_get_cplsize(flagged, i, NULL);
        cpl_table_set_int(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLAG, index, n);
        int fy = (int)cpl_table_get_double(fibre_loc_table,
                                           MOO_FIBRE_LOC_TABLE_Y, index, NULL);
        double flux_y =
            cpl_table_get_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX,
                                 index, NULL);
        cpl_table_select_all(detect_loc_table);
        if (index < (nrow - 1)) {
            int fy_after = (int)cpl_table_get_double(fibre_loc_table,
                                                     MOO_FIBRE_LOC_TABLE_Y,
                                                     index + 1, NULL);

            nbsel1 = cpl_table_and_selected_int(detect_loc_table,
                                                MOO_DETECT_LOC_TABLE_YMIN,
                                                CPL_EQUAL_TO, fy_after);

            if (nbsel1 > 0) {
                cpl_array *sel = cpl_table_where_selected(detect_loc_table);
                for (int j = 0; j < nbsel1; j++) {
                    cpl_size idx = cpl_array_get_cplsize(sel, j, NULL);

                    cpl_table_set_int(detect_loc_table,
                                      MOO_DETECT_LOC_TABLE_YMIN, idx, fy);
                    cpl_table_set_double(detect_loc_table,
                                         MOO_DETECT_LOC_TABLE_FLUX_YMIN, idx,
                                         flux_y);
                }
                cpl_array_delete(sel);
            }
        }

        if (index > 0) {
            int fy_before = (int)cpl_table_get_double(fibre_loc_table,
                                                      MOO_FIBRE_LOC_TABLE_Y,
                                                      index - 1, NULL);

            cpl_table_select_all(detect_loc_table);
            nbsel2 = cpl_table_and_selected_int(detect_loc_table,
                                                MOO_DETECT_LOC_TABLE_YMAX,
                                                CPL_EQUAL_TO, fy_before);

            if (nbsel2 > 0) {
                cpl_array *sel = cpl_table_where_selected(detect_loc_table);

                for (int j = 0; j < nbsel2; j++) {
                    cpl_size idx = cpl_array_get_cplsize(sel, j, NULL);
                    cpl_table_set_int(detect_loc_table,
                                      MOO_DETECT_LOC_TABLE_YMAX, idx, fy);
                    cpl_table_set_double(detect_loc_table,
                                         MOO_DETECT_LOC_TABLE_FLUX_YMAX, idx,
                                         flux_y);
                }
                cpl_array_delete(sel);
            }
        }

        if (nbsel1 == 0 && nbsel2 == 0) {
            if (ymin == -1) {
                nb++;
                ymin = fy;
                ymax = fy;
                ysize = 1;
            }
            else if (fy == (ymin + ysize)) {
                if (index < (nrow - 1)) {
                    int fy_after =
                        (int)cpl_table_get_double(fibre_loc_table,
                                                  MOO_FIBRE_LOC_TABLE_Y,
                                                  index + 1, NULL);
                    int ystep = fy_after - fy;
                    ymax = fy;
                    ysize += ystep;
                }
                else {
                    ysize++;
                }
            }
            else {
                cpl_table_set_size(detect_loc_table, nb);
                cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_NUM,
                                  nb - 1, nb);
                cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_YMIN,
                                  nb - 1, ymin);
                cpl_table_select_all(fibre_loc_table);
                cpl_table_and_selected_double(fibre_loc_table,
                                              MOO_FIBRE_LOC_TABLE_Y,
                                              CPL_EQUAL_TO, ymin);
                cpl_array *sel = cpl_table_where_selected(fibre_loc_table);
                cpl_size idx = cpl_array_get_cplsize(sel, 0, NULL);
                cpl_array_delete(sel);
                double flux_ymin =
                    cpl_table_get_double(fibre_loc_table,
                                         MOO_FIBRE_LOC_TABLE_FLUX, idx, NULL);
                cpl_table_set_double(detect_loc_table,
                                     MOO_DETECT_LOC_TABLE_FLUX_YMIN, nb - 1,
                                     flux_ymin);
                cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_YMAX,
                                  nb - 1, ymax);
                cpl_table_select_all(fibre_loc_table);
                cpl_table_and_selected_double(fibre_loc_table,
                                              MOO_FIBRE_LOC_TABLE_Y,
                                              CPL_EQUAL_TO, ymax);
                sel = cpl_table_where_selected(fibre_loc_table);
                idx = cpl_array_get_cplsize(sel, 0, NULL);
                cpl_array_delete(sel);

                double flux_ymax =
                    cpl_table_get_double(fibre_loc_table,
                                         MOO_FIBRE_LOC_TABLE_FLUX, idx, NULL);
                cpl_table_set_double(detect_loc_table,
                                     MOO_DETECT_LOC_TABLE_FLUX_YMAX, nb - 1,
                                     flux_ymax);
                ymin = fy;
                ymax = fy;
                ysize = 1;
                nb++;
            }
        }
    }

    if (ymin > -1) {
        cpl_table_set_size(detect_loc_table, nb);
        cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_NUM, nb - 1,
                          nb);
        cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_YMIN, nb - 1,
                          ymin);

        cpl_table_select_all(fibre_loc_table);
        cpl_table_and_selected_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_Y,
                                      CPL_EQUAL_TO, ymin);
        cpl_array *sel = cpl_table_where_selected(fibre_loc_table);
        cpl_size idx = cpl_array_get_cplsize(sel, 0, NULL);
        cpl_array_delete(sel);
        double flux_ymin =
            cpl_table_get_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX, idx,
                                 NULL);
        cpl_table_set_double(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YMIN,
                             nb - 1, flux_ymin);

        cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_YMAX, nb - 1,
                          ymax);
        cpl_table_select_all(fibre_loc_table);
        cpl_table_and_selected_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_Y,
                                      CPL_EQUAL_TO, ymax);
        sel = cpl_table_where_selected(fibre_loc_table);
        idx = cpl_array_get_cplsize(sel, 0, NULL);
        cpl_array_delete(sel);

        double flux_ymax =
            cpl_table_get_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX, idx,
                                 NULL);
        cpl_table_set_double(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YMAX,
                             nb - 1, flux_ymax);
    }

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_localise_band(moo_single *single,
                   int cx,
                   int band_width,
                   const char *backg_method,
                   int backg_winhsize,
                   int backg_polydeg,
                   double backg_clip_kappalow,
                   double backg_clip_kappaup,
                   int backg_clip_niter,
                   double detect_noisemult_thresh,
                   int detect_niter,
                   float frac,
                   int nb_expected,
                   int *nb_detected,
                   cpl_table **out_detect_loc_table,
                   cpl_table **out_fibre_loc_table)
{
    cpl_table *detect_loc_table = NULL;
    // background fit
    double klo = backg_clip_kappalow;
    double kup = backg_clip_kappaup;
    int niter = backg_clip_niter;

    int polydeg = backg_polydeg;
    // fibre detection
    double kthresh = detect_noisemult_thresh;

    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    cpl_msg_indent_more();
    hdrl_image *himg = moo_single_get_image(single);

    int ny = hdrl_image_get_size_y(himg);
    int y;

    cpl_table *fibre_loc_table = cpl_table_new(ny);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_Y,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX_ERR,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BAD,
                         CPL_TYPE_INT);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BACKGROUND,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BACKGROUND_ERR,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_THRESH,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BCKGFITTED,
                         CPL_TYPE_INT);
    cpl_table_new_column(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLAG,
                         CPL_TYPE_INT);

    detect_loc_table = cpl_table_new(0);
    cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_NUM,
                         CPL_TYPE_INT);
    cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YMIN,
                         CPL_TYPE_INT);
    cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YMAX,
                         CPL_TYPE_INT);
    cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YMIN,
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YMAX,
                         CPL_TYPE_DOUBLE);

    for (y = 1; y <= ny; y++) {
        hdrl_value val;
        hdrl_image *extract = hdrl_image_extract(himg, cx - band_width / 2, y,
                                                 cx + band_width / 2, y);

        val = hdrl_image_get_median(extract);

        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_Y, y - 1,
                             (double)y);
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX, y - 1,
                             val.data);
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX_ERR,
                             y - 1, val.error);
        cpl_table_set_int(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BAD, y - 1,
                          isnan(val.data));
        cpl_table_set_int(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BCKGFITTED,
                          y - 1, 1);
        cpl_table_set_int(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLAG, y - 1, 0);
        hdrl_image_delete(extract);
    }

    // filter bad data
    cpl_table_select_all(fibre_loc_table);
    cpl_table_and_selected_int(fibre_loc_table, MOO_FIBRE_LOC_TABLE_BAD,
                               CPL_EQUAL_TO, 1);
    cpl_table_erase_selected(fibre_loc_table);

    ny = cpl_table_get_nrow(fibre_loc_table);
    if (strcmp(backg_method, MOO_LOCALISE_BACKG_METHOD_POLYNOMIAL) == 0) {
        cpl_msg_info(
            "moo_localise",
            "Background polynomial (%d degree) fit : kappa (lo %f, up %f) \
                     niter: %d",
            polydeg, klo, kup, niter);

        _moo_localise_background_fit_poly(fibre_loc_table, polydeg, klo, kup,
                                          niter);
    }
    else {
        cpl_msg_info("moo_localise",
                     "Background running median fit (winhsize %d)",
                     backg_winhsize);

        _moo_localise_background_fit_rmin(fibre_loc_table, backg_winhsize);
    }


    double *backgrounds =
        cpl_table_get_data_double(fibre_loc_table,
                                  MOO_FIBRE_LOC_TABLE_BACKGROUND);
    double *backgrounds_err =
        cpl_table_get_data_double(fibre_loc_table,
                                  MOO_FIBRE_LOC_TABLE_BACKGROUND_ERR);
    for (int i = 0; i < ny; i++) {
        double t = backgrounds[i] + kthresh * backgrounds_err[i];
        cpl_table_set_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_THRESH, i, t);
    }

    if (strcmp(backg_method, MOO_LOCALISE_BACKG_METHOD_POLYNOMIAL) == 0) {
        for (int n = 1; n <= detect_niter; n++) {
            double thresh = 0;
            cpl_array *flagged =
                _moo_localise_flag_detected(fibre_loc_table, &thresh);

            cpl_table_select_all(fibre_loc_table);
            int nb_sel = cpl_table_and_selected_int(fibre_loc_table,
                                                    MOO_FIBRE_LOC_TABLE_FLAG,
                                                    CPL_EQUAL_TO, 0);
            int lower =
                cpl_table_and_selected_double(fibre_loc_table,
                                              MOO_FIBRE_LOC_TABLE_THRESH,
                                              CPL_GREATER_THAN, thresh);

            if (lower > 0) {
                cpl_msg_info("test",
                             "niter %d/%d : find in sel (%d) "
                             " \
          "
                             "%d threshold points greater than %f",
                             n, detect_niter, nb_sel, lower, thresh);
                cpl_array_delete(flagged);
                break;
            }
            _moo_localise_detect_fibres(flagged, fibre_loc_table,
                                        detect_loc_table, n);
            cpl_array_delete(flagged);
        }
    }
    else {
        cpl_table_unselect_all(fibre_loc_table);
        cpl_table_or_selected(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX,
                              CPL_GREATER_THAN, MOO_FIBRE_LOC_TABLE_THRESH);
        cpl_array *flagged = cpl_table_where_selected(fibre_loc_table);
        int nb_peaks = cpl_array_get_size(flagged);
        cpl_msg_info("moo_localise",
                     "Filter peaks using ktresh (%f) : find %d peaks", kthresh,
                     nb_peaks);
        _moo_localise_detect_fibres(flagged, fibre_loc_table, detect_loc_table,
                                    1);
        cpl_array_delete(flagged);
    }
#if MOO_DEBUG_LOCALISE_BAND
    {
        char *testname = NULL;

        if (strcmp(backg_method, MOO_LOCALISE_BACKG_METHOD_POLYNOMIAL) == 0) {
            testname = cpl_sprintf("%s_FIB_LOC_TABLE_n%d.fits", single->extname,
                                   niter);
        }
        else {
            testname = cpl_sprintf("%s_FIB_LOC_TABLE_h%d.fits", single->extname,
                                   backg_winhsize);
        }
        cpl_table_save(fibre_loc_table, NULL, NULL, testname, CPL_IO_CREATE);
        cpl_free(testname);
    }
#endif
    cpl_propertylist *slist = cpl_propertylist_new();
    cpl_propertylist_append_bool(slist, MOO_DETECT_LOC_TABLE_YMIN, FALSE);
    cpl_table_sort(detect_loc_table, slist);
    cpl_propertylist_delete(slist);

    int nrow = cpl_table_get_nrow(detect_loc_table);
    int nbdetected_valid_flux = 0;
    for (int i = 0; i < nrow; i++) {
        int rej;
        int ymin = cpl_table_get_int(detect_loc_table,
                                     MOO_DETECT_LOC_TABLE_YMIN, i, &rej);
        int ymax = cpl_table_get_int(detect_loc_table,
                                     MOO_DETECT_LOC_TABLE_YMAX, i, &rej);
        if (ymin < ymax) {
            nbdetected_valid_flux++;
        }
        else {
            cpl_table_set_invalid(detect_loc_table,
                                  MOO_DETECT_LOC_TABLE_FLUX_YMIN, i);
        }
    }
    cpl_table_erase_invalid(detect_loc_table);

    cpl_msg_info(
        "moo_localise",
        "Number of detected fibres vs. expected valid fibres : %d / %d",
        nbdetected_valid_flux, nb_expected);

    *nb_detected = nbdetected_valid_flux;
    *out_detect_loc_table = detect_loc_table;
    *out_fibre_loc_table = fibre_loc_table;
    cpl_msg_indent_less();
    return CPL_ERROR_NONE;
}

static fibre *
_moo_localise_bandauto(moo_single *single,
                       int cx,
                       int band_width,
                       const char *backg_method,
                       int backg_winhsize,
                       int backg_polydeg,
                       double backg_clip_kappalow,
                       double backg_clip_kappaup,
                       int backg_clip_niter,
                       int detect_niter,
                       float frac,
                       int nb_expected,
                       int *out_nb_detected,
                       double *out_detected_noisemult_thresh)
{
    fibre *fibres_tab = NULL;
    double low_limit = NAN;
    double high_limit = NAN;
    int max_niter = 50;
    double detect_noisemult_thresh = 1.0;
    int nb_detected = 0;
    cpl_table *detect_loc_table = NULL;
    cpl_table *fibre_loc_table = NULL;
    double klo = backg_clip_kappalow;
    double kup = backg_clip_kappaup;
    int polydeg = backg_polydeg;
    cpl_msg_info("moo_localise",
                 "1) Detect centroid using "
                 "central band (center=%d,width=%d)",
                 cx, band_width);
    cpl_msg_indent_more();

    if (strcmp(backg_method, MOO_LOCALISE_BACKG_METHOD_POLYNOMIAL) == 0) {
        cpl_msg_info("moo_localise",
                     "background method %s "
                     "background fit with degree %d kappa_lo %f kappa_up "
                     "%f niter %d",
                     backg_method, polydeg, klo, kup, backg_clip_niter);
    }
    else {
        cpl_msg_info("moo_localise",
                     "Estimate background with method %s using winhsize %d",
                     backg_method, backg_winhsize);
    }

    for (int niter = 1; niter <= max_niter; niter++) {
        cpl_msg_info(__func__, "do niter %d/%d: using %f range[%f,%f]", niter,
                     max_niter, detect_noisemult_thresh, low_limit, high_limit);
        _moo_localise_band(single, cx, band_width, backg_method, backg_winhsize,
                           backg_polydeg, backg_clip_kappalow,
                           backg_clip_kappaup, backg_clip_niter,
                           detect_noisemult_thresh, detect_niter, frac,
                           nb_expected, &nb_detected, &detect_loc_table,
                           &fibre_loc_table);

        if (nb_detected == nb_expected) {
            break;
        }
        else {
            cpl_table_delete(detect_loc_table);
            cpl_table_delete(fibre_loc_table);
            detect_loc_table = NULL;
            fibre_loc_table = NULL;
        }
        if (nb_detected > nb_expected) {
            low_limit = detect_noisemult_thresh;
            detect_noisemult_thresh *= 2;
        }
        else {
            high_limit = detect_noisemult_thresh;
            detect_noisemult_thresh /= 2;
        }
        if (!isnan(low_limit) && !isnan(high_limit)) {
            detect_noisemult_thresh = 0.5 * (low_limit + high_limit);
        }
    }
    if (nb_detected < nb_expected) {
        cpl_msg_error("moo_localise", "Failure in detection mask computation");
        char *testname = NULL;

        testname = cpl_sprintf("DETECT_LOC_TABLE_%s.fits", single->extname);

        cpl_table_save(detect_loc_table, NULL, NULL, testname, CPL_IO_CREATE);
        cpl_free(testname);
    }
    else {
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YBCKGMIN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YBCKGMAX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table,
                             MOO_DETECT_LOC_TABLE_FLUX_YBCKGMIN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table,
                             MOO_DETECT_LOC_TABLE_FLUX_YBCKGMAX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YTMIN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YTMAX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YTMIN,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YTMAX,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YBARY,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_FLUX_YBARY,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_YWIDTH,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(detect_loc_table, MOO_DETECT_LOC_TABLE_FIBNUM,
                             CPL_TYPE_INT);
        fibres_tab = (fibre *)cpl_calloc(nb_detected, sizeof(fibre));

        for (int f = 0; f < nb_detected; f++) {
            fibres_tab[f].health = MOO_FIBRES_TABLE_HEALTH_GOOD;
        }
    }
    int ny = cpl_table_get_nrow(fibre_loc_table);
    double *tfluxes =
        cpl_table_get_data_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_FLUX);
    double *tposy =
        cpl_table_get_data_double(fibre_loc_table, MOO_FIBRE_LOC_TABLE_Y);
    hdrl_image *himg = moo_single_get_image(single);
    int size_y = hdrl_image_get_size_y(himg);

    cpl_vector *bandflux = cpl_vector_new(size_y);
    cpl_vector_fill(bandflux, NAN);
    for (int i = 0; i < ny; i++) {
        int px = (int)tposy[i];
        double flux = tfluxes[i];
        cpl_vector_set(bandflux, px - 1, flux);
    }
    double *fluxes = cpl_vector_get_data(bandflux);
    if (fibres_tab != NULL) {
        for (int i = 0; i < nb_detected; i++) {
            int ymin = cpl_table_get_int(detect_loc_table,
                                         MOO_DETECT_LOC_TABLE_YMIN, i, NULL);
            int ymax = cpl_table_get_int(detect_loc_table,
                                         MOO_DETECT_LOC_TABLE_YMAX, i, NULL);
            int ysize = ymax - ymin + 1;
            cpl_vector *vfluxes = cpl_vector_new(ysize);
            cpl_vector *pos = cpl_vector_new(ysize);

            for (int j = ymin; j <= ymax; j++) {
                double f = fluxes[j - 1];
                cpl_vector_set(vfluxes, j - ymin, f);
                cpl_vector_set(pos, j - ymin, j);
            }

            cpl_bivector *points = cpl_bivector_wrap_vectors(pos, vfluxes);
            double center = NAN;
            double width = NAN;

            moo_barycenter_fit(points, &center, &width);

            cpl_bivector_unwrap_vectors(points);
            cpl_vector_delete(vfluxes);
            cpl_vector_delete(pos);

            double thresh_low = NAN, thresh_up = NAN, b_ylow = NAN, b_yup,
                   y1 = NAN, y2 = NAN;
            _moo_profile_find_background_flux(bandflux, center, width, frac,
                                              &b_ylow, &b_yup, &thresh_low,
                                              &thresh_up);
            _moo_profile_find_thresh_positions(bandflux, center, thresh_low,
                                               thresh_up, &y1, &y2);


            cpl_table_set_double(detect_loc_table, MOO_DETECT_LOC_TABLE_YTMIN,
                                 i, y1);
            cpl_table_set_double(detect_loc_table, MOO_DETECT_LOC_TABLE_YTMAX,
                                 i, y2);
            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_YBCKGMIN, i, b_ylow);
            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_YBCKGMAX, i, b_yup);

            double fb_ylow = _moo_get_flux(bandflux, b_ylow);

            double fb_yup = _moo_get_flux(bandflux, b_yup);

            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_FLUX_YBCKGMIN, i,
                                 fb_ylow);
            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_FLUX_YBCKGMAX, i, fb_yup);
            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_FLUX_YTMIN, i,
                                 thresh_low);
            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_FLUX_YTMAX, i, thresh_up);

            points = _moo_extract_profile(bandflux, y1, y2);
            moo_barycenter_fit(points, &center, &width);

            cpl_table_set_double(detect_loc_table, MOO_DETECT_LOC_TABLE_YBARY,
                                 i, center);
            double fbary = _moo_get_flux(bandflux, center);
            cpl_table_set_double(detect_loc_table,
                                 MOO_DETECT_LOC_TABLE_FLUX_YBARY, i, fbary);
            cpl_table_set_double(detect_loc_table, MOO_DETECT_LOC_TABLE_YWIDTH,
                                 i, width);
            cpl_bivector_delete(points);

            int fibidx = i;
            fibres_tab[fibidx].ymin = y1;
            fibres_tab[fibidx].ymax = y2;
            fibres_tab[fibidx].center = center;
            fibres_tab[fibidx].width = width;
            int fibnum = fibidx + 1;
            fibres_tab[fibidx].num = fibnum;
            cpl_table_set_int(detect_loc_table, MOO_DETECT_LOC_TABLE_FIBNUM, i,
                              fibnum);
        }
    }
#if MOO_DEBUG_LOCALISE_BAND
    {
        char *testname =
            cpl_sprintf("%s_DETECT_LOC_TABLE.fits", single->extname);
        cpl_table_save(detect_loc_table, NULL, NULL, testname, CPL_IO_CREATE);
        cpl_free(testname);
    }
#endif
    *out_nb_detected = nb_detected;
    *out_detected_noisemult_thresh = detect_noisemult_thresh;
    cpl_vector_delete(bandflux);
    cpl_table_delete(fibre_loc_table);
    cpl_table_delete(detect_loc_table);
    return fibres_tab;
}

static moo_loc_single *
_moo_localise_single(moo_single *single,
                     moo_localise_params *params,
                     cpl_table *loc_fibre_table,
                     int nb,
                     moo_detector_type type,
                     int ntas,
                     const char *filename,
                     int degpoly,
                     double goodptsfrac_min,
                     cpl_array *indexes,
                     float *goodfrac)
{
    moo_loc_single *res = NULL;
    fibre *fibres_tab = NULL;
    cpl_error_code status = CPL_ERROR_NONE;
    int nb_good = 0;
    cpl_image *m_centroids = NULL;
    cpl_image *f_centroids = NULL;
    cpl_image *m_wlow = NULL;
    cpl_image *f_wlow = NULL;
    cpl_image *m_wup = NULL;
    cpl_image *f_wup = NULL;
    cpl_image *flags = NULL;
    cpl_table **all_tracked = NULL;
    cpl_ensure(single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc_fibre_table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_try_check(nb_good = moo_fibres_table_filter_health(loc_fibre_table),
                  " ");
    int *health =
        cpl_table_get_data_int(loc_fibre_table, MOO_FIBRES_TABLE_HEALTH);

    hdrl_image *himg = moo_single_get_image(single);
    cpl_image *img = hdrl_image_get_image(himg);

    int didx = type + (ntas - 1) * 3;
    int nx = cpl_image_get_size_x(img);
    int band_width = params->centralwidth;
    int cx = params->centralpos;
    const char *backg_method = params->backg_method;

    int backg_polydeg = params->backg_polydeg[didx];
    int backg_winhsize = params->backg_winhsize[didx];
    double backg_clip_kappalow = params->backg_clip_kappalow[didx];
    double backg_clip_kappaup = params->backg_clip_kappaup[didx];
    int backg_clip_niter = params->backg_clip_niter[didx];
    double detect_noisemult_thresh = 1;
    int detect_niter = params->detect_niter[didx];
    float frac = params->relativethresh;
    double ref_snr = params->ref_snr[didx];
    int xgap_max = params->xgap_max;
    int nb_failed = 0;
    int nb_detected = 0;
    int nb_broken = 0;
    fibres_tab =
        _moo_localise_bandauto(single, cx, band_width, backg_method,
                               backg_winhsize, backg_polydeg,
                               backg_clip_kappalow, backg_clip_kappaup,
                               backg_clip_niter, detect_niter, frac, nb_good,
                               &nb_detected, &detect_noisemult_thresh);
    cpl_ensure(fibres_tab != NULL, CPL_ERROR_UNSPECIFIED, NULL);

    const char *extname = moo_detector_get_extname(type, ntas);

    res = moo_loc_single_create(filename, extname);
    res->header = cpl_propertylist_new();
    moo_qc_set_qc_detect_noisemult(res->header, detect_noisemult_thresh);
    res->m_centroids = cpl_image_new(nx, nb_detected, CPL_TYPE_DOUBLE);
    res->f_centroids = cpl_image_new(nx, nb_detected, CPL_TYPE_DOUBLE);
    res->m_wlow = cpl_image_new(nx, nb_detected, CPL_TYPE_DOUBLE);
    res->f_wlow = cpl_image_new(nx, nb_detected, CPL_TYPE_DOUBLE);
    res->m_wup = cpl_image_new(nx, nb_detected, CPL_TYPE_DOUBLE);
    res->f_wup = cpl_image_new(nx, nb_detected, CPL_TYPE_DOUBLE);
    res->flags = cpl_image_new(nx, nb_detected, CPL_TYPE_INT);

    double wdiff_lim = params->wdiff_lim;
    double ydiff_lim = params->ydiff_lim;
    int xlim_hwin = params->loc_xlim_hwin;
    double xlim_fracmin = params->loc_xlim_fracmin;
    cpl_msg_info("moo_localise",
                 "2) Loop on fibre localisation (tracking + fitting)");
    cpl_msg_indent_more();
    cpl_msg_info("moo_localise", "min_snr %f xgap_max %d goodptsfrac-min %f",
                 ref_snr, xgap_max, goodptsfrac_min);

    all_tracked = cpl_calloc(nb_detected, sizeof(cpl_table *));

#ifdef _OPENMP

#pragma omp parallel default(none)                                             \
    shared(all_tracked, nb_detected, fibres_tab, himg, cx, degpoly, wdiff_lim, \
               ydiff_lim, xlim_hwin, xlim_fracmin, frac, res, nx, ref_snr,     \
               xgap_max, goodptsfrac_min) reduction(+ : nb_failed, nb_broken)
    {
#pragma omp for
#endif
        for (int f = 1; f <= nb_detected; f++) {
            //for (int f = 504; f <= 504; f++) {
            cpl_error_code lstatus = CPL_ERROR_NONE;
            fibre *fib = &(fibres_tab[f - 1]);
            int num = fib->num;
            lstatus =
                _moo_localise_fibre(fib, himg, cx, degpoly, wdiff_lim,
                                    ydiff_lim, frac, ref_snr, xgap_max,
                                    goodptsfrac_min, xlim_hwin, xlim_fracmin,
                                    res, &all_tracked[f - 1]);
            if (lstatus != CPL_ERROR_NONE) {
                nb_failed += 1;
            }
            else if (fib->health == MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN) {
                nb_broken += 1;
                for (int i = 1; i <= nx; i++) {
                    cpl_image_set(res->m_centroids, i, num, NAN);
                    cpl_image_set(res->m_wlow, i, num, NAN);
                    cpl_image_set(res->m_wup, i, num, NAN);
                    cpl_image_set(res->flags, i, num, MOONS_FLAG_BROKEN_FIBRE);
                    cpl_image_set(res->f_centroids, i, num, NAN);
                    cpl_image_set(res->f_wlow, i, num, NAN);
                    cpl_image_set(res->f_wup, i, num, NAN);
                }
                cpl_msg_info("moo_localise",
                             "New broken fibre %d/%d : goodfrac %f", num,
                             nb_detected, fib->goodfrac);
            }
        }
#ifdef _OPENMP
    }
#endif
#if MOO_DEBUG_LOCALISE_TRACKING
    {
        char *name = cpl_sprintf("%s_TRACKING_LOC_TABLE.fits", res->extname);
        cpl_table *test = NULL;
        int first = 0;
        for (int i = 0; i < nb_detected; i++) {
            cpl_table *t = all_tracked[i];
            if (t != NULL) {
                test = cpl_table_duplicate(t);
                first = i + 1;
                break;
            }
        }
        if (test != NULL) {
            for (int i = first; i < nb_detected; i++) {
                cpl_table *t = all_tracked[i];
                if (t != NULL) {
                    moo_table_append(test, t);
                }
            }
            cpl_table_save(test, NULL, NULL, name, CPL_IO_CREATE);
            cpl_table_delete(test);
        }
        cpl_free(name);
    }
#endif

    int nb_localise = nb_detected - nb_broken;
    cpl_msg_info("moo_localise",
                 "Number of detected fibres after localisation : %d",
                 nb_localise);
    cpl_msg_indent_less();

    if (nb_failed > 0) {
        status = CPL_ERROR_UNSPECIFIED;
        cpl_error_set(__func__, status);
    }
    int fib_idx = 0;

    if ((nb_good == nb_detected) || (nb_good == nb_localise)) {
        m_centroids = cpl_image_new(nx, nb, CPL_TYPE_DOUBLE);
        f_centroids = cpl_image_new(nx, nb, CPL_TYPE_DOUBLE);
        m_wlow = cpl_image_new(nx, nb, CPL_TYPE_DOUBLE);
        f_wlow = cpl_image_new(nx, nb, CPL_TYPE_DOUBLE);
        m_wup = cpl_image_new(nx, nb, CPL_TYPE_DOUBLE);
        f_wup = cpl_image_new(nx, nb, CPL_TYPE_DOUBLE);
        flags = cpl_image_new(nx, nb, CPL_TYPE_INT);

        for (int f = 0; f < nb; f++) {
            int idx = cpl_array_get_cplsize(indexes, f, NULL);
            int h = health[idx];
            int num = f + 1;

            cpl_table_set_int(loc_fibre_table, MOO_FIBRES_TABLE_INDEXEXT, idx,
                              num);

            if (h == MOO_FIBRES_TABLE_HEALTH_BROKEN) {
                for (int i = 1; i <= nx; i++) {
                    cpl_image_set(m_centroids, i, num, NAN);
                    cpl_image_set(m_wlow, i, num, NAN);
                    cpl_image_set(m_wup, i, num, NAN);
                    cpl_image_set(flags, i, num, MOONS_FLAG_BROKEN_FIBRE);
                    cpl_image_set(f_centroids, i, num, NAN);
                    cpl_image_set(f_wlow, i, num, NAN);
                    cpl_image_set(f_wup, i, num, NAN);
                }
            }
            else {
                int fh = fibres_tab[fib_idx].health;
                if ((nb_good == nb_detected) &&
                    (fh == MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN)) {
                    cpl_table_set_int(loc_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                      idx, fh);
                }
                else {
                    while (fibres_tab[fib_idx].health ==
                           MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN) {
                        fib_idx++;
                    }
                }
                int fnum = fibres_tab[fib_idx].num;

                goodfrac[idx] = fibres_tab[fib_idx].goodfrac;

                for (int i = 1; i <= nx; i++) {
                    int rej;
                    double mcentroid =
                        cpl_image_get(res->m_centroids, i, fnum, &rej);
                    double fcentroid =
                        cpl_image_get(res->f_centroids, i, fnum, &rej);
                    double mwlo = cpl_image_get(res->m_wlow, i, fnum, &rej);
                    double mwu = cpl_image_get(res->m_wup, i, fnum, &rej);
                    double fwlo = cpl_image_get(res->f_wlow, i, fnum, &rej);
                    double fwu = cpl_image_get(res->f_wup, i, fnum, &rej);
                    double flag = cpl_image_get(res->flags, i, fnum, &rej);
                    cpl_image_set(m_centroids, i, num, mcentroid);
                    cpl_image_set(m_wlow, i, num, mwlo);
                    cpl_image_set(m_wup, i, num, mwu);
                    cpl_image_set(flags, i, num, flag);
                    cpl_image_set(f_centroids, i, num, fcentroid);
                    cpl_image_set(f_wlow, i, num, fwlo);
                    cpl_image_set(f_wup, i, num, fwu);
                }
                fib_idx++;
            }
        }
        cpl_image_delete(res->m_centroids);
        res->m_centroids = m_centroids;
        cpl_image_delete(res->f_centroids);
        res->f_centroids = f_centroids;
        cpl_image_delete(res->m_wlow);
        res->m_wlow = m_wlow;
        cpl_image_delete(res->m_wup);
        res->m_wup = m_wup;
        cpl_image_delete(res->flags);
        res->flags = flags;
        cpl_image_delete(res->f_wlow);
        res->f_wlow = f_wlow;
        cpl_image_delete(res->f_wup);
        res->f_wup = f_wup;
    }
    else {
        cpl_error_set_message(
            "moo_localise", CPL_ERROR_UNSPECIFIED,
            "Localise fibres (%d) and expected fibres (%d) doesnt match",
            nb_localise, nb_good);
        status = CPL_ERROR_UNSPECIFIED;
    }

    cpl_free(fibres_tab);

moo_try_cleanup:
    if (all_tracked != NULL) {
        for (int i = 0; i < nb_detected; i++) {
            if (all_tracked[i] != NULL) {
                cpl_table_delete(all_tracked[i]);
            }
        }
        cpl_free(all_tracked);
    }
    if (status != CPL_ERROR_NONE) {
        moo_loc_single_delete(res);
    }
    return res;
}

static moo_loc *
_moo_localise_create_guess(moo_det *det,
                           moo_localise_params *params,
                           const char *locname,
                           int badpix_level)
{
    cpl_array *indexes = NULL;
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(det != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(locname != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_table *fibres_table = moo_det_get_fibre_table(det);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    const char *filename = locname;
    moo_loc *loc = NULL;

    moo_try_check(loc = moo_loc_create(filename), " ");

    int i, j;
    cpl_msg_info("moo_localise", "Create GUESS localisation");
    cpl_msg_indent_more();

    cpl_table *loc_fibre_table = cpl_table_duplicate(fibres_table);

    loc->fibre_table = loc_fibre_table;
    moo_fibres_table_add_locguess_cols(loc_fibre_table);
    cpl_table_new_column(loc_fibre_table, MOO_FIBRES_TABLE_INDEXEXT,
                         CPL_TYPE_INT);
    const char *goodfracname[] = { MOO_FIBRES_TABLE_GOODPTSFRAC_RI,
                                   MOO_FIBRES_TABLE_GOODPTSFRAC_YJ,
                                   MOO_FIBRES_TABLE_GOODPTSFRAC_H };

    for (i = 1; i <= 2; i++) {
        indexes = moo_fibres_table_get_spectro_index(loc_fibre_table, i);
        int nbtotal = cpl_array_get_size(indexes);

        for (j = 0; j < 3; j++) {
            int idx = (i - 1) * 3 + j;
            double goodptsfrac_min = params->goodptsfrac_min[idx] * 100.;
            int dx = params->polydeg[idx];
            moo_single *s1 = moo_det_load_single(det, j, i, badpix_level);
            float *goodfrac =
                cpl_table_get_data_float(loc_fibre_table, goodfracname[j]);

            if (s1 != NULL) {
                cpl_msg_info("moo_localise",
                             "Localising %s : Total number of fibres from "
                             "FIBRE_TABLE : %d",
                             moo_detector_get_extname(j, i), nbtotal);
                cpl_msg_indent_more();
                moo_loc_single *ls = NULL;

                moo_try_check(ls = _moo_localise_single(s1, params,
                                                        loc_fibre_table,
                                                        nbtotal, j, i, filename,
                                                        dx, goodptsfrac_min,
                                                        indexes, goodfrac),
                              " ");

                if (ls != NULL) {
                    moo_loc_single_compute_qc_guess(ls, dx);
                    moo_loc_add_single(loc, ls, j, i, params->keep_points);
                }
                cpl_msg_indent_less();
            }
        }

        cpl_array_delete(indexes);
    }

    cpl_msg_indent_less();
    cpl_table_select_all(loc_fibre_table);
    int nb_newly_broken =
        cpl_table_and_selected_int(loc_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                   CPL_EQUAL_TO,
                                   MOO_FIBRES_TABLE_HEALTH_NEWLY_BROKEN);

    if (nb_newly_broken > 0) {
        indexes = cpl_table_where_selected(loc_fibre_table);
        for (i = 0; i < nb_newly_broken; i++) {
            int idx = cpl_array_get_cplsize(indexes, i, NULL);
            cpl_table_set_int(loc_fibre_table, MOO_FIBRES_TABLE_HEALTH, idx,
                              MOO_FIBRES_TABLE_HEALTH_BROKEN);
        }
        cpl_array_delete(indexes);
    }
    moo_loc_add_fibre_table(loc, loc_fibre_table);
    // dump error from the state

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        if (indexes != NULL) {
            cpl_array_delete(indexes);
        }
        moo_loc_delete(loc);
        loc = NULL;
    }
    return loc;
}
/**END CREATE_GUESS*************************************************************/

/*----------------------------------------------------------------------------*/
/**
  @brief    To localise the centroid of fibre spectrum on frames
  @param    det _DET_ frame to localize
  @param    guess_loc FF_TRACE_GUESS frame or NULL
  @param    params the localisation parameters
  @param    locname the file name of the result
  @return   the FF_TRACE or NULL in error case

 * This function is to find the localisation of the centroid of each fibre spectrum
 * on CCD Frame, then generate the FF_TRACE. It is able be excuted in two cases:
 * 1. construct the FF_TRACE from scratch
 * 2. construct the FF_TRACE based on user input FF_TRACE_GUESS
 * _Flags considered as bad :BADPIX_COSMETIC | BADPIX_HOT

_Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if _DET_ has not a fibre table
 */
/*----------------------------------------------------------------------------*/
moo_loc *
moo_localise(moo_det *det,
             const cpl_frame *guess_loc,
             moo_localise_params *params,
             const char *locname)
{
    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_loc *result = NULL;
    moo_loc *guess = NULL;

    int badpix_level =
        MOO_BADPIX_COSMETIC | MOO_BADPIX_HOT | MOO_BADPIX_OUTSIDE_DATA_RANGE;

    if (guess_loc != NULL) {
        moo_try_check(guess = moo_loc_load(guess_loc), " ");
        moo_try_check(result = _moo_localise_from_guess(det, guess, params,
                                                        locname, badpix_level),
                      " ");
    }
    else {
        moo_try_check(result = _moo_localise_create_guess(det, params, locname,
                                                          badpix_level),
                      " ");
    }

moo_try_cleanup:
    moo_loc_delete(guess);
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_loc_delete(result);
        result = NULL;
    }
    return result;
}
/**@}*/
