/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_mask.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_mask MASK format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_mask
 * 
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_mask.h"
 * @endcode
 */

 
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_mask  
  @return   1 newly allocated moo_mask or NULL in case of an error

  The returned object must be deallocated using moo_mask_delete().  
 
 */
/*----------------------------------------------------------------------------*/
moo_mask* moo_mask_new(void)
{
    moo_mask* res = cpl_calloc(1,sizeof(moo_mask));        
    return res;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the cpl_mask associated to given type,num
  @param    self    moo_mask to read
  @param    type the detector type
  @param    num the TAS number
  @return   the relevant cpl_mask or NULL
 
 */
/*----------------------------------------------------------------------------*/
cpl_mask* moo_mask_get(moo_mask* self,moo_detector_type type, int num)
{
    cpl_mask* result = NULL;
    
    cpl_ensure(self!=NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(num>=1 && num<=2, CPL_ERROR_ILLEGAL_INPUT, NULL);    
    
    result = self->data[type+(num-1)*3];
        
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the cpl_mask associated to given type,num
  @param    self    moo_mask to read
  @param    type the detector type
  @param    num the TAS number
  @param    mask the new cpl_mask to set
 
 */
/*----------------------------------------------------------------------------*/
cpl_error_code moo_mask_set(moo_mask* self,moo_detector_type type, int num,
        cpl_mask* mask)
{        
    cpl_ensure_code(self!=NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num>=1 && num<=2, CPL_ERROR_ILLEGAL_INPUT);    
    
    self->data[type+(num-1)*3] = mask;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_mask
  @param    self    moo_mask to delete
  @return   void

  If the moo_mask @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void moo_mask_delete(moo_mask* self)
{
    if ( self!= NULL){ 
        int i;
        
        for(i = 0; i< 6; i++){            
            if ( self->data[i]!= NULL){
                cpl_mask_delete(self->data[i]);
            }        
        }
        
        cpl_free(self);        
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_mask to a FITS file
  @param    self moo_mask to write to disk or NULL
  @param    filename Name of the file to write  
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_map to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void moo_mask_save(moo_mask* self, const char* filename)
{   
    if ( self != NULL){                  
        moo_fits_create(filename);
        for(int i = 0; i< 2; i++){
            for(int j = 0; j< 3; j++){                
                const char* extname = moo_detector_get_extname(j, i+1);            
                moo_fits_write_extension_mask(self->data[j+i*3], filename, 
                        MASK_EXTENSION, extname, NULL);
            }
        }                
    }
}

/**@}*/
