/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <assert.h>
#include "moo_masklist.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_masklist MASK_LIST format
 * @ingroup moo_data
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_masklist  
  @return   1 newly allocated moo_masklist or NULL in case of an error

  The returned object must be deallocated using moo_detlist_delete().  
 
 */
moo_masklist *
moo_masklist_new(void)
{
    return (moo_masklist *)cpl_calloc(1, sizeof(moo_masklist));
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_masklist from the given MASK frameset
  @return   1 newly allocated moo_masklist or NULL in case of an error

  The returned object must be deallocated using moo_masklist_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
moo_masklist *
moo_masklist_create(int size)
{
    cpl_ensure(size > 0, CPL_ERROR_NULL_INPUT, NULL);
    int i;
    moo_masklist *res = moo_masklist_new();

    res->list = cpl_calloc(size, sizeof(moo_mask));
    res->size = size;
    for (i = 0; i < size; i++) {
        res->list[i] = moo_mask_new();
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_masklist
  @brief    Get the number of MASK in the masklist
  @param    self    the list of MASK
  @return   The number of MASK or -1 on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_size
moo_masklist_get_size(const moo_masklist *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    assert(self->size >= 0);

    return self->size;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_masklist
  @brief    Get the MASK at the position i in the list
  @param    self    the list of MASK
  @param    i       position in the list
  @return   MASK element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
moo_mask *
moo_masklist_get(moo_masklist *self, int i)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(i >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int size = moo_masklist_get_size(self);
    cpl_ensure(i < size, CPL_ERROR_ILLEGAL_INPUT, NULL);

    return self->list[i];
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_masklist
  @brief    Get the CPL_MASK at the position i,type,num in the list
  @param    self    the list of MASK
  @param    i       position in the list
  @param    type    the detector type
  @param    num     the number of TAS
  @return   MASK element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
cpl_mask *
moo_masklist_get_mask(moo_masklist *self,
                      int i,
                      moo_detector_type type,
                      int num)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    moo_mask *mask = moo_masklist_get(self, i);
    return moo_mask_get(mask, type, num);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free all memory used by a moo_masklist object including the MASK
  @param    self    The MASK list or NULL
  @return   void    

 */
void
moo_masklist_delete(moo_masklist *self)
{
    if (self != NULL) {
        for (int i = 0; i < self->size; i++) {
            moo_mask_delete(self->list[i]);
        }
        cpl_free(self->list);
        cpl_free(self);
    }

    return;
}

/**@}*/
