/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_multifit_nlin.h>
#include "moo_utils.h"
#include "moo_params.h"
#include "moo_single.h"
#include "moo_psf_single.h"
#include "moo_badpix.h"
#include "moo_fibres_table.h"
#include "moo_model_flat.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
double
moo_image_get_linear_y_interpolated(cpl_image *image, double y, int x, int *rej)
{
    double flux = NAN;

    cpl_ensure(image != NULL, CPL_ERROR_NULL_INPUT, NAN);
    cpl_ensure(rej != NULL, CPL_ERROR_NULL_INPUT, NAN);

    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_x(image);
    cpl_ensure(x >= 1 && x <= nx, CPL_ERROR_ILLEGAL_INPUT, NAN);
    cpl_ensure(y >= 1 && y <= ny, CPL_ERROR_ILLEGAL_INPUT, NAN);

    int y1 = (int)floor(y);
    int y2 = y1 + 1;

    double flux1 = cpl_image_get(image, x, y1, rej);

    double flux2 = cpl_image_get(image, x, y2, rej);

    if (*rej == CPL_BINARY_0) {
        flux = flux1 + (y - y1) * (flux2 - flux1);
    }
    return flux;
}

double
moo_image_get_pixel_y_interpolated(cpl_image *image,
                                   double y,
                                   int x,
                                   double step,
                                   int *rej)
{
    double flux = NAN;

    cpl_ensure(image != NULL, CPL_ERROR_NULL_INPUT, NAN);
    cpl_ensure(rej != NULL, CPL_ERROR_NULL_INPUT, NAN);

    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_x(image);
    cpl_ensure(x >= 1 && x <= nx, CPL_ERROR_ILLEGAL_INPUT, NAN);
    cpl_ensure(y >= 1 && y <= ny, CPL_ERROR_ILLEGAL_INPUT, NAN);

    double ystart = y - step / 2.0;
    double ystop = y + step / 2.0;

    int pix1 = (int)floor(ystart + 0.5);
    int pix2 = (int)floor(ystop + 0.5);

    if (pix1 == pix2) {
        flux = cpl_image_get(image, x, pix1, rej);
    }
    else {
        double frac1 = (pix1 + 0.5 - ystart) / step;
        double frac2 = (ystop - (pix2 - 0.5)) / step;
        double flux1 = cpl_image_get(image, x, pix1, rej);
        double flux2 = cpl_image_get(image, x, pix2, rej);

        flux = flux1 * frac1 + flux2 * frac2;
        // cpl_msg_info("test","get %f * %d (%f) + %f * %d (%f) = %f",frac1,pix1,flux1,frac2,pix2,flux2,flux);
    }

    if (*rej == CPL_BINARY_1) {
        flux = NAN;
    }

    return flux;
}

double
moo_image_get_pixel2_y_interpolated(cpl_image *image, double y, int x, int *rej)
{
    double flux = NAN;

    cpl_ensure(image != NULL, CPL_ERROR_NULL_INPUT, NAN);
    cpl_ensure(rej != NULL, CPL_ERROR_NULL_INPUT, NAN);

    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_x(image);
    cpl_ensure(x >= 1 && x <= nx, CPL_ERROR_ILLEGAL_INPUT, NAN);
    cpl_ensure(y >= 1 && y <= ny, CPL_ERROR_ILLEGAL_INPUT, NAN);

    int pix = (int)floor(y + 0.5);

    flux = cpl_image_get(image, x, pix, rej);

    if (*rej == CPL_BINARY_1) {
        flux = NAN;
    }

    return flux;
}

double
moo_image_get_pixel_yerror_interpolated(cpl_image *error, double y, int x)
{
    double err = NAN;

    cpl_ensure(error != NULL, CPL_ERROR_NULL_INPUT, NAN);

    int nx = cpl_image_get_size_x(error);
    int ny = cpl_image_get_size_x(error);
    cpl_ensure(x >= 1 && x <= nx, CPL_ERROR_ILLEGAL_INPUT, NAN);
    cpl_ensure(y >= 1 && y <= ny, CPL_ERROR_ILLEGAL_INPUT, NAN);

    int y1 = (int)floor(y + 0.5);

    int rej;
    err = cpl_image_get(error, x, y1, &rej);

    return err;
}

double
moo_image_get_linear_yerror_interpolated(cpl_image *error, double y, int x)
{
    double err = NAN;

    cpl_ensure(error != NULL, CPL_ERROR_NULL_INPUT, NAN);

    int nx = cpl_image_get_size_x(error);
    int ny = cpl_image_get_size_x(error);
    cpl_ensure(x >= 1 && x <= nx, CPL_ERROR_ILLEGAL_INPUT, NAN);
    cpl_ensure(y >= 1 && y <= ny, CPL_ERROR_ILLEGAL_INPUT, NAN);

    int y1 = (int)floor(y);
    int y2 = y1 + 1;

    int rej;
    double err1 = cpl_image_get(error, x, y1, &rej);

    double err2 = cpl_image_get(error, x, y2, &rej);

    err = sqrt((y - y1) * err1 * (y - y1) * err1 +
               (y2 - y) * err2 * (y2 - y) * err2);

    return err;
}

static void
_moo_fit_model_flat_fibre(cpl_image *rec_fluxes, int deg_poly)
{
    int nx = cpl_image_get_size_x(rec_fluxes);
    int ny = cpl_image_get_size_y(rec_fluxes);

    cpl_bivector *points = cpl_bivector_new(nx);
    cpl_vector *xpos = cpl_bivector_get_x(points);
    cpl_vector *fluxes = cpl_bivector_get_y(points);

    int *flag = cpl_calloc(nx, sizeof(int));

    for (int x = 0; x < nx; x++) {
        cpl_vector_set(xpos, x, x + 1);
    }

    double *data = cpl_image_get_data(rec_fluxes);
    for (int y = 0; y < ny; y++) {
        for (int x = 0; x < nx; x++) {
            double val = data[x + y * nx];
            cpl_vector_set(fluxes, x, val);
            if (isnan(val)) {
                flag[x] = 1;
            }
            else {
                flag[x] = 0;
            }
        }
        double min = moo_vector_get_min(fluxes, flag);
        double max = moo_vector_get_max(fluxes, flag);

        moo_tchebychev_fit(points, flag, deg_poly, 1, nx, min, max);

        for (int x = 0; x < nx; x++) {
            double val = cpl_vector_get(fluxes, x);
            data[x + y * nx] = val;
        }
    }

    cpl_bivector_delete(points);
    cpl_free(flag);
}

static void
_moo_smooth_model_flat_fibre(cpl_image *rec_fluxes, double s)
{
    int nx = cpl_image_get_size_x(rec_fluxes);
    int ny = cpl_image_get_size_y(rec_fluxes);

    cpl_vector *fluxes = cpl_vector_new(nx);

    double *data = cpl_image_get_data(rec_fluxes);
    for (int y = 0; y < ny; y++) {
        int nbnan = 0;
        for (int x = 0; x < nx; x++) {
            double val = data[x + y * nx];
            if (isnan(val)) {
                nbnan++;
            }
            cpl_vector_set(fluxes, x, val);
        }

        cpl_vector *filter = cpl_vector_new(nx - nbnan);

        int filteridx = 0;
        for (int x = 0; x < nx; x++) {
            double val = cpl_vector_get(fluxes, x);

            if (!isnan(val)) {
                cpl_vector_set(filter, filteridx, val);
                filteridx++;
            }
        }

        cpl_vector *res = moo_hpfilter(filter, s);

        nbnan = 0;

        for (int x = 0; x < nx; x++) {
            double val = cpl_vector_get(fluxes, x);
            if (!isnan(val)) {
                val = cpl_vector_get(res, x - nbnan);
            }
            else {
                nbnan++;
            }
            data[x + y * nx] = val;
        }
        cpl_vector_delete(filter);
        cpl_vector_delete(res);
    }

    cpl_vector_delete(fluxes);
}

static void
_moo_median_filter_model_flat_fibre(cpl_image *rec_fluxes,
                                    int winhsize,
                                    int kappa)
{
    int nx = cpl_image_get_size_x(rec_fluxes);
    int ny = cpl_image_get_size_y(rec_fluxes);

    cpl_mask *mask = cpl_mask_new(nx, ny);

    cpl_mask *orig = cpl_image_get_bpm(rec_fluxes);

    cpl_vector *fluxes = cpl_vector_new(nx);
    int window_size = winhsize * 2 + 1;

    cpl_vector *winflux = cpl_vector_new(window_size);
    for (int y = 1; y <= ny; y++) {
        for (int x = 1; x <= nx; x++) {
            int rej;
            double val = cpl_image_get(rec_fluxes, x, y, &rej);
            cpl_vector_set(fluxes, x - 1, val);
        }

        for (int x = 0; x < window_size; x++) {
            double val = cpl_vector_get(fluxes, x);
            cpl_vector_set(winflux, x, val);
        }
        double median = moo_vector_get_percentile(winflux, 0.5);
        double p1 = moo_vector_get_percentile(winflux, 0.1585);
        double p2 = moo_vector_get_percentile(winflux, 0.8415);
        double sigma = (p2 - p1) / 2;

        for (int x = 0; x < winhsize + 1; x++) {
            double val = cpl_vector_get(fluxes, x);
            double diff = val - median;
            if (fabs(diff) > kappa * sigma) {
                cpl_mask_set(mask, x + 1, y, CPL_BINARY_1);
            }
        }
        for (int x = winhsize + 2; x < nx - winhsize - 1; x++) {
            for (int i = x - winhsize; i <= x + winhsize; i++) {
                double val = cpl_vector_get(fluxes, i);
                cpl_vector_set(winflux, i - x + winhsize, val);
            }
            double val = cpl_vector_get(winflux, winhsize);
            if (!isnan(val)) {
                median = moo_vector_get_percentile(winflux, 0.5);
                p1 = moo_vector_get_percentile(winflux, 0.1585);
                p2 = moo_vector_get_percentile(winflux, 0.8415);
                sigma = (p2 - p1) / 2;

                double diff = val - median;
                if (fabs(diff) > kappa * sigma) {
                    cpl_mask_set(mask, x + 1, y, CPL_BINARY_1);
                }
            }
        }
    }
    int count = cpl_mask_count(mask);
    cpl_msg_info(__func__, "new cosmics %d", count);
    cpl_mask_or(orig, mask);
    cpl_mask_delete(mask);
    cpl_vector_delete(fluxes);
    cpl_vector_delete(winflux);
}

static void
_moo_median_filter2_model_flat_fibre(cpl_image *rec_fluxes, int winhsize)
{
    int nx = cpl_image_get_size_x(rec_fluxes);
    int ny = cpl_image_get_size_y(rec_fluxes);

    cpl_vector *fluxes = cpl_vector_new(nx);

    for (int y = 1; y <= ny; y++) {
        for (int x = 1; x <= nx; x++) {
            int rej;
            double val = cpl_image_get(rec_fluxes, x, y, &rej);
            cpl_vector_set(fluxes, x - 1, val);
        }

        cpl_vector *res = moo_median_filter(fluxes, winhsize);

        for (int x = 1; x <= nx; x++) {
            double val = cpl_vector_get(res, x - 1);
            cpl_image_set(rec_fluxes, x, y, val);
        }

        cpl_vector_delete(res);
    }
    cpl_vector_delete(fluxes);
}

static void
_moo_savgol_model_flat_fibre(cpl_image *rec_fluxes,
                             int polyorder,
                             int window_length)
{
    int nx = cpl_image_get_size_x(rec_fluxes);
    int ny = cpl_image_get_size_y(rec_fluxes);

    cpl_vector *fluxes = cpl_vector_new(nx);

    for (int y = 1; y <= ny; y++) {
        for (int x = 1; x <= nx; x++) {
            int rej;
            double val = cpl_image_get(rec_fluxes, x, y, &rej);

            if (rej != 0) {
                val = NAN;
            }
            cpl_vector_set(fluxes, x - 1, val);
        }
        if (y == 1) {
            char *testname = cpl_sprintf("befsavgol.txt");
            FILE *test = fopen(testname, "w");
            fprintf(test, "#x f\n");

            for (int x = 0; x < nx; x++) {
                double val = cpl_vector_get(fluxes, x);
                fprintf(test, "%d %f\n", x, val);
            }

            fclose(test);
            cpl_free(testname);
        }
        cpl_vector *res = moo_savgol_filter(fluxes, window_length, polyorder);
        if (y == 1) {
            char *testname = cpl_sprintf("savgol.txt");
            FILE *test = fopen(testname, "w");
            fprintf(test, "#x f\n");

            for (int x = 0; x < nx; x++) {
                double val = cpl_vector_get(res, x);
                fprintf(test, "%d %f\n", x, val);
            }

            fclose(test);
            cpl_free(testname);
        }

        for (int x = 0; x < nx; x++) {
            double val = cpl_vector_get(res, x);
            cpl_image_set(rec_fluxes, x + 1, y, val);
        }
        cpl_vector_delete(res);
    }

    cpl_vector_delete(fluxes);
}

static void
_moo_extract_error_model_flat_fibre(hdrl_image *image,
                                    int numfib,
                                    double *centroids,
                                    int nstep_lo,
                                    int nstep_up,
                                    double step,
                                    cpl_image *rec_fluxes)
{
    int nx = hdrl_image_get_size_x(image);
    cpl_image *error = hdrl_image_get_error(image);

    for (int x = 1; x <= nx; x++) {
        double centroid = centroids[(numfib - 1) * nx + x - 1];

        if (!isnan(centroid)) {
            for (int y = -nstep_lo; y <= nstep_up; y++) {
                double yc = centroid + y * step;
                double flux =
                    moo_image_get_pixel_yerror_interpolated(error, yc, x);
                cpl_image_set(rec_fluxes, x, y + nstep_lo + 1, flux);
            }
        }
        else {
            for (int y = -nstep_lo; y <= nstep_up; y++) {
                cpl_image_set(rec_fluxes, x, y + nstep_lo + 1, NAN);
            }
        }
    }
}

static void
_moo_rectify_fibre(hdrl_image *image,
                   int numfib,
                   double *centroids,
                   int nstep_lo,
                   int nstep_up,
                   double step,
                   cpl_image *rec_fluxes)
{
    int nx = hdrl_image_get_size_x(image);
    cpl_image *data = hdrl_image_get_image(image);

    for (int x = 1; x <= nx; x++) {
        double centroid = centroids[(numfib - 1) * nx + x - 1];

        if (!isnan(centroid)) {
            for (int y = -nstep_lo; y <= nstep_up; y++) {
                double yc = centroid + y * step;
                int rej = 0;
                // double flux = moo_image_get_linear_y_interpolated( data, yc, x, &rej);
                // double flux = moo_image_get_pixel_y_interpolated( data, yc, x, step, &rej);
                double flux =
                    moo_image_get_pixel2_y_interpolated(data, yc, x, &rej);
                cpl_image_set(rec_fluxes, x, y + nstep_lo + 1, flux);
            }
        }
        else {
            for (int y = -nstep_lo; y <= nstep_up; y++) {
                cpl_image_set(rec_fluxes, x, y + nstep_lo + 1, NAN);
            }
        }
    }
}

static int
_moo_model_f(const gsl_vector *px, void *data, gsl_vector *f)
{
    cpl_bivector *bvdata = (cpl_bivector *)data;
    size_t n = cpl_bivector_get_size(bvdata);
    cpl_vector *vy = cpl_bivector_get_y(bvdata);
    cpl_vector *vx = cpl_bivector_get_x(bvdata);

    double sigma = gsl_vector_get(px, 0);
    double cexp = gsl_vector_get(px, 1);
    double A = gsl_vector_get(px, 2);
    double c0 = gsl_vector_get(px, 3);
    double b = gsl_vector_get(px, 4);

    moo_model *m = moo_model_new(sigma, cexp, A, c0, b);

    size_t i;

    for (i = 0; i < n; i++) {
        /* Model Yi = A * exp(-lambda * i) + b */
        double x = cpl_vector_get(vx, i);
        double y = cpl_vector_get(vy, i);

        double Yx = moo_model_eval(m, x);

        gsl_vector_set(f, i, Yx - y);
    }
    moo_model_delete(m);

    return GSL_SUCCESS;
}

static int
_moo_model_df(const gsl_vector *px, void *data, gsl_matrix *J)
{
    cpl_bivector *bvdata = (cpl_bivector *)data;
    size_t n = cpl_bivector_get_size(bvdata);
    cpl_vector *vx = cpl_bivector_get_x(bvdata);

    double sigma = gsl_vector_get(px, 0);
    double cexp = gsl_vector_get(px, 1);
    double A = gsl_vector_get(px, 2);
    double c0 = gsl_vector_get(px, 3);
    double b = gsl_vector_get(px, 4);

    moo_model *m = moo_model_new(sigma, cexp, A, c0, b);
    size_t i;

    for (i = 0; i < n; i++) {
        /* Jacobian matrix J(i,j) = dfi / dxj, */
        /* where fi = (Yi - yi)/sigma[i],      */
        /*       Yi = A * exp(-lambda * i) + b  */
        /* and the xj are the parameters (A,lambda,b) */
        double x = cpl_vector_get(vx, i);
        double dA = moo_model_dA_eval(m, x);
        double dcexp = moo_model_dcexp_eval(m, x);
        double dsigma = moo_model_dsigma_eval(m, x);
        double dc0 = moo_model_dc0_eval(m, x);

        gsl_matrix_set(J, i, 0, dsigma);
        gsl_matrix_set(J, i, 1, dcexp);
        gsl_matrix_set(J, i, 2, dA);
        gsl_matrix_set(J, i, 3, dc0);
        gsl_matrix_set(J, i, 4, 1.0);
    }
    moo_model_delete(m);

    return GSL_SUCCESS;
}

static int
_moo_model_df2(const gsl_vector *px, void *data, gsl_matrix *J)
{
    cpl_bivector *bvdata = (cpl_bivector *)data;
    size_t n = cpl_bivector_get_size(bvdata);
    cpl_vector *vx = cpl_bivector_get_x(bvdata);

    double sigma = gsl_vector_get(px, 0);
    double cexp = gsl_vector_get(px, 1);
    double A = gsl_vector_get(px, 2);
    double c0 = gsl_vector_get(px, 3);
    double b = gsl_vector_get(px, 4);

    moo_model *m = moo_model_new(sigma, cexp, A, c0, b);
    size_t i;

    for (i = 0; i < n; i++) {
        /* Jacobian matrix J(i,j) = dfi / dxj, */
        /* where fi = (Yi - yi)/sigma[i],      */
        /*       Yi = A * exp(-lambda * i) + b  */
        /* and the xj are the parameters (A,lambda,b) */
        double x = cpl_vector_get(vx, i);
        double dA = moo_model_dA_eval(m, x);
        double dc0 = moo_model_dc0_eval(m, x);

        gsl_matrix_set(J, i, 0, 0);
        gsl_matrix_set(J, i, 1, 0);
        gsl_matrix_set(J, i, 2, dA);
        gsl_matrix_set(J, i, 3, dc0);
        gsl_matrix_set(J, i, 4, 1.0);
    }
    moo_model_delete(m);

    return GSL_SUCCESS;
}

static moo_model *
_moo_model_fit(cpl_bivector *data,
               int (*fit_func)(const gsl_vector *, void *, gsl_matrix *),
               double isigma,
               double icexp)
{
    moo_model *model = NULL;

    if (data != NULL) {
        int N = cpl_bivector_get_size(data);
        cpl_vector *dy = cpl_bivector_get_y(data);
        double min = cpl_vector_get_min(dy);
        double max = cpl_vector_get_max(dy);
        double A = max - min;
        double b = min;
        const gsl_multifit_fdfsolver_type *T = gsl_multifit_fdfsolver_lmsder;
        gsl_multifit_fdfsolver *s;
        int info;

        const size_t p = 5;

        gsl_matrix *J = gsl_matrix_alloc(N, p);
        gsl_matrix *covar = gsl_matrix_alloc(p, p);

        gsl_multifit_function_fdf f;
        double x_init[5] = { isigma, icexp, A, 0.0, b };
        gsl_vector_view x = gsl_vector_view_array(x_init, p);

        const double xtol = 1e-9;
        const double gtol = 1e-9;
        const double ftol = 0.0;

        f.f = &_moo_model_f;
        f.df = fit_func;
        f.n = N;
        f.p = p;
        f.params = data;

        s = gsl_multifit_fdfsolver_alloc(T, N, p);

        /* initialize solver with starting point */
        gsl_multifit_fdfsolver_set(s, &f, &x.vector);

        /* compute initial residual norm */
        gsl_multifit_fdfsolver_residual(s);

        /* solve the system with a maximum of 50 iterations */
        gsl_multifit_fdfsolver_driver(s, 50, xtol, gtol, ftol, &info);

        gsl_multifit_fdfsolver_jac(s, J);
        gsl_multifit_covar(J, 0.0, covar);

        /*
        double chi = gsl_blas_dnrm2(res_f);

        #define FIT(i) gsl_vector_get(s->x, i)
        #define ERR(i) sqrt(gsl_matrix_get(covar,i,i))
          {
            double dof = N - p;
            double c = GSL_MAX_DBL(1, chi / sqrt(dof));

            fprintf(stderr, "chisq/dof = %g\n",  pow(chi, 2.0) / dof);

            fprintf (stderr, "sigma  = %.5f +/- %.5f\n", FIT(0), c*ERR(0));
            fprintf (stderr, "cexp   = %.5f +/- %.5f\n", FIT(1), c*ERR(1));
            fprintf (stderr, "A      = %.5f +/- %.5f\n", FIT(2), c*ERR(2));
            fprintf (stderr, "c0     = %.5f +/- %.5f\n", FIT(3), c*ERR(3));
            fprintf (stderr, "B      = %.5f +/- %.5f\n", FIT(4), c*ERR(4));
          }

        fprintf (stderr, "status = %s\n", gsl_strerror (status));
         */
        double sigma = gsl_vector_get(s->x, 0);
        double cexp = gsl_vector_get(s->x, 1);
        A = gsl_vector_get(s->x, 2);
        double c0 = gsl_vector_get(s->x, 3);
        b = gsl_vector_get(s->x, 4);

        model = moo_model_new(sigma, cexp, A, c0, b);
        gsl_multifit_fdfsolver_free(s);
        gsl_matrix_free(covar);
        gsl_matrix_free(J);
    }
    return model;
}

moo_model *
moo_model_new(double sigma, double cexp, double A, double c0, double b)
{
    moo_model *res = cpl_calloc(1, sizeof(moo_model));

    res->sigma = sigma;
    res->cexp = cexp;
    res->A = A;
    res->c0 = c0;
    res->b = b;

    return res;
}

void
moo_model_delete(moo_model *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

double
moo_model_eval(moo_model *model, double x)
{
    double res = NAN;

    cpl_ensure(model != NULL, CPL_ERROR_NULL_INPUT, res);

    double sigma = model->sigma;
    double cexp = model->cexp;
    double A = model->A;
    double c0 = model->c0;
    double b = model->b;

    res = (A / (sqrt(CPL_MATH_2PI) * sigma)) *
              exp(-0.5 * pow((fabs(x - c0) / sigma), cexp)) +
          b;

    return res;
}

double
moo_model_dA_eval(moo_model *model, double x)
{
    double res = NAN;

    cpl_ensure(model != NULL, CPL_ERROR_NULL_INPUT, res);

    double sigma = model->sigma;
    double cexp = model->cexp;
    double c0 = model->c0;

    double absxc0 = fabs(x - c0);

    res = (1 / (sqrt(CPL_MATH_2PI) * sigma)) *
          exp(-0.5 * pow((absxc0 / sigma), cexp));

    return res;
}

double
moo_model_dsigma_eval(moo_model *model, double x)
{
    double res = NAN;

    cpl_ensure(model != NULL, CPL_ERROR_NULL_INPUT, res);

    double sigma = model->sigma;
    double cexp = model->cexp;
    double c0 = model->c0;
    double A = model->A;

    double absxc0 = fabs(x - c0);

    res = -(A * pow(sigma, -cexp - 2) *
            (2 * pow(sigma, cexp) - cexp * pow(absxc0, cexp)) *
            exp(-pow(absxc0, cexp) / (2 * pow(sigma, cexp)))) /
          (2 * sqrt(CPL_MATH_2PI));
    return res;
}

double
moo_model_dcexp_eval(moo_model *model, double x)
{
    double res = NAN;

    cpl_ensure(model != NULL, CPL_ERROR_NULL_INPUT, res);

    double sigma = model->sigma;
    double cexp = model->cexp;
    double c0 = model->c0;
    double A = model->A;

    if (x == c0) {
        res = 0;
    }
    else {
        double absxc0 = fabs(x - c0);

        res = -(A * pow(sigma, -cexp - 1) * pow(absxc0, cexp) *
                log(absxc0 / sigma) *
                exp(-pow(absxc0, cexp) / (2. * pow(sigma, cexp)))) /
              (2. * sqrt(CPL_MATH_2PI));
    }
    return res;
}

double
moo_model_dc0_eval(moo_model *model, double x)
{
    double res = NAN;

    cpl_ensure(model != NULL, CPL_ERROR_NULL_INPUT, res);

    double sigma = model->sigma;
    double cexp = model->cexp;
    double c0 = model->c0;
    double A = model->A;

    double absxc0 = fabs(x - c0);

    if (x == c0) {
        res = 0;
    }
    else {
        res = (A * cexp * pow(sigma, -cexp - 1) *
               exp(-pow(absxc0, cexp) / (2 * pow(sigma, cexp))) *
               pow(absxc0, cexp)) /
              (2 * sqrt(CPL_MATH_2PI) * (x - c0));
    }
    return res;
}

static void
_moo_model_fibre(hdrl_image *image,
                 int numfib,
                 double *centroids,
                 cpl_image *fwlo,
                 cpl_image *fwup,
                 int nstep_lo,
                 int nstep_up,
                 double step,
                 cpl_image *rec_fluxes,
                 int win_hsize,
                 int xstep)
{
    int nx = hdrl_image_get_size_x(image);

    cpl_vector *vX = cpl_vector_new(nx);
    cpl_vector *vCexp = cpl_vector_new(nx);
    cpl_vector *vSigma = cpl_vector_new(nx);

    int model_size = 0;

    for (int x = win_hsize + 1; x <= nx - win_hsize; x += xstep) {
        cpl_bivector *data = NULL;

        for (int x2 = x - win_hsize; x2 <= x + win_hsize; x2++) {
            int prej;
            double centroid = centroids[(numfib - 1) * nx + x2 - 1];
            double wlo = cpl_image_get(fwlo, x2, numfib, &prej);
            double wup = cpl_image_get(fwup, x2, numfib, &prej);
            int yl = floor(centroid - wlo);
            int yu = ceil(centroid + wup);

            int ysize = yu - yl + 1;
            cpl_bivector *tmp_data = cpl_bivector_new(ysize);
            cpl_vector *tmp_y = cpl_bivector_get_x(tmp_data);
            cpl_vector *tmp_flux = cpl_bivector_get_y(tmp_data);

            int tmp_size = 0;
            int nbrej = 0;
            if (!isnan(centroid)) {
                for (int y = yl; y <= yu; y++) {
                    int rej = 0;
                    hdrl_value flux = hdrl_image_get_pixel(image, x2, y, &rej);
                    if (rej == 0) {
                        cpl_vector_set(tmp_y, tmp_size, y);
                        cpl_vector_set(tmp_flux, tmp_size, flux.data);
                        tmp_size++;
                    }
                    else {
                        nbrej++;
                    }
                }
            }

            double frac = 1 - (double)nbrej / (double)(tmp_size + nbrej);

            if (tmp_size > 4 && frac > 0.8) {
                cpl_vector_set_size(tmp_y, tmp_size);
                cpl_vector_set_size(tmp_flux, tmp_size);
                double min = cpl_vector_get_min(tmp_flux);
                cpl_vector_subtract_scalar(tmp_y, centroid);

                cpl_vector_subtract_scalar(tmp_flux, min);

                double sum = cpl_vector_get_sum(tmp_flux);

                if (sum > 0) {
                    cpl_vector_divide_scalar(tmp_flux, sum);

                    if (data == NULL) {
                        data = cpl_bivector_duplicate(tmp_data);
                    }
                    else {
                        int size = cpl_bivector_get_size(data);
                        cpl_vector *dx = cpl_bivector_get_x(data);
                        cpl_vector *dy = cpl_bivector_get_y(data);
                        cpl_vector_set_size(dx, size + tmp_size);
                        cpl_vector_set_size(dy, size + tmp_size);
                        for (int i = size; i < size + tmp_size; i++) {
                            cpl_vector_set(dx, i,
                                           cpl_vector_get(tmp_y, i - size));
                            cpl_vector_set(dy, i,
                                           cpl_vector_get(tmp_flux, i - size));
                        }
                    }
                }
            }
            cpl_bivector_delete(tmp_data);
        }

        moo_model *model = _moo_model_fit(data, &_moo_model_df, 1., 2.);

        if (model != NULL) {
            cpl_vector_set(vX, model_size, x);
            cpl_vector_set(vCexp, model_size, model->cexp);
            cpl_vector_set(vSigma, model_size, model->sigma);
            model_size++;
        }

        moo_model_delete(model);
        cpl_bivector_delete(data);
    }

    cpl_vector_set_size(vX, model_size);
    cpl_vector_set_size(vCexp, model_size);
    cpl_vector_set_size(vSigma, model_size);

    cpl_vector *resCexp = moo_median_filter(vCexp, 5);
    cpl_vector *resSigma = moo_median_filter(vSigma, 5);

    for (int x = 1; x <= nx; x++) {
        int prej;
        double centroid = centroids[(numfib - 1) * nx + x - 1];
        double wlo = cpl_image_get(fwlo, x, numfib, &prej);
        double wup = cpl_image_get(fwup, x, numfib, &prej);
        int yl = floor(centroid - wlo);
        int yu = ceil(centroid + wup);

        int ysize = yu - yl + 1;

        if (!isnan(centroid)) {
            cpl_bivector *tmp_data = cpl_bivector_new(ysize);
            cpl_vector *tmp_y = cpl_bivector_get_x(tmp_data);
            cpl_vector *tmp_flux = cpl_bivector_get_y(tmp_data);

            int tmp_size = 0;
            int nbrej = 0;

            for (int y = yl; y <= yu; y++) {
                int rej = 0;
                hdrl_value flux = hdrl_image_get_pixel(image, x, y, &rej);
                if (rej == 0) {
                    cpl_vector_set(tmp_y, tmp_size, y);
                    cpl_vector_set(tmp_flux, tmp_size, flux.data);
                    tmp_size++;
                }
                else {
                    nbrej++;
                }
            }
            double frac = (double)nbrej / (double)(nbrej + tmp_size);

            if (tmp_size > 4 && frac < 0.2) {
                double sigma = 1.0;
                double cexp = 2.0;

                cpl_vector_set_size(tmp_flux, tmp_size);
                cpl_vector_set_size(tmp_y, tmp_size);
                cpl_vector_subtract_scalar(tmp_y, centroid);

                int idx = cpl_vector_find(vX, x);
                sigma = cpl_vector_get(resSigma, idx);
                cexp = cpl_vector_get(resCexp, idx);

                moo_model *model =
                    _moo_model_fit(tmp_data, &_moo_model_df2, sigma, cexp);
                for (int i = -nstep_lo; i <= nstep_up; i++) {
                    double y = i * step;
                    double v = moo_model_eval(model, y);

                    cpl_image_set(rec_fluxes, x, i + nstep_lo + 1, v);
                }

                moo_model_delete(model);
            }
            else {
                for (int i = -nstep_lo; i <= nstep_up; i++) {
                    cpl_image_set(rec_fluxes, x, i + nstep_lo + 1, NAN);
                }
            }
            cpl_bivector_delete(tmp_data);
        }
        else {
            for (int i = -nstep_lo; i <= nstep_up; i++) {
                cpl_image_set(rec_fluxes, x, i + nstep_lo + 1, NAN);
            }
        }
    }

    cpl_vector_delete(vX);
    cpl_vector_delete(vCexp);
    cpl_vector_delete(vSigma);

    cpl_vector_delete(resCexp);
    cpl_vector_delete(resSigma);
}

static moo_psf_single *
_moo_model_flat_single(moo_single *det,
                       moo_loc_single *loc,
                       const char *filename,
                       moo_model_flat_params *params,
                       int deg_poly,
                       const int *health)
{
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *fcentroids = moo_loc_single_get_f_centroids(loc);
    cpl_image *fwlo = moo_loc_single_get_f_wlo(loc);
    cpl_image *fwup = moo_loc_single_get_f_wup(loc);
    hdrl_image *image = moo_single_get_image(det);

    int nb_fibres = cpl_image_get_size_y(fcentroids);
    int nx = cpl_image_get_size_x(fcentroids);

    cpl_image_reject_value(fwlo, CPL_VALUE_NAN);
    cpl_image_reject_value(fwup, CPL_VALUE_NAN);

    double max_fwlo = cpl_image_get_max(fwlo);
    double max_fwup = cpl_image_get_max(fwup);

    cpl_msg_indent_more();
    cpl_msg_info("moo_model_flat", "Use Y localisation limits: %.3f,%.3f pix",
                 -max_fwlo, max_fwup);

    int nstep_lo = (int)ceil(max_fwlo * params->oversamp);
    int nstep_up = (int)ceil(max_fwup * params->oversamp);

    int ny = nstep_lo + nstep_up + 1;

    moo_psf_single *res = moo_psf_single_create(filename, det->extname);

    double *centroids = cpl_image_get_data(fcentroids);
    double step = 1.0 / params->oversamp;

    cpl_image **imgtab = cpl_calloc(nb_fibres, sizeof(cpl_image *));
    const char *extname = moo_detector_get_extname(det->type, det->ntas);

    int win_hsize = params->winhsize;
    int xstep = params->xstep;

    cpl_msg_info("moo_model_flat",
                 "Fitting profile shape parameters using step %d and window "
                 "half size %d along X axis",
                 xstep, win_hsize);
    cpl_msg_indent_less();

#if MOO_DEBUG_MODEL_FLAT
    int test_fibers[] = { 10, 250, 500 };
    for (int ti = 306; ti <= 308; ti++) {
        int numfib = ti;

        char *testname =
            cpl_sprintf("%s_%d_localise_step1.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);
        char *testname1b =
            cpl_sprintf("%s_%d_localise_align_centroid_step1.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);
        char *testmodelname =
            cpl_sprintf("%s_%d_model_parameters_step1.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);
        char *testmodel2name =
            cpl_sprintf("%s_%d_model_step1.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);
        char *testmodel3name =
            cpl_sprintf("%s_%d_model_step2.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);
        char *testmodelp2name =
            cpl_sprintf("%s_%d_model_parameters_step2.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);
        FILE *test = fopen(testname, "w");
        fprintf(test, "#x x2 y flux err qual\n");

        FILE *test1 = fopen(testname1b, "w");
        fprintf(test1, "#x x2 y flux centroid min sum\n");

        FILE *test2 = fopen(testmodelname, "w");
        fprintf(test2, "#x centroid A sigma c0 cexp b\n");

        FILE *test3 = fopen(testmodel2name, "w");
        fprintf(test3, "#x y flux\n");

        FILE *test4 = fopen(testmodel3name, "w");
        fprintf(test4, "#x y yr flux\n");

        FILE *testp2 = fopen(testmodelp2name, "w");
        fprintf(testp2, "#x centroid A sigma c0 cexp b\n");

        cpl_vector *vX = cpl_vector_new(nx);
        cpl_vector *vA = cpl_vector_new(nx);
        cpl_vector *vCexp = cpl_vector_new(nx);
        cpl_vector *vSigma = cpl_vector_new(nx);
        cpl_vector *vB = cpl_vector_new(nx);
        cpl_vector *vC0 = cpl_vector_new(nx);

        int model_size = 0;

        for (int x = win_hsize + 1; x <= nx - win_hsize; x += xstep) {
            cpl_bivector *data = NULL;

            for (int x2 = x - win_hsize; x2 <= x + win_hsize; x2++) {
                int prej;
                double centroid = centroids[(numfib - 1) * nx + x2 - 1];
                double wlo = cpl_image_get(fwlo, x2, numfib, &prej);
                double wup = cpl_image_get(fwup, x2, numfib, &prej);
                int yl = floor(centroid - wlo);
                int yu = ceil(centroid + wup);

                int ysize = yu - yl + 1;
                cpl_bivector *tmp_data = cpl_bivector_new(ysize);
                cpl_vector *tmp_y = cpl_bivector_get_x(tmp_data);
                cpl_vector *tmp_flux = cpl_bivector_get_y(tmp_data);

                int tmp_size = 0;
                int nbrej = 0;

                if (!isnan(centroid)) {
                    for (int y = yl; y <= yu; y++) {
                        int rej = 0;
                        hdrl_value flux =
                            hdrl_image_get_pixel(image, x2, y, &rej);
                        if (rej == 0) {
                            fprintf(test, "%d %d %d %f %f %d\n", x, x2, y,
                                    flux.data, flux.error, rej);

                            cpl_vector_set(tmp_y, tmp_size, y);
                            cpl_vector_set(tmp_flux, tmp_size, flux.data);
                            tmp_size++;
                        }
                        else {
                            nbrej++;
                        }
                    }
                }
                double frac = 1 - (double)nbrej / (double)(tmp_size + nbrej);

                if (tmp_size >= 4 && frac > 0.9) {
                    cpl_vector_set_size(tmp_y, tmp_size);
                    cpl_vector_set_size(tmp_flux, tmp_size);

                    double min = cpl_vector_get_min(tmp_flux);

                    cpl_vector_subtract_scalar(tmp_y, centroid);
                    cpl_vector_subtract_scalar(tmp_flux, min);
                    double sum = cpl_vector_get_sum(tmp_flux);

                    if (sum > 0) {
                        cpl_vector_divide_scalar(tmp_flux, sum);

                        for (int y = 0; y < tmp_size; y++) {
                            double ry = cpl_vector_get(tmp_y, y);
                            double rf = cpl_vector_get(tmp_flux, y);
                            fprintf(test1, "%d %d %f %f %f %f %f\n", x, x2, ry,
                                    rf, centroid, min, sum);
                        }

                        if (data == NULL) {
                            data = cpl_bivector_duplicate(tmp_data);
                        }
                        else {
                            int size = cpl_bivector_get_size(data);
                            cpl_vector *dx = cpl_bivector_get_x(data);
                            cpl_vector *dy = cpl_bivector_get_y(data);
                            cpl_vector_set_size(dx, size + tmp_size);
                            cpl_vector_set_size(dy, size + tmp_size);
                            for (int i = size; i < size + tmp_size; i++) {
                                cpl_vector_set(dx, i,
                                               cpl_vector_get(tmp_y, i - size));
                                cpl_vector_set(dy, i,
                                               cpl_vector_get(tmp_flux,
                                                              i - size));
                            }
                        }
                    }
                }

                cpl_bivector_delete(tmp_data);
            }

            if (data != NULL) {
                moo_model *model = _moo_model_fit(data, &_moo_model_df, 1., 2.);
                if (model != NULL) {
                    double centroid = centroids[(numfib - 1) * nx + x - 1];
                    fprintf(test2, "%d %f %f %f %f %f %f\n", x, centroid,
                            model->A, model->sigma, model->c0, model->cexp,
                            model->b);
                    cpl_vector_set(vX, model_size, x);
                    cpl_vector_set(vA, model_size, model->A);
                    cpl_vector_set(vCexp, model_size, model->cexp);
                    cpl_vector_set(vSigma, model_size, model->sigma);
                    cpl_vector_set(vB, model_size, model->b);
                    cpl_vector_set(vC0, model_size, model->c0);

                    model_size++;

                    for (double y = -5; y <= 5; y += 0.1) {
                        double v = moo_model_eval(model, y);

                        fprintf(test3, "%d %f %f\n", x, y, v);
                    }
                }

                moo_model_delete(model);
                cpl_bivector_delete(data);
            }
        }

        cpl_vector_set_size(vX, model_size);
        cpl_vector_set_size(vA, model_size);
        cpl_vector_set_size(vCexp, model_size);
        cpl_vector_set_size(vSigma, model_size);
        cpl_vector_set_size(vB, model_size);
        cpl_vector_set_size(vC0, model_size);

        cpl_vector *resA = moo_median_filter(vA, 5);
        cpl_vector *resCexp = moo_median_filter(vCexp, 5);
        cpl_vector *resSigma = moo_median_filter(vSigma, 5);
        cpl_vector *resB = moo_median_filter(vB, 5);
        cpl_vector *resC0 = moo_median_filter(vC0, 5);
        {
            char *filtername =
                cpl_sprintf("%s_%d_model_parameters_step1_medianfilter.txt",
                            moo_detector_get_extname(det->type, det->ntas),
                            numfib);
            FILE *testm = fopen(filtername, "w");
            fprintf(testm, "#x A cexp sigma c0 b\n");

            for (int i = 0; i < model_size; i++) {
                double x = cpl_vector_get(vX, i);
                double A = cpl_vector_get(resA, i);
                double cexp = cpl_vector_get(resCexp, i);
                double sigma = cpl_vector_get(resSigma, i);
                double c0 = cpl_vector_get(resC0, i);
                double b = cpl_vector_get(resB, i);
                fprintf(testm, "%f %f %f %f %f %f\n", x, A, cexp, sigma, c0, b);
            }
            fclose(testm);
            cpl_free(filtername);
        }

        fclose(test);
        cpl_free(testname);
        fclose(test1);
        cpl_free(testname1b);
        fclose(test2);
        cpl_free(testmodelname);
        cpl_free(testmodel3name);

        testname =
            cpl_sprintf("%s_%d_localise_step2.txt",
                        moo_detector_get_extname(det->type, det->ntas), numfib);

        test = fopen(testname, "w");
        fprintf(test, "#x y flux err qual\n");
        cpl_image *rec_fluxes = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

        for (int x = 1; x <= nx; x++) {
            int prej;
            double centroid = centroids[(numfib - 1) * nx + x - 1];
            double wlo = cpl_image_get(fwlo, x, numfib, &prej);
            double wup = cpl_image_get(fwup, x, numfib, &prej);
            int yl = floor(centroid - wlo);
            int yu = ceil(centroid + wup);

            int ysize = yu - yl + 1;

            if (!isnan(centroid)) {
                cpl_bivector *tmp_data = cpl_bivector_new(ysize);
                cpl_vector *tmp_y = cpl_bivector_get_x(tmp_data);
                cpl_vector *tmp_flux = cpl_bivector_get_y(tmp_data);

                int tmp_size = 0;
                int nbrej = 0;

                for (int y = yl; y <= yu; y++) {
                    int rej = 0;
                    hdrl_value flux = hdrl_image_get_pixel(image, x, y, &rej);
                    fprintf(test, "%d %d %f %f %d\n", x, y, flux.data,
                            flux.error, rej);
                    if (rej == 0) {
                        cpl_vector_set(tmp_y, tmp_size, y);
                        cpl_vector_set(tmp_flux, tmp_size, flux.data);
                        tmp_size++;
                    }
                    else {
                        nbrej++;
                    }
                }

                double frac = (double)nbrej / (double)(nbrej + tmp_size);

                if (tmp_size > 4 && frac < 0.2) {
                    double sigma = 1.0;
                    double cexp = 2.0;

                    cpl_vector_set_size(tmp_flux, tmp_size);
                    cpl_vector_set_size(tmp_y, tmp_size);
                    cpl_vector_subtract_scalar(tmp_y, centroid);

                    int idx = cpl_vector_find(vX, x);
                    sigma = cpl_vector_get(resSigma, idx);
                    cexp = cpl_vector_get(resCexp, idx);

                    moo_model *model =
                        _moo_model_fit(tmp_data, &_moo_model_df2, sigma, cexp);

                    fprintf(testp2, "%d %f %f %f %f %f %f\n", x, centroid,
                            model->A, model->sigma, model->c0, model->cexp,
                            model->b);

                    for (int i = -nstep_lo; i <= nstep_up; i++) {
                        double y = i * step;
                        double v = moo_model_eval(model, y);
                        fprintf(test4, "%d %f %d %f\n", x, y + centroid, i, v);
                        cpl_image_set(rec_fluxes, x, i + nstep_lo + 1, v);
                    }

                    moo_model_delete(model);
                }
                else {
                    for (int i = -nstep_lo; i <= nstep_up; i++) {
                        double y = i * step;
                        fprintf(test4, "%d %f %d %f\n", x, y + centroid, i,
                                NAN);
                        cpl_image_set(rec_fluxes, x, i + nstep_lo + 1, NAN);
                    }
                }
                cpl_bivector_delete(tmp_data);
            }
            else {
                fprintf(test4, "%d %f %d %f\n", x, NAN, 0, NAN);
                for (int i = -nstep_lo; i <= nstep_up; i++) {
                    cpl_image_set(rec_fluxes, x, i + nstep_lo + 1, NAN);
                }
            }
        }

        _moo_median_filter2_model_flat_fibre(rec_fluxes, 10);
        {
            char *testmodelp3name =
                cpl_sprintf("%s_%d_model_step3.txt",
                            moo_detector_get_extname(det->type, det->ntas),
                            numfib);
            FILE *test5 = fopen(testmodelp3name, "w");
            fprintf(test5, "#x yr flux\n");
            for (int x = 1; x <= nx; x++) {
                for (int y = 1; y <= ny; y++) {
                    int rej;
                    double v = cpl_image_get(rec_fluxes, x, y, &rej);
                    fprintf(test5, "%d %d %f\n", x, y - nstep_lo - 1, v);
                }
            }
            cpl_free(testmodelp3name);
            fclose(test5);
            testmodelp3name =
                cpl_sprintf("%s_%d_model_step3.fits",
                            moo_detector_get_extname(det->type, det->ntas),
                            numfib);
            cpl_image_save(rec_fluxes, testmodelp3name, CPL_TYPE_FLOAT, NULL,
                           CPL_IO_CREATE);
            cpl_free(testmodelp3name);
        }
        cpl_image_delete(rec_fluxes);
        cpl_vector_delete(vX);
        cpl_vector_delete(vA);
        cpl_vector_delete(vCexp);
        cpl_vector_delete(vSigma);
        cpl_vector_delete(vC0);
        cpl_vector_delete(vB);

        cpl_vector_delete(resA);
        cpl_vector_delete(resCexp);
        cpl_vector_delete(resSigma);
        cpl_vector_delete(resB);
        cpl_vector_delete(resC0);

        fclose(test4);
        cpl_free(testmodel2name);

        cpl_free(testname);
        fclose(test);

        fclose(testp2);
        cpl_free(testmodelp2name);
    }
#endif

/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions where
 * these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_fibres, health, nx, ny, image, centroids, \
                                nstep_lo, nstep_up, step, imgtab, det,   \
                                win_hsize, xstep, fwup, fwlo, extname)
#else
#pragma omp parallel default(none)                                          \
    shared(nb_fibres, health, nx, ny, image, centroids, nstep_lo, nstep_up, \
               step, imgtab, det, win_hsize, xstep, fwup, fwlo, extname)
#endif
    {
#pragma omp for
#endif
        for (int i = 1; i <= nb_fibres; i++) {
            int h = health[i - 1];
            cpl_image *rec_fluxes = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
            if (h == 0) {
                double *data = cpl_image_get_data(rec_fluxes);
                for (int j = 0; j < nx * ny; j++) {
                    data[j] = NAN;
                }
            }
            else {
                /*_moo_extract_error_model_flat_fibre(image, i, centroids, nstep_lo, nstep_up,
            step, rec_fluxes,nb_fibres);
        {
            char* testname = cpl_sprintf("error_%d.fits",i);
            cpl_image_save(rec_fluxes,testname,CPL_TYPE_DOUBLE,header,CPL_IO_CREATE);
            cpl_free(testname);
        }*/
                /*_moo_rectify_fibre(image, i, centroids, nstep_lo, nstep_up,
            step, rec_fluxes);*/
                cpl_msg_info(__func__, "Modelling profiles %s: fibre %d",
                             extname, i);
                _moo_model_fibre(image, i, centroids, fwlo, fwup, nstep_lo,
                                 nstep_up, step, rec_fluxes, win_hsize, xstep);

                _moo_median_filter2_model_flat_fibre(rec_fluxes, 10);

                //_moo_fit_model_flat_fibre(rec_fluxes, deg_poly);
                //_moo_smooth_model_flat_fibre(rec_fluxes, 1600);

                //_moo_median_filter_model_flat_fibre(rec_fluxes,10,5);
                //_moo_savgol_model_flat_fibre(rec_fluxes,3,21);
            }
            imgtab[i - 1] = rec_fluxes;
        }
#ifdef _OPENMP
    }
#endif

    cpl_imagelist *cube = cpl_imagelist_new();
    for (int i = 0; i < nb_fibres; i++) {
        cpl_imagelist_set(cube, imgtab[i], i);
    }
    cpl_free(imgtab);
    res->cube = cube;
    moo_psf_single_set_cube_ref(res, nstep_lo + 1, step);
#if MOO_DEBUG_MODEL_FLAT
    {
        double crpix2, cd2_2;
        moo_psf_single_get_cube_ref(res, &crpix2, &cd2_2);

        char *testname =
            cpl_sprintf("%s_neg_cube.txt",
                        moo_detector_get_extname(det->type, det->ntas));
        FILE *test = fopen(testname, "w");
        fprintf(test, "#fibnum x y yc flux\n");

        for (int f = 0; f < nb_fibres; f++) {
            cpl_image *img = cpl_imagelist_get(cube, f);
            int cy = cpl_image_get_size_y(img);

            for (int y = 1; y <= cy; y++) {
                int rej = 0;

                for (int x = 1; x <= nx; x++) {
                    double centroid = centroids[f * nx + x - 1];
                    if (!isnan(centroid)) {
                        double yc = (y - crpix2) * cd2_2 + centroid;

                        double flux = cpl_image_get(img, x, y, &rej);
                        if (flux < 0) {
                            fprintf(test, "%d %d %f %d %f\n", f + 1, x, yc, y,
                                    flux);
                        }
                    }
                }
            }
        }
        fclose(test);
        cpl_free(testname);
    }
#endif
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in modelize flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    To extract 2D FF and model it, resulting MASTER_FLAT
  @param    det _DET_ frame to extract
  @param    loc _LOC_ localisation of the fibres
  @param    params the model flat parameters
  @param    filename the name of the result
  @return   the _PSF_ model flat

 * This function is to model the extracted FF, and resultsing a modelled FF,
 * i.e., the MasterFlat.
 *
 * _Flags considered as bad : BADPIX_COSMETIC | BADPIX_HOT
 *
 * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the _LOC_ file has not a fibre table
 */
/*----------------------------------------------------------------------------*/
moo_psf *
moo_model_flat(moo_det *det,
               moo_loc *loc,
               moo_model_flat_params *params,
               const char *filename)
{
    moo_psf *result = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_COSMETIC | MOO_BADPIX_HOT;
    // int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE| MOO_BADPIX_COSMETIC;

    cpl_msg_info(__func__, "Modelizing flat using oversamp %d",
                 params->oversamp);
    result = moo_psf_create(filename);

    cpl_table *fibres_table = moo_loc_get_fibre_table(loc);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_msg_indent_more();
    for (int i = 1; i <= 2; i++) {
        cpl_table_unselect_all(fibres_table);
        int fibname = i;
        cpl_table_or_selected_int(fibres_table, MOO_FIBRES_TABLE_SPECTRO,
                                  CPL_EQUAL_TO, fibname);
        cpl_table *selected = cpl_table_extract_selected(fibres_table);
        const int *health =
            cpl_table_get_data_int_const(selected, MOO_FIBRES_TABLE_HEALTH);

        for (int j = 0; j < 3; j++) {
            moo_single *det_single =
                moo_det_load_single(det, j, i, badpix_level);
            moo_loc_single *loc_single = moo_loc_get_single(loc, j, i);
            if (det_single != NULL && loc_single != NULL) {
                cpl_msg_info(__func__, "Modelizing flat for extension %s",
                             moo_detector_get_extname(j, i));
                moo_psf_single *psf_single =
                    _moo_model_flat_single(det_single, loc_single,
                                           result->filename, params, 9, health);
                moo_psf_add_single(result, psf_single, j, i);
            }
        }
        cpl_table_delete(selected);
    }
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in model flat");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}
/**@}*/
