/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <telluriccorr.h>

#include "moo_molecfit.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
#include "moo_molectable.h"
#include "moo_fibres_table.h"
#include "mf_model.h"
#include "moo_kernel.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static cpl_table *
_molecfit_data_create(double crpix1,
                      double crval1,
                      double cd1_1,
                      cpl_image *data,
                      cpl_image *data_err,
                      int idrbn)
{
    cpl_table *result = NULL;
    int nx = cpl_image_get_size_x(data);

    result = cpl_table_new(nx);
    cpl_table_new_column(result, "WAVE", CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, "FLUX", CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, "ERR", CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, "mask", CPL_TYPE_INT);

    for (int i = 0; i < nx; i++) {
        int rej1;
        double w = (i - crpix1) * cd1_1 + crval1;
        double f = cpl_image_get(data, i + 1, idrbn, &rej1);
        double e = cpl_image_get(data_err, i + 1, idrbn, &rej1);
        cpl_table_set_double(result, "WAVE", i, w);
        cpl_table_set_double(result, "FLUX", i, f);
        cpl_table_set_double(result, "ERR", i, e);
        cpl_table_set_int(result, "mask", i, rej1);
    }
    return result;
}

static cpl_error_code
_moo_molecfit_update_mf_conf(mf_configuration *config,
                             moo_molecfit_model_params *params)
{
    mf_parameters_inputs *inputs = &(config->parameters->inputs);
    inputs->wlg_to_micron = 0.001;
    cpl_free(inputs->column_dflux);
    cpl_free(inputs->column_lambda);
    cpl_free(inputs->column_flux);
    cpl_free(inputs->wavelengths_frame);
    inputs->column_dflux = cpl_strdup("ERR");
    inputs->column_lambda = cpl_strdup("WAVE");
    inputs->column_flux = cpl_strdup("FLUX");
    inputs->wavelengths_frame = cpl_strdup("AIR");
    inputs->transmission = CPL_TRUE;
    inputs->clean_flux = CPL_FALSE;
    inputs->silent_external_bins = CPL_TRUE;

    /*config->parameters->directories.telluric_path="/home/haigron/pipelines";
  config->parameters->directories.telluriccorr_data_path="/home/haigron/pipelines/share/molecfit/data/";
  */
    mf_parameters_fitting *fitting = &(config->parameters->fitting);
    fitting->ftol = 1E-10;
    fitting->xtol = 1E-10;
    fitting->flux_unit = 0;
    fitting->fit_telescope_background.fit = CPL_TRUE;
    fitting->fit_telescope_background.const_val = 0.1;
    fitting->fit_continuum.fit = CPL_TRUE;
    if (params) {
        fitting->fit_continuum.const_val = params->continuum_const;
        fitting->kern_mode = params->kern_mode;
        fitting->kern_fac = params->kern_fac;
        fitting->var_kern = params->var_kern;
    }
    else {
        fitting->fit_continuum.const_val = 1.;
        fitting->kern_mode = CPL_FALSE;
        fitting->kern_fac = 3;
        fitting->var_kern = CPL_TRUE;
    }
    fitting->fit_wavelenght.fit = CPL_TRUE;
    fitting->fit_wavelenght.n = 1;
    fitting->fit_wavelenght.const_val = 0.0;
    fitting->fit_gauss.fit = CPL_TRUE;
    fitting->fit_gauss.const_val = 1.0;

    fitting->fit_res_box.fit = CPL_FALSE;
    fitting->fit_res_box.const_val = 0;
    fitting->fit_lorentz.fit = CPL_FALSE;
    fitting->fit_lorentz.const_val = 0;

    fitting->fit_chips[0] = CPL_TRUE;

    mf_parameters_instrumental *instrumental =
        &(config->parameters->instrumental);
    cpl_free(instrumental->slit_width.key);
    cpl_free(instrumental->pixel_scale.key);
    instrumental->slit_width.key = cpl_strdup("NONE");
    instrumental->slit_width.value = 1.0;
    instrumental->pixel_scale.key = cpl_strdup("NONE");
    instrumental->pixel_scale.value = 0.015;

    mf_parameters_atmospheric *atmospheric = &(config->parameters->atmospheric);
    cpl_free(atmospheric->ref_atm);
    atmospheric->ref_atm = cpl_strdup("equ.fits");
    atmospheric->layers = CPL_TRUE;
    atmospheric->emix = 5.;
    atmospheric->pwv = -1;

    mf_io_lnfl_config *lnfl = config->lnfl;
    cpl_free(lnfl->line_db);
    lnfl->line_db = cpl_strdup("aer_v_3.8.1.2");
    lnfl->line_db_fmt = 100.0;

    mf_io_lblrtm_config *lblrtm = config->lblrtm;
    lblrtm->alfal0 = 0.0;
    lblrtm->altd[0] = 0;
    lblrtm->altd[1] = 0;
    lblrtm->icntnm = 5;
    lblrtm->avmass = 0;
    lblrtm->avtrat = 2.;
    lblrtm->beta = 0;
    lblrtm->delv = 1.;
    lblrtm->dptfac = 0.001;
    lblrtm->dptmin = 2e-4;
    lblrtm->h[0] = 2.635;
    lblrtm->h[1] = 0.;
    lblrtm->hobs = 0.;
    lblrtm->hspace = 120.;
    lblrtm->iaersl = 0.;
    lblrtm->ipunch = 0;
    lblrtm->itype = 3;
    lblrtm->len = 0;
    lblrtm->model = 0;
    lblrtm->mpts = 5;
    lblrtm->noprnt = 0;
    lblrtm->nozero = 0;
    lblrtm->npts = 5;
    lblrtm->range = 0.;
    lblrtm->re = 0.;
    lblrtm->sample = 4;
    lblrtm->sremis[0] = 0.;
    lblrtm->sremis[1] = 0.;
    lblrtm->sremis[2] = 0.;
    lblrtm->srrefl[0] = 0.;
    lblrtm->srrefl[1] = 0.;
    lblrtm->srrefl[2] = 0.;
    lblrtm->tbound = 0.;
    lblrtm->tdiff[0] = 5.;
    lblrtm->tdiff[1] = 8.;
    lblrtm->v[0] = 1.9;
    lblrtm->v[1] = 2.4;

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_molecfit_update_mf_conf_with_propertylist(
    mf_configuration *config,
    cpl_propertylist *parameters_header)
{
    cpl_ensure_code(config != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parameters_header != NULL, CPL_ERROR_NULL_INPUT);

    // mf_parameters_inputs *inputs = &(config->parameters->inputs);
    //  mf_parameters_fitting *fitting = &(config->parameters->fitting);
    //  mf_parameters_ambient *ambient = &(config->parameters->ambient);
    //  mf_parameters_instrumental *instrumental =
    //      &(config->parameters->instrumental);
    // mf_parameters_atmospheric *atmospheric = &(config->parameters->atmospheric);

    /*** Set Molecfit configuration with recipe parameters ***/

#if 0
  /*** Directories ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELLURICCORR_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->telluric_path) cpl_free(directories->telluric_path);
      directories->telluric_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->telluric_path);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELLURICCORR_DATA_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->telluriccorr_data_path) cpl_free(directories->telluriccorr_data_path);
      directories->telluriccorr_data_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->telluriccorr_data_path);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TMP_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->tmp_path) cpl_free(directories->tmp_path);
      directories->tmp_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->tmp_path);
  }

  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OUTPUT_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->output_path) cpl_free(directories->output_path);
      directories->output_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->output_path);
  }*/

  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OUTPUT_NAME;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->output_name) cpl_free(directories->output_name);
      directories->output_name = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->output_name);
  }*/
#endif
#if 0
  /*** Input parameters ***/
  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OMP_NUM_THREADS;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->omp_num_threads = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, inputs->omp_num_threads);
  }*/

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_SILENT_EXTERNAL_BINS;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->silent_external_bins = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, inputs->silent_external_bins);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TRANSMISSION;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->transmission = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, inputs->transmission);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_LAMBDA;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_lambda) cpl_free(inputs->column_lambda);
      inputs->column_lambda = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_lambda);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_FLUX;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_flux) cpl_free(inputs->column_flux);
      inputs->column_flux = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_flux);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_DFLUX;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_dflux) cpl_free(inputs->column_dflux);
      inputs->column_dflux = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_dflux);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_MASK;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_mask) cpl_free(inputs->column_mask);
      inputs->column_mask = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_mask);
  }


  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_DEFAULT_ERROR;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->default_error = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, inputs->default_error);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLG_TO_MICRON;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->wlg_to_micron = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, inputs->wlg_to_micron);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WAVELENGTH_FRAME;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->wavelengths_frame) cpl_free(inputs->wavelengths_frame);
      inputs->wavelengths_frame = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->wavelengths_frame);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_ERF_RV_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->observing_erv_rv.key) cpl_free(inputs->observing_erv_rv.key);
      inputs->observing_erv_rv.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->observing_erv_rv.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_ERF_RV_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->observing_erv_rv.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, inputs->observing_erv_rv.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CLEAN_MODEL_FLUX;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->clean_flux = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, inputs->clean_flux);
  }



  /*** Convergence criterion ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FTOL;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->ftol = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->ftol);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_XTOL;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->xtol = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->xtol);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FLUX_UNIT;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->flux_unit = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->flux_unit);
  }


  /*** Telescope background ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_TELESCOPE_BACK;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_telescope_background.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_telescope_background.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELESCOPE_BACK_CONST;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_telescope_background.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->fit_telescope_background.const_val);
  }


  /*** Continuum ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_CONTINUUM;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_continuum.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_continuum.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CONTINUUM_N;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_continuum.n = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->fit_continuum.n);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CONTINUUM_CONST;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_continuum.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_continuum.const_val);
  }

  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_BARY_RV;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->obs_bary_rv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->obs_bary_rv);
  }
  */


  /*** Wavelength solution fit/adjustment ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_WLC;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_wavelenght.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_wavelenght.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_N;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_wavelenght.n = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->fit_wavelenght.n);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_CONST;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_wavelenght.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_wavelenght.const_val);
  }

  /* p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_REF;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (fitting->wlc_ref) cpl_free(fitting->wlc_ref);
      fitting->wlc_ref = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, fitting->wlc_ref);
  } */


  /*** Default kernel: Boxcar kernel ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_RES_BOX;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_res_box.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_res_box.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_BOX;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_res_box.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_res_box.const_val);
  }


  /*** Default kernel: Gaussian kernel ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_GAUSS;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_gauss.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_gauss.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_GAUSS;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_gauss.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_gauss.const_val);
  }


  /*** Default kernel: Lorentz kernel ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_LORENTZ;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_lorentz.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_lorentz.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_LORENTZ;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_lorentz.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_lorentz.const_val);
  }


  /*** Default kernels: Generic parameters ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_MODE;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->kern_mode = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->kern_mode);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_FAC;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->kern_fac = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->kern_fac);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_VAR_KERN;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->var_kern = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->var_kern);
  }


  /*** Ambient ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVING_DATE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->observing_date.key) cpl_free(ambient->observing_date.key);
      ambient->observing_date.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->observing_date.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVING_DATE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->observing_date.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->observing_date.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_UTC_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->utc.key) cpl_free(ambient->utc.key);
      ambient->utc.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->utc.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_UTC_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->utc.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->utc.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELESCOPE_ANGLE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->telescope_angle.key) cpl_free(ambient->telescope_angle.key);
      ambient->telescope_angle.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->telescope_angle.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELESCOPE_ANGLE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->telescope_angle.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->telescope_angle.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RELATIVE_HUMIDITY_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->relative_humidity.key) cpl_free(ambient->relative_humidity.key);
      ambient->relative_humidity.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->relative_humidity.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RELATIVE_HUMIDITY_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->relative_humidity.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->relative_humidity.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PRESSURE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->pressure.key) cpl_free(ambient->pressure.key);
      ambient->pressure.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->pressure.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PRESSURE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->pressure.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->pressure.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TEMPERATURE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->temperature.key) cpl_free(ambient->temperature.key);
      ambient->temperature.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->temperature.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TEMPERATURE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->temperature.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->temperature.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_MIRROR_TEMPERATURE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->mirror_temperature.key) cpl_free(ambient->mirror_temperature.key);
      ambient->mirror_temperature.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->mirror_temperature.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_MIRROR_TEMPERATURE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->mirror_temperature.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->mirror_temperature.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_ELEVATION_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->elevation.key) cpl_free(ambient->elevation.key);
      ambient->elevation.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->elevation.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_ELEVATION_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->elevation.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->elevation.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LONGITUDE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->longitude.key) cpl_free(ambient->longitude.key);
      ambient->longitude.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->longitude.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LONGITUDE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->longitude.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->longitude.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LATITUDE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->latitude.key) cpl_free(ambient->latitude.key);
      ambient->latitude.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->latitude.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LATITUDE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->latitude.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->latitude.value);
  }


  /*** Instrumental ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_SLIT_WIDTH_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (instrumental->slit_width.key) cpl_free(instrumental->slit_width.key);
      instrumental->slit_width.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, instrumental->slit_width.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_SLIT_WIDTH_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      instrumental->slit_width.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, instrumental->slit_width.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PIXEL_SCALE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (instrumental->pixel_scale.key) cpl_free(instrumental->pixel_scale.key);
      instrumental->pixel_scale.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, instrumental->pixel_scale.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PIXEL_SCALE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      instrumental->pixel_scale.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, instrumental->pixel_scale.value);
  }


  /*** Atmospheric ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_REFERENCE_ATMOSPHERIC;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (atmospheric->ref_atm) cpl_free(atmospheric->ref_atm);
      atmospheric->ref_atm = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, atmospheric->ref_atm);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_GDAS_PROFILE;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (atmospheric->gdas_prof) cpl_free(atmospheric->gdas_prof);
      atmospheric->gdas_prof = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, atmospheric->gdas_prof);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LAYERS;
  if (cpl_propertylist_has(parameters_header, p)) {
      atmospheric->layers = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, atmospheric->layers);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_EMIX;
  if (cpl_propertylist_has(parameters_header, p)) {
      atmospheric->emix = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, atmospheric->emix);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PWV;
  if (cpl_propertylist_has(parameters_header, p)) {
      atmospheric->pwv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, atmospheric->pwv);
  }


  /*** LNFL ***/
  mf_io_lnfl_config *lnfl = mf_config->lnfl;

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LNFL_LINE_DB;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (lnfl->line_db) cpl_free(lnfl->line_db);
      lnfl->line_db = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, lnfl->line_db);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LNFL_LINE_DB_FMT;
  if (cpl_propertylist_has(parameters_header, p)) {
      lnfl->line_db_fmt = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lnfl->line_db_fmt);
  }


  /*** LBLRTM ***/
  mf_io_lblrtm_config *lblrtm = mf_config->lblrtm;

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ICNTNM;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->icntnm = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->icntnm);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_IAERSL;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->iaersl = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->iaersl);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_MPTS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->mpts = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->mpts);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_NPTS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->npts = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->npts);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_V1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->v[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->v[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_V2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->v[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->v[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SAMPLE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sample = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->sample);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ALFAL0;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->alfal0 = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->alfal0);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_AVMASS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->avmass = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->avmass);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_DPTMIN;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->dptmin = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->dptmin);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_DPTFAC;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->dptfac = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->dptfac);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_TBOUND;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->tbound = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->tbound);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SREMIS1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sremis[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->sremis[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SREMIS2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sremis[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->sremis[1]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SREMIS3;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sremis[2] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->sremis[2]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SRREFL1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->srrefl[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->srrefl[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SRREFL2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->srrefl[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->srrefl[1]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SRREFL3;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->srrefl[2] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->srrefl[2]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_MODEL;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->model = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->model);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ITYPE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->itype = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->itype);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_NOZERO;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->nozero = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->nozero);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_NOPRNT;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->noprnt = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->noprnt);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_IPUNCH;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->ipunch = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->ipunch);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_RE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->re = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->re);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_HSPACE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->hspace = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->hspace);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LATITUDE_VALUE; // MF_LBLRTM_REF_LAT;   // MF_PARAMETERS_LATITUDE_VALUE
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->ref_lat = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->ref_lat);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_ELEVATION_VALUE; // MF_LBLRTM_H1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->h[0] =  ambient->elevation.value /1000.00 ; // cpl_propertylist_get_double(parameters_header, p);
      /*cpl_propertylist_update_double(pl, p, lblrtm->h[0]);*/
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_H2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->h[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->h[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_RANGE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->range = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->range);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_BETA;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->beta = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->beta);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_LEN;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->len = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->len);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_HOBS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->hobs = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->hobs);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_AVTRAT;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->avtrat = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->avtrat);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_TDIFF1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->tdiff[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->tdiff[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_TDIFF2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->tdiff[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->tdiff[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ALTD1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->altd[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->altd[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ALTD2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->altd[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->altd[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_DELV;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->delv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->delv);
  }


  /*** Update the Molecfit configuration with the header ***/
  cpl_error_code e = mf_parameters_config_update_with_header_keywords(mf_config->parameters, data_header);


  cpl_msg_info(cpl_func, "Columns : wave[%s], flux[%s], dflux[%s], mask[%s]", inputs->column_lambda, inputs->column_flux, inputs->column_dflux, inputs->column_mask);


  /* Check possible errors */
  if (!cpl_errorstate_is_equal(pre_state) || e != CPL_ERROR_NONE) {
      mf_configuration_delete(mf_config);
      cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                            "Error loading Molecfit default configuration");
      return NULL;
  }
#endif
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_molecfit_update_mf_conf_with_winc(mf_configuration *config,
                                       cpl_table *inc_wranges)
{
    // mf_parameters_inputs *inputs = &(config->parameters->inputs);

    mf_parameters_fitting *fitting = &(config->parameters->fitting);
    int nrange = cpl_table_get_nrow(inc_wranges);
    int max_order = 0;
    for (int i = 0; i < nrange; i++) {
        cpl_boolean flag = CPL_TRUE;

        int ival =
            cpl_table_get_int(inc_wranges, MF_COL_WAVE_RANGE_CONT_FIT, i, NULL);
        int order = cpl_table_get_int(inc_wranges, MF_COL_WAVE_RANGE_CONT_ORDER,
                                      i, NULL);
        if (order > max_order) {
            max_order = order;
        }
        if (ival == 0)
            flag = CPL_FALSE;
        fitting->fit_ranges[i] = flag;
        fitting->cont_poly_order[i] = order;
        fitting->cont_coeffs[i][0] = fitting->fit_continuum.const_val;
        fitting->cont_coeffs[i][1] = 1;
    }
    fitting->fit_continuum.n = max_order;
    fitting->fit_chips[0] = CPL_TRUE;

    return CPL_ERROR_NONE;
}

static void
_mf_parameters_fit_dump(const char *name, mf_parameters_fit *fit, FILE *out)
{
    fprintf(out, "%s.fit : %d\n", name, fit->fit);
    fprintf(out, "%s.n   : %lld\n", name, fit->n);
    fprintf(out, "%s.const_val : %e\n", name, fit->const_val);
}

static void
_mf_parameters_key_dump(const char *name, mf_parameters_keyword *key, FILE *out)
{
    fprintf(out, "%s.key :   %s\n", name, key->key);
    fprintf(out, "%s.value : %e\n", name, key->value);
}

static cpl_error_code
_moo_molecfit_mf_conf_dump(mf_configuration *config, int nrange, FILE *out)
{
    mf_parameters_inputs *inputs = &(config->parameters->inputs);
    fprintf(out, "---INPUTS\n");
    fprintf(out, "wlg_to_micron:              %f\n", inputs->wlg_to_micron);

    fprintf(out, "column_lambda:              %s\n", inputs->column_lambda);
    fprintf(out, "column_flux:                %s\n", inputs->column_flux);
    fprintf(out, "column_dflux:               %s\n", inputs->column_dflux);
    fprintf(out, "column_mask:                %s\n", inputs->column_mask);
    fprintf(out, "default_error:              %e\n", inputs->default_error);
    fprintf(out, "mask_binary:                %d\n", inputs->mask_binary);
    fprintf(out, "mask_threshold:             %d\n", inputs->mask_threshold);
    fprintf(out, "wavelengths_frame:          %s\n", inputs->wavelengths_frame);
    fprintf(out, "transmission :              %d\n", inputs->transmission);
    fprintf(out, "clean_flux :                %d\n", inputs->clean_flux);
    fprintf(out, "silent_external_bins :      %d\n",
            inputs->silent_external_bins);
    _mf_parameters_key_dump("observing_erv_rv", &(inputs->observing_erv_rv),
                            out);

    fprintf(out, "\n---DIRECTORIES\n");
    fprintf(out, "telluric_path :          %s\n",
            config->parameters->directories.telluric_path);
    fprintf(out, "telluriccorr_data_path : %s\n",
            config->parameters->directories.telluriccorr_data_path);
    fprintf(out, "tmp_path :               %s\n",
            config->parameters->directories.tmp_path);

    mf_parameters_fitting *fitting = &(config->parameters->fitting);
    fprintf(out, "\n---FITTING\n");
    fprintf(out, "ftol :      %e\n", fitting->ftol);
    fprintf(out, "xtol :      %e\n", fitting->xtol);
    fprintf(out, "flux_unit : %d\n", fitting->flux_unit);
    _mf_parameters_fit_dump("fit_telescope_background",
                            &(fitting->fit_telescope_background), out);
    _mf_parameters_fit_dump("fit_continuum", &(fitting->fit_continuum), out);
    _mf_parameters_fit_dump("fit_wavelength",
                            &(fitting->fit_telescope_background), out);
    _mf_parameters_fit_dump("fit_gauss", &(fitting->fit_gauss), out);
    _mf_parameters_fit_dump("fit_res_box", &(fitting->fit_res_box), out);
    _mf_parameters_fit_dump("fit_lorentz", &(fitting->fit_lorentz), out);
    fprintf(out, "kern_mode :   %d\n", fitting->kern_mode);
    fprintf(out, "kern_fac :    %e\n", fitting->kern_fac);
    fprintf(out, "var_kern :    %d\n", fitting->var_kern);
    fprintf(out, "expert_mode :  %d\n", fitting->expert_mode);
    fprintf(out, "fit_n_cflags : %d\n", fitting->fit_n_cflags);
    fprintf(out, "fit_n_rflags : %d\n", fitting->fit_n_rflags);

    for (int i = 0; i < nrange; i++) {
        fprintf(out, "fit_ranges     [%d] :  %d\n", i, fitting->fit_ranges[i]);
        fprintf(out, "fit_chips      [%d] :  %d\n", i, fitting->fit_chips[i]);
        fprintf(out, "cont_poly_order[%d] :  %d\n", i,
                fitting->cont_poly_order[i]);
        for (int j = 0; j < fitting->cont_poly_order[i]; j++) {
            fprintf(out, "cont_coeffs    [%d][%d] :  %f\n", i, j,
                    fitting->cont_coeffs[i][j]);
        }
        fprintf(out, "wlc_poly_order [%d] :  %d\n", i,
                fitting->wlc_poly_order[i]);
        fprintf(out, "wlc_coeffs     [%d][0] :  %f\n", i,
                fitting->wlc_coeffs[i][0]);
    }

    mf_parameters_instrumental *instrumental =
        &(config->parameters->instrumental);
    fprintf(out, "\n---INSTRUMENTAL\n");
    _mf_parameters_key_dump("slit_width", &(instrumental->slit_width), out);
    _mf_parameters_key_dump("pixel_scale", &(instrumental->pixel_scale), out);

    mf_parameters_atmospheric *atmospheric = &(config->parameters->atmospheric);
    fprintf(out, "\n---ATMOSPHERIC\n");

    fprintf(out, "ref_atm :   %s\n", atmospheric->ref_atm);
    fprintf(out, "layers :    %d\n", atmospheric->layers);
    fprintf(out, "emix :      %e\n", atmospheric->emix);
    fprintf(out, "pwv :       %e\n", atmospheric->pwv);
    fprintf(out, "gdas_prof : %s\n", atmospheric->gdas_prof);

    mf_io_lnfl_config *lnfl = config->lnfl;
    fprintf(out, "\n---LNLF\n");
    fprintf(out, "line_db : %s\n", lnfl->line_db);
    fprintf(out, "line_db_fmt : %d\n", lnfl->line_db_fmt);

    mf_io_lblrtm_config *lblrtm = config->lblrtm;
#if 0
  int                        icntnm;                 /* Continua and Rayleigh extinction [0,1,2,3,4,5]              */
  int                        iaersl;                 /* Aerosols [0,1]                                              */
  int                        mpts;                   /* Number of optical depth values                              */
  int                        npts;                   /* Number of values for each panel                             */
  double                     v[2];                   /* Ending wavenumber value for the calculation                 */
  int                        sample;                 /* Number of sample points per mean halfwidth [between 1 to 4] */
  double                     alfal0;                 /* Average collision broadened halfwidth [cm-1/atm]            */
  double                     avmass;                 /* Average molecular mass [amu] for Doppler halfwidth          */
  double                     dptmin;                 /* Min molecular optical depth below lines will be rejected    */
  double                     dptfac;                 /* Factor multiplying molecular continuum optical depth        */
  double                     tbound;                 /* Temperature of boundary [K]                                 */
  double                     sremis[3];              /* Emissivity coefficients                                     */
  double                     srrefl[3];              /* Reflectivity coefficients                                   */
  int                        model;                  /* Atmospheric profile [0,1,2,3,4,5,6]                         */
  int                        itype;                  /* Type of path [1,2,3]                                        */
  int                        nozero;                 /* Zeroing of small amounts of absorbers [0,1]                 */
  int                        noprnt;                 /* Do not print output? [0,1]                                  */
  int                        ipunch;                 /* Write out layer data to TAPE7 [0,1]                         */
  double                     re;                     /* Radius of earth [km]                                        */
  double                     hspace;                 /* Altitude definition for space [km]                          */
  double                     ref_lat;                /* Latitude of location of calculation [degrees] [-90.-90]     */
  double                     h[2];                   /* h[0] = Observer altitude, h[1] = Upper height limit; [km]   */
  double                     range;                  /* Length of a straight path from H1 to H2 [km]                */
  double                     beta;                   /* Earth centered angle from H1 to H2 [degrees]                */
  int                        len;                    /* Path length [0,1]                                           */
  double                     hobs;                   /* Height of observer                                          */
  double                     avtrat;                 /* Maximum Voigt width ratio across a layer                    */
  double                     tdiff[2];               /* Maximum layer temperature difference at ALTD1, ALTD2 [K]    */
  double                     altd[2];                /* Altitude of TDIFF1, TDIFF2 [km]                             */
  double                     delv;                   /* Number of wavenumbers [cm-1] per major division             */
#endif
    fprintf(out, "\n---LBLRTM\n");
    fprintf(out, "alfal0 :   %e\n", lblrtm->alfal0);
    fprintf(out, "altd[0] :  %e\n", lblrtm->altd[0]);
    fprintf(out, "altd[1] :  %e\n", lblrtm->altd[1]);
    fprintf(out, "avmass :   %e\n", lblrtm->avmass);
    fprintf(out, "avtrat :   %e\n", lblrtm->avtrat);
    fprintf(out, "beta :     %e\n", lblrtm->beta);
    fprintf(out, "dptfac :   %e\n", lblrtm->dptfac);
    fprintf(out, "dptmin :   %e\n", lblrtm->dptmin);
    fprintf(out, "h[0] :     %e\n", lblrtm->h[0]);
    fprintf(out, "h[1] :     %e\n", lblrtm->h[1]);
    fprintf(out, "hobs :     %e\n", lblrtm->hobs);
    fprintf(out, "hspace :   %e\n", lblrtm->hspace);
    fprintf(out, "iaersl :   %d\n", lblrtm->iaersl);
    fprintf(out, "icntnm :   %d\n", lblrtm->icntnm);
    fprintf(out, "ipunch :   %d\n", lblrtm->ipunch);
    fprintf(out, "itype :    %d\n", lblrtm->itype);
    fprintf(out, "len :      %d\n", lblrtm->len);
    fprintf(out, "model :    %d\n", lblrtm->mpts);
    fprintf(out, "npprnt :   %d\n", lblrtm->noprnt);
    fprintf(out, "nozero :   %d\n", lblrtm->nozero);
    fprintf(out, "npts :     %d\n", lblrtm->npts);
    fprintf(out, "range :    %e\n", lblrtm->range);
    fprintf(out, "re :       %e\n", lblrtm->re);
    fprintf(out, "sample :   %d\n", lblrtm->sample);
    fprintf(out, "sremis[0] :%e\n", lblrtm->sremis[0]);
    fprintf(out, "sremis[1] :%e\n", lblrtm->sremis[1]);
    fprintf(out, "sremis[2] :%e\n", lblrtm->sremis[2]);
    fprintf(out, "srrefl[0] :%e\n", lblrtm->srrefl[0]);
    fprintf(out, "srrefl[1] :%e\n", lblrtm->srrefl[1]);
    fprintf(out, "srrefl[2] :%e\n", lblrtm->srrefl[2]);
    fprintf(out, "tbound    :%e\n", lblrtm->tbound);
    fprintf(out, "tdiff[0]  :%e\n", lblrtm->tdiff[0]);
    fprintf(out, "tdiff[1]  :%e\n", lblrtm->tdiff[1]);
    fprintf(out, "v[0]      :%e\n", lblrtm->v[0]);
    fprintf(out, "v[1]      :%e\n", lblrtm->v[1]);

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the relative flux calibration to 1D rebinned spectra
  @param    sci_frame the sky Input wavelength calibrated, sky corrected frame
  @param    modename the instrument mode (LR or HR)
  @param    molecule_frame the MOLECULE frame
  @param    winc_frame the WAVE_INCLUDE frame
  @param    pmt_atm_fitted The ATM fitted result
  @param    best_fitted_model The BEST FITTED model result
  @param    params the molecfit model parameters
  @return   The best fitted parameters result

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_molectable *
moo_molecfit_model(const cpl_frame *sci_frame,
                   const char *modename,
                   const cpl_frame *molecule_frame,
                   const cpl_frame *winc_frame,
                   moo_molectable **pmt_atm_fitted,
                   moo_molectable **best_fitted_model,
                   moo_molecfit_model_params *params)
{
    mf_configuration *mf_config = NULL;
    moo_molectable *mt_molecules = NULL;
    moo_molectable *mt_winc = NULL;
    moo_molectable *mt_lblrtm_input = NULL;
    moo_molectable *mt_excprange_input = NULL;
    moo_molectable *mt_excwrange_input = NULL;

    moo_molectable *mt_gdas_interp = NULL;
    moo_molectable *mt_gdas_after = NULL;
    moo_molectable *mt_gdas_before = NULL;
    moo_molectable *mt_atm_combined = NULL;
    moo_molectable *mt_atm_fitted = NULL;
    moo_molectable *mt_atm_standard = NULL;
    moo_molectable *mt_best_fitted = NULL;
    moo_molectable *mt_best_model = NULL;

    cpl_table *molecules_table = NULL;
    cpl_table *winc_table = NULL;
    cpl_table *spec = NULL;
    cpl_table *gdas_user = NULL;
    cpl_table *atm_profile_standard = NULL;
    cpl_table *atm_profile_combined = NULL;
    cpl_error_code err = CPL_ERROR_NONE;
    moo_sci *sci = NULL;
    mf_model_results *results = NULL;
    cpl_table *mdata = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(pmt_atm_fitted != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_msg_info(__func__, "use molecfit model");
    mf_config = mf_configuration_create();
    sci = moo_sci_create(sci_frame);

    mt_molecules = moo_molectable_load(molecule_frame, modename);
    mt_winc = moo_molectable_load(winc_frame, modename);

    mt_lblrtm_input = moo_molectable_new();
    mt_excprange_input = moo_molectable_new();
    mt_excwrange_input = moo_molectable_new();

    mt_gdas_after = moo_molectable_new();
    mt_gdas_interp = moo_molectable_new();
    mt_gdas_before = moo_molectable_new();
    mt_atm_combined = moo_molectable_new();
    mt_atm_standard = moo_molectable_new();

    mt_atm_fitted = moo_molectable_new();
    mt_best_fitted = moo_molectable_new();
    mt_best_model = moo_molectable_new();
    mt_atm_fitted->primary_header = cpl_propertylist_new();
    mt_best_fitted->primary_header = cpl_propertylist_new();
    mt_best_model->primary_header = cpl_propertylist_new();


    for (int type = 0; type < 3; type++) {
        moo_sci_single *sci_single = moo_sci_get_single(sci, type);
        molecules_table = moo_molectable_get_data(mt_molecules, type);
        winc_table = moo_molectable_get_data(mt_winc, type);
        if (sci_single != NULL && molecules_table != NULL &&
            winc_table != NULL) {
            cpl_msg_info(__func__, "Do molecfit_model for %s",
                         sci_single->extname);
            cpl_msg_indent_more();
            cpl_propertylist *header = moo_sci_single_get_header(sci_single);
            double cd1_1 = moo_pfits_get_cd1_1(header);
            double crpix1 = moo_pfits_get_crpix1(header);
            double crval1 = moo_pfits_get_crval1(header);

            // int nrange = cpl_table_get_nrow(winc_table);
            err = mf_parameters_config_update_with_header_keywords(
                mf_config->parameters, sci->primary_header);

            if (err != CPL_ERROR_NONE) {
                cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                      "Error updating molecfit parameters with "
                                      "the raw primary header configuration");
            }
            _moo_molecfit_update_mf_conf(mf_config, params);
            _moo_molecfit_update_mf_conf_with_winc(mf_config, winc_table);
            moo_sci_single_load(sci_single, MOO_BADPIX_GOOD);
            hdrl_image *hdata = moo_sci_single_get_image(sci_single);
            cpl_image *img = hdrl_image_get_image(hdata);
            cpl_image *ierr = hdrl_image_get_error(hdata);
            mdata = _molecfit_data_create(crpix1, crval1, cd1_1, img, ierr, 1);
            /* Running molecfit */
            cpl_msg_info(cpl_func, "MNB About to call mf_model, err=%d", err);


            if (!err) {
                cpl_table *exc_wranges = NULL;
                cpl_table *exc_pranges = NULL;
                cpl_propertylist *header_kernel = NULL;
                cpl_matrix *kernel = NULL;

                /* Create input molecfit spec format */
                cpl_table *spec_telluriccorr_lblrtm =
                    mf_spectrum_create(mf_config->parameters, mdata);
                spec = spec_telluriccorr_lblrtm;

#if MOO_DEBUG_MOLECFIT_MODEL
                const char *arm = moo_detector_get_name(type);
                if (atm_profile_standard != NULL) {
                    char *testname =
                        cpl_sprintf("ATM_PROFILE_STANDARD_INPUT_%s.fits", arm);
                    cpl_table_save(atm_profile_standard, NULL, NULL, testname,
                                   CPL_IO_CREATE);
                    cpl_free(testname);
                }
                if (atm_profile_combined != NULL) {
                    char *testname =
                        cpl_sprintf("ATM_PROFILE_COMBINED_INPUT_%s.fits", arm);
                    cpl_table_save(atm_profile_combined, NULL, NULL, testname,
                                   CPL_IO_CREATE);
                    cpl_free(testname);
                }
                if (gdas_user != NULL) {
                    char *testname =
                        cpl_sprintf("GDAS_USER_INPUT_%s.fits", arm);
                    cpl_table_save(gdas_user, NULL, NULL, testname,
                                   CPL_IO_CREATE);
                    cpl_free(testname);
                }
                moo_molectable_set_data(mt_lblrtm_input, type, spec);
                moo_molectable_set_data(mt_excprange_input, type, exc_pranges);
                moo_molectable_set_data(mt_excwrange_input, type, exc_wranges);
                if (sci_single->header) {
                    char *testname =
                        cpl_sprintf("HEADER_SPEC_INPUT_%s.txt", arm);
                    FILE *f = fopen(testname, "w");
                    cpl_propertylist_dump(sci_single->header, f);
                    fclose(f);
                    cpl_free(testname);
                }
                if (header_kernel != NULL) {
                    char *testname =
                        cpl_sprintf("HEADER_KERNEL_INPUT_%s.txt", arm);
                    FILE *f = fopen(testname, "w");
                    cpl_propertylist_dump(header_kernel, f);
                    fclose(f);
                    cpl_free(testname);
                }
                if (kernel != NULL) {
                    char *testname = cpl_sprintf("KERNEL_INPUT_%s.txt", arm);
                    FILE *f = fopen(testname, "w");
                    cpl_matrix_dump(kernel, f);
                    fclose(f);
                    cpl_free(testname);
                }
                if (mf_config != NULL) {
                    char *testname = cpl_sprintf("MF_CONFIG_INPUT_%s.txt", arm);
                    cpl_msg_info("test", "save config %s", testname);
                    FILE *f = fopen(testname, "w");
                    _moo_molecfit_mf_conf_dump(mf_config, nrange, f);
                    fclose(f);
                    cpl_free(testname);
                }
#endif
                results = mf_model(
                    mf_config, molecules_table, sci_single->header,
                    spec, /* const cpl_table        *spec_telluriccorr    */
                    winc_table, /* cpl_table              *inc_wranges          */
                    exc_wranges, /* cpl_table              *exc_wranges          */
                    exc_pranges, /* cpl_table              *exc_wranges          */
                    header_kernel, /* const cpl_propertylist *header_kernel        */
                    kernel, /* const cpl_matrix       *kernel               */
                    gdas_user, /* const cpl_table        *gdas_user            */
                    atm_profile_standard, /* const cpl_table        *atm_profile_standard */
                    atm_profile_combined); /* const cpl_table        *atm_profile_combined */
                err = cpl_error_get_code();

                if (!err && results != NULL) {
                    moo_molectable_set_data(mt_gdas_interp, type,
                                            results->gdas_interpolate);
                    moo_molectable_set_data(mt_gdas_before, type,
                                            results->gdas_before);
                    moo_molectable_set_data(mt_gdas_after, type,
                                            results->gdas_after);
                    moo_molectable_set_data(mt_atm_standard, type,
                                            results->atm_profile_standard);
                    moo_molectable_set_data(mt_atm_combined, type,
                                            results->atm_profile_combined);
                    moo_molectable_set_data(mt_atm_fitted, type,
                                            results->atm_profile_fitted);
                    moo_molectable_set_data(mt_best_fitted, type, results->res);
                    moo_molectable_set_data(mt_best_model, type, results->spec);
                }

                if (results != NULL) {
                    if (results->atm_profile_combined != NULL) {
                        cpl_table_delete(results->atm_profile_combined);
                        results->atm_profile_combined = NULL;
                    }
                    if (results->atm_profile_standard != NULL) {
                        cpl_table_delete(results->atm_profile_standard);
                        results->atm_profile_standard = NULL;
                    }
                    mf_model_results_delete(results);
                    results = NULL;
                }
                cpl_table_delete(exc_pranges);
                cpl_table_delete(exc_wranges);
                cpl_table_delete(spec);
                spec = NULL;
            }

            cpl_table_delete(mdata);
            mdata = NULL;
        }
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        moo_molectable_delete(mt_best_fitted);
        moo_molectable_delete(mt_atm_fitted);
        moo_molectable_delete(mt_best_model);
        mt_best_fitted = NULL;
        mt_atm_fitted = NULL;
        mt_best_model = NULL;
        cpl_table_delete(mdata);
        mf_model_results_delete(results);
    }
    *pmt_atm_fitted = mt_atm_fitted;
    *best_fitted_model = mt_best_model;
    moo_molectable_delete(mt_molecules);
    moo_molectable_delete(mt_winc);
    moo_molectable_delete(mt_lblrtm_input);
    moo_molectable_delete(mt_excprange_input);
    moo_molectable_delete(mt_excwrange_input);
    moo_molectable_delete(mt_gdas_interp);
    moo_molectable_delete(mt_gdas_after);
    moo_molectable_delete(mt_gdas_before);
    moo_molectable_delete(mt_atm_standard);
    moo_molectable_delete(mt_atm_combined);
    mf_configuration_delete(mf_config);
    moo_sci_delete(sci);
    return mt_best_fitted;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the molcecfit calctrans to the RBN to compute telle telluric correction
  @param    rbn_frame The RBN SPECTRA
  @param    modename mode LR or HR
  @param    molecules_frame The MOLECULES frame
  @param    atm_frame the ATM frame (from model)
  @param    best_fit_frame the BEST_FIT frame (from model)
  @param    kernel_lib_frame the KERNEL frame
  @param    params the calctrans parameters
  @return   Flux calibrated frame

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/

moo_telluric *
moo_molecfit_calctrans(const cpl_frame *rbn_frame,
                       const char *modename,
                       const cpl_frame *molecules_frame,
                       const cpl_frame *atm_frame,
                       const cpl_frame *best_fit_frame,
                       const cpl_frame *kernel_lib_frame,
                       moo_molecfit_calctrans_params *params)
{
    moo_rbn *rbn = NULL;

    mf_configuration *mf_config = NULL;
    moo_molectable *mt_molecules = NULL;
    moo_molectable *mt_atm = NULL;
    moo_molectable *mt_bfit = NULL;
    moo_kernel *mkernel = NULL;
    cpl_table *mdata = NULL;
    moo_molectable *mt_lblrtm_results = NULL;
    cpl_error_code err = CPL_ERROR_NONE;
    cpl_array *sel = NULL;
    moo_telluric *result = NULL;
    int filter_skyfibre = 1;
    cpl_ensure(rbn_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(atm_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(best_fit_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    filter_skyfibre = params->filter_skyfibre;
    mf_config = mf_configuration_create();

    rbn = moo_rbn_create(rbn_frame);
    mt_molecules = moo_molectable_load(molecules_frame, modename);
    mt_atm = moo_molectable_load(atm_frame, NULL);
    mt_bfit = moo_molectable_load(best_fit_frame, NULL);
    mt_lblrtm_results = moo_molectable_new();

    cpl_table *fibre_table = moo_rbn_get_fibre_table(rbn);
    mkernel = moo_kernel_load(kernel_lib_frame);

    const char *snrcolnames[] = { MOO_FIBRES_TABLE_MEDSNR_RI_EXT,
                                  MOO_FIBRES_TABLE_MEDSNR_YJ_EXT,
                                  MOO_FIBRES_TABLE_MEDSNR_H_EXT };
    const char *bandcolnames[] = { MOO_TELLURIC_EXT_RI, MOO_TELLURIC_EXT_YJ,
                                   MOO_TELLURIC_EXT_H };
    result = moo_telluric_new();
    result->primary_header = cpl_propertylist_new();
    result->telluric_table = cpl_table_new(0);

    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_INDEXRBN,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_SPECTRO,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_BAND,
                         CPL_TYPE_STRING);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_TELLURIC,
                         CPL_TYPE_INT);

    for (int type = 0; type < 3; type++) {
        moo_rbn_single *rbn_single = moo_rbn_get_single(rbn, type);
        cpl_table *molecules_table =
            moo_molectable_get_data(mt_molecules, type);

        cpl_table *atm_parameters_table = moo_molectable_get_data(mt_atm, type);

        cpl_table *best_fit_parameters_table =
            moo_molectable_get_data(mt_bfit, type);

        if (rbn_single != NULL && molecules_table != NULL &&
            atm_parameters_table != NULL && best_fit_parameters_table != NULL) {
            cpl_table *telluric_table = NULL;
            double min_snr = params->min_snr[type];
            const char *band = bandcolnames[type];
            cpl_table_unselect_all(fibre_table);
            int size = 0;
            for (int i = 0; i < cpl_table_get_nrow(fibre_table); i++) {
                int rej;
                double val = cpl_table_get_float(fibre_table, snrcolnames[type],
                                                 i, &rej);
                if (!isnan(val)) {
                    cpl_table_select_row(fibre_table, i);
                    size++;
                }
            }

            size =
                cpl_table_and_selected_int(fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                           CPL_NOT_EQUAL_TO,
                                           MOO_FIBRES_TABLE_HEALTH_BROKEN);
            if (filter_skyfibre) {
                size = cpl_table_and_selected_string(fibre_table,
                                                     MOO_FIBRES_TABLE_TYPE,
                                                     CPL_NOT_EQUAL_TO,
                                                     MOO_FIBRES_TABLE_TYPE_SKY);
                size = cpl_table_and_selected_string(
                    fibre_table, MOO_FIBRES_TABLE_TYPE, CPL_NOT_EQUAL_TO,
                    MOO_FIBRES_TABLE_TYPE_SKYCONTAM);
            }
            size = cpl_table_and_selected_float(fibre_table, snrcolnames[type],
                                                CPL_GREATER_THAN, min_snr);

            sel = cpl_table_where_selected(fibre_table);

            cpl_msg_info(__func__,
                         "Do molecfit_calctrans for %s (%d selected fibres)",
                         rbn_single->extname, size);
            cpl_msg_indent_more();
            cpl_propertylist *header = moo_rbn_single_get_header(rbn_single);
            double cd1_1 = moo_pfits_get_cd1_1(header);
            double crpix1 = moo_pfits_get_crpix1(header);
            double crval1 = moo_pfits_get_crval1(header);

            err = mf_parameters_config_update_with_header_keywords(
                mf_config->parameters, rbn->primary_header);

            if (err != CPL_ERROR_NONE) {
                cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                      "Error updating molecfit parameters with "
                                      "the raw primary header configuration");
            }

            _moo_molecfit_update_mf_conf(mf_config, NULL);
            moo_rbn_single_load(rbn_single, MOO_BADPIX_GOOD);
            hdrl_image *hdata = moo_rbn_single_get_image(rbn_single);
            cpl_image *img = hdrl_image_get_image(hdata);
            cpl_image *ierr = hdrl_image_get_error(hdata);
            int nx = cpl_image_get_size_x(img);
            telluric_table = cpl_table_new(size);

            cpl_table_new_column(telluric_table, MOO_TELLURIC_TABLE_INDEXRBN,
                                 CPL_TYPE_INT);
            cpl_table_new_column(telluric_table, MOO_TELLURIC_TABLE_SPECTRO,
                                 CPL_TYPE_INT);
            cpl_table_new_column(telluric_table, MOO_TELLURIC_TABLE_BAND,
                                 CPL_TYPE_STRING);
            cpl_table_new_column(telluric_table, MOO_TELLURIC_TABLE_TELLURIC,
                                 CPL_TYPE_INT);

            cpl_propertylist *theader = cpl_propertylist_new();
            cpl_propertylist_copy_property(theader, header, MOO_PFITS_CD1_1);
            cpl_propertylist_copy_property(theader, header, MOO_PFITS_CRVAL1);
            cpl_propertylist_copy_property(theader, header, MOO_PFITS_CRPIX1);
            cpl_propertylist_copy_property(theader, header, MOO_PFITS_CUNIT1);
            result->data_header[type] = theader;

            cpl_image *im = cpl_image_new(nx, size, CPL_TYPE_DOUBLE);
            moo_telluric_set_image(result, type, im);
            mf_calctrans_lblrtm_results *lblrtm_results = NULL;
            const double wl_start = -1.;
            const double wl_end = -1.;
            const cpl_propertylist *spec_telluriccorr_head = rbn_single->header;

            if (size > 0) {
                cpl_msg_info(__func__, "Do LBLRTM for %s", rbn_single->extname);
                int idx = cpl_array_get_cplsize(sel, 0, NULL);
                int idrbn =
                    cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_INDEXRBN,
                                      idx, NULL);

                mdata = _molecfit_data_create(crpix1, crval1, cd1_1, img, ierr,
                                              idrbn);

                /* Create input molecfit spec format */
                cpl_table *spec_telluriccorr =
                    mf_spectrum_create(mf_config->parameters, mdata);

                lblrtm_results =
                    mf_calctrans_lblrtm(mf_config, spec_telluriccorr,
                                        molecules_table, wl_start, wl_end,
                                        atm_parameters_table,
                                        best_fit_parameters_table);
                cpl_table_delete(spec_telluriccorr);
                cpl_table_delete(mdata);
            }

            if (!(lblrtm_results)) {
                err = cpl_error_set_message(
                    cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                    "Unexpected error in the Molecfit call "
                    "mf_calctrans_lblrtm(...)");
            }
            else {
                cpl_size n_range = mf_config->parameters->internal.n_range;

                if (n_range != 1) {
                    err = cpl_error_set_message(
                        cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                        "Unexpected n_ranges in the "
                        "mf_calctrans_lblrtm(...) Molecfit execution : "
                        "n_ranges = %lld (Mandatory == 1)",
                        n_range);
                }
                else {
                    err = cpl_error_get_code();
                }
                if (!err) {
                    for (int i = 0; i < size; i++) {
                        mf_calctrans_convolution_results *convolution_results =
                            NULL;
                        int idx = cpl_array_get_cplsize(sel, i, NULL);
                        int idrbn = cpl_table_get_int(fibre_table,
                                                      MOO_FIBRES_TABLE_INDEXRBN,
                                                      idx, NULL);
                        const int spectro =
                            cpl_table_get_int(fibre_table,
                                              MOO_FIBRES_TABLE_SPECTRO, idx,
                                              NULL);
                        cpl_table_set_int(telluric_table,
                                          MOO_TELLURIC_TABLE_INDEXRBN, i,
                                          idrbn);
                        cpl_table_set_int(telluric_table,
                                          MOO_TELLURIC_TABLE_SPECTRO, i,
                                          spectro);
                        cpl_table_set_string(telluric_table,
                                             MOO_TELLURIC_TABLE_BAND, i, band);
                        cpl_table_set_int(telluric_table,
                                          MOO_TELLURIC_TABLE_TELLURIC, i,
                                          i + 1);

                        cpl_msg_info(__func__, "Do convolution for %d:%d",
                                     idrbn, cpl_error_get_code());

                        mdata = _molecfit_data_create(crpix1, crval1, cd1_1,
                                                      img, ierr, idrbn);
                        /* Create input molecfit spec format */
                        cpl_table *spec_telluriccorr =
                            mf_spectrum_create(mf_config->parameters, mdata);
                        cpl_msg_info(
                            cpl_func,
                            "Convolve input spectrum "
                            "(ext_orig[ATM_PARAMETERS/BEST_FIT_PARAMETERS]");
                        cpl_propertylist *header_kernel = NULL;
                        cpl_matrix *kernel = NULL;

                        header_kernel =
                            moo_kernel_get_header(mkernel, type, 1, idrbn);

                        kernel = moo_kernel_get_matrix(mkernel, type, 1, idrbn);
                        convolution_results =
                            mf_calctrans_convolution(mf_config->parameters,
                                                     lblrtm_results,
                                                     spec_telluriccorr_head,
                                                     spec_telluriccorr,
                                                     header_kernel, kernel,
                                                     wl_start, wl_end,
                                                     best_fit_parameters_table);
                        if (!convolution_results) {
                            err = cpl_error_set_message(
                                cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                                "Unexpected error in the Molecfit call "
                                "mf_calctrans_convolution(...)");
                        }
                        else {
                            cpl_table *telluric_data =
                                convolution_results->spec_telluriccorr_format;
                            for (int k = 0; k < nx; k++) {
                                double mt = cpl_table_get(telluric_data,
                                                          "mtrans", k, NULL);
                                cpl_image_set(im, k + 1, i + 1, mt);
                            }
#if MOO_DEBUG_MOLECFIT_CALCTRANS
                            {
                                char *testname =
                                    cpl_sprintf("TELLURICDATA_%s_%d.fits",
                                                rbn_single->extname, idrbn);
                                cpl_table_save(telluric_data, NULL, NULL,
                                               testname, CPL_IO_CREATE);
                                cpl_free(testname);
                            }
#endif
                            err = cpl_error_get_code();
                        }

                        cpl_propertylist_delete(header_kernel);
                        cpl_matrix_delete(kernel);
                        mf_calctrans_convolution_results_delete(
                            convolution_results);
                        cpl_table_delete(spec_telluriccorr);
                        cpl_table_delete(mdata);
                    }
                    mf_calctrans_lblrtm_results_delete(lblrtm_results);
                }
            }

            int nrow = cpl_table_get_nrow(result->telluric_table);
            cpl_table_insert(result->telluric_table, telluric_table, nrow);

            cpl_table_delete(telluric_table);
            cpl_array_delete(sel);
        }
    }

    moo_kernel_delete(mkernel);
    mf_configuration_delete(mf_config);
    moo_molectable_delete(mt_molecules);
    moo_molectable_delete(mt_atm);
    moo_molectable_delete(mt_bfit);
    moo_molectable_delete(mt_lblrtm_results);
    moo_rbn_delete(rbn);

    return result;
}
/**@}*/
