/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_PFITS_H
#define MOO_PFITS_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include "moo_detector.h"

#define MOO_PFITS_FLUXCAL                   "FLUXCAL"
#define MOO_PFITS_FLUXCAL_UNCALIBRATED      "UNCALIBRATED"
#define MOO_PFITS_FLUXCAL_ABSOLUTE          "ABSOLUTE"

#define MOO_PFITS_FLUXCAL_C                 "flux calibration"
#define MOO_PFITS_WAVELMIN                  "WAVELMIN"
#define MOO_PFITS_WAVELMIN_C                "Minimum of wavelength in [nm]"
#define MOO_PFITS_WAVELMAX                  "WAVELMAX"
#define MOO_PFITS_WAVELMAX_C                "Maximum of wavelength in [nm]"
#define MOO_PFITS_SPECBIN                   "SPEC_BIN"
#define MOO_PFITS_SPECBIN_C                 "Average spectral coordinate bin size in [nm]"

#define MOO_PFITS_RA                        "RA"
#define MOO_PFITS_DEC                       "DEC"

#define MOO_PFITS_CDELT1                    "CDELT1"
#define MOO_PFITS_CDELT2                    "CDELT2"
#define MOO_PFITS_CD1_1                     "CD1_1"
#define MOO_PFITS_CD1_2                     "CD1_2"
#define MOO_PFITS_CD2_1                     "CD2_1"
#define MOO_PFITS_CD2_2                     "CD2_2"
#define MOO_PFITS_CRPIX1                    "CRPIX1"
#define MOO_PFITS_CRPIX2                    "CRPIX2"
#define MOO_PFITS_CRVAL1                    "CRVAL1"
#define MOO_PFITS_CRVAL2                    "CRVAL2"
#define MOO_PFITS_CTYPE1                    "CTYPE1"
#define MOO_PFITS_CTYPE2                    "CTYPE2"
#define MOO_PFITS_CUNIT1                    "CUNIT1"
#define MOO_PFITS_CUNIT2                    "CUNIT2"
#define MOO_PFITS_EXTNAME                   "EXTNAME"
#define MOO_PFITS_EXPTIME                   "EXPTIME"
#define MOO_PFITS_EXPTIME_C                 "Exposure time"
#define MOO_PFITS_ARCFILE                   "ARCFILE"
#define MOO_PFITS_NAXIS                     "NAXIS"
#define MOO_PFITS_NAXIS1                    "NAXIS1"
#define MOO_PFITS_NAXIS2                    "NAXIS2"
#define MOO_PFITS_BUNIT                     "BUNIT"

#define MOO_PFITS_UTC                       "UTC"
#define MOO_PFITS_TMSTART                   "TM-START"

#define MOO_PFITS_DATE                      "DATE"
#define MOO_PFITS_DATEOBS                   "DATE-OBS"
#define MOO_PFITS_MJDOBS                    "MJD-OBS"
#define MOO_PFITS_DET_DIT                   "ESO DET SEQ DIT"
#define MOO_PFITS_DET_NDIT                  "ESO DET NDIT"
#define MOO_PFITS_DET_OUTPUTS               "ESO DET OUTPUTS"
#define MOO_PFITS_DET_LIVE                  "ESO DET LIVE"

#define MOO_PFITS_DET_OUTI_RON              "ESO DET OUT%d RON"
#define MOO_PFITS_DET_OUTI_GAIN             "ESO DET OUT%d GAIN"
#define MOO_PFITS_DET_OUTI_NX               "ESO DET OUT%d NX"
#define MOO_PFITS_DET_OUTI_NY               "ESO DET OUT%d NY"
#define MOO_PFITS_DET_OUTI_PRSCX            "ESO DET OUT%d PRSCX"
#define MOO_PFITS_DET_OUTI_OVSCX            "ESO DET OUT%d OVSCX"
#define MOO_PFITS_DET_OUTI_PRSCY            "ESO DET OUT%d PRSCY"
#define MOO_PFITS_DET_OUTI_OVSCY            "ESO DET OUT%d OVSCY"
#define MOO_PFITS_DET_OUTI_X                "ESO DET OUT%d X"
#define MOO_PFITS_DET_OUTI_Y                "ESO DET OUT%d Y"

#define MOO_PFITS_DET_CHIP_RON              "ESO DET CHIP RON"
#define MOO_PFITS_DET_CHIP_OUTI_GAIN        "ESO DET CHIP OUT%d GAIN"
#define MOO_PFITS_DET_CHIP_OUTI_NX          "ESO DET CHIP OUT%d NX"
#define MOO_PFITS_DET_CHIP_OUTI_NY          "ESO DET CHIP OUT%d NY"
#define MOO_PFITS_DET_CHIP_OUTI_PRSCX       "ESO DET CHIP OUT%d PRSCX"
#define MOO_PFITS_DET_CHIP_OUTI_OVSCX       "ESO DET CHIP OUT%d OVSCX"
#define MOO_PFITS_DET_CHIP_OUTI_PRSCY       "ESO DET CHIP OUT%d PRSCY"
#define MOO_PFITS_DET_CHIP_OUTI_OVSCY       "ESO DET CHIP OUT%d OVSCY"
#define MOO_PFITS_DET_CHIP_OUTI_X           "ESO DET CHIP OUT%d X"
#define MOO_PFITS_DET_CHIP_OUTI_Y           "ESO DET CHIP OUT%d Y"

#define MOO_PFITS_DET_CHIP_NX               "ESO DET CHIP NX"
#define MOO_PFITS_DET_CHIP_NY               "ESO DET CHIP NY"
#define MOO_PFITS_DET_CHIP_LIVE             "ESO DET CHIP LIVE"

#define MOO_PFITS_INS_SLIT_OFFSET           "ESO INS SLIT NO"
#define MOO_PFITS_INS_MODE                  "ESO INS TAS MODE"

#define MOO_PFITS_OBS_START                 "ESO OBS START"

#define MOO_PFITS_HDUCLASS                  "HDUCLASS"
#define MOO_PFITS_HDUDOC                    "HDUDOC"
#define MOO_PFITS_HDUVERS                   "HDUVERS"
#define MOO_PFITS_HDUCLAS1                  "HDUCLAS1"
#define MOO_PFITS_HDUCLAS2                  "HDUCLAS2"
#define MOO_PFITS_HDUCLAS3                  "HDUCLAS3"
#define MOO_PFITS_SCIDATA                   "SCIDATA"
#define MOO_PFITS_ERRDATA                   "ERRDATA"
#define MOO_PFITS_QUALDATA                  "QUALDATA"
#define MOO_PFITS_QUALMASK                  "QUALMASK"

#define MOO_PFITS_HDUCLASS_V                "ESO"
#define MOO_PFITS_HDUCLASS_C                "Identification of the data format"
#define MOO_PFITS_HDUDOC_V                  "DICD"
#define MOO_PFITS_HDUDOC_C                  "Document describing the data format"
#define MOO_PFITS_HDUVERS_V                 "DICD version 6"
#define MOO_PFITS_HDUVERS_C                 "Specific version of the document"
#define MOO_PFITS_HDUCLAS1_V                "IMAGE"
#define MOO_PFITS_HDUCLAS1_C                "Image data format"
#define MOO_PFITS_HDUCLAS2_DATA             "DATA"
#define MOO_PFITS_HDUCLAS2_DATA_C           "Identification as science extension"
#define MOO_PFITS_SCIDATA_C                 "Name of its science extension"
#define MOO_PFITS_ERRDATA_C                 "Name of its error extension"
#define MOO_PFITS_QUALDATA_C                "Name of its quality extension"
#define MOO_PFITS_HDUCLAS2_ERROR            "ERROR"
#define MOO_PFITS_HDUCLAS2_ERROR_C          "Identification as error extension"
#define MOO_PFITS_HDUCLAS3_ERROR_MSE        "RMSE"
#define MOO_PFITS_HDUCLAS3_ERROR_MSE_C      "Error type: root mean square error"
#define MOO_PFITS_HDUCLAS2_QUALITY          "QUALITY"
#define MOO_PFITS_HDUCLAS2_QUALITY_C        "Identification as quality extension"
#define MOO_PFITS_HDUCLAS3_QUALITY_FLAG32   "FLAG32BIT"
#define MOO_PFITS_HDUCLAS3_QUALITY_FLAG32_C "Data quality type"
#define MOO_PFITS_QUALMASK_C                "Mask number to identify bad pixels"

#define MOO_PFITS_TEL_AIRM_START            "ESO TEL AIRM START"
#define MOO_PFITS_TEL_AIRM_START_C          "Airmass at start time"
#define MOO_PFITS_TEL_AIRM_END              "ESO TEL AIRM END"
#define MOO_PFITS_TEL_AIRM_END_C            "Airmass at end time"

#define MOO_PFITS_PRO_NCOADD                "ESO PRO NCOADD"
#define MOO_PFITS_PRO_NCOADD_C              "Number of coadded frames"

#define MOO_PFITS_PRO_WAVESOL_MODEL         "ESO PRO WAVESOL MODEL"
#define MOO_PFITS_PRO_WAVESOL_MODEL_C       "Polynomial model of wave solution"
#define MOO_PFITS_PRO_WAVESOL_DEGX          "ESO PRO WAVESOL DEGX"
#define MOO_PFITS_PRO_WAVESOL_DEGX_C        "Polynomial X degree"
#define MOO_PFITS_PRO_WAVESOL_DEGY          "ESO PRO WAVESOL DEGY"
#define MOO_PFITS_PRO_WAVESOL_DEGY_C        "Polynomial Y degree"
#define MOO_PFITS_PRO_WAVEMAP_GUESS         "ESO PRO WAVEMAP GUESS"
#define MOO_PFITS_PRO_WAVEMAP_GUESS_C       "Model parameters from WAVEMAP_GUESS"

#define MOO_PFITS_PRO_REC1_ID               "ESO PRO REC1 ID"

typedef struct
{
    float ron;
    float gain;
    int nx;
    int ny;
    int x;
    int y;
    int dx;
    int dy;
    int prscx;
    int prscy;
    int ovscx;
    int ovscy;

} moo_output;

typedef struct
{
    moo_output *outputs;
    /* naxis */
    int nb;
    int minx;
    int miny;
} moo_outputs;

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

cpl_error_code moo_pfits_set_fluxcal(cpl_propertylist *plist, const char *val);
cpl_error_code moo_pfits_set_wlmin(cpl_propertylist *plist, double wlmin);
cpl_error_code moo_pfits_set_wlmax(cpl_propertylist *plist, double wlmax);
cpl_error_code moo_pfits_set_spec_bin(cpl_propertylist *plist, double specbin);
const char *moo_pfits_get_arcfile(const cpl_propertylist *plist);
const char *moo_pfits_get_date(const cpl_propertylist *plist);
const char *moo_pfits_get_dateobs(const cpl_propertylist *plist);
double moo_pfits_get_mjdobs(const cpl_propertylist *plist);
double moo_pfits_get_dit(const cpl_propertylist *plist);
int moo_pfits_get_ndit(const cpl_propertylist *plist);
int moo_pfits_get_live(const cpl_propertylist *plist);

int moo_pfits_get_det_outputs(const cpl_propertylist *plist);
float moo_pfits_get_det_outi_ron(const cpl_propertylist *plist, int i);
float moo_pfits_get_det_outi_gain(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_nx(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_ny(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_x(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_y(const cpl_propertylist *plist, int i);
cpl_error_code moo_pfits_set_det_outi_x(cpl_propertylist *plist, int i, int v);
cpl_error_code moo_pfits_set_det_outi_y(cpl_propertylist *plist, int i, int v);
int moo_pfits_get_det_outi_prscx(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_prscy(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_ovscx(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_outi_ovscy(const cpl_propertylist *plist, int i);

float moo_pfits_get_det_chip_ron(const cpl_propertylist *plist);
float moo_pfits_get_det_chip_outi_gain(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_nx(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_ny(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_x(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_y(const cpl_propertylist *plist, int i);
cpl_error_code
moo_pfits_set_det_chip_outi_x(cpl_propertylist *plist, int i, int v);
cpl_error_code
moo_pfits_set_det_chip_outi_y(cpl_propertylist *plist, int i, int v);
int moo_pfits_get_det_chip_outi_prscx(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_prscy(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_ovscx(const cpl_propertylist *plist, int i);
int moo_pfits_get_det_chip_outi_ovscy(const cpl_propertylist *plist, int i);

int moo_pfits_get_det_chip_nx(const cpl_propertylist *plist);
int moo_pfits_get_det_chip_ny(const cpl_propertylist *plist);
int moo_pfits_get_det_chip_live(const cpl_propertylist *plist);

int moo_pfits_get_naxis(const cpl_propertylist *plist);
int moo_pfits_get_naxis1(const cpl_propertylist *plist);
int moo_pfits_get_naxis2(const cpl_propertylist *plist);

double moo_pfits_get_exptime(const cpl_propertylist *plist);
cpl_error_code moo_pfits_update_exptime(cpl_propertylist *plist, double v);
cpl_error_code moo_pfits_update_ra(cpl_propertylist *plist, double v);
cpl_error_code moo_pfits_update_dec(cpl_propertylist *plist, double v);

int moo_pfits_get_slit_offset(const cpl_propertylist *plist);
moo_mode_type moo_pfits_get_mode(const cpl_propertylist *plist);
double moo_pfits_get_crval1(const cpl_propertylist *plist);
double moo_pfits_get_crpix2(const cpl_propertylist *plist);
double moo_pfits_get_crpix1(const cpl_propertylist *plist);
double moo_pfits_get_cdelt2(const cpl_propertylist *plist);
double moo_pfits_get_cdelt1(const cpl_propertylist *plist);
double moo_pfits_get_cd1_1(const cpl_propertylist *plist);
double moo_pfits_get_cd2_2(const cpl_propertylist *plist);

const char *moo_pfits_get_extname(const cpl_propertylist *plist);

cpl_error_code moo_pfits_append_hduclass(cpl_propertylist *plist);
cpl_error_code moo_pfits_append_hduclass_data(cpl_propertylist *plist,
                                              moo_detector_type type,
                                              int ntas);
cpl_error_code
moo_pfits_append_hduclass_error(cpl_propertylist *plist,
                                moo_detector_type type,
                                int ntas,
                                const cpl_propertylist *sci_header);
cpl_error_code
moo_pfits_append_hduclass_quality(cpl_propertylist *plist,
                                  moo_detector_type type,
                                  int ntas,
                                  const cpl_propertylist *sci_header,
                                  int mask);

double moo_pfits_get_tel_airm_start(const cpl_propertylist *plist);
double moo_pfits_get_tel_airm_end(const cpl_propertylist *plist);

cpl_error_code moo_pfits_update_ncoadd(cpl_propertylist *plist, int nb);
int moo_pfits_get_pro_ncoadd(const cpl_propertylist *plist);

const char *moo_pfits_get_pro_wavesol_model(const cpl_propertylist *plist);
const char *moo_pfits_get_pro_wavesol_degx(const cpl_propertylist *plist);
const char *moo_pfits_get_pro_rec1_id(const cpl_propertylist *plist);
int moo_pfits_get_pro_wavesol_degy(const cpl_propertylist *plist);

moo_outputs *moo_outputs_new(int nb);
void moo_outputs_delete(moo_outputs *res);
moo_outputs *moo_outputs_load(cpl_propertylist *header, moo_detector_type type);
cpl_error_code moo_outputs_get_det_size(moo_outputs *self, int *nx, int *ny);
cpl_image *moo_outputs_create_det(moo_outputs *self,
                                  cpl_image *raw,
                                  int nx,
                                  int ny,
                                  moo_detector_type type);

const char *moo_pfits_get_obs_start(const cpl_propertylist *plist);
#endif
