/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_dfs.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_params.h"
#include "moo_badpix.h"
#include "moo_single.h"
#include "moo_det.h"
#include "moo_detlist.h"
#include "moo_fits.h"
#include "moo_detector.h"
#include "moo_drl.h"
#include "moo_prepare.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static cpl_image *
_moo_prepare_get_qual_extension(const char *filename,
                                const char *extname,
                                moo_outputs *outputs,
                                cpl_image *data,
                                moo_detector_type type)
{
    cpl_image *rqual = NULL;
    cpl_image *qual = NULL;

    if (data != NULL) {
        int nx = cpl_image_get_size_x(data);
        int ny = cpl_image_get_size_y(data);

        if (filename != NULL) {
            rqual = moo_fits_load_extension_image(filename, "QUAL", extname,
                                                  CPL_TYPE_INT);
            int qnx = cpl_image_get_size_x(rqual);
            int qny = cpl_image_get_size_y(rqual);

            if (qnx == nx && qny == ny) {
                qual = cpl_image_duplicate(rqual);
            }
            else {
                qual = moo_outputs_create_det(outputs, rqual, nx, ny, type);
            }
            cpl_image_delete(rqual);
        }
        if (qual == NULL) {
            qual = cpl_image_new(nx, ny, CPL_TYPE_INT);
        }
    }
    return qual;
}

static cpl_error_code
_moo_prepare_image_output(cpl_image *idata, moo_output out)
{
    cpl_ensure_code(idata != NULL, CPL_ERROR_NULL_INPUT);
    int x = out.dx - 1;
    int y = out.dy - 1;
    int nx = out.nx;
    int ny = out.ny;
    double gain = out.gain;
    /* ADU to e- */
    int size_x = cpl_image_get_size_x(idata);

    double *data = cpl_image_get_data_double(idata);

    for (int j = y; j < (y + ny); j++) {
        for (int i = x; i < (x + nx); i++) {
            data[i + j * size_x] = MOO_TO_ELECTRON(data[i + j * size_x], gain);
        }
    }

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_prepare_error_output(cpl_image *ierr,
                          cpl_image *idata,
                          cpl_image *ibias,
                          moo_output out)
{
    cpl_ensure_code(ierr != NULL, CPL_ERROR_NULL_INPUT);
    int x = out.dx - 1;
    int y = out.dy - 1;
    int nx = out.nx;
    int ny = out.ny;
    double ron = out.ron;
    double *err = cpl_image_get_data_double(ierr);
    int size_x = cpl_image_get_size_x(ierr);
    const double *data = cpl_image_get_data_double_const(idata);

    if (ibias != NULL) {
        const double *biasdata = cpl_image_get_data_double_const(ibias);
        for (int j = y; j < (y + ny); j++) {
            for (int i = x; i < (x + nx); i++) {
                int idx = i + j * size_x;
                err[idx] = sqrt(ron * ron + fabs(data[idx] - biasdata[idx]));
            }
        }
    }
    else {
        for (int j = y; j < (y + ny); j++) {
            for (int i = x; i < (x + nx); i++) {
                int idx = i + j * size_x;
                err[idx] = ron;
            }
        }
    }

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_prepare_single_adu(moo_single *single, const char *const qual_filename_rp)
{
    cpl_image *data = NULL;
    cpl_image *rdata = NULL;
    cpl_image *err = NULL;
    cpl_image *qual_rp = NULL;
    moo_outputs *outputs = NULL;

    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    const char *extname = single->extname;
    int num = single->ntas;
    /* header */
    cpl_propertylist *header = moo_single_get_header(single);

    cpl_propertylist_update_string(header, MOO_PFITS_BUNIT,
                                   MOO_PREPARE_ADU_BUNIT);
    moo_pfits_append_hduclass_data(header, single->type, num);
    outputs = moo_outputs_load(header, single->type);

    int nx = moo_pfits_get_det_chip_nx(header);
    int ny = moo_pfits_get_det_chip_ny(header);
    if (single->type != MOO_TYPE_RI) {
        moo_outputs_get_det_size(outputs, &nx, &ny);
    }
    moo_try_check(rdata =
                      moo_fits_load_extension_image(single->filename, NULL,
                                                    extname, CPL_TYPE_DOUBLE),
                  " ");
    data = moo_outputs_create_det(outputs, rdata, nx, ny, single->type);
    cpl_msg_info("moo_prepare", "  Creating %s", extname);
    cpl_msg_info("moo_prepare", "  Creating ERR_%s", extname);

    err = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    single->image = hdrl_image_create(data, err);
    cpl_msg_info("moo_prepare", "  Creating QUAL_%s", extname);
    moo_try_check(qual_rp = _moo_prepare_get_qual_extension(qual_filename_rp,
                                                            extname, outputs,
                                                            data, single->type),
                  " ");
    single->qual = qual_rp;

moo_try_cleanup:
    cpl_image_delete(rdata);
    cpl_image_delete(data);
    cpl_image_delete(err);
    moo_outputs_delete(outputs);
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_prepare_single(moo_single *single,
                    const char *const qual_filename_nl,
                    const char *const masterbias_filename)
{
    cpl_image *data = NULL;
    cpl_image *err = NULL;
    cpl_image *biasdata = NULL;
    cpl_image *qual_nl = NULL;
    moo_outputs *outputs = NULL;

    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    const char *extname = single->extname;
    /* header */
    cpl_propertylist *header = moo_single_get_header(single);
    if (!cpl_propertylist_has(header, MOO_PFITS_CDELT1)) {
        double cd1_1 = moo_pfits_get_cd1_1(header);
        cpl_propertylist_append_double(header, MOO_PFITS_CDELT1, cd1_1);
    }
    if (!cpl_propertylist_has(header, MOO_PFITS_CDELT2)) {
        double cd2_2 = moo_pfits_get_cd2_2(header);
        cpl_propertylist_append_double(header, MOO_PFITS_CDELT2, cd2_2);
    }
    cpl_propertylist_update_string(header, MOO_PFITS_BUNIT,
                                   MOO_PREPARE_COUNT_BUNIT);

    outputs = moo_outputs_load(header, single->type);

    hdrl_image *image = moo_single_get_image(single);
    data = hdrl_image_get_image(image);
    err = hdrl_image_get_error(image);

    cpl_msg_info("moo_prepare", "  Creating %s", extname);

    for (int i = 0; i < outputs->nb; i++) {
        if (single->type == MOO_TYPE_RI) {
            moo_pfits_set_det_outi_x(header, i + 1, outputs->outputs[i].dx);
            moo_pfits_set_det_outi_y(header, i + 1, outputs->outputs[i].dy);
        }
        else {
            moo_pfits_set_det_chip_outi_x(header, i + 1,
                                          outputs->outputs[i].dx);
            moo_pfits_set_det_chip_outi_y(header, i + 1,
                                          outputs->outputs[i].dy);
        }
        _moo_prepare_image_output(data, outputs->outputs[i]);
    }

    cpl_msg_info("moo_prepare", "  Creating ERR_%s", extname);

    if (single->type == MOO_TYPE_RI) {
        if (masterbias_filename != NULL) {
            biasdata = moo_fits_load_extension_image(masterbias_filename, NULL,
                                                     extname, CPL_TYPE_DOUBLE);
        }
    }
    else {
        int nx = cpl_image_get_size_x(data);
        int ny = cpl_image_get_size_y(data);
        biasdata = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    }

    for (int i = 0; i < outputs->nb; i++) {
        _moo_prepare_error_output(err, data, biasdata, outputs->outputs[i]);
    }

    cpl_msg_info("moo_prepare", "  Creating QUAL_%s", extname);
    moo_try_check(qual_nl = _moo_prepare_get_qual_extension(qual_filename_nl,
                                                            extname, outputs,
                                                            data, single->type),
                  " ");

    cpl_image_or(single->qual, single->qual, qual_nl);

    if (single->type == MOO_TYPE_RI && masterbias_filename != NULL) {
        cpl_image *bqual =
            moo_fits_load_extension_image(masterbias_filename, "QUAL", extname,
                                          CPL_TYPE_INT);
        cpl_image_or(single->qual, single->qual, bqual);
        cpl_image_delete(bqual);
    }

moo_try_cleanup:
    if (biasdata != NULL) {
        cpl_image_delete(biasdata);
    }
    cpl_image_delete(qual_nl);
    moo_outputs_delete(outputs);
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    This function transforms RAW frames in DET frames attaching
the default bad pixel map and an error image
  @param    rawframe the RAW cpl frame to prepare
  @param    badpixmask_rp the filename of the bad pixel mask RP or NULL
  @param    badpixmask_nl the filename of the bad pixel mask NL or NULL
  @param    masterbias the MASTER_BIAS frame or NULL if not
  @param    cube_frame the LINEARITY_COEFF_CUBE frame or NULL if no correction
  @param    params the prepare parameters
  @return   the _DET_ corresponding to the given raw files

  This function is placed at the beginning of every recipe, it transforms an input
 * RAW frame in a DET frame attaching to it additional extensions: a bad pixel map,
 * and an error image for each detector extension. The bad pixel map is the default
 * one or a zero image if the default map does not exist. The error image is
 * built summing the Read Out Noise and the shot noise.
 *
  * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if raw frame is not from GROUP RAW
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_prepare(const cpl_frame *rawframe,
            const char *const badpixmask_rp,
            const char *const badpixmask_nl,
            const cpl_frame *masterbias,
            const cpl_frame *cube_frame,
            moo_prepare_params *params)
{
    moo_det *dres = NULL;

    const char *masterbias_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(rawframe != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(rawframe);
    const char *tag = cpl_frame_get_tag(rawframe);
    cpl_frame_group group = cpl_frame_get_group(rawframe);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(group == CPL_FRAME_GROUP_RAW, CPL_ERROR_ILLEGAL_INPUT, NULL);
    dres = moo_prepare_adu(rawframe, badpixmask_rp, params);
    moo_try_check(moo_correct_detlin(dres, cube_frame), " ");
#if DEBUG_PREPARE_ADU_LIN
    {
        moo_det_save(dres, "ADU_LIN.fits");
    }
#endif
    cpl_msg_info(__func__, "Preparing %s", filename);

    cpl_msg_indent_more();
    if (badpixmask_nl != NULL) {
        cpl_msg_info(__func__, "Using BP_MAP_NL %s", badpixmask_nl);
    }

    if (masterbias != NULL) {
        masterbias_filename = cpl_frame_get_filename(masterbias);
        cpl_msg_info(__func__, "Using MASTER_BIAS %s", masterbias_filename);
    }
    for (int i = 0; i < 3; i++) {
        for (int j = 1; j <= 2; j++) {
            moo_single *single = moo_det_get_single(dres, i, j);
            if (single != NULL) {
                int ignored = params->ignore_detector[i + (j - 1) * 3];
                if (ignored == 0) {
                    moo_try_check(_moo_prepare_single(single, badpixmask_nl,
                                                      masterbias_filename),
                                  "Error in %s",
                                  moo_detector_get_extname(i, j));
                }
            }
        }
    }
    cpl_msg_indent_less();
#if DEBUG_PREPARE_ADU_LIN
    {
        moo_det_save(dres, "DET.fits");
    }
#endif
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_det_delete(dres);
        dres = NULL;
        cpl_msg_error(__func__, "Error in prepare");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_errorstate_set(prestate);
    }
    return dres;
}

static cpl_error_code
_moo_create_primary_header(cpl_propertylist *header, const char *filename)
{
    cpl_error_code status = CPL_ERROR_NONE;

    moo_try_check(status = moo_qc_set_is_linearcor(header, CPL_FALSE), " ");
    moo_try_check(status = moo_qc_set_is_p2pcor(header, CPL_FALSE), " ");
    moo_qc_set_frame_raw1(header, filename);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    This function transforms RAW frames in DET ADU frames attaching
the default bad pixel
  @param    rawframe the RAW cpl frame to prepare
  @param    badpixmask_rp the filename of the bad pixel mask RP or NULL
  @param    params the prepare parameters
  @return   the _DET_ corresponding to the given raw files

  This function is placed at the beginning of every recipe, it transforms an input
 * RAW frame in a DET frame attaching to it additional extensions: a bad pixel map,
 * and an error image for each detector extension. The bad pixel map is the default
 * one or a zero image if the default map does not exist. The error image is
 * built summing the Read Out Noise and the shot noise.
 *
  * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if raw frame is not from GROUP RAW
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_prepare_adu(const cpl_frame *rawframe,
                const char *const badpixmask_rp,
                moo_prepare_params *params)
{
    moo_det *dres = NULL;
    int live = 0;
    int clive = 0;
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(rawframe != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(rawframe);
    const char *tag = cpl_frame_get_tag(rawframe);
    cpl_frame_group group = cpl_frame_get_group(rawframe);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(group == CPL_FRAME_GROUP_RAW, CPL_ERROR_ILLEGAL_INPUT, NULL);

    dres = moo_det_create(rawframe);
    _moo_create_primary_header(dres->primary_header, filename);

    cpl_msg_info(__func__, "Preparing in ADU %s", filename);
    cpl_msg_indent_more();
    if (badpixmask_rp != NULL) {
        cpl_msg_info(__func__, "Using BP_MAP_RP %s", badpixmask_rp);
    }

    for (int i = 0; i < 3; i++) {
        for (int j = 1; j <= 2; j++) {
            moo_single *single = moo_det_get_single(dres, i, j);
            int ignored = params->ignore_detector[i + (j - 1) * 3];
            if (single != NULL) {
                cpl_propertylist *header = moo_single_get_header(single);
                moo_try_check(live = moo_pfits_get_live(header), " ");
                ignored = ignored || !live;
                moo_try_check(clive = moo_pfits_get_det_chip_live(header), " ");
                ignored = ignored || !clive;
                if (ignored == 0) {
                    moo_try_check(_moo_prepare_single_adu(single,
                                                          badpixmask_rp),
                                  "Error in %s",
                                  moo_detector_get_extname(i, j));
                }
                else {
                    moo_det_set_single(dres, i, j, NULL);
                }
            }
        }
    }

    cpl_msg_indent_less();

#if DEBUG_PREPARE_ADU

    {
        moo_det_save(dres, "ADU.fits");
    }
#endif

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_det_delete(dres);
        dres = NULL;
        cpl_msg_error(__func__, "Error in prepare");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_errorstate_set(prestate);
    }
    return dres;
}
/**@}*/
