#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_dfs.h"
#include "moo_det.h"
#include "moo_loc.h"
#include "moo_map.h"
#include "moo_params.h"
#include "moo_utils.h"
#include "moo_badpix.h"
#include "moo_products.h"
#include "moo_region.h"
#include "moo_resp.h"
#include "moo_telluric.h"
#include "moo_target_table.h"
#include "irplib_ksigma_clip.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_products  Moons products management
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    create a moo_product object for a recipe
  @param    framelist the recipe input frames list
  @param    parlist the recipe input parameters list
  @param    recid the recipe id
  @param    pipeline_id the pipeline id
  @return a new moo_product object
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
moo_products *
moo_products_new(cpl_frameset *framelist,
                 const cpl_parameterlist *parlist,
                 const char *recid,
                 const char *pipeline_id)
{
    cpl_ensure(framelist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(recid != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pipeline_id != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_products *res = cpl_calloc(1, sizeof(moo_products));
    res->framelist = framelist;
    res->temporarylist = cpl_frameset_new();
    res->parlist = parlist;
    res->recid = recid;
    res->pipeline_id = pipeline_id;
    res->params = moo_params_new(MOO_PRODUCTS_PIPEID, recid);
    res->keep_temp = moo_params_get_keep_temp(res->params, res->parlist);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    get the moo_params object
  @param    self the moo_product object
  @return the object for managing the recipe parameters
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
const moo_params *
moo_products_get_params(const moo_products *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    return self->params;
}

/*----------------------------------------------------------------------------*/
/**
  @brief create a frame as a recipe product
  @param self the moo_product object
  @param frame the product frame
  @return the relevant error_code
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
static cpl_frame *
_moo_products_create_frame(const moo_products *self,
                           cpl_propertylist *header,
                           cpl_frame_level level,
                           const char *tag,
                           const char *filename,
                           const cpl_frame *inherit_frame)
{
    cpl_frame *product_frame = cpl_frame_new();

    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, level);
    cpl_frame_set_tag(product_frame, tag);

    if (filename != NULL) {
        cpl_frame_set_filename(product_frame, filename);
    }
    else {
        char *name = cpl_sprintf("%s.fits", tag);
        cpl_frame_set_filename(product_frame, name);
        cpl_free(name);
    }
    cpl_error_code status =
        cpl_dfs_setup_product_header(header, product_frame, self->framelist,
                                     self->parlist, self->recid,
                                     self->pipeline_id, DICTIONARY_ID,
                                     inherit_frame);

    if (status != CPL_ERROR_NONE) {
        cpl_frame_delete(product_frame);
        product_frame = NULL;
    }
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief add a frame to the recipe products
  @param self the moo_product object
  @param frame the product frame
  @return the relevant error_code
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_products_add_frame(const moo_products *self, cpl_frame *frame)
{
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(self != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(frame != NULL, CPL_ERROR_ILLEGAL_INPUT);

    cpl_frame_level level = cpl_frame_get_level(frame);

    if (level == CPL_FRAME_LEVEL_TEMPORARY) {
        status = cpl_frameset_insert(self->temporarylist, frame);
    }
    else if (level == CPL_FRAME_LEVEL_INTERMEDIATE && self->keep_temp == 0) {
        status = cpl_frameset_insert(self->temporarylist, frame);
    }
    else {
        status = cpl_frameset_insert(self->framelist, frame);
    }
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a DET object
  @param    self the moo_product object
  @param    det the DET object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add(moo_products *self,
                 moo_det *det,
                 cpl_frame_level level,
                 const char *tag,
                 const char *filename,
                 const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(det->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, det->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_det_save(det, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(product_frame);
    }
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    This function creates a product from a LOC structure
  @return   the product frame

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_loc(moo_products *self,
                     moo_loc *loc,
                     int keep_points,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = cpl_frame_new();

    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, level);
    cpl_frame_set_tag(product_frame, tag);
    cpl_frame_set_filename(product_frame, filename);

    cpl_dfs_setup_product_header(loc->primary_header, product_frame,
                                 self->framelist, self->parlist, self->recid,
                                 self->pipeline_id, DICTIONARY_ID,
                                 inherit_frame);

    moo_loc_save(loc, filename, keep_points);

    if (level == CPL_FRAME_LEVEL_TEMPORARY) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else if (level == CPL_FRAME_LEVEL_INTERMEDIATE && self->keep_temp == 0) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else {
        cpl_frameset_insert(self->framelist, product_frame);
    }

    return product_frame;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a EXT object
  @param    self the moo_product object
  @param    ext the EXT object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_ext(moo_products *self,
                     moo_ext *ext,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ext->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, ext->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_ext_save(ext, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");
moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a RBN object
  @param    self the moo_product object
  @param    rbn the RBN object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_rbn(moo_products *self,
                     moo_rbn *rbn,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rbn->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, rbn->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_rbn_save(rbn, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a SCI object
  @param    self the moo_product object
  @param    sci the SCI object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_sci(moo_products *self,
                     moo_sci *sci,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sci != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sci->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, sci->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_sci_save(sci, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a RESP object
  @param    self the moo_product object
  @param    resp the RESP object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_resp(moo_products *self,
                      moo_resp *resp,
                      cpl_frame_level level,
                      const char *tag,
                      const char *filename,
                      const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(resp != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(resp->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, resp->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_resp_save(resp, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a TELLURIC object
  @param    self the moo_product object
  @param    tell the TELLURic object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_telluric(moo_products *self,
                          moo_telluric *tell,
                          cpl_frame_level level,
                          const char *tag,
                          const char *filename,
                          const cpl_frame *inherit_frame)
{
    cpl_frame *product_frame = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tell != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tell->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, tell->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");

    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_telluric_save(tell, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a EXT object
  @param    self the moo_product object
  @param    map the MAP object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @param    rbn the RBN to update the line table
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_map(moo_products *self,
                     moo_map *map,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame,
                     moo_rbn *rbn)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(map != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(map->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, map->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_map_update_linetable(map, rbn);
    moo_try_check(moo_map_save(map, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a MOLECTABLE object
  @param    self the moo_product object
  @param    mtable the MAP object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_molectable(moo_products *self,
                            moo_molectable *mtable,
                            cpl_frame_level level,
                            const char *tag,
                            const char *filename,
                            const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(mtable != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(mtable->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, mtable->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_molectable_save(mtable, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a BPM object
  @param    self the moo_product object
  @param    bpm the BPM object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_bpm(moo_products *self,
                     moo_bpm *bpm,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(bpm != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(bpm->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, bpm->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_bpm_save(bpm, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a SATURATE MAP object
  @param    self the moo_product object
  @param    saturate the SATURATE MAP object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_saturate_map(moo_products *self,
                              moo_saturate_map *saturate,
                              cpl_frame_level level,
                              const char *tag,
                              const char *filename,
                              const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(saturate != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(saturate->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, saturate->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_saturate_map_save(saturate, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a CUBE object
  @param    self the moo_product object
  @param    cube the CUBE object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_cube(moo_products *self,
                      moo_cube *cube,
                      cpl_frame_level level,
                      const char *tag,
                      const char *filename,
                      const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cube->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, cube->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");
    moo_try_check(moo_cube_save(cube, pname), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a TARGET_TABLE object
  @param    self the moo_product object
  @param    ttable the TARGET_TABLE object product
  @param    level the level of product frame
  @param    tag the tag of product frame
  @param    filename the name of product frame or NULL for using tag
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_target_table(moo_products *self,
                              moo_target_table *ttable,
                              cpl_frame_level level,
                              const char *tag,
                              const char *filename,
                              const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ttable != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ttable->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;
    const char *pname = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, ttable->primary_header,
                                                 level, tag, filename,
                                                 inherit_frame),
                  " ");
    moo_try_check(pname = cpl_frame_get_filename(product_frame), " ");

    moo_try_check(moo_target_table_save(ttable, pname), " ");

    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}

cpl_frame *
moo_products_add_f2f(moo_products *self,
                     moo_f2f *f2f,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(f2f != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(f2f->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = cpl_frame_new();

    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, level);
    cpl_frame_set_tag(product_frame, tag);
    cpl_frame_set_filename(product_frame, filename);

    cpl_dfs_setup_product_header(f2f->primary_header, product_frame,
                                 self->framelist, self->parlist, self->recid,
                                 self->pipeline_id, DICTIONARY_ID,
                                 inherit_frame);

    moo_f2f_save(f2f, filename);

    if (level == CPL_FRAME_LEVEL_TEMPORARY) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else if (level == CPL_FRAME_LEVEL_INTERMEDIATE && self->keep_temp == 0) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else {
        cpl_frameset_insert(self->framelist, product_frame);
    }

    return product_frame;
}

cpl_frame *
moo_products_add_psf(moo_products *self,
                     moo_psf *psf,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(psf != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(psf->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = cpl_frame_new();

    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, level);
    cpl_frame_set_tag(product_frame, tag);
    cpl_frame_set_filename(product_frame, filename);

    cpl_dfs_setup_product_header(psf->primary_header, product_frame,
                                 self->framelist, self->parlist, self->recid,
                                 self->pipeline_id, DICTIONARY_ID,
                                 inherit_frame);

    moo_psf_save(psf, filename);

    if (level == CPL_FRAME_LEVEL_TEMPORARY) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else if (level == CPL_FRAME_LEVEL_INTERMEDIATE && self->keep_temp == 0) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else {
        cpl_frameset_insert(self->framelist, product_frame);
    }

    return product_frame;
}

cpl_frame *
moo_products_add_raw(moo_products *self,
                     moo_raw *raw,
                     cpl_frame_level level,
                     const char *tag,
                     const char *filename,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = cpl_frame_new();

    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, level);
    cpl_frame_set_tag(product_frame, tag);
    cpl_frame_set_filename(product_frame, filename);

    cpl_dfs_setup_product_header(raw->primary_header, product_frame,
                                 self->framelist, self->parlist, self->recid,
                                 self->pipeline_id, DICTIONARY_ID,
                                 inherit_frame);
    moo_raw_save(raw, filename);
    if (level == CPL_FRAME_LEVEL_TEMPORARY) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else if (level == CPL_FRAME_LEVEL_INTERMEDIATE && self->keep_temp == 0) {
        cpl_frameset_insert(self->temporarylist, product_frame);
    }
    else {
        cpl_frameset_insert(self->framelist, product_frame);
    }
    return product_frame;
}

void
moo_products_delete(moo_products *self)
{
    if (self != NULL) {
        int i;
        int size = cpl_frameset_get_size(self->temporarylist);

        for (i = 0; i < size; i++) {
            const cpl_frame *f =
                cpl_frameset_get_position_const(self->temporarylist, i);
            const char *name = cpl_frame_get_filename(f);
            remove(name);
        }
        cpl_frameset_delete(self->temporarylist);
        moo_params_delete(self->params);
        cpl_free(self);
    }
}

static double
moo_region_ron(hdrl_imagelist *biaslist,
               int llx,
               int lly,
               int urx,
               int ury,
               const char *method)
{
    double ron = 0.0, mean = 0.0, stdev = 0.0;
    double sks_low = 3;
    double sks_iter = 10;
    int nbias = hdrl_imagelist_get_size(biaslist);
    cpl_vector *region_ron = cpl_vector_new(nbias - 1);

    for (int i = 0; i < nbias - 1; i++) {
        const hdrl_image *h1 = hdrl_imagelist_get_const(biaslist, i);
        const hdrl_image *h2 = hdrl_imagelist_get_const(biaslist, i + 1);
        const cpl_image *i1 = hdrl_image_get_image_const(h1);
        const cpl_image *i2 = hdrl_image_get_image_const(h2);
        cpl_image *diff = cpl_image_subtract_create(i1, i2);
        if (strcmp(method, MOO_RON_ESTIMATION_METHOD_LOCAL) == 0) {
            stdev = moo_image_get_ron(diff, llx, lly, urx, ury, 100, 4, 0.1, 5);
        }
        else {
            cpl_error_code error;

            error = irplib_ksigma_clip(diff, llx, lly, urx, ury, sks_low,
                                       sks_iter, 1e-5, &mean, &stdev);

            cpl_ensure_code(!error, error);
        }
        ron = stdev / sqrt(2);
        cpl_vector_set(region_ron, i, ron);
        cpl_image_delete(diff);
    }
    ron = cpl_vector_get_mean(region_ron);
    cpl_vector_delete(region_ron);
    return ron;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    This function creates the master bias frame as a product and
 * essentially produces a standard output as well as it computes and writes
 * the different QC parameters.
  @param    det the MASTER_BIAS det
  @param    bias_list the BIAS det list
  @param    bias_params the masterbias parameter
  @param    products the product structure to save product file
  @return   the DET result

 *  _Flags considered as bad : BADPIX_COSMETIC | BADPIX_NON_LINEAR
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_masterbias(moo_det *det,
               moo_detlist *bias_list,
               moo_bias_params *bias_params,
               moo_products *products)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(det->primary_header != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bias_list != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bias_params != NULL, CPL_ERROR_NULL_INPUT);
    int i;

    int badpix_level = MOO_BADPIX_COSMETIC | MOO_BADPIX_NON_LINEAR;

    cpl_msg_indent_more();

    cpl_msg_info(__func__, "Use method %s for estimate RON",
                 bias_params->ron_estimation_method);

    for (i = 0; i < 2; i++) {
        moo_single *s = det->ri[i];
        if (s != NULL) {
            moo_detlist_load_single(bias_list, MOO_TYPE_RI, i + 1,
                                    badpix_level);
            hdrl_imagelist *ilist =
                moo_detlist_get_image(bias_list, MOO_TYPE_RI, i + 1);

            cpl_propertylist *header = moo_single_get_header(s);

            moo_outputs *outputs = moo_outputs_load(header, MOO_TYPE_RI);

            moo_single_load(s, badpix_level);
            for (int ri = 0; ri < outputs->nb; ri++) {
                int llx = outputs->outputs[ri].x;
                int lly = outputs->outputs[ri].y;
                int urx = outputs->outputs[ri].x + outputs->outputs[ri].nx - 1;
                int ury = outputs->outputs[ri].y + outputs->outputs[ri].ny - 1;

                hdrl_image *image =
                    hdrl_image_extract(s->image, llx, lly, urx, ury);
                hdrl_value mean = hdrl_image_get_mean(image);
                double stdev = hdrl_image_get_stdev(image);

                double mad, median;
                median = cpl_image_get_mad(hdrl_image_get_image(image), &mad);
                hdrl_image_delete(image);
                moo_qc_set_mbias_avg(s->header, ri + 1, mean.data);
                moo_qc_set_mbias_med(s->header, ri + 1, median);
                moo_qc_set_mbias_rms(s->header, ri + 1, stdev);
                moo_qc_set_mbias_mad(s->header, ri + 1, mad);

                double ron_raw =
                    moo_region_ron(ilist, llx, lly, urx, ury,
                                   bias_params->ron_estimation_method);
                moo_qc_set_ron_raw(s->header, ri + 1, ron_raw);
                cpl_msg_info(__func__, "RI_%d OUT%d RON RAW : %f e-(%f ADU)",
                             i + 1, ri + 1, ron_raw,
                             ron_raw * outputs->outputs[ri].gain);
                double ron_master = mad * CPL_MATH_STD_MAD;
                moo_qc_set_ron_master(s->header, ri + 1, ron_master);
                cpl_msg_info(__func__,
                             "RI_%d OUT%d RON MASTER : %f e- (%f ADU)", i + 1,
                             ri + 1, ron_master,
                             ron_master * outputs->outputs[ri].gain);
            }
            moo_outputs_delete(outputs);
            hdrl_imagelist_unwrap(ilist);
            moo_detlist_free_single(bias_list, MOO_TYPE_RI, i + 1);
        }
    }
    cpl_msg_indent_less();
    moo_products_add(products, det, CPL_FRAME_LEVEL_FINAL,
                     MOONS_TAG_MASTER_BIAS, NULL, NULL);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    This function creates the master dark frame as a product and
 * essentially produces a standard output as well as it computes and writes
 * the different QC parameters.
  @param    det the _DET_ MASTER_DARK
  @param    products the product structure to save product file
  @param    mode VIS (0) or NIR(1)
  @return   the relevant error code or CPL_ERROR_NONE

 *  _Flags considered as bad : BADPIX_COSMETIC | BADPIX_HOT
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_masterdark(moo_det *det, moo_products *products, int mode)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(det->primary_header != NULL, CPL_ERROR_NULL_INPUT);

    int badpix_level =
        MOO_BADPIX_COSMETIC | MOO_BADPIX_HOT | MOO_BADPIX_OUTSIDE_DATA_RANGE;
    int i;
    float exptime = moo_pfits_get_exptime(det->primary_header);

    for (i = 0; i < 2; i++) {
        moo_single *s = det->ri[i];
        if (s != NULL) {
            moo_single_load(s, badpix_level);
            cpl_msg_info(__func__, "Normalizing RI_%d by exptime %f", i + 1,
                         exptime);

            /* normalise all pixels */
            cpl_image *image = hdrl_image_get_image(s->image);
            cpl_mask *mask = cpl_image_set_bpm(image, NULL);
            hdrl_image_div_scalar(s->image, (hdrl_value){ exptime, 0. });
            cpl_image_set_bpm(image, mask);

            moo_qc_set_mdark_normalise_factor(s->header, exptime);
            moo_pfits_update_exptime(det->primary_header, 1.0);

            hdrl_value mean = hdrl_image_get_mean(s->image);
            hdrl_value median = hdrl_image_get_median(s->image);
            double stdev = hdrl_image_get_stdev(s->image);
            moo_qc_set_mdark_avg(s->header, mean.data);
            moo_qc_set_mdark_med(s->header, median.data);
            moo_qc_set_mdark_rms(s->header, stdev);
            float gain = moo_pfits_get_det_outi_gain(s->header, 1);
            double mdark_current = MOO_TO_ADU(mean.data, gain) * 3600.0;
            moo_qc_set_mdark_current(s->header, mdark_current);
            cpl_msg_info(__func__, "RI%d MDARK CURRENT %f", i + 1,
                         mdark_current);
        }
        s = det->yj[i];
        if (s != NULL) {
            moo_single_load(s, badpix_level);
            double dit = moo_pfits_get_dit(s->header);
            int ndit = moo_pfits_get_ndit(s->header);
            hdrl_value mean = hdrl_image_get_mean(s->image);
            hdrl_value median = hdrl_image_get_median(s->image);
            double stdev = hdrl_image_get_stdev(s->image);
            moo_qc_set_mdark_avg(s->header, mean.data);
            moo_qc_set_mdark_med(s->header, median.data);
            moo_qc_set_mdark_rms(s->header, stdev);
            float gain = moo_pfits_get_det_chip_outi_gain(s->header, 1);
            moo_qc_set_mdark_current(s->header, MOO_TO_ADU(mean.data, gain) /
                                                    (dit * ndit) * 3600.0);
        }
        s = det->h[i];
        if (s != NULL) {
            moo_single_load(s, badpix_level);
            double dit = moo_pfits_get_dit(s->header);
            int ndit = moo_pfits_get_ndit(s->header);
            hdrl_value mean = hdrl_image_get_mean(s->image);
            hdrl_value median = hdrl_image_get_median(s->image);
            double stdev = hdrl_image_get_stdev(s->image);
            moo_qc_set_mdark_avg(s->header, mean.data);
            moo_qc_set_mdark_med(s->header, median.data);
            moo_qc_set_mdark_rms(s->header, stdev);
            float gain = moo_pfits_get_det_chip_outi_gain(s->header, 1);
            moo_qc_set_mdark_current(s->header, MOO_TO_ADU(mean.data, gain) /
                                                    (dit * ndit) * 3600.0);
        }
    }

    if (mode == 0) {
        moo_products_add(products, det, CPL_FRAME_LEVEL_FINAL,
                         MOONS_TAG_MASTER_DARK_VIS, NULL, NULL);
    }
    else {
        moo_products_add(products, det, CPL_FRAME_LEVEL_FINAL,
                         MOONS_TAG_MASTER_DARK_NIR, NULL, NULL);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    create a product from a MOLECTABLE object
  @param    self the moo_product object
  @param    s1d the s1d object product
  @param    tag the tag of product frame
  @param    inherit_frame frame to inherit keywords
  @return the product frame
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if one of the inputs is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_frame *
moo_products_add_s1d(moo_products *self,
                     moo_s1d *s1d,
                     const char *tag,
                     const cpl_frame *inherit_frame)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(s1d != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(s1d->primary_header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frame *product_frame = NULL;

    moo_try_check(product_frame =
                      _moo_products_create_frame(self, s1d->primary_header,
                                                 CPL_FRAME_LEVEL_FINAL, tag,
                                                 s1d->filename, inherit_frame),
                  " ");
    moo_try_check(moo_s1d_save(s1d), " ");
    moo_try_check(moo_products_add_frame(self, product_frame), " ");

moo_try_cleanup:
    return product_frame;
}
/**@}*/
