/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_dfs.h"
#include "moo_pfits.h"
#include "moo_raw.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_raw  RAW format
 * @ingroup moo_data
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
moo_raw_ext *
moo_raw_ext_new(void)
{
    moo_raw_ext *raw_ext = cpl_calloc(1, sizeof(moo_raw_ext));
    return raw_ext;
}

void
moo_raw_ext_delete(moo_raw_ext *self)
{
    if (self != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        if (self->data != NULL) {
            cpl_image_delete(self->data);
        }
        cpl_free(self);
    }
}

moo_raw_ext *
moo_raw_ext_create(const char *filename, const char *extname, cpl_type type)
{
    moo_raw_ext *res = NULL;

    cpl_size qnum = cpl_fits_find_extension(filename, extname);

    if (qnum > 0) {
        res = moo_raw_ext_new();
        res->header = cpl_propertylist_load(filename, qnum);
        int naxis = moo_pfits_get_naxis(res->header);
        if (naxis > 0) {
            res->data = cpl_image_load(filename, type, 0, qnum);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_raw  
  @return   1 newly allocated moo_row or NULL in case of an error

  The returned object must be deallocated using moo_det_delete().  
 
 */
/*----------------------------------------------------------------------------*/
moo_raw *
moo_raw_new(void)
{
    moo_raw *raw = cpl_calloc(1, sizeof(moo_raw));
    return raw;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_raw
  @param    self    moo_raw to delete
  @return   void

  If the moo_det @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_raw_delete(moo_raw *self)
{
    if (self != NULL) {
        int i;
        for (i = 0; i < 2; i++) {
            moo_raw_ext_delete(self->ri[i]);
            moo_raw_ext_delete(self->yj[i]);
            moo_raw_ext_delete(self->h[i]);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        cpl_free(self);
    }
}

cpl_error_code
moo_raw_ext_avg(moo_raw_ext *a, moo_raw_ext *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    if (a->data != NULL && b->data != NULL) {
        cpl_image_add(a->data, b->data);
        cpl_image_divide_scalar(a->data, 2.0);
    }
    return cpl_error_get_code();
}

cpl_error_code
moo_raw_ext_save(moo_raw_ext *self, const char *filename, cpl_type type)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);

    if (self->data != NULL) {
        cpl_image_save(self->data, filename, type, self->header, CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(self->header, filename, CPL_IO_EXTEND);
    }
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    prepare moons raw frames  
  @param    rawframe the RAW cpl frame to prepare
  @return   the _RAW_ corresponding to the given raw file
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT if an input parameter is a NULL pointer
  - CPL_ERROR_ILLEGAL_INPUT if the rawframe group is not CPL_FRAME_GROUP_RAW 
 */
/*----------------------------------------------------------------------------*/
moo_raw *
moo_raw_create(const cpl_frame *rawframe)
{
    cpl_ensure(rawframe != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(rawframe);
    const char *tag = cpl_frame_get_tag(rawframe);
    cpl_frame_group group = cpl_frame_get_group(rawframe);

    // message par defaut en pile et return
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(group == CPL_FRAME_GROUP_RAW, CPL_ERROR_ILLEGAL_INPUT, NULL);


    moo_raw *res = moo_raw_new();

    res->primary_header = cpl_propertylist_load(filename, 0);

    res->ri[0] = moo_raw_ext_create(filename, "RI_1", CPL_TYPE_FLOAT);
    res->ri[1] = moo_raw_ext_create(filename, "RI_2", CPL_TYPE_FLOAT);

    res->yj[0] = moo_raw_ext_create(filename, "YJ_1", CPL_TYPE_DOUBLE);
    res->yj[1] = moo_raw_ext_create(filename, "YJ_2", CPL_TYPE_DOUBLE);

    res->h[0] = moo_raw_ext_create(filename, "H_1", CPL_TYPE_DOUBLE);
    res->h[1] = moo_raw_ext_create(filename, "H_2", CPL_TYPE_DOUBLE);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add to RAW frames (result in a)
  @param    a _RAW_ frame containing the result of (a+b)
  @param    b _RAW_ frame to add
  @return  relevant error code or CPL_ERROR_NONE
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT if an input parameter is a NULL pointer
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_raw_avg(moo_raw *a, moo_raw *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    int i;
    for (i = 0; i < 2; i++) {
        if (a->ri[i] != NULL && b->ri[i] != NULL) {
            moo_try_check(moo_raw_ext_avg(a->ri[i], b->ri[i]),
                          "Error adding extension RI %d", i);
        }
        if (a->yj[i] != NULL && b->yj[i] != NULL) {
            moo_try_check(moo_raw_ext_avg(a->yj[i], b->yj[i]),
                          "Error adding extension YJ %d", i);
        }
        if (a->h[i] != NULL && b->h[i] != NULL) {
            moo_try_check(moo_raw_ext_avg(a->h[i], b->h[i]),
                          "Error adding extension H %d", i);
        }
    }
moo_try_cleanup:
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_raw to a FITS file
  @param    self          moo_raw to write to disk or NULL
  @param    filename      Name of the file to write  
  @return   error code

  This function saves a moo_raw to a FITS file, using cfitsio.
  Only not NULL extensions are written.
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT if input parameter is a NULL pointer
  
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_raw_save(moo_raw *self, const char *filename)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->primary_header != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);

    int i;
    for (i = 0; i < 2; i++) {
        if (self->ri[i] != NULL) {
            moo_try_check(moo_raw_ext_save(self->ri[i], filename,
                                           CPL_TYPE_FLOAT),
                          "Can't save RI %d", i);
        }
        if (self->yj[i] != NULL) {
            moo_try_check(moo_raw_ext_save(self->yj[i], filename,
                                           CPL_TYPE_FLOAT),
                          "Can't save YJ %d", i);
        }
        if (self->h[i] != NULL) {
            moo_try_check(moo_raw_ext_save(self->h[i], filename,
                                           CPL_TYPE_FLOAT),
                          "Can't save H %d", i);
        }
    }
moo_try_cleanup:
    return cpl_error_get_code();
}
/**@}*/
