/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <assert.h>
#include "moo_scilist.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_scilist SCI_LIST format
 * @ingroup moo_data
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_scilist
  @return   1 newly allocated moo_scilist or NULL in case of an error

  The returned object must be deallocated using moo_scilist_delete().

 */
moo_scilist *
moo_scilist_new(void)
{
    return (moo_scilist *)cpl_calloc(1, sizeof(moo_scilist));
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup  moo_scilist
  @brief    Remove a SCI from a SCI list
  @param    self   The moo_scilist
  @param    pos      The list position (from 0 to number of SCI-1)
  @return   The pointer to the removed SCI or NULL in error case

  The specified SCI is not deallocated, it is simply removed from the
  list. The pointer to the SCI is returned to let the user decide to
  deallocate it or not.
  Eventually, the SCI will have to be deallocated with moo_sci_delete().

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    images in self
 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_scilist_unset(moo_scilist *self, cpl_size pos)
{
    moo_sci *out;
    cpl_size i;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pos >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(pos < self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    /* Get pointer to SCI to be removed */
    out = self->list[pos];

    /* Move the following SCI one position towards zero */
    for (i = pos + 1; i < self->size; i++) {
        self->list[i - 1] = self->list[i];
    }

    /* Decrement of the size */
    self->size--;

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Empty an moo_scilist and deallocate all its SCI
  @param    self  The SCI list or NULL
  @return   Nothing
  @see  moo_scilist_delete()
  @note If @em self is @c NULL nothing is done and no error is set.

  After the call the moo_sci list can be populated again. It must eventually
  be deallocated with a call to moo_scilist_delete().

 */
/*----------------------------------------------------------------------------*/
void
moo_scilist_empty(moo_scilist *self)
{
    if (self != NULL) {
        while (self->size > 0) { /* An iteration may unset more than 1 image */
            int i = self->size - 1;
            moo_sci *del = moo_scilist_unset(self, i);

            moo_sci_delete(del);

            /* If this image was inserted more than once into the list,
               the other insertions must be unset without a delete. */
            while (--i >= 0) {
                if (self->list[i] == del) {
                    /* This image was inserted more than once in the list */
                    (void)moo_scilist_unset(self, i);
                }
            }
        }
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Free memory used by a moo_scilist object, except the SCI
  @param    self    The SCI list or NULL
  @return   Nothing
  @see moo_scilist_empty()
  @note The caller must have pointers to all SCI in the list and is
        reponsible for their deallocation. If @em self is @c NULL nothing is
        done and no error is set.
 */
/*----------------------------------------------------------------------------*/
void
moo_scilist_unwrap(moo_scilist *self)
{
    if (self != NULL) {
        cpl_free(self->list);
        cpl_free(self);
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free all memory used by a moo_scilist object including the SCI
  @param    self    The SCI list or NULL
  @return   void
  @see      moo_scilist_empty(), moo_scilist_unwrap()
  If the moo_scilist @em self is @c NULL, nothing is done and no error is set.

 */
void
moo_scilist_delete(moo_scilist *self)
{
    if (self != NULL) {
        moo_scilist_empty(self);
        moo_scilist_unwrap(self);
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Get the number of SCI in the scilist
  @param    self    the list of SCI
  @return   The number of SCI or -1 on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_size
moo_scilist_get_size(const moo_scilist *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    assert(self->size >= 0);

    return self->size;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Insert a SCI into an moo_scilist
  @param    self    The moo_scilist
  @param    sci     The SCI to insert
  @param    pos     The list position (from 0 to number of SCI)
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  It is allowed to specify the position equal to the number of SCI in the
  list. This will increment the size of the scilist.

  No action occurs if a SCI is inserted more than once into the same
  position. It is allowed to insert the same SCI into two different
  positions in a list.

  The SCI is inserted at the position pos in the SCI list. If the SCI
  already there is only present in that one location in the list, then the
  SCI is deallocated.

  The added SCI is owned by the moo_scilist object, which deallocates it
  moo_scilist_delete is called. Other option is to use moo_scilist_unset to
  recover ownership of the SCI, in which case the moo_scilist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    SCI in scilist
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_scilist_set(moo_scilist *self, moo_sci *sci, cpl_size pos)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sci, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pos >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(pos <= self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Do nothing if the sci is already there */
    if (pos < self->size && sci == self->list[pos])
        return CPL_ERROR_NONE;

    if (pos == self->size) {
        self->size++;
        self->list =
            cpl_realloc(self->list, (size_t)self->size * sizeof(moo_sci *));
    }
    else {
        /* Check if the sci at the position to be overwritten
           is present in only one position */
        int i;

        for (i = 0; i < self->size; i++) {
            if (i != pos && self->list[i] == self->list[pos])
                break;
        }
        if (i == self->size) {
            /* The image at the position to be overwritten
               is present in only one position, so delete it */
            moo_sci_delete(self->list[pos]);
        }
    }

    self->list[pos] = sci;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Insert a SCI a the end of moo_scilist
  @param    self    The moo_scilist
  @param    sci     The SCI to insert
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
  @see  moo_scilist_set()

  No action occurs if a SCI is inserted more than once into the same
  position. It is allowed to insert the same SCI into two different
  positions in a list.

  The SCI is inserted at the end in the SCI list. If the SCI
  already there is only present in that one location in the list, then the
  SCI is deallocated.

  The added SCI is owned by the moo_scilist object, which deallocates it
  moo_scilist_delete is called. Other option is to use moo_scilist_unset to
  recover ownership of the SCI, in which case the moo_scilist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_scilist_push(moo_scilist *self, moo_sci *sci)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    moo_scilist_set(self, sci, self->size);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_scilist from the given SCI frameset
  @return   1 newly allocated moo_scilist or NULL in case of an error

  The returned object must be deallocated using moo_scilist_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
moo_scilist *
moo_scilist_create(cpl_frameset *frameset)
{
    cpl_ensure(frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int i;
    moo_scilist *res = moo_scilist_new();
    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame =
            cpl_frameset_get_position_const(frameset, i);
        moo_sci *sci = moo_sci_create(current_frame);
        moo_scilist_push(res, sci);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_scilist from the given SCI frameset ignoring SCI with 0 targets
  @return   1 newly allocated moo_scilist or NULL in case of an error

  The returned object must be deallocated using moo_scilist_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
moo_scilist *
moo_scilist_create_clean(cpl_frameset *frameset)
{
    cpl_ensure(frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int i;
    moo_scilist *res = moo_scilist_new();
    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame =
            cpl_frameset_get_position_const(frameset, i);
        moo_sci *sci = moo_sci_create(current_frame);
        moo_target_table *target_table = moo_sci_get_target_table(sci);
        if (target_table != NULL) {
            int nrow = cpl_table_get_nrow(target_table->table);
            if (nrow > 0) {
                moo_scilist_push(res, sci);
            }
            else {
                moo_sci_delete(sci);
            }
        }
        else {
            moo_sci_delete(sci);
        }
    }
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Get the SCI at the position i in the list
  @param    self    the list of SCI
  @param    i       position in the list
  @return   SCI element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_scilist_get(moo_scilist *self, int i)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(i >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int size = moo_scilist_get_size(self);
    cpl_ensure(i < size, CPL_ERROR_ILLEGAL_INPUT, NULL);

    return self->list[i];
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Get the SCI SINGLE at the position i in the list for the given detector
  @param    self    the list of SCI
  @param    i       position in the list
  @param    type       detector type
  @return   SCI_SINGLE element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
moo_sci_single *
moo_scilist_get_single(moo_scilist *self, int i, moo_detector_type type)
{
    moo_sci_single *result = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(i >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int size = moo_scilist_get_size(self);
    cpl_ensure(i < size, CPL_ERROR_ILLEGAL_INPUT, NULL);

    if (self->list[i] != NULL) {
        result = moo_sci_get_single(self->list[i], type);
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Load the type part for all SCI in the scilist
  @param    self    the list of SCI
  @param    type    the type of extension to load
  @param    level   Mask error level
  @return   The error code or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_scilist_load_single(const moo_scilist *self,
                        moo_detector_type type,
                        int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    int i;
    for (i = 0; i < self->size; i++) {
        moo_sci *sci = self->list[i];
        moo_sci_load_single(sci, type, level);
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Get the all the images of the type part in the scilist
  @param    self    the list of SCI
  @param    type     the type of extension to load
  @return   The  image list or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
hdrl_imagelist *
moo_scilist_get_image(const moo_scilist *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    hdrl_imagelist *res = NULL;
    int i;

    res = hdrl_imagelist_new();

    for (i = 0; i < self->size; i++) {
        moo_sci *sci = self->list[i];
        moo_sci_single *single = NULL;
        moo_try_check(single = moo_sci_get_single(sci, type), " ");

        if (single != NULL) {
            hdrl_image *img = moo_sci_single_get_image(single);
            hdrl_imagelist_set(res, img, i);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        hdrl_imagelist_delete(res);
        res = NULL;
    }
    return res;
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Get the all the QUAL of the type part in the scilist
  @param    self    the list of SCI
  @param    type     the type of extension to load
  @return   The  QUAL list or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_scilist_get_qual(const moo_scilist *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_imagelist *res = NULL;
    int i;

    res = cpl_imagelist_new();

    for (i = 0; i < self->size; i++) {
        moo_sci *sci = self->list[i];
        moo_sci_single *single = NULL;
        moo_try_check(single = moo_sci_get_single(sci, type), " ");

        if (single != NULL) {
            cpl_image *qual = moo_sci_single_get_qual(single);
            cpl_imagelist_set(res, qual, i);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_imagelist_delete(res);
        res = NULL;
    }
    return res;
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_scilist
  @brief    Get the all the SKY of the type part in the scilist
  @param    self    the list of SCI
  @param    type     the type of extension to load
  @return   The  SKY list or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_scilist_get_sky(const moo_scilist *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_imagelist *res = NULL;
    int i;

    res = cpl_imagelist_new();

    for (i = 0; i < self->size; i++) {
        moo_sci *sci = self->list[i];
        moo_sci_single *single = NULL;
        moo_try_check(single = moo_sci_get_single(sci, type), " ");

        if (single != NULL) {
            cpl_image *sky = moo_sci_single_get_sky(single);
            cpl_imagelist_set(res, sky, i);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_imagelist_delete(res);
        res = NULL;
    }
    return res;
    return res;
}
/**@}*/
