/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_single.h"
#include "moo_det.h"
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_single  DET_SINGLE format
 * @ingroup moo_det
 *
 * This module provides functions to create, use, and destroy a @em moo_single
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_single.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_single
  @param type the type of detector
  @param ntas number of spectrograph [1,2]
  @return   1 newly allocated moo_single or NULL in case of an error

  The returned object must be deallocated using moo_single_delete().

  _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
 */
/*----------------------------------------------------------------------------*/
moo_single *
moo_single_new(moo_detector_type type, int ntas)
{
    cpl_ensure(ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    moo_single *res = (moo_single *)cpl_calloc(1, sizeof(moo_single));
    res->badpix_mask = MOO_BADPIX_GOOD;
    res->type = type;
    res->ntas = ntas;
    const char *extname = moo_detector_get_extname(type, ntas);
    res->extname = extname;
    res->header = cpl_propertylist_new();
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_single from the given filename
  @param filename name of DET file
  @param type the type of detector
  @param ntas number of spectrograph [1,2]
  @return   1 newly allocated moo_single or NULL in case of an error

  If the filename doesnt have the extension name a NULL object is returned.
  The returned object must be deallocated using moo_single_delete().

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if ntas>2
  - CPL_ERROR_FILE_IO if filename not a fits file
 */
/*----------------------------------------------------------------------------*/
moo_single *
moo_single_create(const char *filename, moo_detector_type type, int ntas)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *extname = NULL;
    moo_single *single = NULL;
    cpl_size ext_num = -1;
    moo_try_check(extname = moo_detector_get_extname(type, ntas), " ");
    moo_try_check(ext_num = cpl_fits_find_extension(filename, extname), " ");

    if (ext_num > 0) {
        cpl_propertylist *header = cpl_propertylist_load(filename, ext_num);

        /* remove unsupported KW */
        cpl_propertylist_erase_regexp(header, "CONTINUE*", 0);
        if (!cpl_propertylist_has(header, MOO_PFITS_CUNIT1)) {
            cpl_propertylist_append_string(header, MOO_PFITS_CUNIT1,
                                           MOO_SINGLE_CUNIT1);
        }
        int naxis = moo_pfits_get_naxis(header);
        if (naxis > 0) {
            single = moo_single_new(type, ntas);
            single->filename = filename;
            cpl_propertylist_append(single->header, header);
        }
        cpl_propertylist_delete(header);
    }
moo_try_cleanup:
    return single;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load data in a single structure
  @param    self the SINGLE to load
  @param    level the bad pixel level
  @return   error code or CPL_ERROR_NONE

  If the filename doesnt have the extension name a NULL object is returned.
  The returned object must be deallocated using moo_detlist_delete().

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the frame group is not CPL_FRAME_GROUP_PRODUCT
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_load(moo_single *self, unsigned int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    self->badpix_mask = level;

    if ((self->filename != NULL) && (self->extname != NULL)) {
        if (self->header == NULL) {
            cpl_size extnum =
                cpl_fits_find_extension(self->filename, self->extname);
            if (extnum > 0) {
                self->header = cpl_propertylist_load(self->filename, extnum);
            }
        }
        if (self->qual == NULL) {
            const char *extname =
                moo_detector_get_qual_extname(self->type, self->ntas);
            cpl_size extnum = cpl_fits_find_extension(self->filename, extname);
            if (extnum > 0) {
                self->qual =
                    cpl_image_load(self->filename, CPL_TYPE_INT, 0, extnum);
            }
        }
        if (self->image == NULL) {
            cpl_image *data = NULL;
            cpl_image *err = NULL;

            data =
                moo_fits_load_extension_image(self->filename, NULL,
                                              self->extname, CPL_TYPE_DOUBLE);
            err = moo_fits_load_extension_image(self->filename, MOO_ERR,
                                                self->extname, CPL_TYPE_DOUBLE);

            if (data != NULL && err != NULL) {
                self->image = hdrl_image_create(data, err);
                cpl_image_delete(data);
                cpl_image_delete(err);
            }
        }
    }
    if (self->image != NULL && self->qual != NULL) {
        cpl_mask *mask = hdrl_image_get_mask(self->image);
        moo_badpix_to_mask(self->qual, mask, level);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_single
  @param    self    moo_single to delete
  @return   void

  If the moo_single @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_single_delete(moo_single *self)
{
    if (self != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        if (self->image != NULL) {
            hdrl_image_delete(self->image);
        }
        if (self->qual != NULL) {
            cpl_image_delete(self->qual);
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Get the DATA part of single DET
  @param    self    moo_single to read from
  @return   1 newly allocated cpl_image

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_single_get_data(moo_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_image *res = NULL;

    if (self->image != NULL) {
        res = cpl_image_duplicate(hdrl_image_get_image(self->image));
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Get the ERROR part of single DET
  @param    self    moo_single to read from
  @return   1 newly allocated cpl_image

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_single_get_err(moo_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *res =
        moo_fits_load_extension_image(self->filename, MOO_ERR, self->extname,
                                      CPL_TYPE_DOUBLE);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Apply the given mask on the QUAL part
  @param    self    moo_single to read from
  @param    mask the given mask
  @param    level the bad pixel level
  @return   the error code

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the level < 0
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_apply_mask(moo_single *self, cpl_mask *mask, unsigned int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mask != NULL, CPL_ERROR_NULL_INPUT);

    cpl_image *qual = moo_single_get_qual(self);
    moo_mask_to_badpix(qual, mask, level);
    if (self->image) {
        cpl_mask *hmask = hdrl_image_get_mask(self->image);
        cpl_mask_or(hmask, mask);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Get the QUAL part of single DET
  @param    self    moo_single to read from
  @return   the QUAL part of DET

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_single_get_qual(moo_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    return self->qual;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Get the IMAGE part (DATA,ERR) of single DET
  @param    self    moo_single to read from
  @return   the image part of single DET

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
hdrl_image *
moo_single_get_image(moo_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return self->image;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Free memory associate to this single DET
  @param    self    moo_single where want to free memory
  @return   error_code or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
    - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_free(moo_single *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    if (self->filename != NULL && self->extname != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
            self->header = NULL;
        }
        if (self->image != NULL) {
            hdrl_image_delete(self->image);
            self->image = NULL;
        }
        if (self->qual != NULL) {
            cpl_image_delete(self->qual);
            self->qual = NULL;
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_single to a FITS file
  @param    self          moo_single to write to disk or NULL
  @param    filename      Name of the file to write
  @param    type      the detector type (RI, YJ, H)
  @param    ntas      the number id of spectrograph

  This function saves a moo_det to a FITS file, using cfitsio.
  Only not NULL extensions are written.

*/
/*----------------------------------------------------------------------------*/
void
moo_single_save(moo_single *self,
                const char *filename,
                moo_detector_type type,
                int ntas)
{
    const char *extname = moo_detector_get_extname(type, ntas);

    if (self != NULL) {
        if (self->image != NULL) {
            if (self->header == NULL) {
                self->header = cpl_propertylist_new();
                cpl_propertylist_append_string(self->header, MOO_PFITS_EXTNAME,
                                               extname);
                cpl_propertylist_append_double(self->header, MOO_PFITS_EXPTIME,
                                               self->exptime);
            }
            else if (cpl_propertylist_has(self->header, MOO_PFITS_EXTNAME) ==
                     0) {
                cpl_propertylist_append_string(self->header, MOO_PFITS_EXTNAME,
                                               extname);
            }
            cpl_image_save(hdrl_image_get_image(self->image), filename,
                           CPL_TYPE_FLOAT, self->header, CPL_IO_EXTEND);

            cpl_propertylist *err_header = cpl_propertylist_new();

            moo_pfits_append_hduclass_error(err_header, type, ntas,
                                            self->header);
            moo_fits_write_extension_image(hdrl_image_get_error(self->image),
                                           filename, MOO_ERR, extname,
                                           CPL_TYPE_FLOAT, err_header);
            cpl_propertylist_delete(err_header);
        }
        else {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME, extname);
            cpl_propertylist_save(h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
            moo_fits_write_extension_image(NULL, filename, MOO_ERR, extname,
                                           CPL_TYPE_FLOAT, NULL);
        }
        cpl_propertylist *qual_header = cpl_propertylist_new();
        moo_pfits_append_hduclass_quality(qual_header, type, ntas, self->header,
                                          self->badpix_mask);
        moo_fits_write_extension_image(self->qual, filename, MOO_QUAL, extname,
                                       CPL_TYPE_INT, qual_header);
        cpl_propertylist_delete(qual_header);
    }
    else {
        cpl_propertylist *h = cpl_propertylist_new();
        cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME, extname);
        cpl_propertylist_save(h, filename, CPL_IO_EXTEND);
        cpl_propertylist_delete(h);

        moo_fits_write_extension_image(NULL, filename, MOO_ERR, extname,
                                       CPL_TYPE_FLOAT, NULL);

        moo_fits_write_extension_image(NULL, filename, MOO_QUAL, extname,
                                       CPL_TYPE_INT, NULL);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Dump structural information of a Single DET
  @param    self    single to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_dump(const moo_single *self, FILE *stream)
{
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    if (self != NULL) {
        fprintf(stream, "---MOO_SINGLE\n");
        fprintf(stream, "extname %s\n", self->extname);
        fprintf(stream, "filename %s\n", self->filename);
        fprintf(stream, "image: ");
        if (self->image != NULL) {
            cpl_image_dump_structure(hdrl_image_get_image(self->image), stream);
        }
        else {
            fprintf(stream, "null\n");
        }
        fprintf(stream, "header %p\n", (void *)self->header);
        fprintf(stream, "qual: ");
        if (self->qual != NULL) {
            cpl_image_dump_structure(self->qual, stream);
        }
        else {
            fprintf(stream, "null\n");
        }
        fprintf(stream, "mask %p\n", (void *)self->mask);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Subtract two single DET
  @param    a    single to subtract
  @param    b    single which is subtract
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_sub(moo_single *a, moo_single *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    hdrl_image *a_img = moo_single_get_image(a);
    hdrl_image *b_img = moo_single_get_image(b);

    hdrl_image_sub_image(a_img, b_img);

    cpl_image *qual1 = moo_single_get_qual(a);
    cpl_image *qual2 = moo_single_get_qual(b);
    cpl_image_or(a->qual, qual1, qual2);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Divide two single DET
  @param    a    single to divide
  @param    b    single which is divide
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_divide(moo_single *a, moo_single *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    hdrl_image *a_img = moo_single_get_image(a);
    hdrl_image *b_img = moo_single_get_image(b);

    hdrl_image_div_image(a_img, b_img);

    cpl_image *qual1 = moo_single_get_qual(a);
    cpl_image *qual2 = moo_single_get_qual(b);
    cpl_image_or(a->qual, qual1, qual2);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Elementwise multiplication of a single with a scalar.
  @param    a    single to be modified
  @param    scalar    Number to multiply with
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_multiply_scalar(moo_single *a, double scalar)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    hdrl_image *a_img = moo_single_get_image(a);

    hdrl_value v;
    v.data = scalar;
    v.error = 0;
    hdrl_image_mul_scalar(a_img, v);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Add two single DET
  @param    a    single to add
  @param    b    single which is add
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_sum(moo_single *a, moo_single *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    hdrl_image *a_img = moo_single_get_image(a);
    hdrl_image *b_img = moo_single_get_image(b);

    hdrl_image_add_image(a_img, b_img);

    cpl_image *qual1 = moo_single_get_qual(a);
    cpl_image *qual2 = moo_single_get_qual(b);

    cpl_image_or(a->qual, qual1, qual2);

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Mean of two single DET
  @param    a    first single to mean wich contanins the result of the operation
  @param    b    second single to mean
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_mean(moo_single *a, moo_single *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    hdrl_image *a_img = moo_single_get_image(a);
    hdrl_image *b_img = moo_single_get_image(b);

    hdrl_image_add_image(a_img, b_img);
    cpl_image *img = hdrl_image_get_image(a_img);
    cpl_image *err = hdrl_image_get_error(a_img);
    cpl_image_divide_scalar(img, 2.);
    cpl_image_divide_scalar(err, 2.);
    cpl_image *qual1 = moo_single_get_qual(a);
    cpl_image *qual2 = moo_single_get_qual(b);

    cpl_image_or(a->qual, qual1, qual2);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_single
  @brief    Flag low snr pixels from single DET structure
  @param    self single where to flag min_snr pixels
  @param    min_snr    second single to mean
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_single_filter_snr(moo_single *self, double min_snr)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    hdrl_image *img = moo_single_get_image(self);

    cpl_mask *img_mask = hdrl_image_get_mask(img);

    cpl_image *im = hdrl_image_get_image(img);
    cpl_image *err = hdrl_image_get_error(img);
    cpl_image *snr = cpl_image_divide_create(im, err);

    int nx = cpl_image_get_size_x(snr);
    int ny = cpl_image_get_size_y(snr);
    cpl_mask *mask = cpl_mask_new(nx, ny);

    for (int j = 1; j <= ny; j++) {
        for (int i = 1; i <= nx; i++) {
            int rej;
            double val = cpl_image_get(snr, i, j, &rej);
            if (fabs(val) >= DBL_EPSILON && val < min_snr) {
                cpl_mask_set(img_mask, i, j, CPL_BINARY_1);
                cpl_mask_set(mask, i, j, CPL_BINARY_1);
            }
        }
    }
    cpl_image *qual = moo_single_get_qual(self);
    moo_mask_to_badpix(qual, mask, MOO_BADPIX_LOW_QE);
    cpl_mask_delete(mask);
    cpl_image_delete(snr);
    return CPL_ERROR_NONE;
}


moo_single *
moo_single_add_create(moo_single *a, moo_single *b)
{
    cpl_ensure(a != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(b != NULL, CPL_ERROR_NULL_INPUT, NULL);
    moo_single *result = moo_single_new(a->type, a->ntas);

    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CRVAL2);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CD1_2);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CD2_1);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CD2_2);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CUNIT1);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(result->header, a->header, MOO_PFITS_BUNIT);

    result->image = hdrl_image_duplicate(moo_single_get_image(a));
    result->qual = cpl_image_duplicate(moo_single_get_qual(a));

    moo_single_sum(result, b);

    return result;
}

double
moo_single_get_exptime(moo_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);

    return moo_pfits_get_exptime(self->header);
}

cpl_propertylist *
moo_single_get_header(moo_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    if (self->header == NULL) {
        if (self->extname != NULL && self->filename != NULL) {
            self->header = moo_fits_load_extension_header(self->filename, "",
                                                          self->extname);
        }
        if (self->header == NULL) {
            self->header = cpl_propertylist_new();
        }
    }
    return self->header;
}

cpl_error_code
moo_single_set_exptime(moo_single *self, double val)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->header != NULL, CPL_ERROR_NULL_INPUT);

    return moo_pfits_update_exptime(self->header, val);
}
/**@}*/
