/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_SINGLE_H
#define MOO_SINGLE_H


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include "moo_detector.h"

#define MOO_SINGLE_CUNIT1 "pixel"
/*-----------------------------------------------------------------------------
                                   Types
 -----------------------------------------------------------------------------*/
typedef struct
{
    cpl_propertylist *header;
    hdrl_image *image;
    cpl_image *qual;
    cpl_mask *mask;
    /* conversion from eletcrons to ADU */
    float gain;
    /* exposure time */
    float exptime;
    /* ron */
    double ron;
    /* conad */
    double conad;
    const char *filename;
    const char *extname;
    unsigned int badpix_mask;
    moo_detector_type type;
    int ntas;
} moo_single;


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
moo_single *moo_single_new(moo_detector_type type, int ntas);
moo_single *
moo_single_create(const char *filename, moo_detector_type type, int ntas);
cpl_error_code moo_single_load(moo_single *self, unsigned int level);

cpl_error_code moo_single_free(moo_single *self);
void moo_single_delete(moo_single *self);

hdrl_image *moo_single_get_image(moo_single *self);
cpl_image *moo_single_get_data(moo_single *self);
cpl_image *moo_single_get_qual(moo_single *self);

cpl_error_code
moo_single_apply_mask(moo_single *self, cpl_mask *mask, unsigned int level);

void moo_single_save(moo_single *self,
                     const char *filename,
                     moo_detector_type type,
                     int ntas);

cpl_error_code moo_single_dump(const moo_single *self, FILE *stream);

cpl_error_code moo_single_sub(moo_single *a, moo_single *b);
cpl_error_code moo_single_divide(moo_single *a, moo_single *b);
cpl_error_code moo_single_multiply_scalar(moo_single *a, double scalar);
cpl_error_code moo_single_sum(moo_single *a, moo_single *b);
cpl_error_code moo_single_mean(moo_single *a, moo_single *b);
cpl_error_code moo_single_div_image(moo_single *a, cpl_image *b);
cpl_error_code moo_single_filter_snr(moo_single *self, double min_snr);

moo_single *moo_single_add_create(moo_single *a, moo_single *b);

cpl_propertylist *moo_single_get_header(moo_single *self);
double moo_single_get_exptime(moo_single *self);
cpl_error_code moo_single_set_exptime(moo_single *self, double val);
#endif
