/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_sky_lines_list.h"
#include "moo_fits.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_sky_lines_list SKY_LINES_LIST table
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_sky_lines_list
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_sky_lines_list.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_sky_lines_list
  @return   1 newly allocated moo_sky_lines_list or NULL in case of an error

  The returned object must be deallocated using moo_sky_lines_list_delete().

 */
/*----------------------------------------------------------------------------*/
moo_sky_lines_list* moo_sky_lines_list_new(void)
{
    moo_sky_lines_list* res = cpl_calloc(1,sizeof(moo_sky_lines_list));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a SKY_LINES_LIST frame and create a moo_sky_lines_list
  @param frame the SKY_LINES_LIST frame
  @return   1 newly allocated moo_sky_lines_list or NULL in case of an error

  The returned object must be deallocated using moo_sky_lines_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_sky_lines_list* moo_sky_lines_list_load(const cpl_frame* frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT,
               NULL);
    const char* filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT,
               NULL);

    moo_sky_lines_list* res = moo_sky_lines_list_new();
    res->filename = cpl_strdup(filename);

    return res;

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_sky_lines_list
  @param    self    moo_sky_lines_list to delete
  @return   void

  If the moo_sky_lines_list @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void moo_sky_lines_list_delete(moo_sky_lines_list* self)
{
  if ( self!= NULL){
    if ( self->filename!= NULL){
      cpl_free(self->filename);
    }
    if ( self->linecat!= NULL){
      cpl_table_delete(self->linecat);
    }
    if ( self->free_zones!= NULL){
      cpl_table_delete(self->free_zones);
    }
    cpl_free(self);
  }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get free zones for a specific wave range
  @param    self    moo_sky_lines_list
  @param    wmin the minimum wavelength for free zones
  @param    wmax the maximum wavelength for free zones
  @param    zwmin a pointer on free zones minimum wavelength
  @param    zwmax a pointer on free zones maximum wavelength
  @param    sel array of indexes of request free zones
  @return   void

 */
/*----------------------------------------------------------------------------*/
cpl_error_code moo_sky_lines_list_get_free_zones(moo_sky_lines_list* self,
  double wmin, double wmax, double** zwmin, double** zwmax, cpl_array** sel)
{
  cpl_ensure_code(self!=NULL, CPL_ERROR_NULL_INPUT);
  cpl_errorstate prev_state = cpl_errorstate_get();

  if (self->free_zones==NULL){
    moo_try_check(self->free_zones = moo_fits_load_extension_table(self->filename,
      NULL, MOO_SKY_LINES_LIST_FREEZONE_TNAME)," ");
  }
  cpl_table_select_all(self->free_zones);
  cpl_table_and_selected_double(self->free_zones,MOO_SKY_LINES_LIST_WMAX_COLNAME,
    CPL_GREATER_THAN,wmin);
  cpl_table_and_selected_double(self->free_zones,MOO_SKY_LINES_LIST_WMIN_COLNAME,
    CPL_LESS_THAN,wmax);
  *zwmin = cpl_table_get_data_double(self->free_zones,
    MOO_SKY_LINES_LIST_WMIN_COLNAME);
  *zwmax = cpl_table_get_data_double(self->free_zones,
    MOO_SKY_LINES_LIST_WMAX_COLNAME);
  *sel = cpl_table_where_selected(self->free_zones);
  
  moo_try_cleanup:
    if ( !cpl_errorstate_is_equal(prev_state)){
      cpl_msg_error(__func__,"Can't load skylines for filename %s",
        self->filename);
      cpl_errorstate_set(prev_state);
    }
    return CPL_ERROR_NONE;
}
/**@}*/
