/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_sub_sky_stare.h"
#include "moo_fibres_table.h"
#include "moo_f2f.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
#include "moo_skycorr.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
/* create all sky image */
static hdrl_image *
_moo_create_median_sky(cpl_array *sky_indexes, int *indexrbn, hdrl_image *rbn)
{
    hdrl_image *median_sky = NULL;

    cpl_ensure(rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sky_indexes != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nx = hdrl_image_get_size_x(rbn);
    int nb_sky = (int)cpl_array_get_size(sky_indexes);

    if (nb_sky > 0) {
        hdrl_image *allsky_img = hdrl_image_new(nx, nb_sky);
        if (allsky_img == NULL) {
            return NULL;
        }

        for (int i = 1; i <= nb_sky; i++) {
            int idx = cpl_array_get_cplsize(sky_indexes, i - 1, NULL);
            int rbn_idx = indexrbn[idx];

            for (int j = 1; j <= nx; j++) {
                int rej;
                hdrl_value rbn_img_val =
                    hdrl_image_get_pixel(rbn, j, rbn_idx, &rej);

                if (rej) {
                    hdrl_image_reject(allsky_img, j, i);
                }
                else {
                    hdrl_image_set_pixel(allsky_img, j, i, rbn_img_val);
                }
            }
        }

        median_sky = moo_image_collapse_median_create(allsky_img);
        hdrl_image_delete(allsky_img);
    }
    else {
        median_sky = hdrl_image_new(nx, 1);
    }
    return median_sky;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    This function selects sky fibres around a target object
  @param    targname Name of the target object
  @param    indextarg Index of the target in the TARGET_TABLE
  @param    obj_index Index RBN of the target in the TARGET_TABLE
  @param    obj_targalpha Alpha position of the target
  @param    obj_targdelta Delta position of the target
  @param    target_table the TARGET TABLE
  @param    index_colname the name of the INDEX column in table
  @param    sky_stare_table the SKY fibres results table
  @param    type the detector type RI,YJ or H
  @param    maxdistslit the maximum distance on the slit for the sky fibres
  @param    mintrans the minimum value of transmission for the sky fibres
  @param    min_sky the minimum of sky fibres to find
  @param    radius the radius around the object position to find the sky fibres
  @param    step_r the step to increase rhe radius to search the sky fibres
  @param    f2f the fibre to fibre transmission
  @param    allsky_indexes the indexes of sky fibres in target_table
  @return   the selected sky indexes
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_array *
moo_select_sky(const char *targname,
               int indextarg,
               int obj_index,
               double obj_targalpha,
               double obj_targdelta,
               cpl_table *target_table,
               const char *index_colname,
               cpl_table *sky_stare_table,
               moo_detector_type type,
               int maxdistslit,
               double mintrans,
               int min_sky,
               double radius,
               double step_r,
               int target_table_idx,
               moo_f2f *f2f,
               cpl_array *allsky_indexes)
{
    cpl_array *sky_indexes = NULL;

    cpl_ensure(maxdistslit >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int *indexes = cpl_table_get_data_int(target_table, index_colname);
    double *targalpha =
        cpl_table_get_data_double(target_table, MOO_FIBRES_TABLE_TARGALPHA);
    double *targdelta =
        cpl_table_get_data_double(target_table, MOO_FIBRES_TABLE_TARGDELTA);
    sky_indexes = cpl_array_duplicate(allsky_indexes);
    int nb_sky = cpl_array_get_size(sky_indexes);
    int nstart = cpl_table_get_nrow(sky_stare_table);
    cpl_table_set_size(sky_stare_table, nstart + nb_sky);
    int nstop = cpl_table_get_nrow(sky_stare_table);

    for (int i = nstart; i < nstop; i++) {
        cpl_table_set_string(sky_stare_table, MOO_SKY_STARE_TARGNAME, i,
                             targname);
        cpl_table_set_int(sky_stare_table, MOO_SKY_STARE_INDEXTARG, i,
                          indextarg);
        cpl_table_set_int(sky_stare_table, MOO_SKY_STARE_INDEXRBN, i,
                          obj_index);
    }
    cpl_table_unselect_all(sky_stare_table);
    cpl_array *obj_sel = NULL;
    // filter in slit
    if (maxdistslit >= 0) {
        cpl_table_select_all(target_table);
        cpl_table_and_selected_int(target_table, index_colname, CPL_EQUAL_TO,
                                   obj_index);

        obj_sel = cpl_table_where_selected(target_table);
        int tobj_idx = cpl_array_get_cplsize(obj_sel, 0, NULL);
        cpl_array_delete(obj_sel);
        obj_sel = NULL;
        int obj_spectro =
            cpl_table_get_int(target_table, MOO_FIBRES_TABLE_SPECTRO, tobj_idx,
                              NULL);

        for (int i = 1; i <= nb_sky; i++) {
            int idx = cpl_array_get_cplsize(sky_indexes, i - 1, NULL);
            int idx_t = nstart + i - 1;
            int sky_index = indexes[idx];
            cpl_table_select_all(target_table);
            cpl_table_and_selected_int(target_table, index_colname,
                                       CPL_EQUAL_TO, sky_index);
            obj_sel = cpl_table_where_selected(target_table);
            int tsky_idx = cpl_array_get_cplsize(obj_sel, 0, NULL);
            cpl_array_delete(obj_sel);
            obj_sel = NULL;
            const int sky_spectro =
                cpl_table_get_int(target_table, MOO_FIBRES_TABLE_SPECTRO,
                                  tsky_idx, NULL);

            cpl_table_set_int(sky_stare_table, MOO_SKY_STARE_SKYINDEXRBN, idx_t,
                              sky_index);
            if (obj_spectro == sky_spectro) {
                int dslit = (int)fabs((double)sky_index - (double)obj_index);
                cpl_table_set_int(sky_stare_table, MOO_SKY_STARE_SKYDSLIT,
                                  idx_t, dslit);
                if (dslit == 0) {
                    double sky_alpha = targalpha[idx];
                    double sky_delta = targdelta[idx];
                    double d = moo_sky_distance(obj_targalpha, obj_targdelta,
                                                sky_alpha, sky_delta);
                    if (d < 1E-9) {
                        cpl_table_select_row(sky_stare_table, idx_t);
                        cpl_array_set_invalid(sky_indexes, i - 1);
                    }
                }
                else if (dslit > maxdistslit) {
                    cpl_table_select_row(sky_stare_table, idx_t);
                    cpl_array_set_invalid(sky_indexes, i - 1);
                }
            }
            else {
                cpl_array_set_invalid(sky_indexes, i - 1);
                cpl_table_select_row(sky_stare_table, idx_t);
            }
        }
    }
    int nb_valid_distslit = nb_sky - cpl_array_count_invalid(sky_indexes);

    if (nb_valid_distslit < min_sky) {
        cpl_array_delete(sky_indexes);
        sky_indexes = NULL;
        cpl_error_set_message(
            "sub_sky_stare", CPL_ERROR_EOL,
            "for indextarg %d after using filter with maxdistslit=%d : "
            "not enough sky fibre (%d) for object indexrbn: %d",
            indextarg, maxdistslit, nb_valid_distslit, obj_index);
        return sky_indexes;
    }
    // filter in slit
    if (f2f != NULL && mintrans > 0) {
        cpl_array *f2f_sel = NULL;
        const char *f2f_trans_colname = moo_f2f_get_trans_colname(f2f, type);

        for (int i = 0; i < nb_sky; i++) {
            int rej;
            int idx_t = nstart + i;
            int idx = cpl_array_get_cplsize(sky_indexes, i, &rej);
            int sky_index = indexes[idx];

            cpl_table_select_all(f2f->table);
            cpl_table_and_selected_int(f2f->table, index_colname, CPL_EQUAL_TO,
                                       sky_index);
            f2f_sel = cpl_table_where_selected(f2f->table);
            int f2f_sky_idx = cpl_array_get_cplsize(f2f_sel, 0, NULL);
            cpl_array_delete(f2f_sel);
            f2f_sel = NULL;
            float sky_trans =
                cpl_table_get(f2f->table, f2f_trans_colname, f2f_sky_idx, NULL);

            cpl_table_set_double(sky_stare_table, MOO_SKY_STARE_SKYTRANS, idx_t,
                                 sky_trans);
            if (rej == 0) {
                if (sky_trans < mintrans) {
                    cpl_table_select_row(sky_stare_table, idx_t);
                    cpl_array_set_invalid(sky_indexes, i);
                }
            }
        }
    }

    int nb_valid_trans = nb_sky - cpl_array_count_invalid(sky_indexes);
    if (nb_valid_trans < min_sky) {
        cpl_array_delete(sky_indexes);
        sky_indexes = NULL;
        cpl_error_set_message(
            "sub_sky_stare", CPL_ERROR_EOL,
            "for indextarg %d after using filter with mintrans=%f : "
            "not enough sky fibre (%d) for object indexrbn: %d",
            indextarg, mintrans, nb_valid_trans, obj_index);
        return sky_indexes;
    }

    // filter in sky
    for (int i = 0; i < nb_sky; i++) {
        int rej;
        int idx = cpl_array_get_cplsize(sky_indexes, i, &rej);
        int idx_t = nstart + i;

        if (rej == 0) {
            double sky_alpha = targalpha[idx];
            double sky_delta = targdelta[idx];
            double d = moo_sky_distance(obj_targalpha, obj_targdelta, sky_alpha,
                                        sky_delta);
            // rad to ARCMIN
            d = d * CPL_MATH_DEG_RAD * 60;
            cpl_table_set_double(sky_stare_table, MOO_SKY_STARE_RADIUS, idx_t,
                                 radius);
            cpl_table_set_double(sky_stare_table, MOO_SKY_STARE_SKYDIST, idx_t,
                                 d);
            if (d > radius) {
                cpl_table_select_row(sky_stare_table, idx_t);
                cpl_array_set_invalid(sky_indexes, i);
            }
        }
    }

    int nb_valid = nb_sky - cpl_array_count_invalid(sky_indexes);

    if (nb_valid >= min_sky) {
        cpl_table_erase_selected(sky_stare_table);
        return sky_indexes;
    }
    else if (step_r > 0) {
        cpl_array_delete(sky_indexes);
        cpl_table_set_size(sky_stare_table, nstart);
        sky_indexes =
            moo_select_sky(targname, indextarg, obj_index, obj_targalpha,
                           obj_targdelta, target_table, index_colname,
                           sky_stare_table, type, maxdistslit, mintrans,
                           min_sky, radius + step_r, step_r, target_table_idx,
                           f2f, allsky_indexes);
    }
    else {
        cpl_array_delete(sky_indexes);
        sky_indexes = NULL;
        cpl_error_set_message(
            "sub_sky_stare", CPL_ERROR_EOL,
            "for indextarg %d "
            "filtering distslit: %d valid sky fibre, "
            "filtering mintrans: %d valid sky fibre, "
            "after using sky ditance filter with radius=%f : "
            "not enough sky fibre (%d) for object indexrbn: %d",
            indextarg, nb_valid_distslit, nb_valid_trans, radius, nb_valid,
            obj_index);
        return sky_indexes;
    }
    return sky_indexes;
}

static moo_sci_single *
_moo_sub_sky_stare_single(moo_rbn_single *obja_rbn_single,
                          cpl_table *obja_fibre_table,
                          double ea,
                          moo_target_table *target_table,
                          moo_rbn_single *objb_rbn_single,
                          cpl_table *objb_fibre_table,
                          double eb,
                          moo_f2f *f2f,
                          const cpl_frame *solflux_frame,
                          const cpl_frame *airglow_group_frame,
                          const cpl_frame *airglow_var_frame,
                          moo_sub_sky_stare_params *params,
                          cpl_array *obja_sky_indexes,
                          cpl_array *objb_sky_indexes,
                          cpl_array *targ_obj_indexes,
                          cpl_propertylist *pheader,
                          const char *filename,
                          int ispaired)
{
    moo_sci_single *sci_single = NULL;
    cpl_array *valid_targets_indexes = NULL;
    cpl_table *sky_stare_table = NULL;
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(obja_rbn_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_rbn_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(obja_sky_indexes != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_sky_indexes != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(targ_obj_indexes != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nb_target = (int)cpl_array_get_size(targ_obj_indexes);

    if (nb_target == 0) {
        cpl_msg_warning("sub_sky_stare", "No targets in stare mode");
        cpl_table_set_size(target_table->table, 0);
        return NULL;
    }

    sci_single = moo_sci_single_new(obja_rbn_single->type);
    double crpix1 = moo_pfits_get_crpix1(obja_rbn_single->header);
    double crval1 = moo_pfits_get_crval1(obja_rbn_single->header);
    double cd1_1 = moo_pfits_get_cd1_1(obja_rbn_single->header);

    moo_pfits_append_hduclass_data(sci_single->header, sci_single->type, -1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CD1_2);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CUNIT1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_BUNIT);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CRVAL2);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CD2_1);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CD2_2);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_WAVELMIN);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_WAVELMAX);
    cpl_propertylist_copy_property(sci_single->header, obja_rbn_single->header,
                                   MOO_PFITS_SPECBIN);
    int nx = hdrl_image_get_size_x(obja_rbn_single->image);

    sci_single->image = hdrl_image_new(nx, nb_target);
    sci_single->sky = cpl_image_new(nx, nb_target, CPL_TYPE_DOUBLE);
    sci_single->qual = cpl_image_new(nx, nb_target, CPL_TYPE_INT);
    sci_single->filename = filename;

    cpl_image *sci_img = hdrl_image_get_image(sci_single->image);
    /* create the mask if empty */
    cpl_image_get_bpm(sci_img);

    cpl_image *sci_err_img = hdrl_image_get_error(sci_single->image);
    cpl_image *sci_qual_img = sci_single->qual;

    int *indexrbnt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_INDEXRBN);
    int *pindexrbnt = cpl_table_get_data_int(target_table->table,
                                             MOO_TARGET_TABLE_PAIREDINDEXRBN);
    int *indexestarg =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG);
    const char **fibret =
        cpl_table_get_data_string_const(target_table->table,
                                        MOO_TARGET_TABLE_FIBRE);
    const char **targnames =
        cpl_table_get_data_string_const(target_table->table,
                                        MOO_TARGET_TABLE_TARGNAME);
    double *targalphat = cpl_table_get_data_double(target_table->table,
                                                   MOO_TARGET_TABLE_TARGALPHA);
    double *targdeltat = cpl_table_get_data_double(target_table->table,
                                                   MOO_TARGET_TABLE_TARGDELTA);
    /* add objects -sky */
    const char **obstypet =
        cpl_table_get_data_string_const(target_table->table,
                                        MOO_TARGET_TABLE_OBSTYPE);

    cpl_table_unselect_all(target_table->table);
    for (int i = 0; i < nb_target; i++) {
        int obj_idx = cpl_array_get_cplsize(targ_obj_indexes, i, NULL);
        cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG,
                          obj_idx, i + 1);
        cpl_table_set_string(target_table->table, MOO_TARGET_TABLE_SUBSKYMODE,
                             obj_idx, MOO_TARGET_TABLE_SUBSKY_STARE);
        const char *obstype = obstypet[obj_idx];
        int is_stare = moo_is_obstype_stare(obstype);

        cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_NBSKY, obj_idx,
                          0);
        cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_MAX_DIST_SLIT,
                          obj_idx, 0);
        cpl_table_set_double(target_table->table, MOO_TARGET_TABLE_MEDSNRRI_SCI,
                             obj_idx, 0);

        cpl_table_set_double(target_table->table, MOO_TARGET_TABLE_MEDSNRYJ_SCI,
                             obj_idx, 0);
        cpl_table_set_double(target_table->table, MOO_TARGET_TABLE_MEDSNRH_SCI,
                             obj_idx, 0);
        if (is_stare == 1) {
            cpl_table_select_row(target_table->table, obj_idx);
        }
    }

    valid_targets_indexes = cpl_table_where_selected(target_table->table);
    int nb_valid_targets = cpl_array_get_size(valid_targets_indexes);
    if (strcmp(params->method, MOO_SUB_SKY_STARE_METHOD_NONE) != 0) {
        sky_stare_table = cpl_table_new(0);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_INDEXTARG,
                             CPL_TYPE_INT);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_TARGNAME,
                             CPL_TYPE_STRING);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_INDEXRBN,
                             CPL_TYPE_INT);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYINDEXRBN,
                             CPL_TYPE_INT);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYDSLIT,
                             CPL_TYPE_INT);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYTRANS,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_RADIUS,
                             CPL_TYPE_DOUBLE);
        cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYDIST,
                             CPL_TYPE_DOUBLE);
        int *nbsky =
            cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_NBSKY);
        int *maxdslit = cpl_table_get_data_int(target_table->table,
                                               MOO_TARGET_TABLE_MAX_DIST_SLIT);
        float *maxdsky =
            cpl_table_get_data_float(target_table->table,
                                     MOO_TARGET_TABLE_MAX_DIST_SKY);

        for (int i = 0; i < nb_valid_targets; i++) {
            int idx = cpl_array_get_cplsize(valid_targets_indexes, i, NULL);
            int indextarg = indexestarg[idx];
            int indexrbn = indexrbnt[idx];

            const char *targname = targnames[idx];
            const char *obstype = obstypet[idx];
            double obj_targalpha = targalphat[idx];
            double obj_targdelta = targdeltat[idx];
            cpl_array *sky_indexes = NULL;
            cpl_table *obj_table = NULL;
            cpl_table *allsky_table = target_table->table;
            cpl_array *allsky_indexes = NULL;
            const char *allsky_table_indexcol = MOO_FIBRES_TABLE_INDEXRBN;

            if (strcmp(MOO_OBSTYPE_OBSNODSTARE, obstype) == 0 ||
                strcmp(MOO_OBSTYPE_OBSPSTARE, obstype) == 0 ||
                strcmp(MOO_OBSTYPE_SKYRESPSTARE, obstype) == 0 ||
                (strcmp(MOO_OBSTYPE_OBSSWITCHSTARE, obstype) == 0 &&
                 ispaired)) {
                obj_table = objb_fibre_table;
                allsky_table = obja_fibre_table;
                allsky_indexes = obja_sky_indexes;
            }
            else {
                obj_table = obja_fibre_table;
                allsky_table = objb_fibre_table;
                allsky_indexes = objb_sky_indexes;
            }

            cpl_table_select_all(obj_table);
            cpl_table_and_selected_int(obj_table, MOO_FIBRES_TABLE_INDEXRBN,
                                       CPL_EQUAL_TO, indexrbn);
            cpl_array *sel = cpl_table_where_selected(obj_table);
            int ft_idx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);
            const char *fibre =
                cpl_table_get_string(obj_table, MOO_FIBRES_TABLE_FIBRE, ft_idx);
            cpl_table_set_string(target_table->table, MOO_TARGET_TABLE_FIBRE,
                                 idx, fibre);
            moo_target_table_set_snr(target_table, idx, obj_table, ft_idx);
            moo_try_check(sky_indexes = moo_select_sky(
                              targname, indextarg, indexrbn, obj_targalpha,
                              obj_targdelta, allsky_table,
                              allsky_table_indexcol, sky_stare_table,
                              obja_rbn_single->type, params->maxdistslit,
                              params->min_trans, params->min_sky,
                              params->radius_sky, params->step_r, idx, f2f,
                              allsky_indexes),
                          " ");

            int nb_sky = cpl_table_and_selected_int(sky_stare_table,
                                                    MOO_SKY_STARE_INDEXTARG,
                                                    CPL_EQUAL_TO, indextarg);

            int max_dslit = 0;
            float max_dsky = 0.0;
            if (nb_sky > 0) {
                cpl_table *subtable =
                    cpl_table_extract_selected(sky_stare_table);
                max_dslit =
                    cpl_table_get_column_max(subtable, MOO_SKY_STARE_SKYDSLIT);
                max_dsky =
                    cpl_table_get_column_max(subtable, MOO_SKY_STARE_SKYDIST);
                int all = cpl_array_get_size(allsky_indexes);
                cpl_msg_debug(
                    __func__,
                    "---do %s %d %d %.3f %.3f %s => all:%d, sel:%d, %d",
                    targname, indextarg, indexrbn, obj_targalpha, obj_targdelta,
                    obstype, all, nb_sky, max_dslit);
                cpl_table_delete(subtable);
            }
            nbsky[idx] = nb_sky;
            maxdslit[idx] = max_dslit;
            maxdsky[idx] = max_dsky;
            cpl_array_delete(sky_indexes);
        }
    }
/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions
 * where these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_target, indexrbnt, pindexrbnt, fibret,          \
                                indexestarg, target_table, obstypet, ispaired, \
                                nb_valid_targets, valid_targets_indexes,       \
                                sky_stare_table, obja_rbn_single,              \
                                objb_rbn_single, params, sci_single,           \
                                sci_qual_img, crpix1, crval1, cd1_1, sci_img,  \
                                sci_err_img, nx, airglow_var_frame, pheader,   \
                                solflux_frame, airglow_group_frame, ea, eb)
#else
#pragma omp parallel default(none)                                         \
    shared(nb_target, indexrbnt, pindexrbnt, fibret, indexestarg,          \
               target_table, obstypet, ispaired, nb_valid_targets,         \
               valid_targets_indexes, sky_stare_table, obja_rbn_single,    \
               objb_rbn_single, params, sci_single, sci_qual_img, crpix1,  \
               crval1, cd1_1, sci_img, sci_err_img, nx, airglow_var_frame, \
               pheader, solflux_frame, airglow_group_frame, ea, eb)
#endif
    {
#pragma omp for
#endif

        for (int i = 0; i < nb_valid_targets; i++) {
            hdrl_image *median_sky = NULL;
            int idx = cpl_array_get_cplsize(valid_targets_indexes, i, NULL);
            int rbn_idx = indexrbnt[idx];
            int targ_idx = indexestarg[idx];
            const char *obstype = obstypet[idx];
            const char *fibre = fibret[idx];
            moo_rbn_single *obj_rbn_single = NULL;
            moo_rbn_single *sky_rbn_single = NULL;
            double sky_rescale = 1.0;
            const char *method = params->method;

            if (strcmp(MOO_OBSTYPE_OBSNODSTARE, obstype) == 0 ||
                strcmp(MOO_OBSTYPE_OBSPSTARE, obstype) == 0 ||
                strcmp(MOO_OBSTYPE_SKYRESPSTARE, obstype) == 0 ||
                (strcmp(MOO_OBSTYPE_OBSSWITCHSTARE, obstype) == 0 &&
                 ispaired)) {
                cpl_msg_debug(__func__, "---do B-A");
                obj_rbn_single = objb_rbn_single;
                sky_rbn_single = obja_rbn_single;
            }
            else {
                cpl_msg_debug(__func__, "---do A-B");
                obj_rbn_single = obja_rbn_single;
                sky_rbn_single = objb_rbn_single;
                sky_rescale = ea / eb;
            }
            cpl_image *obj_rbn_img =
                hdrl_image_get_image(obj_rbn_single->image);
            cpl_image *obj_rbn_err_img =
                hdrl_image_get_error(obj_rbn_single->image);
            cpl_image *obj_rbn_qual_img = obj_rbn_single->qual;
            if (strcmp(method, MOO_SUB_SKY_STARE_METHOD_NONE) == 0) {
                double *rbn_img_data = cpl_image_get_data_double(obj_rbn_img);
                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double rbn_img_val =
                        rbn_img_data[j - 1 + (rbn_idx - 1) * nx];
                    double sci_flux = rbn_img_val;
                    cpl_image_set(sci_img, j, targ_idx, sci_flux);
                    double rbn_err_val =
                        cpl_image_get(obj_rbn_err_img, j, rbn_idx, &rej);
                    double err = rbn_err_val;
                    cpl_image_set(sci_err_img, j, targ_idx, err);
                    double rbn_qual_val =
                        cpl_image_get(obj_rbn_qual_img, j, rbn_idx, &rej);

                    cpl_image_set(sci_qual_img, j, targ_idx, rbn_qual_val);
                }
            }
            else {
                cpl_array *sky_indexes = NULL;
                cpl_table *private_table = cpl_table_duplicate(sky_stare_table);
                cpl_table_select_all(private_table);
                int *sky_indexrbn =
                    cpl_table_get_data_int(private_table,
                                           MOO_SKY_STARE_SKYINDEXRBN);
                int nb_sky = cpl_table_and_selected_int(private_table,
                                                        MOO_SKY_STARE_INDEXRBN,
                                                        CPL_EQUAL_TO, rbn_idx);
                int max_dslit = 0;
                if (nb_sky > 0) {
                    cpl_table *subtable =
                        cpl_table_extract_selected(private_table);
                    max_dslit =
                        cpl_table_get_column_max(subtable,
                                                 MOO_SKY_STARE_SKYDSLIT);
                    cpl_table_delete(subtable);
                }
                cpl_msg_debug(__func__,
                              "---do indextarg %d indexrbn %d nb_sky %d "
                              "max_dist_slit %d %d",
                              targ_idx, rbn_idx, nb_sky, max_dslit,
                              cpl_error_get_code());

                sky_indexes = cpl_table_where_selected(private_table);

                median_sky = _moo_create_median_sky(sky_indexes, sky_indexrbn,
                                                    sky_rbn_single->image);

                hdrl_value srescale = { .data = sky_rescale, .error = 0 };
                hdrl_image_mul_scalar(median_sky, srescale);
                if (nb_sky == 0) {
                    cpl_msg_info(
                        "sub_sky_stare",
                        "no SKY fibre for fibre indexrbn %d with name %s "
                        ": median sky is 0",
                        rbn_idx, fibre);
                }
                cpl_table_delete(private_table);
                cpl_array_delete(sky_indexes);

                if (median_sky != NULL) {
                    if (strcmp(method, MOO_SUB_SKY_STARE_METHOD_SIMPLE) == 0) {
                        double *rbn_img_data =
                            cpl_image_get_data_double(obj_rbn_img);
                        for (int j = 1; j <= nx; j++) {
                            int rej;
                            hdrl_value sky_val =
                                hdrl_image_get_pixel(median_sky, j, 1, &rej);
                            cpl_image_set(sci_single->sky, j, targ_idx,
                                          sky_val.data);
                            double rbn_img_val =
                                rbn_img_data[j - 1 + (rbn_idx - 1) * nx];
                            double sci_flux = rbn_img_val - sky_val.data;
                            cpl_image_set(sci_img, j, targ_idx, sci_flux);
                            double rbn_err_val =
                                cpl_image_get(obj_rbn_err_img, j, rbn_idx,
                                              &rej);
                            double err = sqrt(rbn_err_val * rbn_err_val +
                                              sky_val.error * sky_val.error);
                            cpl_image_set(sci_err_img, j, targ_idx, err);
                            double rbn_qual_val =
                                cpl_image_get(obj_rbn_qual_img, j, rbn_idx,
                                              &rej);

                            cpl_image_set(sci_qual_img, j, targ_idx,
                                          rbn_qual_val);
                        }
                    }
                    else {
                        moo_skycorr(obj_rbn_single->image, median_sky, targ_idx,
                                    rbn_idx, crpix1, crval1, cd1_1, pheader,
                                    solflux_frame, airglow_group_frame,
                                    airglow_var_frame, params->sk,
                                    sci_single->image, sci_single->sky);
                    }
                    hdrl_image_delete(median_sky);
                }
            }
        }
#ifdef _OPENMP
    }
#endif

moo_try_cleanup:
#if MOO_DEBUG_SKY
    if (sky_stare_table != NULL) {
        char *sky_stare_table_name =
            cpl_sprintf("SKY_TABLE_%s_%s", obja_rbn_single->extname, filename);
        cpl_table_save(sky_stare_table, NULL, NULL, sky_stare_table_name,
                       CPL_IO_CREATE);
        cpl_free(sky_stare_table_name);
    }
#endif
    cpl_array_delete(valid_targets_indexes);
    cpl_table_delete(sky_stare_table);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in sky subtraction %s",
                              obja_rbn_single->extname);
        moo_sci_single_delete(sci_single);
        sci_single = NULL;
    }
    return sci_single;
}

static cpl_propertylist *
_moo_create_primary_header(cpl_propertylist *rbn_primary_header)
{
    cpl_propertylist *primary_header = NULL;

    primary_header = cpl_propertylist_new();
    cpl_propertylist_copy_property(primary_header, rbn_primary_header,
                                   MOO_PFITS_EXPTIME);
    cpl_propertylist_copy_property(primary_header, rbn_primary_header,
                                   MOO_PFITS_MJDOBS);
    cpl_propertylist_copy_property_regexp(primary_header, rbn_primary_header,
                                          "TM*", 0);
    cpl_propertylist_copy_property_regexp(primary_header, rbn_primary_header,
                                          "ESO QC IS*", 0);
    cpl_propertylist_copy_property(primary_header, rbn_primary_header,
                                   MOO_PFITS_FLUXCAL);

    return primary_header;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    This function subtracts the sky in wavelength calibrated and
 * extracted science frames obtain in Stare mode observations
  @param    obja_rbn _RBN_ Wavelength calibrated science spectra for object A
  @param    target_table _TARGET_TABLE_ Attached target table
  @param    objb_rbn _RBN_ Wavelength calibrated science spectra for object B
  @param    f2f the F2F (fibre to fibre transmission) table
  @param    solflux_frame the monthly averages of solar radio flux at 10.7 cm frame
  @param    airglow_group_frame the airglow line list frame (use in subsky)
  @param    airglow_var_frame the airglow scaling parameters frame (use in subsky)
  @param    params the sub sky stare parameters
  @param    filename the name of the result SCI   
  @param    ispaired MOO_SCI1D_NOT_PAIRED or MOO_SCI1D_PAIRED
  @return   the _SCI_ result

 * _Flags considered as bad : BADPIX_COSMETIC | BADPIX_HOT
 *
 * _Bad pixels flags_:
  - BADPIX_COSMIC_UNREMOVED is set in SIGCLIP mode

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_sub_sky_stare(moo_rbn *obja_rbn,
                  moo_target_table *target_table,
                  moo_rbn *objb_rbn,
                  moo_f2f *f2f,
                  const cpl_frame *solflux_frame,
                  const cpl_frame *airglow_group_frame,
                  const cpl_frame *airglow_var_frame,
                  moo_sub_sky_stare_params *params,
                  const char *filename,
                  int ispaired)
{
    moo_sci *sci = NULL;

    cpl_ensure(obja_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_array *objb_sky_indexes = NULL;
    cpl_array *obja_sky_indexes = NULL;
    cpl_array *targ_obj_indexes = NULL;
    moo_target_table *sci_target_table = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;

    moo_fits_create(filename);
    sci = moo_sci_new();
    sci->filename = filename;
    sci->primary_header = _moo_create_primary_header(obja_rbn->primary_header);
    cpl_msg_info(
        __func__,
        "Substract sky stare with method %s "
        "using maxdistslit %d, mintrans %f, and radius (%f+%f) and min sky %d",
        params->method, params->maxdistslit, params->min_trans,
        params->radius_sky, params->step_r, params->min_sky);

    if (f2f != NULL) {
        moo_f2f_order_by_indexrbn(f2f);
    }
    double exptimeA[3];
    double exptimeB[3];
    exptimeA[0] = moo_pfits_get_exptime(obja_rbn->primary_header);
    exptimeB[0] = moo_pfits_get_exptime(objb_rbn->primary_header);
    for (int i = 1; i < 3; i++) {
        moo_rbn_single *obja_rbn_single = moo_rbn_get_single(obja_rbn, i);
        moo_rbn_single *objb_rbn_single = moo_rbn_get_single(objb_rbn, i);
        exptimeA[i] = 0;
        exptimeB[i] = 0;
        if (obja_rbn_single != NULL) {
            cpl_propertylist *header =
                moo_rbn_single_get_header(obja_rbn_single);

            double dit = moo_pfits_get_dit(header);
            int ndit = moo_pfits_get_ndit(header);

            exptimeA[i] = ndit * dit;
        }
        if (objb_rbn_single != NULL) {
            cpl_propertylist *header =
                moo_rbn_single_get_header(objb_rbn_single);

            double dit = moo_pfits_get_dit(header);
            int ndit = moo_pfits_get_ndit(header);

            exptimeB[i] = ndit * dit;
        }
    }
    sci_target_table = moo_target_table_duplicate(target_table);
    moo_target_table_set_exptimes(sci_target_table, exptimeA, exptimeB);

    sci->target_table = sci_target_table;

    targ_obj_indexes =
        moo_target_table_select_obj_index(sci_target_table, ispaired);
    objb_sky_indexes =
        moo_target_table_select_sky_index(sci_target_table, objb_rbn);
    obja_sky_indexes =
        moo_target_table_select_sky_index(sci_target_table, obja_rbn);

    cpl_msg_indent_more();
    cpl_table *obja_fibre_table = moo_rbn_get_fibre_table(obja_rbn);
    cpl_table *objb_fibre_table = moo_rbn_get_fibre_table(objb_rbn);
    if (ispaired) {
        moo_target_table_xswitch_paired(sci_target_table);
    }

    for (int i = 0; i < 3; i++) {
        moo_sci_single *sci_single = NULL;
        moo_rbn_single *obja_rbn_single =
            moo_rbn_load_single(obja_rbn, i, badpix_level);
        moo_rbn_single *objb_rbn_single = NULL;

        if (obja_rbn_single != NULL) {
            cpl_msg_info(__func__, "Substract sky stare extension %s",
                         obja_rbn_single->extname);
            objb_rbn_single = moo_rbn_load_single(objb_rbn, i, badpix_level);
            double ea = exptimeA[i];
            double eb = exptimeB[i];
            moo_try_check(sci_single = _moo_sub_sky_stare_single(
                              obja_rbn_single, obja_fibre_table, ea,
                              sci_target_table, objb_rbn_single,
                              objb_fibre_table, eb, f2f, solflux_frame,
                              airglow_group_frame, airglow_var_frame, params,
                              obja_sky_indexes, objb_sky_indexes,
                              targ_obj_indexes, obja_rbn->primary_header,
                              filename, ispaired),
                          " ");
        }
        moo_sci_add_single(sci, i, sci_single);
    }
    cpl_msg_indent_less();
    int nrow = cpl_table_get_nrow(sci_target_table->table);

    for (int i = 0; i < nrow; i++) {
        if (!cpl_table_is_selected(sci_target_table->table, i)) {
            cpl_table_set_invalid(sci_target_table->table,
                                  MOO_TARGET_TABLE_INDEXTARG, i);
        }
    }
    if (nrow > 0) {
        cpl_table_erase_invalid(sci_target_table->table);
    }
    moo_sci_add_target_table(sci, sci_target_table);

moo_try_cleanup:
    cpl_array_delete(obja_sky_indexes);
    cpl_array_delete(objb_sky_indexes);
    cpl_array_delete(targ_obj_indexes);

    if (!cpl_errorstate_is_equal(prestate)) {
        // cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        //  Recover from the error(s) (Reset to prestate))
        // cpl_errorstate_set(prestate);
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in sky subtraction");
        moo_sci_delete(sci);
        sci = NULL;
    }
    return sci;
}
/**@}*/
